package code;

 /** 
 *  Debargo Message used for two phases write  
 **/ 
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

public final class DebargoMsg implements Externalizable, Immutable
{
  
  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;
  
  final static boolean debugging = false;
  private final ObjInvalTarget objId; // This field redundant -- acceptStamp
  // uniquely identifies the write unbound.
  // But include it for sanity checking.
  private final AcceptStamp timestamp;
    
 /** 
 *  empty Constructor for serialization 
 **/ 
  public DebargoMsg(){
    
    this.objId = null; 
    
    this.timestamp = null;
  }

 /** 
 *  Constructor 
 **/ 
  public DebargoMsg(ObjInvalTarget objId_, AcceptStamp timestamp_){
    assert(objId_ instanceof Immutable);
    this.objId = objId_;
    assert(timestamp_ instanceof Immutable);
    this.timestamp = timestamp_; 
  }

 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    //System.out.println(DebargoMsg.class.getDeclaredFields().length);
    assert(DebargoMsg.class.getDeclaredFields().length == 8);

    //the fields we custom serialize remain the same
    Field[] f = new Field[2];  
      try{
        f[0] = DebargoMsg.class.getDeclaredField("objId");
        f[1] = DebargoMsg.class.getDeclaredField("timestamp");
                
        assert (f[0].getType().getName().equals("ObjInvalTarget"));
        assert (f[1].getType().getName().equals("AcceptStamp"));
        
      }catch(NoSuchFieldException ne){
        assert false;
      }
  }

 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- removed from  
 *                   TaggedOutputStream to here 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("DebargoMsg -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

    out.writeObject(this.getObjId().getPath());
    out.writeLong(this.getOffset());
    out.writeLong(this.getLength());
    out.writeLong(this.getTime());
    if(SangminConfig.forkjoin){
      out.writeObject(this.getNodeId());
    } else {
      out.writeLong(this.getNodeId().getIDint());
    }
    
  }

  
 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --originated from  
 *                   TaggedInputStream 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    String objIdString = (String)(in.readObject());
    long  offset = in.readLong();
    long  length = in.readLong();
    long  localClock = in.readLong();
    //long  nodeIdInt = in.readLong();
    NodeId nodeId = SangminConfig.forkjoin?
        (BranchID)in.readObject():new NodeId(in.readLong());
    
      
    Field[] f = new Field[2];
      
      try{

        f[0] = DebargoMsg.class.getDeclaredField("objId");
        f[1] = DebargoMsg.class.getDeclaredField("timestamp");
        
      }catch(NoSuchFieldException ne){
        assert false;
      }
      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        assert false;
      }
      try{
        f[0].set(this, new ObjInvalTarget(new ObjId(objIdString),
                                          offset,
                                          length));
        f[1].set(this, new AcceptStamp(localClock, nodeId));
        
      }catch(IllegalArgumentException ie){
        assert false;
      }catch(IllegalAccessException iae){
        assert false;
      }
      
      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        assert false;
      }
  }

 /** 
 *  Clone this object 
 **/ 
  public Object clone(){
    assert(this instanceof Immutable);
    return this;
  }
    
 /** 
 *  Get NodeId 
 **/ 
  public NodeId getNodeId(){
    return timestamp.getNodeId();
  }

 /** 
 *  Get Time 
 **/ 
  public long getTime(){
    return timestamp.getLocalClock();
  }

 /** 
 *  Get AcceptStamp 
 **/ 
  public AcceptStamp getAcceptStamp(){
    return timestamp;
  }	
 /** 
 *  Get InvalTarget 
 **/ 
  public ObjInvalTarget getInvalTarget(){
    return objId;
  }

 /** 
 *  Get ObjId 
 **/ 
  public final ObjId getObjId() {
    return objId.getObjId();
  }

 /** 
 *  getOffset() -- return the offset 
 **/ 
  public final long
  getOffset(){
    return objId.getOffset();
  }

 /** 
 *  getLength() -- return the length 
 **/ 
  public final long
  getLength(){
    return objId.getLength();
  }

 /** 
 *  Return true if the passed-in object is "equal" to this DebargoMsg 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    DebargoMsg pi = null;

    
    if(obj instanceof DebargoMsg){
      pi = (DebargoMsg)obj;
      result = (this.objId.equals(pi.objId) &&
                this.timestamp.equals(pi.timestamp));
    }
    return(result);
  }

  
	
 /** 
 *  Return a string representation 
 **/ 
  public final String toString(){
    String str = null;
    str = "DebargoMsg:(";
    str += this.objId;
    str += ", ";
    str += this.timestamp;
    str += ")";
    return str;
  }

}

//------------------------------------------------------------------------------
/*$Log: DebargoMsg.java,v $
/*Revision 1.4  2007/05/31 06:02:01  zjiandan
/*add AllPreciseSetsUnit
/*
/*Revision 1.3  2006/09/13 02:44:40  zjiandan
/*move unit test to XUnit.java for junit test.
/*
/*Revision 1.2  2006/04/20 03:52:53  zjiandan
/*Callbacks merged with runTime.
/*
/*Revision 1.1  2005/07/18 05:10:22  zjiandan
/*Embargoed Writes etc. features implementation plus
/*log overhead measurement with disk size and in-memory size.
/**/
//------------------------------------------------------------------------------