package code;

import java.io.*;
import java.net.*;
import java.util.*;

public class DelayOutputStream extends OutputStream{

    private static long delay_ms; 
    private DelayQueue dq;
    private double accumulatedTimeMillis;
    private DelayStreamWorker dsw;
    private OutputStream os;

    public DelayOutputStream(OutputStream os, long delay) throws IOException{
	delay_ms = delay;
	Env.printDebug("Delay is set on " + delay_ms);
	this.os = os;
	dq = new DelayQueue();
	dsw = new DelayStreamWorker(dq, os);
	dsw.start();
    }

    public void write(int b)throws IOException{
	byte [] tmp = new byte[1];
	tmp[0] = (byte)b;
	delayedWrite(tmp, 0, 1);
    }
    
    public void write(byte[] b)throws IOException{
	delayedWrite(b, 0, b.length);
    }

    public void write(byte[] b, int off, int len)throws IOException{
	delayedWrite(b, off, len);
    }

    private void delayedWrite(byte[] b, int off, int len) throws IOException{
	if(delay_ms > 0){
	    WritePacket wp = new WritePacket(b, off, len);
	    wp.setDepartTime(System.currentTimeMillis()+ delay_ms);
	    dq.enqueueWP(wp);
	} else {
	    os.write(b, off, len);
	}
    }


    public static void main(String [] argv){
	int port = 5566;
	if(argv.length < 1){
	    Env.printDebug("Usage: [client][server address]] or [server]");
	}
	try{
	    if(argv[0].equals("client")){
              NodeId myid = new NodeId(9);
		Config.readConfig("ufs.config");
		Socket s = new Socket(argv[1], port);
		DelayOutputStream dos = new DelayOutputStream(s.getOutputStream(), 3000);
		ObjectOutputStream oos = new ObjectOutputStream(dos);
		int c = 0;
		while(true){
		    long t = System.currentTimeMillis();
		    oos.writeLong(t);
//		    oos.writeObject(String.valueOf(c++));
		}
	    } else {
		ServerSocket ss = new ServerSocket(port);
		InputStream is = ((Socket)ss.accept()).getInputStream();
		ObjectInputStream ois = new ObjectInputStream(is);
		while(true){
		    long st = ois.readLong();
		    long et = System.currentTimeMillis();
		    System.out.println((et-st) + "ms");

		}
	    }
	} catch (Exception e){
	    e.printStackTrace();
	}
    }

}

class DelayQueue{
    
    private Vector delayQueue;
    
    public DelayQueue(){
	delayQueue = new Vector();
    }
    
    public synchronized void enqueueWP(WritePacket wp){
	delayQueue.add(wp);
	notifyAll();
    }

    public synchronized WritePacket dequeueWP(){
	while(delayQueue.isEmpty()){
	    try{
		wait();
	    } catch (InterruptedException ie){
		System.err.println(ie.getMessage());
		// keep waiting if codition is still true
	    }
	}
	WritePacket wp = (WritePacket)delayQueue.firstElement();
	WritePacket wp2 = (WritePacket)delayQueue.remove(0);
	assert(wp==wp2);
	assert(wp!=null);
	return wp;
    }
}

class DelayStreamWorker extends Thread{
    
    private DelayQueue dq;
    private OutputStream os;

    public DelayStreamWorker(DelayQueue dq, OutputStream os){
	this.dq = dq;
	this.os = os;
    }

    public void run(){
	while(true){
	    WritePacket wp = dq.dequeueWP();
	    long sleepTime = wp.getDepartTime() - System.currentTimeMillis();
	    /* note: sleepTime can be nagtive when target delay time is very small */
	    while(sleepTime>0){
		try{
		    sleep(sleepTime);
		} catch (InterruptedException ie){
		    System.err.println(ie.getMessage());
		    // keep sleeping if sleepTime is not up
		}
		sleepTime = wp.getDepartTime() - System.currentTimeMillis();
	    }
	    try{
		os.write(wp.getData(), wp.getOff(), wp.getLen());
	    } catch (IOException ioe){
		System.err.println(ioe.getMessage());
	    }
	}
    }

}

class WritePacket{
    private byte[] data;
    private int off;
    private int len;
    private long departTime;

    public WritePacket(byte[] b, int off, int len){
	this.data = new byte[len];
	for(int i=0; i<len; i++){
	    data[i] = b[i];
	}
	this.off = off;
	this.len = len;
    }

    public void setDepartTime(long t){
	this.departTime = t;
    }
    
    public long getDepartTime(){
	return this.departTime;
    }

    public byte[] getData(){
	return data;
    }

    public int getOff(){
	return off;
    }

    public int getLen(){
	return len;
    }
}
