package code;

/* GCForcer
 * 
 * This class can be included by a resource-intensive task
 * to periodically "force" garbage collection.
 *
 * The basic idea is for this class to maintain a <totCharge>
 * of the amount of work done or resources consumed by 
 * the resource intensive task or class. Each time that
 * class consumes resources, it passes GCForcer a <charge>
 * that represents how many resources it is using. 
 * GCForcer adds <charge> to <totCharge> and if the
 * level exceeds a low water mark, calls System.gc().
 * When System.gc() returns, reduce <totCharge>
 * by its value when gc() was called (the assumption is
 * that gc() succeeded in reclaiming all resources used
 * up to that moment.
 * 
 * If the calling class is multithreaded, then normally
 * other threads can continue to make progress while
 * we call System.gc() outside of the monitory.
 * But, if the <totCharge> exceeds a high water mark, then 
 * calls to GCForcer block until System.gc returns.
 * 
 * In theory, this class should not be needed -- java
 * garbage collection should just "do the right thing."
 * But in practice, under periods of heavy load, we can
 * consume a lot of memory very quickly and the gc
 * falls behind. The point of this class is to ensure
 * that it doesn't fall *too* far behind, and if it does
 * slow down the  caller to give it a chance to catch up.
 *
 * In particular, this function was created to make 
 * RandomAccessState more robust. Without this, unit 
 * test 18 would frequently fail with 
 * "java.lang.OutOfMemoryError: Java heap space"
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */


import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.Condition;

public class GCForcer{

  private long totCharge;
  private long lwm;
  private long hwm;
  private final ReentrantLock lock = new ReentrantLock();
  private final Condition gcDone = lock.newCondition();
  private final Condition gcStart = lock.newCondition();
  private Thread worker;
  private boolean timeToDie;
  
 /** 
 *  GCForcer() --  
 *     startGC -- specify the totCharge value above which BG GC should begin 
 *     forceGC -- specify the totCharge value above which caller must wait for 
 *                gc 
 **/ 
  public GCForcer(long startGC, long forceGC){
    lwm = startGC;
    hwm = forceGC;
    assert(lwm > 0);
    assert(hwm >= lwm);
    totCharge = 0;
    worker = null;
    timeToDie = false;
  }

 /** 
 *  startWorker() -- kick off the worker thread. Must be called right 
 *     after initialization. 
 **/ 
  public void startWorker(){
    lock.lock();
    try{
      worker = new GCForcerWorker(this);
      worker.start();
    }
    finally{
      lock.unlock();
    }
  }

 /** 
 *  killWorker() -- Tell the worker thread to die. 
 **/ 
  public void killWorker(){
    lock.lock();
    try{
      timeToDie = true;
      gcStart.signal();
    }
    finally{
      lock.unlock();
    }
  }

  private static int pwcount = 0;
  private static final int pwmax = 10;
 /** 
 *  payCharge() -- update totCharge, kick of gc if needed, and wait if needed 
 **/ 
  public void payCharge(long charge){
    lock.lock();
    try{
      assert(worker != null); // Don't forget to call startWorker()
      totCharge += charge;
      if(totCharge > lwm){
        gcStart.signal();
      }
      while(totCharge > hwm){
        if(pwcount < pwmax){
          Env.performanceWarning("GCForcer payCharge -- caller blocks b/c gc not done yet");
          pwcount++;
          if(pwcount == pwmax){
            Env.performanceWarning("Future GCForcer warnings suppressed");
          }
        }
        gcDone.awaitUninterruptibly();
      }
    }
    finally{
      lock.unlock();
    }
  }

 /** 
 *  waitForWork() -- called by the worker thread 
 **/ 
  private long waitForWork() throws InterruptedException{
    lock.lock();
    try{
      while(!timeToDie && totCharge <= lwm){
        gcStart.awaitUninterruptibly();
      }
      if(timeToDie){
        throw new InterruptedException("GCForcerThread should die now.");
      }
      return totCharge; // The value of totCharge when we started this gc attempt
    }
    finally{
      lock.unlock();
    }
  }

 /** 
 *  doneWithWork() -- called by the worker thread when done 
 **/ 
  protected void doneWithWork(long charge){
    lock.lock();
    try{
      totCharge -= charge;
      gcDone.signalAll();
    }
    finally{
      lock.unlock();
    }
  }


 /** 
 **/ 
 /** 
 *  Class GCForcerWorker 
 **/ 
 /** 
 **/ 
  static int killCount = 0;
  static int maxKillWarn = 3;
  class GCForcerWorker extends Thread{
    private GCForcer gcf;

    private GCForcerWorker(GCForcer g){
      gcf = g;
    }

    public void run(){
      long workToDo;
      while(true){
        try{
          workToDo = gcf.waitForWork();
        }
        catch(InterruptedException e){
          //
          // No more work to do. Someone called GCForcer.killWorker()
          //
          killCount++;
          if(killCount < maxKillWarn){
            Env.inform("GCForcer worker thread exits b/c GCForcer.killWorker called");
            if(killCount == maxKillWarn){
              Env.inform("GCForcer will suppress future killWorker notifications");
            }
          }
          return; // Thread exits
        }
        System.gc();
        gcf.doneWithWork(workToDo);
      }
    }
  }
}
