package code;

 /** 
 *  HierInvalTarget: Represents entities that can be invalidated 
 **/ 
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.util.HashMap;
import java.util.Set;
import java.util.Iterator;
import java.util.Collection;
import java.util.Map;
import java.util.TreeSet;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

//
// HierInvalTarget serialization has to recursively
// serialize and deserialize each child. Therefore we have to customized-serialize
// each field =>
// more natural to use Externalizable instead of Serializable to 
// customize the serializations
//
public final class HierInvalTarget
  implements InvalTarget, Externalizable, Cloneable, Immutable{

 /** 
 *  Private data 
 **/ 
  
  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;

  private final String name;
  private final HashMap children;
  private final boolean includesSelf;
  private final boolean includesChildren;
  
  private static final byte NON_EMPTY_HIER_INVAL_TARGET = 0x4A;
  private static final byte EMPTY_HIER_INVAL_TARGET = 0x4B;
  private static final byte HIER_INVAL_TARGET = 0x55;
  
  
 /** 
 *  Constructor (build an empty InvalTarget) 
 **/ 
  public
  HierInvalTarget(){
    // Constructor that creates an empty invalidate
    this.name = null;
    this.children = null;
    this.includesSelf = false;
    this.includesChildren = false;
  }

 /** 
 *  Constructor 
 **/ 
  public
  HierInvalTarget(ObjectInput ois) throws IOException{
    byte magicNum = 0;
    String myName = null;
    int boolVec = 0;
    int numChildren = 0;
    HashMap myChildren = null;
    HierInvalTarget hitChild = null;

    // Read the magic number
    magicNum = ois.readByte();
    assert((magicNum == HierInvalTarget.NON_EMPTY_HIER_INVAL_TARGET) ||
           (magicNum == HierInvalTarget.EMPTY_HIER_INVAL_TARGET));

    if(magicNum == HierInvalTarget.NON_EMPTY_HIER_INVAL_TARGET){
      // Read this.name
      myName = ois.readUTF();
      this.name = myName;

      // Read this.includesSelf and this.includesChildren
      boolVec = ois.readByte();
      this.includesSelf = ((boolVec & 2) != 0);
      this.includesChildren = ((boolVec & 1) != 0);
      //boolean hasChildren = ((boolVec & 4) != 0);
      boolean moreThan16Children = ((boolVec & 4) != 0);
      // Read the children of this node
      myChildren = new HashMap();
      
        if(moreThan16Children){
          numChildren = ois.readInt();
        }else{
          numChildren = boolVec >> 3;
        }
        for(int i = 0; i < numChildren; i++){
          hitChild = new HierInvalTarget(ois);
          myChildren.put(hitChild.name, hitChild);
        }
      
      this.children = myChildren;
    }else{
      this.name = null;
      this.children = null;
      this.includesSelf = false;
      this.includesChildren = false;
    }
  }

 /** 
 *  Constructor 
 **/ 
  private
  HierInvalTarget(String myName,
                  HashMap myChildren,
                  boolean myIncludesSelf,
                  boolean myIncludesChildren){
    assert((myName != null) && (myChildren != null));
    this.name = myName;
    this.children = myChildren;
    this.includesChildren = myIncludesChildren;
    this.includesSelf = myIncludesSelf;
  }

 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    assert(this.getClass().getDeclaredFields().length == 11);
     
    //the fields we custom serialize remain the same
    Field[] f = new Field[4];  
    try{
      f[0] = this.getClass().getDeclaredField("name");
      f[1] = this.getClass().getDeclaredField("children");
      f[2] = this.getClass().getDeclaredField("includesSelf");
      f[3] = this.getClass().getDeclaredField("includesChildren");

      assert (f[0].getType().getName().equals("java.lang.String"));
      assert (f[1].getType().getName().equals("java.util.HashMap"));
      assert (f[2].getType().getName().equals("boolean"));
      assert (f[3].getType().getName().equals("boolean"));
    }catch(NoSuchFieldException ne){
      assert false;
    }
  }

 /** 
 *  Verify data structure integrity 
 **/ 
  public void
  verifyIntegrity(){
    if(!this.isEmpty()){
      assert(this.name != null);
      assert(this.children != null);
      if(this.includesChildren){
        assert(this.children.isEmpty());
      }
      if(this.children.isEmpty()){
        assert(this.includesChildren || this.includesSelf);
      }

      // Make sure all children are of the same type
      for(Iterator i = this.children.values().iterator(); i.hasNext();){
        assert(i.next() instanceof HierInvalTarget);
      }
    }else{
      assert(this.name == null);
      assert(this.children == null);
      assert(!this.includesSelf);
      assert(!this.includesChildren);
    }
  }


 /** 
 *  Send a copy of myself onto an ObjectOutputStream 
 **/ 
  public void
  copySelfOntoOOS(ObjectOutput oos) throws IOException{

    int boolVec = 0;
    int numChildren = 0;
    HierInvalTarget hitChild = null;

    if(!this.isEmpty()){
      // Send the magic number
      oos.writeByte(HierInvalTarget.NON_EMPTY_HIER_INVAL_TARGET);

      // Send this.name
      oos.writeUTF(this.name);

      // Send this.includesSelf and this.includesChildren
      boolVec = 0;
      if(this.includesSelf){
        boolVec = boolVec | 2;
      }
      if(this.includesChildren){
        boolVec = boolVec | 1;
      }
      if(this.children.size() >= 16){
        boolVec = boolVec | 4;
	  assert ((boolVec & 4) > 0):boolVec;
//	  if(this.children.size() >= 8){
//	        boolVec = boolVec | 8;
//	  }
      }else{
	  assert ((boolVec & 4)==0):boolVec;
      }
      //      System.out.println("WRITING BOOLVEC: " + boolVec);
      

        // Send the children
        numChildren = this.children.size();
        if(this.children.size() < 16){
          int size = (children.size() << 3) & ((byte)120); // the top 3 bits are set (excluding the sign bit)
          boolVec = boolVec | size;
          oos.writeByte(boolVec);
        }else{
          oos.writeByte(boolVec);
          oos.writeInt(numChildren);
        }
        for(Iterator i = this.children.values().iterator(); i.hasNext();){
          hitChild = (HierInvalTarget)i.next();
          hitChild.copySelfOntoOOS(oos);
          numChildren--;
        }
      
      assert(numChildren == 0);
    }else{
      // Send the magic number
      oos.writeByte(HierInvalTarget.EMPTY_HIER_INVAL_TARGET);      
    }
  }
  
  public int onDiskSize(){
    int size = 0;
    if(name != null){
      size+= 1 + name.length() + 2 + 1; // 1 for empty/nonempty, length number of bytes + 2 for length + 1 for flag and
      if(children.size() > 0){
        size += 4; // 4 for children size
      }
    }
    if(children!= null){
      HierInvalTarget hitChild = null;
      for(Iterator i = this.children.values().iterator(); i.hasNext();){
        hitChild = (HierInvalTarget)i.next();
        size += hitChild.onDiskSize();
      }
    }
    return size;
  }


 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- removed from  
 *                   TaggedOutputStream to here 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("HierInvalTarget -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

    // out.defaultWriteObject();
    //System.out.println("HierInvalTarget.writeExternal");

    this.copySelfOntoOOS(out);
  }

  
 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --originated from  
 *                   TaggedInputStream 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    //in.defaultReadObject();
    //System.out.println("HierInvalTarget.readExternal");

    HierInvalTarget hit = new HierInvalTarget(in);
    
    Field[] f = new Field[4];
      
      try{

        f[0] = this.getClass().getDeclaredField("name");
        f[1] = this.getClass().getDeclaredField("children");
        f[2] = this.getClass().getDeclaredField("includesSelf");
        f[3] = this.getClass().getDeclaredField("includesChildren");
      }catch(NoSuchFieldException ne){
        assert false;
      }
      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        assert false;
      }
      try{
        f[0].set(this, hit.name);
        f[1].set(this, hit.children);
        f[2].setBoolean(this, hit.includesSelf);
        f[3].setBoolean(this, hit.includesChildren);

      }catch(IllegalArgumentException ie){
        assert false;
      }catch(IllegalAccessException iae){
        assert false;
      }
      
      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        assert false;
      }
  }
  
  public static boolean matchClassCode(byte code){
    return code == HIER_INVAL_TARGET;
    
  }
  
  public static byte getClassCode(){
    return HIER_INVAL_TARGET;
  }

/*
  //-------------------------------------------------------------------------
  // Serialization -- optimization to reduce the pikles in the serialization 
  //                  and improve the performance -- removed from 
  //                  TaggedOutputStream to here
  //-------------------------------------------------------------------------
  private void writeObject(ObjectOutputStream out)
    throws IOException{
    // out.defaultWriteObject();
    System.out.println("HierInvalTarget.writeObject");
    this.copySelfOntoOOS(out);
  }

  
  //-------------------------------------------------------------------------
  // Serialization --  optimization to reduce the pikles in the serialization 
  //                  and improve the performance --originated from 
  //                  TaggedInputStream
  //-------------------------------------------------------------------------
  private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException{
    //in.defaultReadObject();
    System.out.println("HierInvalTarget.readObject");
    HierInvalTarget hit = new HierInvalTarget(in);
    this.name = hit.name;
    this.children = hit.children;
    this.includesSelf = hit.includesSelf;
    this.includesChildren = hit.includesChildren;

  }
*/
 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    /*
      HashMap newChildren = null;
      Set childNames = null;
      String name = null;
      HierInvalTarget child = null;
      HierInvalTarget childClone = null;
      HierInvalTarget selfClone = null;

      newChildren = new HashMap();
      childNames = this.children.keySet();
      for(Iterator i = childNames.iterator(); i.hasNext();){
      name = (String)i.next();
      child = (HierInvalTarget)this.children.get(name);
      childClone = (HierInvalTarget)child.clone();
      newChildren.put(name, childClone);
      }
      assert(newChildren.size() == this.children.size());
      selfClone = new HierInvalTarget(this.name,
      newChildren,
      this.includesSelf,
      this.includesChildren);
      return(selfClone);
    */
    return(this);
  }

 /** 
 *  Make a HierInvalTarget object from the given array of Strings 
 **/ 
  public static HierInvalTarget
  makeHierInvalTarget(String[] allNames){
    String completeName = "";

    assert(allNames.length > 0);
    completeName = allNames[0];
    for(int i = 1; i < allNames.length; i++){
      completeName = completeName + ":" + allNames[i];
    }
    return(HierInvalTarget.makeHierInvalTarget(completeName));
  }

 /** 
 *  Make a HierInvalTarget object from the given ":"-separated set of paths 
 **/ 
  public static HierInvalTarget
  makeHierInvalTarget(String completeName){
    int startIndex = 0;
    int endIndex = 0;
    boolean done = false;
    String fullName = null;
    FileNameTokenizer fnt = null;
    HierInvalTarget hit = null;
    HierInvalTarget newHIT = null;

    startIndex = -1;
    endIndex = 0;
    done = false;
    while(!done){
      endIndex = completeName.indexOf(":", startIndex + 1);
      if(endIndex >= 0){
        fullName = completeName.substring(startIndex + 1, endIndex);
      }else{
        fullName = completeName.substring(startIndex + 1);
        done = true;
      }
      startIndex = endIndex;
      fnt = new FileNameTokenizer(fullName);
      if(hit == null){
        hit = HierInvalTarget.makeHierInvalTarget(fnt);
      }else{
        newHIT = HierInvalTarget.makeHierInvalTarget(fnt);
        hit = hit.getCompleteUnion(newHIT);
      }
    }
    return(hit);
  }

 /** 
 *  Method inherited from InvalTarget (See InvalTarget.java) 
 *  Return true if this invalidate intersects with "ss" 
 *  
 *  Note: We return true only if the intersection of the two is non-empty. 
 *  Thus, the intersection of an empty invalidate with an empty ss is false. 
 **/ 
  public boolean
  intersects(SubscriptionSet ss){
    boolean result = false;
    Map.Entry childEntry = null;
    String childName = null;
    HierInvalTarget child = null;
    SubscriptionSet ssChild = null;
    if (ss == null ){
      return false;
    }

    if(this.isEmpty()){
      result = false;
    }else if(ss.isEmpty()){
      result = false;
    }else if(ss.hasNoChildren()){
      assert(this.name.equals(ss.getName()));
      if(ss.containsChildren()){
        result = (this.includesChildren ||
                  (!this.children.isEmpty()) ||
                  (this.includesSelf && ss.containsSelf()));
      }else if(this.includesSelf){
        assert(ss.containsSelf());
        result = true;
      }else{
        assert(ss.containsSelf());
        result = false;
      }
    }else if(this.children.isEmpty()){
      if(this.includesChildren){
        result = (ss.containsChildren() ||
                  (!ss.hasNoChildren()) ||
                  (this.includesSelf && ss.containsSelf()));
      }else if(ss.containsSelf()){
        assert(this.includesSelf);
        result = true;
      }else{
        assert(this.includesSelf);
        result = false;
      }
    }else{
      // Neither includes all children; look for common children
      assert((!this.includesChildren) && (!ss.containsChildren()));
      assert((!this.children.isEmpty()) || this.includesSelf);
      assert((!ss.hasNoChildren()) || ss.containsSelf());
      result = this.includesSelf && ss.containsSelf();
      for(Iterator i = this.children.entrySet().iterator();
          i.hasNext() && (!result);){
        childEntry = (Map.Entry)i.next();
        childName = (String)childEntry.getKey();
        child = (HierInvalTarget)childEntry.getValue();
        ssChild = ss.getChild(childName);
        if(ssChild != null){
          result = child.intersects(ssChild);
        }
      }
    }
    return(result);
  }

 /** 
 *  Method inherited from InvalTarget (See InvalTarget.java) 
 *  Rules: 
 *  (1) The intersection of an ObjInvalTarget and HierInvalTarget must 
 *      return either (a) an ObjInvalTarget if the invalidated object 
 *      lies in the intersection, or (b) an empty HierInvalTarget otherwise. 
 *  (2) If the caller passes in two instances of ObjInvalTarget to the 
 *      intersection method, we assert false unless they are equal. 
 *  (3) The intersection of two HierInvalTarget objects is another 
 *      HierInvalTarget object. 
 **/ 
  public InvalTarget
  getIntersection(InvalTarget it){
    ObjInvalTarget objIT = null;
    ObjId objId = null;
    HierInvalTarget hierIT = null;
    MultiObjInvalTarget multiObjIT = null;
    InvalTarget result = null;

    assert((it instanceof ObjInvalTarget) ||
           (it instanceof HierInvalTarget) ||
           (it instanceof MultiObjInvalTarget));
    if(it instanceof ObjInvalTarget){
      objIT = (ObjInvalTarget)it;
      objId = objIT.getObjId();
      if(this.invalidatesObj(objId.getPath())){
        result = objIT;
      }else{
        result = new HierInvalTarget();
      }
    }else if(it instanceof HierInvalTarget){
      hierIT = (HierInvalTarget)it;
      result = this.getHITIntersection(hierIT);
      if(result == null){
        // getHITIntersection() returns null if the intersection is
        // the empty set. Therefore, we create an empty HierInvalTarget
        // to explicitly indicate that the intersection contained nothing.
        result = new HierInvalTarget();
      }
    }else{
      // Pass the call onto MultObjInvalTarget.getIntersection()
      multiObjIT = (MultiObjInvalTarget)it;
      result = multiObjIT.getIntersection(this);
    }
    return(result);
  }

 /** 
 *  Method inherited from InvalTarget (See InvalTarget.java) 
 **/ 
  public InvalTarget
  getUnion(InvalTarget it, SubscriptionSet ss){
    ObjInvalTarget objIT = null;
    HierInvalTarget hierIT = null;
    ObjId objId = null;
    InvalTarget result = null;
    MultiObjInvalTarget multiObjIT = null;

    assert((it instanceof ObjInvalTarget) ||
           (it instanceof HierInvalTarget) ||
           (it instanceof MultiObjInvalTarget));
    if(it instanceof ObjInvalTarget){
      objIT = (ObjInvalTarget)it;
      objId = objIT.getObjId();
      hierIT = HierInvalTarget.makeHierInvalTarget(objId.getPath());
      result = this.getHITUnion(hierIT, ss);
    }else if(it instanceof HierInvalTarget){
      hierIT = (HierInvalTarget)it;
      result = this.getHITUnion(hierIT, ss);
    }else{
      // Pass the call onto MultiObjInvalTarget.getUnion(...);
      multiObjIT = (MultiObjInvalTarget)it;
      result = multiObjIT.getUnion(this, ss);
    }
    return(result);
  }

 /** 
 *  Return true if this inval target is empty 
 **/ 
  public boolean
  isEmpty(){
    return(this.name == null);
  }

 /** 
 *  Return the name of this InvalTarget node 
 **/ 
  public String
  getName(){
    return(this.name);
  }

 /** 
 *  Return the child with this name. If such a child does not exist, 
 *  return null 
 **/ 
  public HierInvalTarget
  getChild(String newName){
    assert(!this.isEmpty());
    return((HierInvalTarget)this.children.get(newName));
  }

 /** 
 *  Return a HashMap containing all children 
 **/ 
  public HashMap
  getChildren(){
    assert(!this.isEmpty());
    return(this.children);
  }

 /** 
 *  Return true if this node is included in the HierInvalTarget 
 **/ 
  public boolean
  containsSelf(){
    return(this.includesSelf);
  }

 /** 
 *  Return true if this nodes includes all subchildren 
 **/ 
  public boolean
  containsChildren(){
    return(this.includesChildren);
  }

 /** 
 *  Get the intersection of this HierInvalTarget and "hit" 
 *  This recursive method returns null if the intersection of "this" 
 *  and "hit" contains nothing. The caller must know this. 
 **/ 
  private HierInvalTarget
  getHITIntersection(HierInvalTarget hit){
    HierInvalTarget result = null;
    HashMap newChildren = null;
    Map.Entry childEntry = null;
    String childName = null;
    HierInvalTarget child = null;
    HierInvalTarget newChild = null;
    HierInvalTarget hitChild = null;

    if(this.isEmpty() || hit.isEmpty()){
      result = null;
    }else if(hit.children.isEmpty()){
      assert(this.name.equals(hit.name));
      if(hit.includesChildren){
        // If "hit" contains all children, include children of "this"
        if(this.includesChildren ||
           (!this.children.isEmpty()) ||
           (this.includesSelf || hit.includesSelf)){
          result = new HierInvalTarget(this.name,
                                       this.children,
                                       (this.includesSelf && hit.includesSelf),
                                       this.includesChildren);
        }else{
          // "this" is a leaf node that does not include its children
          assert(this.includesSelf);
          result = null;
        }
      }else if(this.includesSelf){
        // Both "hit" and "this" are included in the intersection but
        // the children of "this" aren't.
        assert(hit.includesSelf);
        result = new HierInvalTarget(hit.name,
                                     hit.children,
                                     true,
                                     false);
      }else{
        assert(hit.includesSelf);
        result = null;
      }
    }else if(this.children.isEmpty()){
      if(this.includesChildren){
        // If "this" contains all children, include all of "hit"
        if(hit.includesChildren ||
           (!hit.children.isEmpty()) ||
           (this.includesSelf && hit.includesSelf)){
          result = new HierInvalTarget(hit.name,
                                       hit.children,
                                       (this.includesSelf && hit.includesSelf),
                                       hit.includesChildren);
        }else{
          // "hit" includes only itself and "this" includes only its children
          assert(hit.includesSelf);
          result = null;
        }
      }else if(hit.includesSelf){
        // Both "hit" and "this" are included in the intersection but
        // the children of "hit" aren't.
        assert(this.includesSelf);
        result = new HierInvalTarget(this.name,
                                     this.children,
                                     true,
                                     false);
      }else{
        assert(this.includesSelf);
        result = null;
      }
    }else{
      // Neither includes all children; look for common children
      assert((!this.includesChildren) && (!hit.includesChildren));
      assert((!this.children.isEmpty()) || this.includesSelf);
      assert((!hit.children.isEmpty()) || hit.includesSelf);
      newChildren = new HashMap();
      for(Iterator i = this.children.entrySet().iterator(); i.hasNext();){
        childEntry = (Map.Entry)i.next();
        childName = (String)childEntry.getKey();
        child = (HierInvalTarget)childEntry.getValue();
        hitChild = (HierInvalTarget)hit.children.get(childName);
        if(hitChild != null){
          newChild = child.getHITIntersection(hitChild);
          if(newChild != null){
            newChildren.put(childName, newChild);
          }
        }
      }
      if((!newChildren.isEmpty()) || (this.includesSelf && hit.includesSelf)){
        // The current node should be included in the intersection
        result = new HierInvalTarget(this.name,
                                     newChildren,
                                     this.includesSelf && hit.includesSelf,
                                     false);
      }else{
        // "this" and "hit" share no common children
        result = null;
      }
    }
    return(result);
  }

 /** 
 *  Get the union of this HierInvalTarget and "hit" based on the value 
 *  in ss, which represents what the reciever of the unioned invalidate 
 *  message is interested in. 
 *  What this method does tends to look like Voodoo, but it adheres to a 
 *  few simple principles: 
 *  (1) Do not convert any component of "this" and "hit" that lies in "ss" 
 *      into an imprecise equivalent. 
 *  (2) For any component that does not lie in "ss", we try to convert it 
 *      to as precise an equivalent as possible. Thus, if "this" and "hit" 
 *      have a large overlap in data that they invalidate, the amount of 
 *      data invalidated by the resulting InvalTarget will not be much 
 *      larger than that invalidated by "this" and "hit" individually. 
 *      However, if "this" and "hit" do not have a large overlap, our 
 *      resulting invalidate may be much more imprecise than the two. 
 **/ 
  private HierInvalTarget
  getHITUnion(HierInvalTarget hit, SubscriptionSet ss){
    HierInvalTarget unionHIT = null;
    Collection hitChildren = null;
    Collection thisChildren = null;
    HierInvalTarget thisChild = null;
    HierInvalTarget hitChild = null;
    HierInvalTarget newChild = null;
    SubscriptionSet ssChild = null;
    HashMap newChildren = null;

    if((!this.isEmpty()) && (!hit.isEmpty()) && (!ss.isEmpty())){
      assert(this.name.equals(hit.name));
      assert((this != null) && (hit != null) && (ss != null));
      if(this.children.isEmpty()){
        // "this" ended up at a leaf node
        if(this.includesChildren){
          unionHIT = new HierInvalTarget(this.name,
                                         new HashMap(), 
                                         this.includesSelf || hit.includesSelf,
                                         true);
        }else{
          // Might as well copy this branch from "hit"
          assert(this.includesSelf);
          unionHIT = new HierInvalTarget(hit.name,
                                         hit.children,
                                         true,
                                         hit.includesChildren);
        }
      }else if(hit.children.isEmpty()){
        // "hit" ended up at a leaf node
        if(hit.includesChildren){
          unionHIT = new HierInvalTarget(hit.name,
                                         new HashMap(),
                                         this.includesSelf || hit.includesSelf,
                                         true);
        }else{
          // Might as well copy this branch from "this"
          assert(hit.includesSelf);
          unionHIT = new HierInvalTarget(this.name,
                                         this.children,
                                         true,
                                         this.includesChildren);
        }
      }else if(ss.hasNoChildren()){
        // "ss" hit a leaf node
        if(ss.containsChildren()){
          // All children of "this" and "it" from this node onward
          // belong to the union
          unionHIT = this.getCompleteUnion(hit);
        }else{
          assert(ss.containsSelf());
          unionHIT = this.unionUntilDiverge(hit);
        }
      }else{
        // Neither this, hit, nor ss are leaf nodes
        assert(!this.includesChildren):"HierInvalTarget::getHITUnion:hit:"+hit + " this:" + this + "ss:" + ss;
        assert(!hit.includesChildren):"HierInvalTarget::getHITUnion:hit:"+hit + " this:" + this + "ss:" + ss;
        assert(!ss.containsChildren()):"HierInvalTarget::getHITUnion:hit:"+hit + " this:" + this + "ss:" + ss;
        newChildren = new HashMap();
        thisChildren = this.children.values();
        for(Iterator i = thisChildren.iterator(); i.hasNext();){
          thisChild = (HierInvalTarget)i.next();
          hitChild = hit.getChild(thisChild.name);
          ssChild = ss.getChild(thisChild.name);
          /*
            System.out.println("hitChild = " + hitChild + ", " +
            "thisChild = " + thisChild + ", " +
            "ssChild = " + ssChild);
          */
          if(hitChild != null){
            if(ssChild != null){
              // "ss" and "hit" both have children by this name. Continue
              // recursively.
              newChild = thisChild.getHITUnion(hitChild, ssChild);
            }else{
              // "ss" has no child by this name, so we continue copying
              // from "this" and "it" until they diverge.
              newChild = thisChild.unionUntilDiverge(hitChild);
            }
          }else{
            // "hitChild" is null; "ssChild" may or may not be null
            if((ssChild != null) && (ssChild.containsChildren())){
              // The rest of "this" is contained in "ss", so we clone
              // it as it is.
              assert(ssChild.hasNoChildren());
              newChild = (HierInvalTarget)thisChild.clone();
            }else{
              // Clone "this" based on how much of it "ss" indicates
              // the receiver is interested in.
              newChild = thisChild.cloneChopExcluded(ssChild);
            }
          }
          newChildren.put(newChild.name, newChild);
        }
        hitChildren = hit.children.values();
        for(Iterator i = hitChildren.iterator(); i.hasNext();){
          hitChild = (HierInvalTarget)i.next();
          thisChild = this.getChild(hitChild.name);
          if(!newChildren.containsKey(hitChild.name)){
            ssChild = ss.getChild(hitChild.name);
            if((ssChild != null) && (ssChild.containsChildren())){
              newChild = (HierInvalTarget)hitChild.clone();
            }else{
              newChild = hitChild.cloneChopExcluded(ssChild);
            }
          }
          newChildren.put(newChild.name, newChild);
        }
        unionHIT = new HierInvalTarget(this.name,
                                       newChildren,
                                       this.includesSelf || hit.includesSelf,
                                       false);
      }
    }else if(ss.isEmpty()){
      unionHIT = new HierInvalTarget(); // Empty HierInvalTarget
    }else if(this.isEmpty()){
      unionHIT = hit;
    }else{
      unionHIT = this;
    }
    return(unionHIT);
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  public String
  toString(){
    return(this.toString(""));
  }

 /** 
 *  Return a new hashCode 
 **/ 
  public int
  hashCode(){
    assert(false);
    return(0);
  }

 /** 
 *  Return true if "obj" is a HierInvalTarget and is equals to this one 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    HierInvalTarget hit = null;
    HierInvalTarget hitChild = null;
    Map.Entry entry = null;

    if(obj instanceof HierInvalTarget){
      hit = (HierInvalTarget)obj;
      result = (this.isEmpty() && hit.isEmpty()) ||
        ((this.name.equals(hit.name)) &&
         (this.includesChildren == hit.includesChildren) &&
         (this.includesSelf == hit.includesSelf) &&
         (this.children.size() == hit.children.size()));
      if(result && (!this.isEmpty())){
        for(Iterator i = this.children.entrySet().iterator();
            (i.hasNext() && result);){
          entry = (Map.Entry)i.next();
          hitChild = (HierInvalTarget)hit.children.get(entry.getKey());
          result = (hitChild != null) && (hitChild.equals(entry.getValue()));
        }
      }
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Return the total number of nodes in this inval target tree 
 **/ 
  public int size(){
    int ret = 1;
    HierInvalTarget hitChild = null;
    if((this.children == null)|| this.children.size()==0){
      return ret;
    }else{
      for(Iterator i = this.children.values().iterator(); i.hasNext();){
        hitChild = (HierInvalTarget)i.next();
        ret += hitChild.size();
      }
      return ret;
    }
  }
  

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing HierInvalTarget...");
    HierInvalTarget.test1();
    HierInvalTarget.test2();
    HierInvalTarget.test3();
    HierInvalTarget.test4();
    HierInvalTarget.test5();
    HierInvalTarget.test6();
    HierInvalTarget.test7();
    HierInvalTarget.test8();
    HierInvalTarget.test9();
    HierInvalTarget.test10();
    HierInvalTarget.test11();
    HierInvalTarget.test12();
    HierInvalTarget.test13();
    HierInvalTarget.test14(); //test serialization
    HierInvalTarget.test15(); // Test union with empty invalidates
    HierInvalTarget.test16(); // Test verifyIntegrity
    HierInvalTarget.test17(); // Test verifyIntegrity
    System.out.println("...Finished");
  }

  
 /** 
 *  Call only the constructor for HierInvalTarget with a 
 *  FileNameTokenizer as its argument 
 **/ 
  private static void
  test1(){
    FileNameTokenizer fnt = null;
    HierInvalTarget hit = null;
    HierInvalTarget hitChild = null;

    // Test 1
    hit = new HierInvalTarget();
    assert hit.toString().equals("<Empty invalidate>");

    // Test 2
    fnt = new FileNameTokenizer("/asdf/asdf");
    hit = HierInvalTarget.makeHierInvalTarget(fnt);
    // Make sure that hit has...
    // (1) "" as its root.
    // (2) exactly 1 child with name "asdf".
    // (3) exactly 1 grandchild with name "asdf".
    // (4) no great-grandchildren.
    assert(hit.name.equals(""));
    assert((hit.children.size() == 1) && hit.children.containsKey("asdf"));
    hitChild = (HierInvalTarget)hit.children.get("asdf");
    assert(hitChild.name.equals("asdf"));
    assert((hitChild.children.size() == 1) &&
           hitChild.children.containsKey("asdf"));
    hitChild = (HierInvalTarget)hitChild.children.get("asdf");
    assert(hitChild.name.equals("asdf"));
    assert(hitChild.children.isEmpty());

    // Test 3
    fnt = new FileNameTokenizer("/asdf/*");
    hit = HierInvalTarget.makeHierInvalTarget(fnt);
    // Make sure that hit...
    // (1) has exactly one child node that contains other children but
    //     includes all subchildren in the invalidate
    // (2) is non-empty
    // (3) does not include itself
    assert(hit.name.equals(""));
    assert(!hit.isEmpty());
    assert((hit.children.size() == 1) && (hit.children.containsKey("asdf")));
    hitChild = (HierInvalTarget)hit.children.get("asdf");
    assert(hitChild.children.isEmpty());
    assert(hitChild.includesChildren);
    assert(!hitChild.includesSelf);
  }

  //-------------------------------------------------------------------------
  // Test the getCompleteUnion() method and the constructor
  // which takes the complete name as a String
  //-------------------------------------------------------------------------
  private static void
  test2(){
    FileNameTokenizer fnt = null;
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    HierInvalTarget unionHIT = null;
    HierInvalTarget child1 = null;
    HierInvalTarget child2 = null;
    HierInvalTarget gChild1 = null;
    HierInvalTarget gChild2 = null;
    HierInvalTarget ggChild1 = null;
    HierInvalTarget gggChild1 = null;

    // Test 1
    fnt = new FileNameTokenizer("/abcd");
    hit1 = HierInvalTarget.makeHierInvalTarget(fnt);
    fnt =  new FileNameTokenizer("/efgh");
    hit2 = HierInvalTarget.makeHierInvalTarget(fnt);
    unionHIT = hit1.getCompleteUnion(hit2);
    // Make sure that unionHIT...
    // (1) has 2 children named "abcd" and "efgh"
    // (2) no grandchildren
    assert(unionHIT.children.size() == 2);
    assert(unionHIT.children.containsKey("abcd"));
    assert(unionHIT.children.containsKey("efgh"));
    child1 = (HierInvalTarget)unionHIT.children.get("abcd");
    child2 = (HierInvalTarget)unionHIT.children.get("efgh");
    assert(child1.children.isEmpty());
    assert(child1.includesSelf);
    assert(child2.children.isEmpty());
    assert(child2.includesSelf);

    // Test 2
    hit1 = HierInvalTarget.makeHierInvalTarget("/abcd:/efgh");
    // Make sure that hit1 has 2 children and 0 grandchildren
    assert(hit1.children.size() == 2);
    assert(hit1.children.get("abcd") != null);
    assert(hit1.children.get("efgh") != null);
    assert(((HierInvalTarget)hit1.children.get("abcd")).children.isEmpty());
    assert(((HierInvalTarget)hit1.children.get("efgh")).children.isEmpty());

    // Test 3
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d/*:/a:/a/c:/a/c/*");
    // Make sure that...
    // (1) hit1 has a single child named "a".
    // (2) "a" has two children named "b" and "c".
    // (3) "b" is not included, but "c" is.
    // (4) "b" has a single child named "c" and a single grandchild named "d".
    // (5) "d" has its children included
    // (6) "c" (child of "a") has its children included
    // (7) None of the other leaf nodes have their children included
    assert(hit1.children.size() == 1);
    child1 = (HierInvalTarget)hit1.children.get("a");
    assert(child1 != null);
    assert(child1.includesSelf && (!child1.includesChildren));
    assert(child1.children.size() == 2);
    gChild1 = (HierInvalTarget)child1.children.get("b");
    gChild2 = (HierInvalTarget)child1.children.get("c");
    assert((gChild1 != null) && (gChild2 != null));
    assert(gChild2.includesSelf && gChild2.includesChildren);
    assert(gChild2.children.isEmpty());
    assert((!gChild1.includesSelf) && (!gChild1.includesChildren));
    assert(gChild1.children.size() == 1);
    ggChild1 = (HierInvalTarget)gChild1.children.get("c");
    assert((ggChild1 != null) && (ggChild1.children.size() == 1));
    assert((!ggChild1.includesSelf) && (!ggChild1.includesChildren));
    gggChild1 = (HierInvalTarget)ggChild1.children.get("d");
    assert((gggChild1 != null) && (gggChild1.children.isEmpty()));
    assert((!gggChild1.includesSelf) && gggChild1.includesChildren);

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a");
    // Simple test; mostly testing the ":" parser. make sure that...
    // (1) hit1 has one child named "a" and no grandchildren.
    assert(hit1.children.size() == 1);
    assert((!hit1.includesSelf) && (!hit1.includesChildren));
    assert(hit1.name.equals(""));
    child1 = (HierInvalTarget)hit1.children.get("a");
    assert((child1 != null) && (child1.name.equals("a")));
    assert(child1.children.isEmpty());
    assert(child1.includesSelf && (!child1.includesChildren));
  }

  //-------------------------------------------------------------------------
  // Test simple methods
  //-------------------------------------------------------------------------
  private static void
  test3(){
    HierInvalTarget hit = null;
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;

    // Test 1
    hit = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/a");
    assert(hit.getName() == hit.name);
    assert(hit.getChildren() == hit.children);
    assert(hit.containsSelf() == hit.includesSelf);
    assert(hit.containsChildren() == hit.includesChildren);
    assert(hit.getChild("a") != null);
    assert(hit.getChild("b") == null);
    assert(!hit.isEmpty());

    // Test 2
    hit = new HierInvalTarget();
    assert(hit.isEmpty());

    // Test 3
    hit = HierInvalTarget.makeHierInvalTarget("/a");
    assert(hit.equals(hit));

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a");
    assert(hit1.equals(hit2));
    assert(hit2.equals(hit1));

    // Test 5
    // NOTE: "/a/b/c/d" is redundant
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/a/b/c/d");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d:/a/b/c/*");
    assert(hit1.equals(hit2));
    assert(hit2.equals(hit1));

    // Test 6
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/*:/a/b/c");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    assert(hit1.equals(hit2));
    assert(hit2.equals(hit1));

    // Test 7
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/*:/b/c");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    assert(!hit1.equals(hit2));
    assert(!hit2.equals(hit1));

    // Test 8
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b:/b/c");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/*:/b/c");
    assert(!hit1.equals(hit2));
    assert(!hit2.equals(hit1));

    // Test 9
    hit1 = HierInvalTarget.makeHierInvalTarget("/a:/a/b:/a/b/*");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d:/a/b/*:/a/b");
    assert(!hit1.equals(hit2));
    assert(!hit2.equals(hit1));

    // Test 10
    hit1 = HierInvalTarget.makeHierInvalTarget("/a:/a/b:/a/b/*");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d:/a/b/*:/a/b:/a");
    assert(hit1.equals(hit2));
    assert(hit2.equals(hit1));
  }

  //-------------------------------------------------------------------------
  // Test getHITIntersection
  //-------------------------------------------------------------------------
  private static void
  test4(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    InvalTarget isect = null;
    HierInvalTarget isectHIT = null;
    HierInvalTarget childHIT1 = null;
    HierInvalTarget childHIT2 = null;
    HierInvalTarget childHIT3 = null;
    HierInvalTarget childHIT4 = null;
    HierInvalTarget testHIT = null;

    // Test 1
    hit1 = new HierInvalTarget();
    hit2 = new HierInvalTarget();
    isect = hit1.getIntersection(hit2);
    assert(isect instanceof HierInvalTarget);
    assert(isect.isEmpty());
    isect = hit2.getIntersection(hit1);
    assert(isect instanceof HierInvalTarget);
    assert(isect.isEmpty());

    // Test 2
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    isect = hit1.getIntersection(hit2);
    assert(isect instanceof HierInvalTarget);
    isectHIT = (HierInvalTarget)isect;
    // Make sure that...
    // (1) The intersection is of type HierInvalTarget
    // (2) The intersection is exactly "/a/b/c/d".
    assert((isectHIT.children.size() == 1) &&
           (isectHIT.children.containsKey("a")));
    childHIT1 = (HierInvalTarget)isectHIT.children.get("a");
    assert((childHIT1.children.size() == 1) &&
           (childHIT1.children.containsKey("b")));
    childHIT2 = (HierInvalTarget)childHIT1.children.get("b");
    assert((childHIT2.children.size() == 1) &&
           (childHIT2.children.containsKey("c")));
    childHIT3 = (HierInvalTarget)childHIT2.children.get("c");
    assert((childHIT3.children.size() == 1) &&
           (childHIT3.children.containsKey("d")));
    childHIT4 = (HierInvalTarget)childHIT3.children.get("d");
    assert(childHIT4.children.isEmpty());
    assert(childHIT4.includesSelf);

    // Test 3
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/b/c/*");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/c/*");
    isect = hit1.getIntersection(hit2);
    assert(isect instanceof HierInvalTarget);
    isectHIT = (HierInvalTarget)isect;
    // Make sure that...
    // (1) isectHIT is of the right type
    // (2) isectHIT is the HierInvalTarget made from "/a/b/c/*"
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b/c/*");
    assert(isectHIT.equals(testHIT));
    assert(testHIT.equals(hit2.getIntersection(hit1)));

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c");
    hit2 = HierInvalTarget.makeHierInvalTarget("/c/b/a");
    isectHIT = (HierInvalTarget)hit1.getIntersection(hit2);
    assert(isectHIT.equals(new HierInvalTarget()));
    assert(isectHIT.equals(hit2.getIntersection(hit1)));

    // Test 5
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/*:/b/c/*");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d/*:/b/c/d:/a/b/c");
    isectHIT = (HierInvalTarget)hit1.getIntersection(hit2);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/d/*:/b/c/d");
    assert(isectHIT.equals(testHIT));
    assert(testHIT.equals(hit2.getIntersection(hit1)));

    // Test 6
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/d:/b/c/*");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/e:/b/c/*:/a/b/c");
    isectHIT = (HierInvalTarget)hit1.getIntersection(hit2);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b/c:/b/c/*");
    assert(isectHIT.equals(testHIT));
    assert(testHIT.equals(hit2.getIntersection(hit1)));

    // Test 7
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/d");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/e");
    isectHIT = (HierInvalTarget)hit1.getIntersection(hit2);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b/c");
    assert(isectHIT.equals(testHIT));
    assert(testHIT.equals(hit2.getIntersection(hit1)));
  }

  //-------------------------------------------------------------------------
  // Test unionUntilDiverge
  //-------------------------------------------------------------------------
  private static void
  test5(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    HierInvalTarget unionHIT = null;
    HierInvalTarget testHIT = null;

    // Test 1
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    unionHIT = hit1.unionUntilDiverge(hit1);
    assert(unionHIT.equals(hit1));

    // Test 2
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b");
    unionHIT = hit1.unionUntilDiverge(hit2);
    assert(unionHIT.equals(hit1));
    unionHIT = hit2.unionUntilDiverge(hit1);
    assert(unionHIT.equals(hit1));

    // Test 3
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c");
    unionHIT = hit1.unionUntilDiverge(hit2);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/*");
    assert(testHIT.equals(unionHIT));
    unionHIT = hit2.unionUntilDiverge(hit1);
    assert(testHIT.equals(unionHIT));

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/*");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    unionHIT = hit1.unionUntilDiverge(hit2);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/*");
    assert(testHIT.equals(unionHIT));
    unionHIT = hit2.unionUntilDiverge(hit1);
    assert(testHIT.equals(unionHIT));

    // Test 5
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/a/b/d/e:/a/b/d/g");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/a/b/d/f");
    unionHIT = hit1.unionUntilDiverge(hit2);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/a/b/d/*");
    assert(testHIT.equals(unionHIT));
    unionHIT = hit2.unionUntilDiverge(hit1);
    assert(testHIT.equals(unionHIT));
  }

  //-------------------------------------------------------------------------
  // Test cloneChopExcluded
  //-------------------------------------------------------------------------
  private static void
  test6(){
    SubscriptionSet ss = null;
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    HierInvalTarget choppedHIT = null;

    // Test 1
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/c/d/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d/e");
    choppedHIT = hit1.cloneChopExcluded(ss);
    assert(choppedHIT.equals(hit1));

    // Test 2
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/c/d");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d/e");
    choppedHIT = hit1.cloneChopExcluded(ss);
    assert(choppedHIT.equals(hit1));

    // Test 3
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/c");
    choppedHIT = hit1.cloneChopExcluded(ss);
    assert(choppedHIT.equals(hit1));

    // Test 4
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/c/d/e/f");
    choppedHIT = hit1.cloneChopExcluded(ss);
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c/*");
    assert(choppedHIT.equals(hit2));

    // Test 5
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/c/d/e/f/*");
    choppedHIT = hit1.cloneChopExcluded(ss);
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c/*");
    assert(choppedHIT.equals(hit2));

    // Test 6
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/*");
    choppedHIT = hit1.cloneChopExcluded(ss);
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/*");
    assert(choppedHIT.equals(hit2));

    // Test 7
    ss = SubscriptionSet.makeSubscriptionSet("/a/c/*:/a/b/*:/b/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/c/d/*:/a/b/q/m:/a/l:/b/x/*");
    choppedHIT = hit1.cloneChopExcluded(ss);
    hit2 = HierInvalTarget.makeHierInvalTarget("/c/*:/a/b/q/m:/a/l:/b/x/*");
    assert(choppedHIT.equals(hit2));
  }

  //-------------------------------------------------------------------------
  // Test getUnion (HierInvalTarget and HierInvalTarget)
  //-------------------------------------------------------------------------
  private static void
  test7(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    HierInvalTarget testHIT = null;
    HierInvalTarget unionHIT = null;
    SubscriptionSet ss = null;

    // Test 1
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c");
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b:/a/c");
    // Both hit1 and hit2 lie in ss, so both should be in the union
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));

    // Test 2
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/*");
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/*");
    // hit1 should subsume hit2 and lie in ss
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));

    // Test 3
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d");
    ss = SubscriptionSet.makeSubscriptionSet("/a/b");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/c/d");
    // hit2 continues beyond hit1 and is not included in ss. We note
    // that our union method can legally return "/a/b:/a/b/*" but we return
    // "/a/b:/a/b/c/d" because it doesn't cost too much more and may
    // help the receiver.
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d");
    ss = SubscriptionSet.makeSubscriptionSet("/a/b");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/c/d");
    // hit2 continues beyond hit1 and it not included in ss. We note
    // that our union method can legally return "/a/b:/a/b/*" but we return
    // "/a/b:/a/b/c/d" because it doesn't cost too much more and may
    // help the receiver.
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));

    // Test 5
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d");
    ss = SubscriptionSet.makeSubscriptionSet("/a");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/*");
    // hit2 continues beyond hit1 and is not included in ss. Here,
    // because when we follow "this", "hit" and "ss" down the tree
    // "ss" runs out early, we truncate "this" and "hit" at the
    // earliest possible place.
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));

    // Test 6
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c/d");
    ss = SubscriptionSet.makeSubscriptionSet("/a");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/*");
    // hit2 continues beyond hit1 and is not included in ss. Again,
    // truncate the union of "this" and "hit" as soon as possible.
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));

    // Test 7
    // Create a somewhat more complex HierInvalTarget
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/c/d");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c/d/*:/a/b/c");
    ss = SubscriptionSet.makeSubscriptionSet("/a/b");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/c:" +
                                                  "/a/b/c/*:/a/c/*");
    unionHIT = hit1.getHITUnion(hit2, ss);
    assert(testHIT.equals(unionHIT));
    assert(testHIT.equals(hit2.getHITUnion(hit1, ss)));
  }

  //-------------------------------------------------------------------------
  // Test invalidatesObj (and its usage in getIntersection)
  //-------------------------------------------------------------------------
  private static void
  test8(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    InvalTarget it1 = null;
    ObjInvalTarget oit1 = null;
    ObjInvalTarget oit2 = null;
    boolean invalidated = false;

    // Test 1
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    invalidated = hit1.invalidatesObj("/a/b");
    assert(!invalidated);

    // Test 2
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    invalidated = hit1.invalidatesObj("/a/b/c");
    assert(invalidated);

    // Test 3
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/*");
    invalidated = hit1.invalidatesObj("/a");
    assert(!invalidated);

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    invalidated = hit1.invalidatesObj("/a");
    assert(!invalidated);

    // Test 5
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    invalidated = hit1.invalidatesObj("/a/b");
    assert(invalidated);

    // Test 6
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/*");
    oit1 = new ObjInvalTarget(new ObjId("/a/b"), 10, 100);
    it1 = hit1.getIntersection(oit1);
    assert(it1 instanceof ObjInvalTarget);
    oit2 = (ObjInvalTarget)it1;
    assert(oit1.equals(oit2));
    assert(oit2.equals(oit1));

    // Test 7
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    oit1 = new ObjInvalTarget(new ObjId("/a/b/c"), 10, 100);
    it1 = hit1.getIntersection(oit1);
    assert(it1 instanceof HierInvalTarget);
    hit2 = (HierInvalTarget)it1;
    assert(hit2.isEmpty());

    // Test 8
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/*:/b/c/d/*:/e/f");
    oit1 = new ObjInvalTarget(new ObjId("/b/c/d/e/f/g/h/i/j"), 10, 100);
    it1 = hit1.getIntersection(oit1);
    assert(it1 instanceof ObjInvalTarget);
    oit2 = (ObjInvalTarget)it1;
    assert(oit1.equals(oit2));
    assert(oit2.equals(oit1));
  }

  //-------------------------------------------------------------------------
  // Test getUnion (the union of an ObjInvalTarget and HierInvalTarget)
  //-------------------------------------------------------------------------
  private static void
  test9(){
    HierInvalTarget hit1 = null;
    HierInvalTarget testHIT = null;
    ObjInvalTarget oit1 = null;
    SubscriptionSet ss1 = null;
    InvalTarget it1 = null;

    // Test 1
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/*");
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    oit1 = new ObjInvalTarget(new ObjId("/a/b/c"), 0, 100);
    it1 = hit1.getUnion(oit1, ss1);
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b");
    assert(testHIT.equals(it1));
  }

  //-------------------------------------------------------------------------
  // Test intersects
  //-------------------------------------------------------------------------
  private static void
  test10(){
    HierInvalTarget hit1 = null;
    SubscriptionSet ss1 = null;
    InvalTarget isect = null;
    HierInvalTarget isectHIT = null;
    HierInvalTarget childHIT1 = null;
    HierInvalTarget childHIT2 = null;
    HierInvalTarget childHIT3 = null;
    HierInvalTarget childHIT4 = null;

    // Test 1
    hit1 = new HierInvalTarget();
    ss1 = SubscriptionSet.makeSubscriptionSet("");
    assert(!hit1.intersects(ss1));

    // Test 2
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/d");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/b/*");
    assert(hit1.intersects(ss1));

    // Test 3
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c/*:/b/c/*");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/b/c/*:/c/*");
    assert(hit1.intersects(ss1));

    // Test 4
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c");
    ss1 = SubscriptionSet.makeSubscriptionSet("/c/b/a");
    assert(!hit1.intersects(ss1));

    // Test 5
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/*:/b/c/*");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/b/c/d/*:/b/c/d:/a/b/c");
    assert(hit1.intersects(ss1));

    // Test 6
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/d:/b/c/*");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/b/e:/b/c/*:/a/b/c");
    assert(hit1.intersects(ss1));

    // Test 7
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/d");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/b/c:/a/b/c/e");
    assert(hit1.intersects(ss1));

    // Test 8
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b/c:/a/b/c/d");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/b/c/f:/a/b/c/e");
    assert(!hit1.intersects(ss1));

    // Test 9
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    ss1 = SubscriptionSet.makeEmptySet(); // Empty SubscriptionSet
    assert(!hit1.intersects(ss1));

    // Test 10
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b:/a/c/*");
    ss1 = SubscriptionSet.makeEmptySet(); // Empty SubscriptionSet
    assert(!hit1.intersects(ss1));

    // Test 11
    hit1 = new HierInvalTarget();
    ss1 = SubscriptionSet.makeEmptySet(); // Empty SubscriptionSet
    assert(!hit1.intersects(ss1));
  }

  //-------------------------------------------------------------------------
  // Test makeHierInvalTarget(String[] allNames)
  //-------------------------------------------------------------------------
  private static void
  test11(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    String[] allNames1 = null;

    // Don't need to do much testing, since by now we have thoroughly
    // tested makeHierInvalTarget(String completeName)

    // Test 1
    allNames1 = new String[1];
    allNames1[0] = "/apple/*";
    hit1 = HierInvalTarget.makeHierInvalTarget(allNames1);
    assert(hit1.equals(HierInvalTarget.makeHierInvalTarget("/apple/*")));

    // Test 2
    allNames1 = new String[2];
    allNames1[0] = "/a/b";
    allNames1[1] = "/a/d/*";
    hit1 = HierInvalTarget.makeHierInvalTarget(allNames1);
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b:/a/d/*");
    assert(hit1.equals(hit2));
    assert(hit2.equals(hit1));

    // Test 3
    allNames1 = new String[2];
    allNames1[0] = "/a/b";
    allNames1[1] = "/a/b/c";
    hit1 = HierInvalTarget.makeHierInvalTarget(allNames1);
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b/c");
    assert(!hit1.equals(hit2));
    assert(!hit2.equals(hit1));
  }

  //-------------------------------------------------------------------------
  // Test makeHierInvalTarget(String[] allNames)
  //-------------------------------------------------------------------------
  private static void
  test12(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    ObjectOutputStream oos = null;
    ObjectInputStream ois = null;

    // Test 1
    // Try a simple invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/*");
      hit1.copySelfOntoOOS(oos);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      hit2 = new HierInvalTarget(ois);
      assert(hit1 != hit2);

      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }

    // Test 2
    // Try an empty invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 = new HierInvalTarget();
      hit1.copySelfOntoOOS(oos);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      hit2 = new HierInvalTarget(ois);
      assert(hit1 != hit2);
      assert(hit2.equals(hit1));
      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }

    // Test 3
    // Try a slightly more complicated invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 =
        HierInvalTarget.makeHierInvalTarget("/a/b/c/d:/a/c:/e/f/g/*:/e/f");
      hit1.copySelfOntoOOS(oos);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      hit2 = new HierInvalTarget(ois);
      assert(hit1 != hit2);
      assert(hit2.equals(hit1));
      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }

  //-------------------------------------------------------------------------
  // Test size()
  //-------------------------------------------------------------------------
  private static void
  test13(){
    
    HierInvalTarget hit = HierInvalTarget.makeHierInvalTarget("/*");
    assert hit.size()==1;
  
    hit = HierInvalTarget.makeHierInvalTarget("/");
    assert hit.size()==2;
    hit = HierInvalTarget.makeHierInvalTarget("/a:/b:/c:/d");
    assert hit.size()==5;
    hit = HierInvalTarget.makeHierInvalTarget("/a:/a/b:/c:/d");
    assert hit.size()==5;
    hit = HierInvalTarget.makeHierInvalTarget("/:/a/b:/c:/d");
    assert hit.size()==6;
    hit = HierInvalTarget.makeHierInvalTarget("/c:/d:/c:/d");
    assert hit.size()==3;
    
  }

  //-------------------------------------------------------------------------
  // Test custom serialization
  //-------------------------------------------------------------------------
  private static void
  test14(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    ObjectOutputStream oos = null;
    ObjectInputStream ois = null;

    // Test 1
    // Try a simple invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 = HierInvalTarget.makeHierInvalTarget("/a/b:/a/b/*");
      oos.writeObject(hit1);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      try{
        hit2 = (HierInvalTarget)(ois.readObject());
      }catch(ClassNotFoundException ce){
        ce.printStackTrace();
        assert false;
      }
      assert(hit1 != hit2);
      assert(hit2.equals(hit1));
      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }

    // Test 2
    // Try an empty invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 = new HierInvalTarget();
      oos.writeObject(hit1);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      try{
        hit2 = (HierInvalTarget)(ois.readObject());
      }catch(ClassNotFoundException ce){
        ce.printStackTrace();
        assert false;
      }
      assert(hit1 != hit2);
      assert(hit2.equals(hit1));
      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }

    // Test 3
    // Try a slightly more complicated invalidate
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 =
        HierInvalTarget.makeHierInvalTarget("/a/b/c/d:/a/c:/e/f/g/*:/e/f");
      oos.writeObject(hit1);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      try{
        hit2 = (HierInvalTarget)(ois.readObject());
      }catch(ClassNotFoundException ce){
        ce.printStackTrace();
        assert false;
      }
      assert(hit1 != hit2);
      assert(hit2.equals(hit1));
      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
  }

  //-------------------------------------------------------------------------
  // Test taking unions of empty HierInvalTarget objects
  //-------------------------------------------------------------------------
  private static void
  test15(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    HierInvalTarget testHIT = null;
    SubscriptionSet ss = null;

    // Test 1: Make the second invalidate empty
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = new HierInvalTarget(); // Empty invalidate
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b");
    assert(testHIT.equals(hit1.getUnion(hit2, ss)));

    // Test 2: Make the first invalidate empty
    hit1 = new HierInvalTarget(); // Empty invalidate
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c");
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/c");
    assert(testHIT.equals(hit1.getUnion(hit2, ss)));

    // Test 3: Make both invalidates empty
    hit1 = new HierInvalTarget(); // Empty invalidate
    hit2 = new HierInvalTarget(); // Empty invalidate
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    testHIT = new HierInvalTarget(); // Empty invalidate
    assert(testHIT.equals(hit1.getUnion(hit2, ss)));

    // Test 4: Make sure that the ss does not influence the result
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = new HierInvalTarget(); // Empty invalidate
    ss = SubscriptionSet.makeSubscriptionSet("/b/*");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/b");
    assert(testHIT.equals(hit1.getUnion(hit2, ss)));

    // Test 5: Make sure that the ss does not influence the result
    hit1 = new HierInvalTarget(); // Empty invalidate
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c");
    ss = SubscriptionSet.makeSubscriptionSet("/b/*");
    testHIT = HierInvalTarget.makeHierInvalTarget("/a/c");
    assert(testHIT.equals(hit1.getUnion(hit2, ss)));

    // Test 6: Make sure that the ss does not influence the result
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/c");
    ss = SubscriptionSet.makeEmptySet();
    testHIT = new HierInvalTarget();
    assert(testHIT.equals(hit1.getUnion(hit2, ss)));
  }

//-------------------------------------------------------------------------
  // Test writing of large HierInvalTargets
  //-------------------------------------------------------------------------
  private static void
  test17(){
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    ByteArrayOutputStream baos = null;
    ByteArrayInputStream bais = null;
    ObjectOutputStream oos = null;
    ObjectInputStream ois = null;
    try{
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      hit1 = HierInvalTarget.makeHierInvalTarget("/a:/b:/c:/d:/e:/f:/g:/h:/i:/j:/k:/l:/m:/n:/o:/p:/q:/r:/s:/t:/u:/v:/w:/x:/y:/z");

      oos.writeObject(hit1);
      oos.flush();
      oos.close();
      baos.flush();
      baos.close();

      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      try{
        hit2 = (HierInvalTarget)(ois.readObject());
      }catch(ClassNotFoundException ce){
        ce.printStackTrace();
        assert false;
      }
      assert(hit1 != hit2);
      assert(hit2.equals(hit1)): "hit2" + hit2 + " hit1" +hit1;
      assert(hit1.equals(hit2));
    }catch(IOException e){
      e.printStackTrace();
      System.out.println("" + e);
      assert(false);
    }
    // Test 1: Make the second invalidate empty
      }
  
  //-------------------------------------------------------------------------
  // Test the verifyIntegrity method
  //-------------------------------------------------------------------------
  private static void
  test16(){
    HierInvalTarget hit = null;

    // Test 1: "/a/b"
    hit = HierInvalTarget.makeHierInvalTarget("/a/b");
    hit.verifyIntegrity();

    // Test 2: <empty>
    hit = new HierInvalTarget();
    hit.verifyIntegrity();
  }

  //-------------------------------------------------------------------------
  // Return a new HierInvalTarget for a given path. We built this method
  // to take a FileNameTokenizer instead of just a String with the path
  // name to distinguish it from the method that takes as its parameter the
  // entire description of the HierInvalTarget (as ":"-separated path names).
  //-------------------------------------------------------------------------
  private static HierInvalTarget
  makeHierInvalTarget(FileNameTokenizer fnt){
    String token = null;
    String nextToken = null;
    HierInvalTarget parentHIT = null;
    HierInvalTarget hit = null;
    HierInvalTarget firstHIT = null;
    boolean done = false;

    token = fnt.getNextToken();
    if(token != null){
      assert(token.equals(""));
      parentHIT = null;
      done = false;
      while(!done){
        nextToken = fnt.getNextToken();
        if(nextToken == null){
          hit = new HierInvalTarget(token, new HashMap(), true, false);
          done = true;
        }else if(nextToken.equals("*")){
          hit = new HierInvalTarget(token, new HashMap(), false, true);
          assert(fnt.getNextToken() == null);
          done = true;
        }else{
          // Neither token nor nextToken is null
          hit = new HierInvalTarget(token, new HashMap(), false, false);
        }
        if(parentHIT != null){
          parentHIT.children.put(token, hit);
        }else{
          firstHIT = hit;
        }
        parentHIT = hit;
        token = nextToken;
      }
    }else{
      firstHIT = new HierInvalTarget();
    }
    return(firstHIT);
  }

  //-------------------------------------------------------------------------
  // Used for testing/debugging
  //-------------------------------------------------------------------------
  private String
  toString(String pathPrefix){
    String str = null;
    String childStr = null;
    TreeSet childKeys = null;
    String childName = null;
    HierInvalTarget child = null;

    str = "";
    if(this.isEmpty()){
      str += "<Empty invalidate>";
    }else{
      if(this.includesSelf){
        str += pathPrefix + this.name + ":";
      }
      if((this.children.isEmpty()) && this.includesChildren){
        str += pathPrefix + this.name + "/*:";
      }else{
        childKeys = new TreeSet(this.children.keySet());
        
        for(Iterator i = childKeys.iterator(); i.hasNext();){
          childName = (String)i.next();
          child = (HierInvalTarget)this.children.get(childName);
          assert(child != null);
          childStr = child.toString(pathPrefix + this.name + "/");
          str += childStr;
        }
      }
    }
    return(str);
  }

  //-------------------------------------------------------------------------
  // Build an HIT where components of "this" that are in "ss" are
  // cloned but the rest are made imprecise
  //-------------------------------------------------------------------------
  private HierInvalTarget
  cloneChopExcluded(SubscriptionSet ss){
    HashMap newChildren = null;
    HierInvalTarget newHIT = null;
    Collection thisChildren = null;
    HierInvalTarget thisChild = null;
    SubscriptionSet childSS = null;
    HierInvalTarget newChildHIT = null;
    boolean hasChildren = false;

    assert(!this.isEmpty());
    if(ss != null){
      assert(this.name.equals(ss.getName()));
      newChildren = new HashMap();
      if((ss.containsChildren()) || (this.includesChildren)){
        assert((ss.hasNoChildren()) || (this.children.isEmpty()));
        newHIT = (HierInvalTarget)this.clone();
      }else{
        // !this.includesChildren && !ss.containsChildren
        thisChildren = this.children.values();
        for(Iterator i = thisChildren.iterator(); i.hasNext();){
          thisChild = (HierInvalTarget)i.next();
          childSS = ss.getChild(thisChild.getName());
          newChildHIT = thisChild.cloneChopExcluded(childSS);
          newChildren.put(newChildHIT.name, newChildHIT);
        }
        newHIT = new HierInvalTarget(this.name,
                                     newChildren,
                                     false,
                                     this.includesSelf);
      }
    }else{
      // ss == null
      hasChildren = (!this.children.isEmpty()) || this.includesChildren;
      newHIT = new HierInvalTarget(this.name,
                                   new HashMap(),
                                   this.includesSelf,
                                   hasChildren);
      assert((newHIT.includesSelf) || (newHIT.includesChildren));
    }
    return(newHIT);
  }

  //-------------------------------------------------------------------------
  // Form a union of "this" and "hit" but stop when they diverge, but make
  // the resulting union such that the node where they diverge becomes an
  // imprecise invalidate.
  //-------------------------------------------------------------------------
  private HierInvalTarget
  unionUntilDiverge(HierInvalTarget hit){
    boolean sameChildren = false;
    String childName = null;
    Set childNames = null;
    Collection thisChildren = null;
    HierInvalTarget thisChild = null;
    HierInvalTarget hitChild = null;
    HierInvalTarget newChild = null;
    HierInvalTarget unionHIT = null;
    HashMap newChildren = null;

    assert(!this.isEmpty());
    assert(!hit.isEmpty());
    assert(this.name.equals(hit.name));

    // If "this" and "hit" have the same children, we continue. Otherwise,
    // we consider them as having diverged.
    childNames = this.children.keySet();
    sameChildren = (this.children.size() == hit.children.size());
    for(Iterator i = childNames.iterator(); (i.hasNext() && sameChildren);){
      childName = (String)i.next();
      sameChildren = sameChildren && hit.children.containsKey(childName);
    }
    if(sameChildren){
      thisChildren = this.children.values();
      newChildren = new HashMap();
      for(Iterator i = thisChildren.iterator(); i.hasNext();){
        thisChild = (HierInvalTarget)i.next();
        hitChild = hit.getChild(thisChild.name);
        assert(hitChild != null);
        newChild = thisChild.unionUntilDiverge(hitChild);
        newChildren.put(thisChild.name, newChild);
      }
      unionHIT = new HierInvalTarget(this.name,
                                     newChildren,
                                     this.includesSelf || hit.includesSelf,
                                     (this.includesChildren ||
                                      hit.includesChildren));
    }else{
      unionHIT = new HierInvalTarget(this.name,
                                     new HashMap(),
                                     this.includesSelf || hit.includesSelf,
                                     true);
    }
    return(unionHIT);
  }

  //-------------------------------------------------------------------------
  // Take the complete union of "it" and "this"
  //-------------------------------------------------------------------------
  private HierInvalTarget
  getCompleteUnion(HierInvalTarget hit){
    HierInvalTarget unionIT = null;
    HierInvalTarget thisChild = null;
    HierInvalTarget hitChild = null;
    HierInvalTarget newChild = null;
    Collection thisChildValues = null;
    Collection hitChildValues = null;
    HashMap newChildren = null;

    assert(this != null);
    assert(!this.isEmpty());
    assert(hit != null);
    assert(this.name.equals(hit.name));
    if(this.children.isEmpty()){
      if(this.includesChildren){
        unionIT = new HierInvalTarget(this.name,
                                      this.children,
                                      this.includesSelf || hit.includesSelf,
                                      true);
      }else{
        assert(this.includesSelf);
        unionIT = new HierInvalTarget(hit.name,
                                      hit.children, 
                                      true,
                                      hit.includesChildren);
      }
    }else if(hit.children.isEmpty()){
      if(hit.includesChildren){
        unionIT = new HierInvalTarget(hit.name,
                                      hit.children,
                                      this.includesSelf || hit.includesSelf,
                                      true);
      }else{
        assert(hit.includesSelf);
        unionIT = new HierInvalTarget(this.name,
                                      this.children,
                                      true,
                                      this.includesChildren);
      }
    }else{
      thisChildValues = this.children.values();
      newChildren = new HashMap();
      for(Iterator i = thisChildValues.iterator(); i.hasNext();){
        thisChild = (HierInvalTarget)i.next();
        hitChild = (HierInvalTarget)hit.children.get(thisChild.name);
        if(hitChild != null){
          newChild = thisChild.getCompleteUnion(hitChild);
        }else{
          newChild = (HierInvalTarget)thisChild.clone();
        }
        newChildren.put(thisChild.name, newChild);
      }
      hitChildValues = hit.children.values();
      for(Iterator i = hitChildValues.iterator(); i.hasNext();){
        hitChild = (HierInvalTarget)i.next();
        if(!newChildren.containsKey(hitChild.name)){
          newChild = (HierInvalTarget)hitChild.clone();
          newChildren.put(hitChild.name, newChild);
        }
      }
      unionIT = new HierInvalTarget(this.name,
                                    newChildren,
                                    this.includesSelf || hit.includesSelf,
                                    false);
    }
    return(unionIT);
  }

  //-------------------------------------------------------------------------
  // Return true if this object is invalidated by "this"
  //-------------------------------------------------------------------------
  private boolean
  invalidatesObj(String objPath){
    FileNameTokenizer fnt = null;
    String token = null;
    boolean done = false;
    boolean contained = false;
    HierInvalTarget current = null;

    if(this.isEmpty()){
      contained = false;
    }else{
      fnt = new FileNameTokenizer(objPath);
      token = fnt.getNextToken();
      assert((token != null) && (token.equals("")));
      done = false;
      contained = false;
      current = this;
      while(!done){
        token = fnt.getNextToken();
        if(token == null){
          // Reached the last component of the object path
          contained = current.includesSelf;
          done = true;
        }else if(current.children.isEmpty()){
          // Reached a leaf node, but not the end of the object path
          assert(current.includesSelf || current.includesChildren);
          contained = current.includesChildren;
          done = true;
        }else if(current.children.containsKey(token)){
          current = (HierInvalTarget)current.children.get(token);
          done = false;
        }else{
          // The tree doesn't have a branch that follows this object path
          contained = false;
          done = true;
        }
      }
    }
    return(contained);
  }
}
