package code;

/* ImmutableBytes.java
 * 
 * An immutable byte array.
 *
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */

import java.util.Arrays;
import java.io.Serializable;

public class ImmutableBytes implements Serializable, Immutable, Cloneable {

private final byte b[];
 /** 
 *  ImmutableBytes() -- Safe contructor -- makes a copy 
 **/ 
  public 
  ImmutableBytes(byte b_[]){
    b = new byte[b_.length];
    System.arraycopy(b_, 0, b, 0, b_.length);
  }

 /** 
 *  dangerousCreateImmutableBytes() -- Dangerous contructor (factory, actually)  
 *  
 *                     *** DOES NOT make a copy *** 
 *   
 *     *** CALLER MUST DELETE ITS REFERENCE TO THE ARRAY PASSED IN *** 
 **/ 

  public static ImmutableBytes
  dangerousCreateImmutableBytes(byte b_[]){
    ImmutableBytes ib = new ImmutableBytes(b_, true);
    return ib;
  }

 /** 
 *  ImmutableBytes() -- Private helper for dangerous constructor  
 *  Does not make a copy!!! Called by dangerousCreateImmutableBytes() 
 **/ 
  private
  ImmutableBytes(byte b_[], boolean flag){
    assert(flag == true); // Flag is just to distinguish public from priv interface
    b = b_; // DANGER!!!
  }
  
 /** 
 *  toString()  
 **/ 
  public String
  toString(){
    return new String(b);
  }

 /** 
 *  getLength() 
 **/ 
  public int
  getLength(){
    return b.length;
  }

 /** 
 *  getCopyBytes() -- this is the safe thing to do if you want a reference 
 *    to the bytes for anything at all complex. 
 **/ 
  public byte[]
  getCopyBytes(){
    byte copy[] = new byte[b.length];
    System.arraycopy(b, 0, copy, 0, b.length);
    return copy;
  }

 /** 
 *  dangerousGetReferenceToInternalByteArray() -- *** DANGEROUS *** 
 *    Get a reference to internal byte array and avoid making a copy. 
 *    Performance option only -- much safer to use getCopyBytes(). 
 *     
 *  This call must only be used when writing bytes to outside world 
 *  using an interface *known* to not mess with the contents of the 
 *  array. 
 * . 
 *  e.g. 
 *     ImmutableBytes data = ...; 
 *     RandomAccessFile raf = ...; 
 *     byte readOnlyBuffer[] = data.dangerousGetReferenceToInternalByteArray(); 
 *     raf.write(readOnlyBuffer, 0, length); 
 *     readOnlyBuffer = null; 
 **/ 
  public byte[]
  dangerousGetReferenceToInternalByteArray(){
    return b; // Yikes! Caller had better be careful or Immutable violated!
  }

 /** 
 *  clone() 
 **/ 
  public Object
  clone(){
    assert(this instanceof Immutable);
    return this;
  }

 /** 
 *  makeSubset() -- make a new ImmutableBytes covering 
 *  some subrange of the byte array 
 **/ 
public ImmutableBytes 
makeSubset(int skip, int len){
  assert(skip + len <= b.length);
  byte b1[] = new byte[len];
  System.arraycopy(b, skip, b1, 0, len);

  // 
  // Safe to call dangerous constructor b/c this is
  // only reference to b1 and we throw it away on return.
  //
  return ImmutableBytes.dangerousCreateImmutableBytes(b1);
}


 /** 
 *  equals() 
 **/ 
  public boolean
  equals(Object o){
    if(o == null){
      return false;
    }
    if(!(o instanceof ImmutableBytes)){
      return false;
    }
    ImmutableBytes ib = (ImmutableBytes)o;
    if(b == ib.b){
      return true; // Same reference
    }
    if(Arrays.equals(b, ib.b)){
      return true; // Same contents
    }
    return false;
  }

 /** 
 *  main() -- self test 
 **/ 
  public static void
  main(String []a){
    System.out.print("Self test ImmutableBytes.java...");
    byte b1[] = new byte[4];
    b1[0] = 10;
    b1[1] = 9;
    b1[2] = 8;
    b1[3] = 7;

    byte b2[] = new byte[4];
    b2[0] = 10;
    b2[1] = 9;
    b2[2] = 8;
    b2[3] = 7;

    byte b3[] = new byte[4];
    b3[0] = 0;
    b3[1] = 1;
    b3[2] = 2;
    b3[3] = 3;

    ImmutableBytes ib1 = new ImmutableBytes(b1); // Makes copy
    b1[2] = 43;
    ImmutableBytes ib2 = dangerousCreateImmutableBytes(b2);
    b2 = null;
    ImmutableBytes ib3 = dangerousCreateImmutableBytes(b3);
    b3 = null;
    ImmutableBytes ib4 = (ImmutableBytes)ib1.clone();
    assert(ib1.equals(ib2));
    assert(ib2.equals(ib1));
    assert(!ib3.equals(ib1));
    assert(!ib3.equals(ib2));
    assert(!ib2.equals(ib3));
    assert(ib4.equals(ib1));
    
    String s1 = ib1.toString();
    String s2 = ib2.toString();
    String s3 = ib3.toString();
    String s4 = ib4.toString();
    assert(s1.equals(s2));
    assert(s2.equals(s4));
    assert(!s3.equals(s1));

    ImmutableBytes ib5 = ib4.makeSubset(0, ib4.getLength());
    assert(ib5.equals(ib4));
    byte b4[] = new byte[2];
    b4[0] = 2;
    b4[1] = 3;
    ImmutableBytes ib6 = ib3.makeSubset(2, 2);
    ImmutableBytes ib7 = new ImmutableBytes(b4);
    assert(ib7.equals(ib6));
    b4[0] = 0;
    b4[1] = 1;
    ib6 = ib3.makeSubset(0,2);
    ib7 = new ImmutableBytes(b4);
    assert(ib7.equals(ib6));
    
    System.out.println("...PASSED");
  }
}

//---------------------------------------------------------------------------
/* $Log: ImmutableBytes.java,v $
/* Revision 1.5  2005/07/18 05:10:22  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.4  2004/10/13 17:41:36  zjiandan
/* Initial implementation of PersistentLog tested with DataStore stubs.
/*
/* TBD: test recovery with integrated DataStore and RandomAccessState.
/*
/* Revision 1.3  2004/09/10 15:54:49  dahlin
/* RandomAccessState passes test4 (except deliberate tweak of performance bug
/*
/* Revision 1.2  2004/08/18 22:44:43  dahlin
/* Made BoundInval subclass of PreciseInval; RandomAccessState passes 2 self tests
/*
/* Revision 1.1  2004/07/28 20:18:25  dahlin
/* encapsulated byte[] bodies in ImmutableBytes for safety
/*
 */
//---------------------------------------------------------------------------
