package code;

 /** 
 *  Represent an imprecise invalidate message 
 **/ 

import java.io.ObjectOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;
import code.security.SecureImpreciseInv;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

public class ImpreciseInv implements GeneralInv, Immutable
{
  
  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;

  //
  // The three VersionVector should have the same nodeIds
  // the custom serialization only send the nodeId once
  //
  final protected InvalTarget invalTarget; 
  transient final protected AcceptVV startTime;
  transient final protected AcceptVV endTime;
  transient final protected AcceptVV realTime;  
    
 /** 
 *  Constructor    -- for serialization 
 **/ 
  protected ImpreciseInv(){
    invalTarget = null;
    startTime = null;
    endTime = null;
    realTime = null;
  }

 /** 
 *  Constructor    -- temporary for compatable with old code 
 *        note: realTime is set to endTime if not specified 
 **/ 
  public ImpreciseInv(InvalTarget invalTarget_,
                      AcceptVV startTime_,
                      AcceptVV endTime_)
    {
      
        assert(invalTarget_ instanceof Immutable);
	this.invalTarget = invalTarget_; // InvalTarget is immutable
        assert(startTime_ instanceof Immutable);
	this.startTime = startTime_;     // AcceptVV is immutable
	this.endTime = endTime_;         // AcceptVV is immutable
        this.realTime = endTime_;
        assert invalTarget != null;
        assert startTime != null;
        assert endTime != null;
        assert realTime != null;        
    }

 /** 
 *  Constructor 
 **/ 
  public ImpreciseInv(InvalTarget invalTarget_,
                      AcceptVV startTime_,
                      AcceptVV endTime_,
                      AcceptVV realTime_)
    {
        assert(invalTarget_ instanceof Immutable);
	this.invalTarget = invalTarget_; // InvalTarget is immutable
        assert(startTime_ instanceof Immutable);
	this.startTime = startTime_;     // AcceptVV is immutable
	this.endTime = endTime_;         // AcceptVV is immutable
        
        assert(realTime_ instanceof Immutable);
        this.realTime = realTime_;
        assert invalTarget != null;
        assert startTime != null;
        assert endTime != null;
        assert realTime != null;
        assert startTime.size() == endTime.size();
        assert realTime.size() == startTime.size();        
    }

 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    
    //System.out.println(this.getClass().getDeclaredFields().length);
    
    assert(this.getClass().getDeclaredFields().length == 9);

    //the fields we custom serialize remain the same
    Field[] f = new Field[4];
      try{
        f[0] = this.getClass().getDeclaredField("invalTarget");
        f[1] = this.getClass().getDeclaredField("startTime");
        f[2] = this.getClass().getDeclaredField("endTime");
        f[3] = this.getClass().getDeclaredField("realTime");

        assert (f[0].getType().getName().equals("InvalTarget"));
        assert (f[1].getType().getName().equals("AcceptVV"));
        assert (f[2].getType().getName().equals("AcceptVV"));
        assert (f[3].getType().getName().equals("AcceptVV"));
      }catch(NoSuchFieldException ne){
        assert false;
      }
  }

 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- moved from  
 *                   TaggedOutputStream 
 **/ 
  private void writeObject(ObjectOutputStream out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("ImpreciseInv -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

    out.defaultWriteObject();
    
    assert invalTarget instanceof HierInvalTarget;
      
    //startTime endTime
      
    Object token = null;
    
    assert startTime.size() == endTime.size();
    assert realTime.size() == startTime.size();
    int numVVEntriesToSend = 0;
    
    for(VVIterator vvIter = startTime.getIterator();
        vvIter.hasMoreElements();){
      token = vvIter.getNext();
      // No need to send this entry of startTime and endTime if
      // their timestamps are uninitialized
      if((startTime.getStampByIteratorToken(token) !=
          AcceptStamp.BEFORE_TIME_BEGAN)||
         (endTime.getStampByIteratorToken(token) !=
          AcceptStamp.BEFORE_TIME_BEGAN)){
        
        assert (startTime.getStampByIteratorToken(token) !=
                AcceptStamp.BEFORE_TIME_BEGAN);
        
        assert (endTime.getStampByIteratorToken(token) !=
                AcceptStamp.BEFORE_TIME_BEGAN);
        
        assert (realTime.getStampByIteratorToken(token) != 
                AcceptStamp.BEFORE_TIME_BEGAN);
        numVVEntriesToSend++;
      }else{
        assert realTime.getStampByIteratorToken(token) == 
          AcceptStamp.BEFORE_TIME_BEGAN;
      }
    }
    //System.out.println("numVVEntriesToSend: " + numVVEntriesToSend
    //                   + "startTime.size(): " + startTime.size());
    assert numVVEntriesToSend <= startTime.size();
    out.writeInt(numVVEntriesToSend);
    
    for(VVIterator vvIter = startTime.getIterator();
        vvIter.hasMoreElements();){
      token = vvIter.getNext();
      // No need to send this entry of startTime and endTime if
      // their timestamps are uninitialized
      if(startTime.getStampByIteratorToken(token) !=
         AcceptStamp.BEFORE_TIME_BEGAN){
        if(SangminConfig.forkjoin){
          out.writeObject(startTime.getServerByIteratorToken(token));
        } else {
          out.writeLong(startTime.getServerByIteratorToken(token).getIDint());
        }
        out.writeLong(startTime.getStampByIteratorToken(token));
        out.writeLong(endTime.getStampByIteratorToken(token));
        out.writeLong(realTime.getStampByIteratorToken(token));
        numVVEntriesToSend--;
      }
    }
    assert(numVVEntriesToSend == 0);

  }

 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --moved from  
 *                   TaggedInputStream 
 **/ 
  private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(this.getClass().getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }


    in.defaultReadObject();

    
    int vvSize = in.readInt();
    AcceptStamp[] startArr, endArr, realArr;
    startArr = new AcceptStamp[vvSize];
    endArr = new AcceptStamp[vvSize];
    realArr = new AcceptStamp[vvSize];
    for(int i=0; i<vvSize; i++){
      NodeId nodeId = SangminConfig.forkjoin?
          (BranchID)in.readObject():new NodeId(in.readLong());
      startArr[i] = new AcceptStamp(in.readLong(), nodeId);
      endArr[i] = new AcceptStamp(in.readLong(), nodeId);
      realArr[i] = new AcceptStamp(in.readLong(), nodeId);
    }
    Field[] f = new Field[3];
    //startTime = new AcceptVV(startArr);
    //endTime = new AcceptVV(endArr);
    //realTime = new AcceptVV(realArr);
    try{
      
//      f[0] = this.getClass().getDeclaredField("startTime");
//      f[1] = this.getClass().getDeclaredField("endTime");
//      f[2] = this.getClass().getDeclaredField("realTime");
      f[0] = ImpreciseInv.class.getDeclaredField("startTime");
      f[1] = ImpreciseInv.class.getDeclaredField("endTime");
      f[2] = ImpreciseInv.class.getDeclaredField("realTime");
    }catch(NoSuchFieldException ne){
      assert false;
    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      assert false;
    }
    try{
      f[0].set(this, new AcceptVV(startArr));
      f[1].set(this, new AcceptVV(endArr));
      f[2].set(this, new AcceptVV(realArr));
    }catch(IllegalArgumentException ie){
      assert false;
    }catch(IllegalAccessException iae){
      assert false;
    }
    
    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      assert false;
    }
  }

 /** 
 *  METHODS INHERITED FROM GENERALINV: 
 **/ 

 /** 
 *  Return the InvalTarget for this invalidate 
 **/ 
    public final InvalTarget getInvalTarget()
    {
	return(this.invalTarget);
    }

 /** 
 *  Return the start version vector 
 **/ 
    public final AcceptVV getStartVV()
    {
	return(this.startTime);
    }

 /** 
 *  Return the end version vector 
 **/ 
    public final AcceptVV getEndVV()
    {
	return(this.endTime);
    }
 /** 
 *  Return the realTime version vector 
 **/ 
    public final AcceptVV getRTVV()
    {
	return(this.realTime);
    }

  
    //
    // Note: No call for getVV() for a general
    // inval -- need to say whether you mean the start or end
    //

 /** 
 *  Clone this object 
 *  
 *  If we *know* that this object and all objects we reference 
 *  are immutable, then just return this; otherwise perform 
 *      newOne = super.clone();  
 *      now replace any fields in newOne that point to mutable  
 *          objects with a clone of that field 
 *      return newOne 
 *  
 **/ 
  public final Object
  clone(){
    // All of our classes are immutable, so we can make a shallow copy
    return(new ImpreciseInv(this.invalTarget,
                            this.startTime,
                            this.endTime,
                            this.realTime));
  }

 /** 
 *  Return a new invalidate with mutiple operations performed 
 **/ 
  public GeneralInv
  cloneIntersectInvaltargetChopStartEnd(GeneralInv inv){
    InvalTarget invTarget = null;
    GeneralInv intersectInv = null;
    PreciseInv preciseInv = null;
    MultiObjPreciseInv mopi = null;

    // This call should be made for committed invalidates only
    assert((inv.getStartVV().includesAnyPartOf(this.startTime)) &&
           (this.endTime.includesAnyPartOf(inv.getEndVV())));

    invTarget = this.invalTarget.getIntersection(inv.getInvalTarget());
    assert(invTarget != null);
    if(invTarget instanceof HierInvalTarget){
      assert inv.getRTVV().includes(this.getRTVV());
      intersectInv = new ImpreciseInv((HierInvalTarget)invTarget,
                                      inv.getStartVV(),
                                      inv.getEndVV(),
                                      inv.getRTVV());
    }else if(invTarget instanceof ObjInvalTarget){
      assert(inv.isPrecise());
      assert(inv.getStartVV().equals(inv.getEndVV()));
      //preciseInv = (PreciseInv)inv;
      //intersectInv = new PreciseInv((ObjInvalTarget)invTarget,
      //                              preciseInv.getAcceptStamp(),
      //                              preciseInv.getRTAcceptStamp(),
      //                              false);//because this imprecise implies 
      //it's already debargoed
      assert invTarget.equals(inv.getInvalTarget());
      assert this.startTime.includes(inv.getStartVV());
      assert this.endTime.includes(inv.getEndVV());
      intersectInv = inv;
    }else{
      assert(invTarget instanceof MultiObjInvalTarget);
      assert(inv.isPrecise());
      assert this.startTime.includes(inv.getStartVV());
      assert this.endTime.includes(inv.getEndVV());
      //mopi = (MultiObjPreciseInv)inv;
      //intersectInv = new MultiObjPreciseInv((MultiObjInvalTarget)invTarget,
      //                                      mopi.getAcceptStamp(),
      //                                      mopi.getRTAcceptStamp(),
      //                                      false, // Should be debargoed
      //                                      new MOITBoundEntry[0]);
      intersectInv = inv;
    }
    return(intersectInv);
  }

 /** 
 *  Return the portion of an invalidate for one writer whose 
 *  starting stamp is potentially chopped. 
 **/ 
    public GeneralInv getOneWriterSubsetFrom(NodeId node,
					     long startStampInclusive)
    {
        /*
          return an invalidation (possibly this) with 
          interestSet = this.interestSet
          nodeId = node // assert node was in this.VV
          startVV{node} == startStampInclusive
            // assert this.startStamp{node} >= startSTampInclusive
          endVV{node} == this.endVV{node}
          (Note: OK to just return this if we meet all of the other criteria)
        */
        SingleWriterImpreciseInv subsetInv = null;
	long currentStartStamp = 0;
	long currentEndStamp = 0;
	AcceptStamp newStartAcceptStamp = null;
	AcceptStamp endAcceptStamp = null;

	try{
	    currentEndStamp = this.endTime.getStampByServer(node);
	    currentStartStamp = this.startTime.getStampByServer(node);
	    assert(startStampInclusive >= currentStartStamp);
	    assert(startStampInclusive <= currentEndStamp);

	    newStartAcceptStamp = new AcceptStamp(startStampInclusive, node);
	    endAcceptStamp = new AcceptStamp(currentEndStamp, node);
	    subsetInv = new SingleWriterImpreciseInv(this.invalTarget,
						     newStartAcceptStamp,
						     endAcceptStamp);
	}catch(NoSuchEntryException e){
	    System.err.println("Unknown node: " + node);
	    assert(false);
	}
	return(subsetInv);
    }

 /** 
 *  Return true if the entire invalidation is already accounted 
 *  for by the specified vv 
 **/ 
    public boolean isIncludedBy(VV vv)
    {
	//if we are precise, return vv.includes(this.timestamp)
	//if we are imprecise, return vv.includes(this.*endVV*)
	return(vv.includes(this.endTime));
    }

 /** 
 *  Return true if any component of this invalidation is already 
 *  accounted for by the specified vv 
 **/ 
    public boolean anyComponentIncludedBy(VV vv)
    {
	//if we are precise, return vv.includes(this.timestamp)
	//if we are imprecise, true if for any entry e in *startVV*,
	//  vv.includes(e.timestamp)
	Object token = null;
	VVIterator vvIter = null;
	NodeId nodeId = null;
	long timeStamp = 0;
	AcceptStamp acceptStamp = null;
	boolean result = false;

	vvIter = this.startTime.getIterator();
	while(vvIter.hasMoreElements() && (!result)){
	    token = vvIter.getNext();
	    nodeId = this.startTime.getServerByIteratorToken(token);
	    timeStamp = this.startTime.getStampByIteratorToken(token);
	    acceptStamp = new AcceptStamp(timeStamp, nodeId);
	    result = vv.includes(acceptStamp);
	}
	return(result);
    }

 /** 
 *  Return true if this invalidation is committed 
 **/ 
    public boolean isCommitted()
    {
	// Ensure that either the entire message is uncommitted, or it
	// is committed.
	return(false);
    }

 /** 
 *  Return true if this invalidate is precise (which it isn't!) 
 **/ 
    public boolean isPrecise()
    {
	return(false);
    }

 /** 
 *  Return true if this invalidate is bound (which it isn't!) 
 **/ 
    public boolean isBound()
    {
	return(false);
    }

 /** 
 *  Return true if this invalidate is a delete (which it isn't!) 
 **/ 
    public boolean isDelete()
    {
	return(false);
    }
 /** 
 *  Return true if this invalidate is an embargoed write (which it isn't!) 
 **/ 
    public boolean isEmbargoed()
    {
	return(false);
    }
  
    

 /** 
 *  Create a new generalInv that is the union of this and i 
 **/ 
    public GeneralInv newUnion(GeneralInv i, SubscriptionSet ss)
    {
	/*
	  create a new generalInv that is the union of this and i
	  each element of startVV is min of corresponding
                          end        max
	  interest set is union of corresponding
	  assert(either both are uncommitted or both are committed)
	  if both committed 
	     assert no gap between csnEnd of one and csnStart of
	       next (OK to overlap)
	     csnstart = min of both csnStart
	     csnEnd = max of both csnMax
	*/
      
      assert (!i.isEmbargoed());
	InvalTarget unionInvalTarget = null;
	ImpreciseInv impreciseInv = null;
	AcceptVV newStartVV = null;
	AcceptVV newEndVV = null;
        AcceptVV newRTVV = null;

	// Create an InvalTarget for the union
	unionInvalTarget = this.invalTarget.getUnion(i.getInvalTarget(), ss);
	assert(unionInvalTarget != null);
	assert(unionInvalTarget instanceof HierInvalTarget);

	// Assert that either both are committed or neither
	newStartVV = this.startTime.cloneMinVV(i.getStartVV());
	newEndVV = this.endTime.cloneMaxVV(i.getEndVV());
        newRTVV = this.getRTVV().cloneMaxVV(i.getRTVV());
	impreciseInv = new ImpreciseInv((HierInvalTarget)unionInvalTarget,
					newStartVV,
					newEndVV,
                                        newRTVV);
        
	return(impreciseInv);
    }

 /** 
 *  Return -1, 0, or 1 if this is smaller, equal, or larger than 
 *  gi2, respectively, based on mechanism to provide total orders. 
 **/ 
    public final int totalOrderCompare(GeneralInv gi2)
    {
	/*
          return -1 if this < gi2 in the total order of the log
          return +1 if this > gi2 in the total order of the log
          return 0 if this and gi2 are equivalent and can be ordered either
          way (both start at the same time and end at the same time)

          we sort by CSN then by min acceptStamp then by serverId

          (1) CSN

          If this.CSN < gi2.CSN, then return -1
           
          Note that an unassigned CSN has an infinite value)

          For a bulk invalidation, consider the *minimum* CSN included
          in the invalidation.

          If this.CSN == gi2.CSN != CSN_UNASSIGNED, either order 
          is legal. If either is a bulk invalidation, the one with 
          the lower maxCSN goes first. If both have the same maxCSN
          then return 0

          (2) startTime acceptStamp
      
          assert this.CSN == gi2.CSN == CSN_UNASSIGNED

          if the invalidation is precise, consider its acceptStamp
          if the invalidation is imprecise, consider the startVV and
          look at the minimum entry across all nodeId's in startVV

          if they differ, return -1 or 1

          (3) nodeId
          if both minimum accept stamps in step (2) are equal, 
          then compare the node Ids

          if they differ, return -1 or 1

          (4) endtime acceptStamp

          assert csn, startTime, and nodeId are all the same

          send the "more precise" one first --> look at the endTime
          and the one with the smaller max endtime comes first

          if they differ, return -1 or 1
          if they are the same, return 0 (note that if the invals are
          imprecise, they may still differ in elements other than
          their max or min...but that's ok)
	*/
	int result = 0;
	NodeId thisMinNodeId = null;
	NodeId gi2MinNodeId = null;

        if(this.startTime.getMinTimeStamp() <
           gi2.getStartVV().getMinTimeStamp()){
          result = -1;
	}else if(gi2.getStartVV().getMinTimeStamp() <
		 this.startTime.getMinTimeStamp()){
          result = 1;
	}else{
          assert(this.startTime.getMinTimeStamp() ==
                 gi2.getStartVV().getMinTimeStamp());
          thisMinNodeId = ImpreciseInv.getMinNodeId(this.startTime);
          gi2MinNodeId = ImpreciseInv.getMinNodeId(gi2.getStartVV());
          result = thisMinNodeId.compareTo(gi2MinNodeId);
          if(result == 0){
            assert(thisMinNodeId.equals(gi2MinNodeId));
            if(this.endTime.getMaxTimeStamp() <
               gi2.getEndVV().getMaxTimeStamp()){
              result = -1;
            }else if(gi2.getEndVV().getMaxTimeStamp() <
                     this.endTime.getMaxTimeStamp()){
              result = 1;
            }else{
              result = 0;
            }
          }
	}
	return(result);
    }

 /** 
 *  Return the node Id of vv which has the smallest accept stamp 
 **/ 
    private static final NodeId getMinNodeId(VV vv)
    {
	Object token = null;
	VVIterator vvIter = null;
	long minTimeStamp = Long.MAX_VALUE;
	long timeStamp = 0;
	NodeId minNodeId = null;

	vvIter = vv.getIterator();
	while(vvIter.hasMoreElements()){
	    token = vvIter.getNext();
	    timeStamp = vv.getStampByIteratorToken(token);
	    if(timeStamp < minTimeStamp){
		minTimeStamp = timeStamp;
		minNodeId = vv.getServerByIteratorToken(token);
	    }
	}
	assert(minNodeId != null);
	return(minNodeId);
    }

 /** 
 *  Return true if "this" equals "obj" 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    ImpreciseInv bi = null;

    if(obj instanceof ImpreciseInv){
      bi = (ImpreciseInv)obj;
      result = (this.invalTarget.equals(bi.invalTarget) &&
                this.startTime.equalsIgnoreNegatives(bi.startTime) &&
                this.endTime.equalsIgnoreNegatives(bi.endTime));
    }
    return(result);
  }

 /** 
 *  Convert to a string representation 
 **/ 
    public String toString()
    {
      String str = null;
      
      str = "ImpreciseInv : <" + this.invalTarget + ", " 
        + this.startTime + ", " + this.endTime + ", " + this.realTime+">";
      return(str);
    }
  
 /** 
 *  memSize() -- 
 *    Estimate the memory footprint of this object in a cheap while not 
 *    very accurate way. 
 *  
 *  Use SizeOf and TestClass to get the number of overheads: 
 *  336 is the overhead of ImpreciseInv, 152 is the overhead 
 *  of one component in AcceptVV, 192 is the overhead of one component 
 *  in the hierInvalTarget tree structure. 
 **/ 
  public long memSize(){
    int vvSize = startTime.size();
    assert invalTarget instanceof HierInvalTarget;
    int numHIT = ((HierInvalTarget)(invalTarget)).size();
    return 336 + 152*vvSize + 192*numHIT;
  }

 /** 
 *  memSize() -- 
 *    Estimate the memory footprint of this object in a cheap while not 
 *    very accurate way. 
 *  
 *  Use SizeOf and TestClass to get the number of overheads: 
 *  336 is the overhead of ImpreciseInv, 152 is the overhead 
 *  of one component in AcceptVV, 192 is the overhead of one component 
 *  in the hierInvalTarget tree structure. 
 **/ 
  public long estimateByteSize(){
    long ret = 0;
    try{
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      ObjectOutputStream tos = new ObjectOutputStream(baos);
      tos.writeObject(this);
      ret = baos.size();
      tos.close();
      baos.close();
    }catch (Exception e){
    }
    return ret;
  }

  
    

 /** 
 *  chop the invalidation by endVV 
 *  
 *  for startVV[i] > chopVV[i] delete i component in newStartVV and newEndVV 
 *      otherwise if endVV[i] > chopVV[i], set endVV[i] = chopVV[i] 
 **/ 
  public static ImpreciseInv chopImpreciseInv(ImpreciseInv gi, VV chopVV){
    
    AcceptVV startVV = gi.getStartVV();
    AcceptVV endVV = gi.getEndVV();
    AcceptVV rtVV = gi.getRTVV();

    CounterVV newStartVV = new CounterVV();
    CounterVV newEndVV = new CounterVV();
    CounterVV newRTVV = new CounterVV();

    VVIterator startIter = null;
    long chopVVStamp = 0;
    NodeId nodeId = null;
    long startStamp = 0;
    long endStamp = 0;
    Object token = null;
    boolean vvEmpty = true;

    for(startIter = startVV.getIterator(); startIter.hasMoreElements();){
      token = startIter.getNext();
      nodeId = startVV.getServerByIteratorToken(token);
      try{     
        startStamp = startVV.getStampByIteratorToken(token);      
        endStamp = endVV.getStampByServer(nodeId);
      }catch(NoSuchEntryException e){
        assert false;//should never come here.
      }

      try{
        chopVVStamp = chopVV.getStampByServer(nodeId);
      }catch(NoSuchEntryException e){
        continue;// ignore it, only need catch up anything covered by chopVV
      }

      if(startStamp <= chopVVStamp){//ignore all those component with start > chop
        vvEmpty = false;
        newStartVV.setStamp(new AcceptStamp(startStamp, nodeId));
        try{
          newRTVV.setStamp(new AcceptStamp(rtVV.getStampByServer(nodeId), nodeId));
        }catch(NoSuchEntryException e){
          assert false;//should never come here
        }
        if(endStamp > chopVVStamp){
          newEndVV.setStamp(new AcceptStamp(chopVVStamp, nodeId));
        }else{
          newEndVV.setStamp(new AcceptStamp(endStamp, nodeId));
        }
      }
    }
    if (vvEmpty) {
      return null;
    }
    return new ImpreciseInv(gi.getInvalTarget(),
                            newStartVV.cloneAcceptVV(),
                            newEndVV.cloneAcceptVV(),
                            newRTVV.cloneAcceptVV());
  } 

  
  
 
      
}


//---------------------------------------------------------------------------
/* $Log: ImpreciseInv.java,v $
/* Revision 1.26  2007/06/25 05:21:28  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.25  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.24  2006/09/12 14:57:59  zjiandan
/* Move object custom serialization into individual classes from TaggedOutputStream
/* and TaggedInputStream. The new serialization code consumes a little bit more
/* bytes than the original approach because of java's default object information
/* overhead (I have excluded the field name and type information).
/* The major changes are using java reflection API to serialize "private final"
/* fields (which are very common among our objects that need to custom serialization).
/* Also add sanity check for each class so that we will remember to update
/* the serialization methods when we change the class definitions.
/*
/* Revision 1.23  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.22  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.21  2005/07/18 05:10:22  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.20  2005/03/01 01:20:17  nayate
/* Changed testing code for new InvalTarget
/*
/* Revision 1.19  2005/02/28 22:57:58  nayate
/* Modified for new code
/*
/* Revision 1.18  2004/10/22 20:46:54  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.17  2004/10/22 18:13:38  zjiandan
/* cleaned csn from UpdateLog, modified DeleteInv.
/* TBD:
/*    1. clean csn from all subclasses of GeneralInv
/*    2. fix *Inv::cloneIntersectInvaltargetChopStartEnd
/*
/* Revision 1.16  2004/08/18 22:44:43  dahlin
/* Made BoundInval subclass of PreciseInval; RandomAccessState passes 2 self tests
/*
/* Revision 1.15  2004/07/28 14:27:35  dahlin
/* Added sanity checks for immutable objects
/*
/* Revision 1.14  2004/05/13 18:49:17  dahlin
/* Clean SDIMSController by moving SpanningTree directory out as separate class (rather than using raw SDIMSInterface)
/*
/* Revision 1.13  2004/05/11 00:46:56  dahlin
/* Added read-miss-directory updates to SDIMSController
/*
/* Revision 1.12  2004/05/09 20:09:14  dahlin
/* Updated unit tests for WorkQueue, TimeoutQueue, TimeoutQueueWorker, DirectoryInterestSet, SubscribeInvalRequest, InvalSpanningList, AcceptStamp, NodeId, and AcceptVV
/*
/* Revision 1.11  2004/05/04 06:48:08  nayate
/* Added, but didn't test, an implementation of newUnion()
/*
/* Revision 1.10  2004/05/03 19:35:54  nayate
/* Started making changes to allow passing an interest set to all union
/* operations.
/*
/* Revision 1.9  2004/04/27 02:51:04  zjiandan
/* Fix Assert Error.
/*
/* Revision 1.8  2004/04/27 02:50:30  zjiandan
/* Fix assert error
/*
/* Revision 1.7  2004/04/22 08:52:07  nayate
/* Minor change
/*
/* Revision 1.6  2004/04/21 22:36:56  nayate
/* Tested ImpreciseInv
/*
/* Revision 1.5  2004/04/21 19:39:41  nayate
/* Not fully tested ImpreciseInv
/*
/* Revision 1.4  2004/04/18 07:15:57  nayate
/* Added a method to GeneralInv, finished (but didn't test) PreciseInv
/*
/* Revision 1.3  2004/04/16 18:53:22  nayate
/* Stub implementations that compile + minor fixes
/*
/* Revision 1.2  2004/04/15 20:04:24  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
