package code;

/* IncommingConnection --
 *
 *   A data structure for multiplexing inval stream subscriptions at receiver
 *   side. Created when a new connection is created by OutgoingConnectionThread.
 *   
 *   Note, it's possible that One OutgoingConnection might have multiple
 *   IncommingConnections during its lifetime because its worker might 
 *   die and create multiple times. But at most one incommingConnection exists
 *   given any moment as there's only one worker.
 *
 *
 *   A stream pattern:
 *
 *      [
 *       // handshack
 *       INVAL_SOCKET_MAGIC, senderId, intial prevVV, 
 *
 *       
 *       {
 *         //catchup log  
 *         (CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)  |
 *        
 *         //normal traffic
 *         (GeneralInv|UnbindMsg|DebargoMsg)*  |
 *       
 *         //catchup checkpoint
 *         (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)
 *       
 *       }*
 *       
 *       //goodbye
 *       TerminateMsg
 *      ]
 *
 *  Note: it's a single thread worker
 *      every method is not synchronized
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */
import java.net.Socket;
import java.io.IOException;
import java.io.EOFException;
import java.io.OptionalDataException;
import java.io.InvalidClassException;
import java.io.PrintStream;
import java.io.FileOutputStream;
import java.security.PublicKey;

import code.security.SangminConfig;
import code.security.SecurePreciseInv;
import code.security.SecureCheckpoint;

public class IncommingConnection implements ConnectionState{

  protected final static boolean dbgProgress = false;
  protected final static boolean dbgPerformance = false;
  protected final static boolean dbg = true; // Print connection events
  protected final static boolean dbgVerbose = false; // Print recv events

  protected final static boolean measureTime = true;
  protected final static boolean USE_II_FILTER = true;

  // See inner class IncommingConnectionWorker for more debug params
  private Thread worker;


  protected Core core = null;
  protected Controller controller = null;
  private Socket underlyingSocket = null;
  private TaggedInputStream s = null;
  private StreamId streamId = null;



  protected NodeId senderId = null;
  protected CounterVV prevVV = null;
  protected SubscriptionSet ss = null;
  protected SubscriptionSet pendingSet = null;
  protected CounterVV pendingStartVV = null;



 /** 
 *  Constructor -- for test 
 **/ 
  public
  IncommingConnection(StreamId sid, SubscriptionSet ss, AcceptVV prevVV){
    if(dbgProgress){
      System.out.println("IncommingConnection(sid, ss, prevVV):( " + sid 
          + ", " + ss + ", " + prevVV + ") is created"); 
    }
    this.streamId = sid;
    this.ss = ss;
    this.prevVV = new CounterVV(prevVV);

    this.worker = null;


  }

 /** 
 *  Constructor 
 **/ 
  public
  IncommingConnection(TaggedInputStream s,
      Core core,
      Controller controller,
      Socket underlyingSocket,
      StreamId newStreamId){
    if(dbgProgress){
      System.out.println("IncommingConnection(sid):( " + newStreamId + ") is created"); 
    }
    this.s = s;
    this.core = core;
    this.controller = controller;
    this.underlyingSocket = underlyingSocket;
    //this.countEndVVSearchSlow = 0;
    this.streamId = newStreamId;

    this.senderId = null;
    this.prevVV = (CounterVV)(CounterVV.makeVVAllNegatives());
    this.ss = SubscriptionSet.makeEmptySet();
    this.pendingSet = null;
    this.pendingStartVV = null;

    this.worker = null;


  }



 /** 
 *  startWorker() -- kick off the worker thread. Must be called right 
 *     after initialization. 
 **/ 
  public void startWorker(){    
    assert worker==null;
    worker = new IncommingConnectionWorker(this.s, this.underlyingSocket, this);
    worker.start();
  }


 /** 
 *  interruptWorker() 
 **/ 
  public void interruptWorker(){
    try{
      if(this.worker!=null){
        this.worker.interrupt();
        if(dbgProgress){
          System.out.println("IncommingConnectionWorker interrupt() called");
        }
        this.worker = null;
      }
    }catch(SecurityException e){
      //ignore.
    }
  }
 /** 
 *  handle initial prevVV 
 *  
 *  only called by worker 
 **/ 
  protected void initPrevVV(AcceptVV initPrevVVFromSender){
    //
    // make sure that it is safe to accept this stream
    //
    // Handle stream start VV issue:
    // 
    //   Previously in order to be safe, the receiver will reject the stream 
    //   if startvv > receiver's cvv.
    //   The problem is: if sender always sends its cvv in handshake as stream's start vv,
    //   then the connection between an advanced sender and a receiver will always be rejected.
    //
    // two options: 
    //          
    //      (1) always accept the stream by applying an imprecise invalidate 
    //          </*, receiver'scvv, sender's cvv(i.e. stream startVV) > in handshake.
    //	        simple. 
    //   	But the big imprecise might make everything imprecise for a while.
    //          That's ok since the receiver will subscribe all the interested set soon anyways.
    //
    //      (2) Sender uses the first addSubscribeInval's startVV as the initial prevVV instead of 
    //          sender's cvv (most likely <= receiver's cvv)
    //
    //          Receiver always rejects a stream which has initial startvv >= receiver's cvv.
    //     
    //          treat the first SubscribeInval request different from the other request.
    //       
    //          variation: the handshake of a connection between two nodes should be that the sender sends 
    //          the receiver a checkpoint of <startVV= receiver's cvv, endVV=sender's cvv, 
    //          subscribeset=receiver's interested sets>.
    //  
    // For simplicity, pick (1)
    //
    if(!core.getCurrentVV().includes(initPrevVVFromSender)){//the receiver's cvv < sender's cvv
      ImpreciseInv gapInv = this.prepareGapInv(initPrevVVFromSender);
      if(gapInv != null){
        this.applyGI(gapInv);
      }
    }
    //if(SangminConfig.securityLevel == SangminConfig.NONE){
    //  assert core.getCurrentVV().includes(initPrevVVFromSender);
    //}
    //assert core.getCurrentVV().includes(initPrevVVFromSender);
    assert this.ss.isEmpty();
    this.prevVV.advanceTimestamps(initPrevVVFromSender);

    controller.informInvalStreamInitiated(senderId,
        ss,
        prevVV,
        true);
  }

  //-------------------------------------------------------------------------
  // get senderId
  //-------------------------------------------------------------------------
  public NodeId getSenderId(){
    return this.senderId;
  }

  //-------------------------------------------------------------------------
  // get SubscriptionSet
  //-------------------------------------------------------------------------
  public SubscriptionSet getSubscriptionSet(){
    return this.ss;
  }

  //-------------------------------------------------------------------------
  // get PendingSet
  //-------------------------------------------------------------------------
  public SubscriptionSet getPendingSet(){
    return this.pendingSet;
  }

  //-------------------------------------------------------------------------
  // get pendingStartVV
  //-------------------------------------------------------------------------
  public AcceptVV getPendingStartVV(){
    if (this.pendingStartVV == null){
      return null;
    }
    return this.pendingStartVV.cloneAcceptVV();
  }

  //-------------------------------------------------------------------------
  // return con.prevVV -- not synchronized because of single threading
  //-------------------------------------------------------------------------
  public AcceptVV
  getPrevVV(){
    return this.prevVV.cloneAcceptVV();
  }

  //-------------------------------------------------------------------------
  // advance PrevVV with newVV
  //-------------------------------------------------------------------------
  public void
  advancePrevVV(VV newVV){
    this.prevVV.advanceTimestamps(newVV);
  }

  //-------------------------------------------------------------------------
  // add subscriptionset to the incomming connection:
  //     1.add connection pointer to the corresponding place in ISStatus tree
  //       and notifyAll if prevVV == cvv
  //     2.merge the subscriptionset to the connection.ss
  //-------------------------------------------------------------------------
  public void
  addSubscriptionSet(SubscriptionSet newss){
    if(dbgProgress){
      System.out.println("IncommingConnection.addSubscriptionSet:( " + newss+ ") is called"); 
    }
    core.addConnectionToISStatus(newss, this);
    this.ss = ss.getCompleteUnion(newss);
  }

  //-------------------------------------------------------------------------
  // replace subscriptionset to the incomming connection:
  //     1.add connection pointer to the corresponding place in ISStatus tree
  //       and notifyAll if prevVV == cvv
  //     2.merge the subscriptionset to the connection.ss
  //-------------------------------------------------------------------------
  public void
  replaceSubscriptionSet(SubscriptionSet newss){
    if(dbgProgress){
      System.out.println("IncommingConnection.replaceSubscriptionSet:( " + newss+ ") is called"); 
    }
    core.removeConnectionFromISStatus(ss, this, prevVV);
    core.addConnectionToISStatus(newss, this);
    this.ss = newss;
  }

  //-------------------------------------------------------------------------
  //  remove subscriptionset from the incomming connection
  //    1. update ISStatus: remove this pointer, update lpvv
  //    2. remove the ss from the subscriptionset
  //-------------------------------------------------------------------------
  public void
  removeSubscriptionSet(SubscriptionSet rss, VV newLpvv){
    if(dbgProgress){
      System.out.println("IncommingConnection.removeSubscriptionSet:( " + rss
          + ", " + newLpvv + ") is called"); 
    }
    if (!ss.isEmpty()){
      core.removeConnectionFromISStatus(rss, this, newLpvv);
      try{
        this.ss = this.ss.remove(rss, true);
      }catch(IllegalRemoveSubscriptionSetException e){
        assert false : ("" + e);
      }
    }
  }

  //-------------------------------------------------------------------------
  // update ISStatus lpvv for PTree node overlapping by ss
  //-------------------------------------------------------------------------
  public void
  updateISStatusLpVV(SubscriptionSet uss, VV newLpvv){
    if(dbgProgress){
      System.out.println("IncommingConnection.updateISStatusLpVV:( " + uss
          + ", " + newLpvv + ") is called"); 
    }
    core.removeConnectionFromISStatus(uss, null, newLpvv);
  }

  //-------------------------------------------------------------------------
  // Prepare new lpvv, return max of (vv1, vv2-1)
  //-------------------------------------------------------------------------
  public AcceptVV
  prepareNewLpvv(AcceptVV vv1, AcceptVV vv2){
    CounterVV newLpVV = new CounterVV(vv2);
    newLpVV.decrementAllByOne();//(gi.startVV -1)
    newLpVV.advanceTimestamps(vv1);//get component-wise max of prevVV and (gi.startVV-1)
    return newLpVV.cloneAcceptVV();
  }


  //-------------------------------------------------------------------------
  // If receive a precise inval from node A,
  // advance my outgoing connection to A's cvv
  // so that the sender won't receive this invalidate from me again.
  //
  // for more details see designDocs/tbd_after_sosp_2007.txt
  //-------------------------------------------------------------------------
  private void 
  filterRedundantInvForSender(GeneralInv gi){

    if(USE_II_FILTER){
      if ((gi instanceof PreciseInv)
          ||(gi instanceof MultiObjPreciseInv)
          ||(gi instanceof DeleteInv)){
        core.notifyRecvInval(senderId, gi.getEndVV());
      }
    }
  }


  /**
   * return (1) the intersection of gi.invalTarget and targetSS, 
   *           if gi is imprecise 
   *              && the lpvv of the intersection is less than gi.endVV
   *       
   *        (2) null, otherwise
  **/
  private SubscriptionSet getKickedSet(GeneralInv gi, SubscriptionSet targetSS){

    if (!gi.isPrecise() && gi.getInvalTarget().intersects(targetSS)){

      assert gi.getInvalTarget() instanceof HierInvalTarget;

      SubscriptionSet kickedSet = targetSS.getIntersection((HierInvalTarget)(gi.getInvalTarget()));
      assert !kickedSet.isEmpty();
      if(!core.getLpVV(kickedSet).includes(gi.getEndVV())){//only kick off if my lpvv is not catchup
        System.out.println("lpvv: " + core.getLpVV(kickedSet) + " gi.endVV: " + gi.getEndVV());
        return kickedSet;
      }
    }
    return null;
  }


 /** 
 *  Apply an unbindmsg 
 **/ 
  public void
  applyUnbind(UnbindMsg um)
  throws IOException{
    this.core.applyUnbind(um);
  }

 /** 
 *  Apply debargo msg 
 **/ 
  public void
  applyDebargo(DebargoMsg dm) 
  throws IOException{
    this.core.applyDebargo(dm);
  }


  public void applyAdvancingGI(GeneralInv gi){
    // 
    // from the main stream ==> will advance prevVV
    // 

    assert pendingSet == null;
    assert pendingStartVV == null;

    //kick off imprecise subset
    SubscriptionSet kickedSet = this.getKickedSet(gi, this.ss);
    if(kickedSet != null){
      AcceptVV newLpvv = prepareNewLpvv(this.prevVV.cloneAcceptVV(), gi.getStartVV());
      removeSubscriptionSet(kickedSet, newLpvv);
      controller.informSubscribeInvalFailed(senderId, core.getMyNodeId(), 
          kickedSet, newLpvv);
    }
    //advance prevVV
    this.prevVV.advanceTimestamps(gi.getEndVV());
    core.notifyDatastore(this.ss, this.prevVV.cloneAcceptVV());//notify imprecise read thread

  }

  public void applyCatchupGI(GeneralInv gi){
    //
    // from catch up streams
    //

    //apply to pendingSet
    assert pendingStartVV != null;
    SubscriptionSet kickedPendingSet =  this.getKickedSet(gi, this.pendingSet);

    if(kickedPendingSet != null){
      AcceptVV newLpvv = prepareNewLpvv(pendingStartVV.cloneAcceptVV(), gi.getStartVV());

      //kick off from pendingSet
      this.updateISStatusLpVV(kickedPendingSet, newLpvv);
      try{
        pendingSet = pendingSet.remove(kickedPendingSet, true);
      }catch(IllegalRemoveSubscriptionSetException e){
        assert false : ("" + e);
      }

      if(pendingSet.isEmpty()){
        pendingSet = null;
        pendingStartVV = null;
      }
      controller.informSubscribeInvalFailed(senderId, core.getMyNodeId(), 
          kickedPendingSet, newLpvv);
    }

    //advance pendingStartVV
    if(pendingStartVV != null){
      assert !pendingSet.isEmpty();
      pendingStartVV.advanceTimestamps(gi.getEndVV());
    }

  }
 /** 
 *  Apply a general incomming invalidation 
 **/ 
  public void
  applyInsecureGI(GeneralInv gi){

    if(dbgProgress){
      System.out.println(core.getMyNodeId() + 
          " IncommingConnection.applyGI:( " + gi + ") is called"); 
    }

    //
    // advance outgoingConnection cvv to prevent redundant invalidates
    // from sending back to sender again.
    //
    this.filterRedundantInvForSender(gi);
    //
    //note: we must first update the cvv
    //      then update lpvv/prevVV for causal view across objects
    //
    core.applyInval(gi, streamId); 
    if (gi.getEndVV().isAnyPartGreaterThan(this.prevVV)){
      // 
      // from the main stream ==> will advance prevVV
      // 

      applyAdvancingGI(gi);

    }else if (pendingSet != null){
      //
      // from catch up streams
      //
      applyCatchupGI(gi);

    }
    controller.informReceiveInval(gi, senderId);
  }

 /** 
 *  Apply a general incomming invalidation 
 **/ 
 
  public void
  applyGI(GeneralInv gi){
    applyInsecureGI(gi);
  }

    





 /** 
 *  apply Checkpoint 
 **/ 
  public void
  applyCP(TaggedInputStream s, CPStartMsg cpsm)
  throws IOException, ClassNotFoundException, CausalOrderException{
    if(dbgProgress){
      System.out.println("IncommingConnection.applyCP(" + cpsm + "): is called"); 
    }

    long startTime, end, totalTime;
    if(measureTime){
      startTime = System.currentTimeMillis();
    }
    SubscriptionSet cpSS = cpsm.getSubscriptionSet();
    AcceptVV startVV = cpsm.getStartVV();
    try{
      if(startVV.getStampByServer(core.getMyNodeId()) == RMIClient.ACCEPT_STAMP_TO_FILTER_MY_OWN_UPDATES){
        startVV = startVV.makeVVexcludeNode(core.getMyNodeId());
      }
    }
    catch(NoSuchEntryException nsee){
      ; // OK
    }    



    if(this.checkAndIgnoreInvalidCP(s, cpSS, startVV)){
      controller.informSubscribeInvalFailed(senderId,
          core.getMyNodeId(),
          cpSS,
          startVV);//because of the gap between startVV and minlpvv
      if(measureTime){
        end = System.currentTimeMillis();
        totalTime = (end-startTime);
        LatencyWatcher.put("IncommingConnection.ApplyCatchupStreamStartMsg", 
            totalTime);
        startTime = System.currentTimeMillis();

      }
      return;
    }


    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IncommingConnection.ApplyCatchupStreamStartMsg", 
          totalTime);
      startTime = System.currentTimeMillis();
    }

    //
    // start applying the checkpoint stream
    //
    VV cpEndVV = cpsm.getSenderCVV();
    ImpreciseInv gapInv = null;

    // 1. prepare the big imprecise invalidate for Checkpoint
    if(!core.getCurrentVV().includes(cpEndVV)){
      //the receiver is not safe to apply cp
      gapInv = prepareGapInv(cpEndVV);
      if(gapInv != null){
        this.applyGI(gapInv);
      }else{
        assert core.getCurrentVV().includes(cpEndVV);
      }
    }

    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IncommingConnection.preApplyCP", 
          totalTime);
      startTime = System.currentTimeMillis();
    }

    //2. apply checkpoint to datastore
    core.applyCheckpoint(s, cpsm);

    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IC.core.applyCP", 
          totalTime);
      startTime = System.currentTimeMillis();
    }

    //3. attach if possible
    if(core.getLpVV(cpSS).includes(this.prevVV)){
      this.addSubscriptionSet(cpSS);
      controller.informSubscribeInvalSucceeded(senderId,
          core.getMyNodeId(),
          cpSS,
          startVV);
      if(dbgProgress){
        System.out.println("IncommingConnection::applyCP caught up" + cpSS);
      }

    }else{

      controller.informSubscribeInvalFailed(senderId,
          core.getMyNodeId(),
          cpSS,
          startVV);
      if(dbgProgress){
        System.out.println("IncommingConnection::applyCP NOT caught up");
      }
    }

    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IC.afterApplyCP", 
          totalTime);
      startTime = System.currentTimeMillis();
    }


  }


 /** 
 *  called before applying any incoming checkpoint stream 
 *  
 *  check if my lpVV for all interest sets at or below cpPath are  
 *  at least minLpVV 
 *    -- if this is not satisfied, receive all cp until RAS_DONE 
 *  
 *    -- we need to check this because the RAS will filter out those 
 *       PerObjectStates whose timestamps is between this.minlpvv and startVV 
 *       if the ISStatus for this PerObjectStates is "precise" after applying the cp 
 *       the read will get old value instead of the value between this.minlpvv and startVV 
 **/ 
  private boolean checkAndIgnoreInvalidCP(TaggedInputStream s, 
      SubscriptionSet cpSS, 
      AcceptVV startVV)
  throws IOException, ClassNotFoundException, CausalOrderException{
    if(!core.checkMinLPVV(cpSS, startVV)){
      if(dbgProgress){
        System.out.println("IncommingConnection.applyCP: aborting b/c minLPVV. cpSS = " 
            + cpSS.toString() + " startVV = " + startVV 
            + " core minLpVV for IS = " + core.getISStatusMinLpVV(cpSS));
      }

      //skip checkpoint data

      Object next = s.readTaggedObject();


      while (!((next instanceof Long)&&(((Long)next).longValue()==RandomAccessState.RAS_SHIP_DONE))){
        next = s.readTaggedObject();
      }


      return true;
    }
    return false;
  }


 /** 
 *  Called by applyCP -- 
 *  
 *    Prepare a legal imprecise invalidate with endVV.includes(startVV) 
 *    that covering the gap between currentVV and cpEndVV 
 *    i.e. only the component where cpEndVV has larger value than  
 *    core.cvv will be kept in the resulted gapInv 
 *     
 *    if no component is larger, then return null; 
 **/ 
  public ImpreciseInv prepareGapInv(VV cpEndVV){
    CounterVV startVV;
    CounterVV newStartVV= new CounterVV();;
    CounterVV endVV;
    CounterVV newEndVV = new CounterVV();

    VVIterator endIter = null;
    NodeId nodeId = null;
    long startStamp = 0;
    long endStamp = 0;
    Object token = null;

    startVV = new CounterVV(core.getCurrentVV());
    endVV = new CounterVV(cpEndVV);

    for(endIter = endVV.getIterator(); endIter.hasMoreElements();){
      token = endIter.getNext();
      nodeId = endVV.getServerByIteratorToken(token);
      endStamp = endVV.getStampByIteratorToken(token);

      try{
        startStamp = startVV.getStampByServer(nodeId);
      }catch(NoSuchEntryException e){
        startStamp = -1;
      }

      if(startStamp < endStamp){//ignore all those component with start >= end
        newStartVV.setStamp(new AcceptStamp(startStamp+1, nodeId));
        newEndVV.setStamp(new AcceptStamp(endStamp, nodeId));
      }
    }
    if(newStartVV.getSize()==0){
      return null;
    }
    return new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(":/*"),
        newStartVV.cloneAcceptVV(),
        newEndVV.cloneAcceptVV());
  }

  //-------------------------------------------------------------------------
  // apply CatchupStreamStartMsg --
  //       put the pending set into pendingSet if startVV match current lpvv
  // startVV is the exclusion startVV
  //-------------------------------------------------------------------------
  public void
  applyCatchupStreamStartMsg(CatchupStreamStartMsg psm){

    if(dbgProgress){
      System.out.println("IncommingConnection.applyCatchupStreamStartMsg:( " + psm
          + ") is called"); 
    }

    SubscriptionSet newSS = psm.getSubscriptionSet();
    VV newStart = psm.getStartVV();

    assert !newSS.isEmpty();
    SubscriptionSet intersection = newSS.getIntersection(this.ss);

    if (intersection.equals(newSS)){//already in the connection
      controller.informSubscribeInvalSucceeded(senderId,
          core.getMyNodeId(),
          newSS,
          newStart);
      return;
    }


    if(core.getLpVV(newSS).includes(this.prevVV)){//already caught up
      this.addSubscriptionSet(newSS);
      controller.informSubscribeInvalSucceeded(senderId,
          core.getMyNodeId(),
          newSS,
          newStart);
    }else{
      //make sure that lpvv has no gap with the newStart
      if(core.getLpVV(newSS).includes(newStart)){//make sure that lpvv catch up startvv 
        //anytime there should be only one pendingSet.
        assert ((pendingSet == null) && (pendingStartVV == null));
        pendingSet = newSS;
        //System.out.println("pendingSet is "+newSS);
        pendingStartVV = new CounterVV(core.getLpVV(newSS));

      }else{//there's a gap between startvv & lpvv --> can't accept it
        controller.informSubscribeInvalFailed(senderId, core.getMyNodeId(), 
            newSS, newStart);
        Env.warn("Subscription of " + newSS.toString() 
            + "failed due to gap between startVV=" + newStart 
            + " and receiver's lpvv.");
      }
    }
  }




  //-------------------------------------------------------------------------
  // apply CatchupStreamEndMsg
  //     -- put pendingSet into subscriptionset if lpvv match prevVV
  //-------------------------------------------------------------------------
  public void
  applyCatchupStreamEndMsg(CatchupStreamStartMsg start){
    if(dbgProgress){
      System.out.println("IncommingConnection.applyCatchupStreamEndMsg:(start=" + start
          + ") is called with pendingSet " + pendingSet); 
    }

    SubscriptionSet newSS = start.getSubscriptionSet();
    VV newStart = start.getStartVV();
    if(pendingSet!=null){

      // anyone survived during the catch up stream 
      // should be able to add
      if(!pendingSet.isEmpty()){

        this.addSubscriptionSet(pendingSet);
        if(dbgProgress){
          Env.dprint(dbgProgress, "IncomingConnection::applyCatchupStreamEndMsg succeeded in adding"
              + newSS + " with newStart " + newStart);
        }
        controller.informSubscribeInvalSucceeded(senderId,
            core.getMyNodeId(),
            newSS,
            newStart);
      }
      pendingSet = null;
      pendingStartVV = null;
    }
    assert pendingSet == null;
    assert pendingStartVV == null;
  }

  //-------------------------------------------------------------------------
  // clean reference putted in ISStatus
  //  -- called when the connection died
  //-------------------------------------------------------------------------
  public void cleanReference(){

    if(dbg){
      Env.inform("DBG IncommingConnection: closeStreamAndSocket " +
          " sender: " + senderId.toString() +
          " ss: " + ss.toString() +
          " vv: " + prevVV.toString());
    }
    if(senderId != null){
      //
      //gc the reference in ISStatus
      //
      SubscriptionSet removedSS = this.ss;
      this.removeSubscriptionSet(ss, prevVV);
      controller.informInvalStreamTerminated(senderId, 
          removedSS, 
          prevVV,
          true,
          streamId);
    }
  }



  //-------------------------------------------------------------------------
  // return streamId
  //-------------------------------------------------------------------------
  public StreamId 
  getStreamId(){
    return this.streamId;
  }

  //-------------------------------------------------------------------------
  // Compare whether "o" is equal to this object used for PTreeNode.liveCons
  //-------------------------------------------------------------------------
  public boolean
  equals(Object o){
    if(o instanceof IncommingConnection){
      if (this.streamId.equals(((IncommingConnection)o).getStreamId())){
        return true;
      }
    }
    return false;
  }


  //----------------------------------------------------------------------
  // main() -- self test code
  //----------------------------------------------------------------------
  public static void 
  main(String[] argv) {
    System.out.println("IncommingConnection unit tests now in junit framework "
        + "-- run IncommingConnectionUnit");
  }

  //----------------------------------------------------------------------
  // unit test helper function
  //----------------------------------------------------------------------
  public void resetSubscriptionSet(SubscriptionSet newss){
    this.ss = newss;
  }

  //----------------------------------------------------------------------
  // unit test helper function
  //----------------------------------------------------------------------
  public void resetPrevVV(VV newVV){
    this.prevVV = new CounterVV(newVV);
  }

  //----------------------------------------------------------------------
  // unit test helper function
  //----------------------------------------------------------------------
  public void resetPendingSet(){
    this.pendingSet = null;
    this.pendingStartVV = null;
  }


}




//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//Class IncommingConnectionWorker
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
class IncommingConnectionWorker extends Thread implements SocketMagicConstant{

  private Socket underlyingSocket = null;
  private TaggedInputStream s = null;
  private IncommingConnection ic = null;

  private boolean measureTime = IncommingConnection.measureTime; // measure time it takes to apply invals
  private String outFilePrefix = "IncommingConnection_time_";
  private PrintStream out;

  //--------------------------------------------------------------------------
  // Constructor for worker thread
  //--------------------------------------------------------------------------
  IncommingConnectionWorker(TaggedInputStream tis,
      Socket underlyingSocket,
      IncommingConnection ic){
    this.s = tis;
    this.underlyingSocket = underlyingSocket;
    this.ic = ic;

    openFile();
  }


  private static boolean tbdPrinted = false;

  //--------------------------------------------------------------------------
  // handshake for the connection
  // 
  // return false if fail
  //        true if get all information expected
  //--------------------------------------------------------------------------
  private boolean handshake(){
    try{
      Long magic = (Long)s.readTaggedObject();
      if(IncommingConnection.dbgVerbose){
        Env.inform("DBG IncommingConnection: got magic " + magic);
      }
      if((magic.longValue() != INVAL_SOCKET_MAGIC)&&
          (magic.longValue() != INVAL_SOCKET_RECONNECT_MAGIC)){
        Env.remoteAssert(false, "descriptive error msg");
        throw new Exception("Bad magic number to IncommingConnection");
      }
      if(IncommingConnection.dbgVerbose){
        Env.inform("DBG IncommingConnection: get sender ");
      }
      this.ic.senderId = (NodeId)(s.readTaggedObject());
      if(IncommingConnection.dbgVerbose){
        Env.inform("DBG IncommingConnection: got sender " +
            this.ic.senderId);
      }
    }catch(Exception e){
      // 
      // Attempt to connect failed and we don't even know
      // who it was that was trying. Nothing to do but
      // have this thread die. If this was important,
      // then a controller should timeout and retry. 
      //

      e.printStackTrace();

      return false; // Thread exit.
    }

    AcceptVV streamStartVV = null;
    try{
      streamStartVV = (AcceptVV)(s.readTaggedObject());

    }catch(OptionalDataException o){
      o.printStackTrace();
      return false;//Thread exit.
    }catch(IOException i){
      i.printStackTrace();
      return false;//Thread exit.
    }catch(ClassCastException z){
      z.printStackTrace();
      return false; // Thread exit.
    }catch(Exception zz){
      zz.printStackTrace();     
      return false; // Thread exit.
    }

    ic.initPrevVV(streamStartVV);
    return true;
  }

  public void applySecureCheckpoint(SecureCheckpoint sc){
    assert(SangminConfig.securityLevel > SangminConfig.SIGNATURE);
  }

  //-------------------------------------------------------------------------
  // Process an incoming stream consisting of
  // 
  // [INVAL_SOCKET_MAGIC, senderId, intial prevVV, 
  //  {(CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)|
  //  (GeneralInv|UnbindMsg|DebargoMsg)*|
  //  (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)}*
  //  TerminateMsg]
  //-------------------------------------------------------------------------
  public void
  run(){

    if(IncommingConnection.dbgProgress){
      System.out.println("IncommingConnectionWorker starts");
    }

    //
    // handshake
    //
    if(!handshake()){
      Env.remoteAssert(false);
      this.closeStreamAndSocket("IncommingConnection cannot handshake with sender ");
      closeFile();
      return;
    }

    //tbd from InvalRecvWorker.java
    if(!tbdPrinted){
      Env.tbd("TBD: subclass Socket to be a heartbeat Socket \n"
          + "sender sends a msg every 5 seconds \n "
          + "and receiver expects to recv message every 20 seconds or\n"
          + "throws IO Exception OR USE KEEPALIVE OPTION?");
      tbdPrinted = true;
    }





    long handleInvalStart=0;
    long handleInvalEnd=0;

    //
    // Keep catchupstreamstart so that when a catchupstream
    // finishes, we know what we were working on...
    //
    CatchupStreamStartMsg lastCatchupStreamStart = null;

    //
    // Read the sequence of invalidations/CP from stream
    //
    try{

      Object next = s.readTaggedObject();
      while(! (next instanceof TerminateMsg)){

        if(measureTime){
          handleInvalStart = System.currentTimeMillis();
        }
        if(IncommingConnection.dbg){
          Env.dprintln(IncommingConnection.dbg, "Read next : " + next);
        }


        if(next instanceof UnbindMsg){
          ic.applyUnbind((UnbindMsg)next);
        }else if(next instanceof DebargoMsg){
          ic.applyDebargo((DebargoMsg)next);



        }else if(next instanceof CatchupStreamStartMsg){
          // 
          // Right now we do not allow nested catchupstreams.
          // If we want to do that. Receiver needs to keep
          // a stack of start messages...
          //
          Env.remoteAssert(lastCatchupStreamStart == null);
          if(lastCatchupStreamStart != null){
            throw new IOException("Nested catchup stream");
          }
          if(IncommingConnection.dbgPerformance){
            long start = System.currentTimeMillis();
            Env.dprintln(IncommingConnection.dbgPerformance, "@ "+ start 
                + " IncommingConnection -- catchupStreamStartMsg arrives");
          }
          lastCatchupStreamStart = (CatchupStreamStartMsg)next;
          ic.applyCatchupStreamStartMsg(lastCatchupStreamStart);

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();

            LatencyWatcher.put("IncommingConnection.ApplyCatchupStreamStartMsg", 
                (handleInvalEnd - handleInvalStart));

          }
        }else if(next instanceof CatchupStreamEndMsg){

          Env.remoteAssert(lastCatchupStreamStart != null);
          if(lastCatchupStreamStart == null){
            throw new IOException("Mismatched catchup stream");
          }
          if(IncommingConnection.dbgPerformance){
            long start = System.currentTimeMillis();
            Env.dprintln(IncommingConnection.dbgPerformance, "@ "+ start 
                + " IncommingConnection -- catchupStreamEndMsg arrives");
          }
          ic.applyCatchupStreamEndMsg( 
              lastCatchupStreamStart);
          lastCatchupStreamStart = null;

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();

            LatencyWatcher.put("IncommingConnection.ApplyCatchupStreamEndMsg", 
                (handleInvalEnd - handleInvalStart));
          }


        }else if(next instanceof CPStartMsg){
          assert(SangminConfig.securityLevel <= SangminConfig.SIGNATURE);
          
          long start = System.currentTimeMillis();

          ic.applyCP(this.s, (CPStartMsg)next);
          long end, totalTime;

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("IncommingConnection.ApplyCP", (handleInvalEnd - handleInvalStart));
          }

        }else if(next instanceof SecureCheckpoint){
          applySecureCheckpoint((SecureCheckpoint)next);

        }else {
          if(!(next instanceof GeneralInv)){
            throw(new InvalidClassException("" + next.getClass()));
          }

          if(IncommingConnection.dbgPerformance && (next instanceof PreciseInv)){
            long start = System.currentTimeMillis();
            Env.dprintln(IncommingConnection.dbgPerformance, "IncommingReceive @ "+ start 
                + " " + ((PreciseInv)next).getAcceptStamp().toString());
          }
          ic.applyGI((GeneralInv)next);
          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("IncommingConnection.ApplyInval", (handleInvalEnd - handleInvalStart));
          }
        }



        if(IncommingConnection.dbgPerformance){
          long start = System.currentTimeMillis();
          Env.dprintln(IncommingConnection.dbgPerformance, "@ "+ start 
              + " IncommingConnection -- about to read next message");
        }


        next = s.readTaggedObject();


        if(IncommingConnection.dbgPerformance){
          long start = System.currentTimeMillis();
          Env.dprintln(IncommingConnection.dbgPerformance, "@ "+ start 
              + " IncommingConnection -- done reading next message");
        }
      }// while loop for next object

    }catch(EOFException i){

      this.closeStreamAndSocket("terminated because of EOFException " + i.toString());
      closeFile();

      return; // expected case thread exit
    }catch(IOException io){
      if(IncommingConnection.dbg){
        io.printStackTrace();
      }
      this.closeStreamAndSocket("terminated because of IOException " + io.toString());
      closeFile();
      return; // expected case thread exit
    }catch(Exception i3){
      i3.printStackTrace();
      assert false; // Unexpected cases
      this.closeStreamAndSocket("terminated unexcpectedly because of " + i3.toString());
      closeFile();
      return; // thread exit
    }

  }

  //-------------------------------------------------------------------------
  // Close the stream and socket
  //-------------------------------------------------------------------------
  private void
  closeStreamAndSocket(String msg){

    try{
      s.close();
      if(this.underlyingSocket != null){
        this.underlyingSocket.close();
      }
    }catch(Exception e){
      // Ignore errors when trying to close
    }


    ic.cleanReference();
    System.out.println("IncommingConnectionWorker closing stream: " + msg);

  }

  //-------------------------------------------------------------------------
  // openFile
  //-------------------------------------------------------------------------
  public void openFile(){
    try{
      if(measureTime) {
        out = new PrintStream(new FileOutputStream(outFilePrefix + ic.getStreamId() 
            + ".out")); 
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

  //-------------------------------------------------------------------------
  // closeFile
  //-------------------------------------------------------------------------
  public void closeFile(){
    if(measureTime) {
      out.close();
    }
  }

  //-------------------------------------------------------------------------
  // writeToFile
  //-------------------------------------------------------------------------
  public void writeToFile(long duration){
    if(measureTime) {
      out.println(duration);
    }
  }

}







//---------------------------------------------------------------------------
/* $Log: IncommingConnection.java,v $
/* Revision 1.41  2007/11/05 23:32:42  zjiandan
/* fix SubscribeBWunit.
/*
/* Revision 1.40  2007/10/06 18:09:19  zjiandan
/* java version expt1 done.
/*
/* Revision 1.39  2007/10/06 05:29:53  zjiandan
/* Add Coda java hack version.
/*
/* Revision 1.38  2007/09/18 02:41:40  zjiandan
/* *** empty log message ***
/*
/* Revision 1.37  2007/09/10 23:52:22  zjiandan
/* upgrade to newest BerkeleyDB je version.
/*
/* Revision 1.36  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.35  2007/07/12 17:02:32  zjiandan
/* *** empty log message ***
/*
/* Revision 1.34  2007/07/11 22:06:45  zjiandan
/* *** empty log message ***
/*
/* Revision 1.33  2007/07/11 19:08:07  zjiandan
/* clean IncommingConnection
/*
/* Revision 1.32  2007/07/03 07:57:54  zjiandan
/*  more unit tests
/*
/* Revision 1.31  2007/06/29 05:59:59  zjiandan
/* *** empty log message ***
/*
/* Revision 1.30  2007/06/25 05:21:28  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.29  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.28  2007/05/30 20:30:19  dahlin
/* Added checkpoint to SubscribeBWUnit. Changed outgoingconnection checkpoint 
/* send to not send body by default. Told barrier to close sockets when done with them.
/*
/* Revision 1.27  2007/05/25 21:06:01  dahlin
/* fixed 1 second delay on serializing start catchup stream; 
/* SubscribeBWUnit runs much faster -- though still nto as fast as I would like. 
/* Probably also need to convert catchup stream to catchup checkpoint
/*
/* */
//---------------------------------------------------------------------------
