package code;
/* IncommingConnection --
 *
 *   A data structure for multiplexing inval stream subscriptions at receiver
 *   side. Created when a new connection is created by OutgoingConnectionThread.
 *   
 *   Note, it's possible that One OutgoingConnection might have multiple
 *   IncommingConnections during its lifetime because its worker might 
 *   die and create multiple times. But at most one incommingConnection exists
 *   given any moment as there's only one worker.
 *
 *
 *   A stream pattern:
 *
 *      [
 *       // handshack
 *       INVAL_SOCKET_MAGIC, senderId, intial prevVV, initial streamSS
 *
 *       
 *       {
 *         //catchup log  
 *         (CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)  |
 *        
 *         //normal traffic
 *         (GeneralInv|UnbindMsg|DebargoMsg)*  |
 *       
 *         //catchup checkpoint
 *         (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)
 *       
 *       }*
 *       
 *       //goodbye
 *       TerminateMsg
 *      ]
 *
 *  Note: it's a single thread worker
 *      every method is not synchronized
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import java.net.Socket;
import java.io.IOException;



public class IncommingInvalConnection implements ConnectionState{

  public final static boolean dbgProgress = false;
  public final static boolean dbgPerformance = false;
  public final static boolean dbg = false; // Print connection events
  public final static boolean dbgVerbose = false; // Print recv events

  public final static boolean measureTime = true;
  public final static boolean USE_II_FILTER = true;

  // See inner class IncommingInvalConnectionWorker for more debug params
  protected Thread worker;


  protected Core core = null;
  protected Controller controller = null;
  protected Socket underlyingSocket = null;
  protected TaggedInputStream s = null;
  protected StreamId streamId = null;



  protected NodeId senderId = null;
  protected CounterVV prevVV = null;
  protected SubscriptionSet ss = null;
  protected SubscriptionSet pendingSet = null;
  protected CounterVV pendingStartVV = null;



 /** 
 *  Constructor -- for test 
 **/ 
  public
  IncommingInvalConnection(StreamId sid, SubscriptionSet ss, AcceptVV prevVV){
    if(dbgProgress){
      System.out.println("IncommingInvalConnection(sid, ss, prevVV):( " + sid 
          + ", " + ss + ", " + prevVV + ") is created"); 
    }
    this.streamId = sid;
    this.ss = ss;
    this.prevVV = new CounterVV(prevVV);

    this.worker = null;


  }

 /** 
 *  Constructor 
 **/ 
  public
  IncommingInvalConnection(TaggedInputStream s,
      Core core,
      Controller controller,
      Socket underlyingSocket,
      StreamId newStreamId){
    if(dbgProgress){
      System.out.println("IncommingInvalConnection(sid):( " + newStreamId + ") is created"); 
    }
    this.s = s;
    this.core = core;
    this.controller = controller;
    this.underlyingSocket = underlyingSocket;
    //this.countEndVVSearchSlow = 0;
    this.streamId = newStreamId;

    this.senderId = null;
    this.prevVV = (CounterVV)(CounterVV.makeVVAllNegatives());
    this.ss = SubscriptionSet.makeEmptySet();
    this.pendingSet = null;
    this.pendingStartVV = null;

    this.worker = null;


  }



 /** 
 *  startWorker() -- kick off the worker thread. Must be called right 
 *     after initialization. 
 **/ 
  public void startWorker(){    
    assert worker==null;
    worker = new IncommingInvalConnectionWorker(this.s, this.underlyingSocket, this);
    worker.start();
  }


 /** 
 *  interruptWorker() 
 **/ 
  public void interruptWorker(){
    try{
      if(this.worker!=null){
        this.worker.interrupt();
        if(dbgProgress){
          System.out.println("IncommingInvalConnectionWorker interrupt() called");
        }
        this.worker = null;
      }
    }catch(SecurityException e){
      //ignore.
    }
  }
 /** 
 *  handle initial prevVV 
 *  
 *  only called by worker 
 **/ 
  protected boolean initConnection(TaggedInputStream tis, AcceptVV senderVV, SubscriptionSet senderSS){
	// replace by checkpoint
	    // for security, can't simply initiate with any vv
	    // need to get the Checkpoint/security information as well.
	    //
	    /*
	    AcceptVV initPrevVVFromSender = null;
	    SubscriptionSet streamStartSS = null;
	    try{
	      initPrevVVFromSender = (AcceptVV)(s.readTaggedObject());
	      //streamStartSS = (SubscriptionSet)(s.readTaggedObject());
	    }catch(OptionalDataException o){
	      o.printStackTrace();
	      return false;//Thread exit.
	    }catch(IOException i){
	      i.printStackTrace();
	      return false;//Thread exit.
	    }catch(ClassCastException z){
	      z.printStackTrace();
	      return false; // Thread exit.
	    }catch(Exception zz){
	      zz.printStackTrace();     
	      return false; // Thread exit.
	    }
	    
    //
    // make sure that it is safe to accept this stream
    //
    // Handle stream start VV issue:
    // 
    //   Previously in order to be safe, the receiver will reject the stream 
    //   if startvv > receiver's cvv.
    //   The problem is: if sender always sends its cvv in handshake as stream's start vv,
    //   then the connection between an advanced sender and a receiver will always be rejected.
    //
    // two options: 
    //          
    //      (1) always accept the stream by applying an imprecise invalidate 
    //          </*, receiver'scvv, sender's cvv(i.e. stream startVV) > in handshake.
    //	        simple. 
    //   	But the big imprecise might make everything imprecise for a while.
    //          That's ok since the receiver will subscribe all the interested set soon anyways.
    //
    //      (2) Sender uses the first addSubscribeInval's startVV as the initial prevVV instead of 
    //          sender's cvv (most likely <= receiver's cvv)
    //
    //          Receiver always rejects a stream which has initial startvv >= receiver's cvv.
    //     
    //          treat the first SubscribeInval request different from the other request.
    //       
    //          variation: the handshake of a connection between two nodes should be that the sender sends 
    //          the receiver a checkpoint of <startVV= receiver's cvv, endVV=sender's cvv, 
    //          subscribeset=receiver's interested sets>.
    //  
    // IncommingConnection.java For simplicity, pick (1)
    // This class pick (2). Use the first addSubscribeInval startVV as the initial prevVV
    // and the ss as the initial stream SS.
    // don't reject the stream, but create a gap imprecise invalidation
    //
    if(!core.getCurrentVV().includes(initPrevVVFromSender)){//the receiver's cvv < startVV
      ImpreciseInv gapInv = this.prepareGapInv(initPrevVVFromSender);
      if(gapInv != null){
        applyGI(gapInv);
      }
    }   
    assert core.getCurrentVV().includes(initPrevVVFromSender);
    assert this.ss.isEmpty();
    this.prevVV.advanceTimestamps(initPrevVVFromSender);

    controller.informInvalStreamInitiated(senderId,
        ss,
        prevVV,
        true);

    //
    // fall back to the original protocol
    // initialize with sender.cvv and streamSS = empty
    //
    //if(core.getLpVV(initSS).includes(this.prevVV)){
    //  this.addSubscriptionSet(initSS);
    //  controller.informSubscribeInvalSucceeded(senderId,
    //                                           core.getMyNodeId(),
    //                                           initSS,
    //                                           prevVV);
    //  if(dbgProgress){
    //    System.out.println("IncommingInvalConnection::applyCP caught up" + initSS);
    // }
    //
    // }else{
    //
    //  controller.informSubscribeInvalFailed(senderId,
    //                                        core.getMyNodeId(),
    //                                        initSS,
    //                                       prevVV);
    //
    //}
     
     */
	  
    //princem: commented out the following lines of code because the previous code that sangmin used was essntially translating to 
    // returning true
//	  if(!core.getCurrentVV().includes(senderVV)){//the receiver's cvv < startVV
//	      ImpreciseInv gapInv = this.prepareGapInv(senderVV);
//	      if(gapInv != null){
//	        this.applyGI(gapInv);
//	      }
//	    }   
//	    assert core.getCurrentVV().includes(senderVV);
//	    assert this.ss.isEmpty();
//	    this.prevVV.advanceTimestamps(senderVV);
//	      
//	    controller.informInvalStreamInitiated(senderId,
//	                                          ss,
//	                                          prevVV,
//	                                          true);
//	  return this.checkAndApplyInitialCP(tis); //tbd
    return true;
  }

  protected boolean checkAndApplyInitialCP(TaggedInputStream tis){
    try{
      Object next = tis.readTaggedObject();
      if(!(next instanceof CPStartMsg)){
        assert false:""+next;
        return false;
      }else{

        this.applyCP(tis, (CPStartMsg) next);

        return true;
      }
    }catch(Exception e){
      e.printStackTrace();
      return false;
    }
	  
  }
  
  public void applyGI(GeneralInv gi){
    applyInsecureGI(gi);
  }
 /** 
 *  get senderId 
 **/ 
  public NodeId getSenderId(){
    return this.senderId;
  }

 /** 
 *  get SubscriptionSet 
 **/ 
  public SubscriptionSet getSubscriptionSet(){
    return this.ss;
  }

 /** 
 *  get PendingSet 
 **/ 
  public SubscriptionSet getPendingSet(){
    return this.pendingSet;
  }

 /** 
 *  get pendingStartVV 
 **/ 
  public AcceptVV getPendingStartVV(){
    if (this.pendingStartVV == null){
      return null;
    }
    return this.pendingStartVV.cloneAcceptVV();
  }

 /** 
 *  return con.prevVV -- not synchronized because of single threading 
 **/ 
  public AcceptVV
  getPrevVV(){
    return this.prevVV.cloneAcceptVV();
  }

 /** 
 *  advance PrevVV with newVV 
 **/ 
  public void
  advancePrevVV(VV newVV){
    this.prevVV.advanceTimestamps(newVV);
  }

 /** 
 *  add subscriptionset to the incomming connection: 
 *      1.add connection pointer to the corresponding place in ISStatus tree 
 *        and notifyAll if prevVV == cvv 
 *      2.merge the subscriptionset to the connection.ss 
 **/ 
  public void
  addSubscriptionSet(SubscriptionSet newss){
    if(dbgProgress){
      System.out.println("IncommingInvalConnection.addSubscriptionSet:( " + newss+ ") is called"); 
    }
    core.addConnectionToISStatus(newss, this);
    this.ss = ss.getCompleteUnion(newss);
  }

 /** 
 *  replace subscriptionset to the incomming connection: 
 *      1.add connection pointer to the corresponding place in ISStatus tree 
 *        and notifyAll if prevVV == cvv 
 *      2.merge the subscriptionset to the connection.ss 
 **/ 
  public void
  replaceSubscriptionSet(SubscriptionSet newss){
    if(dbgProgress){
      System.out.println("IncommingConnection.replaceSubscriptionSet:( " + newss+ ") is called"); 
    }
    core.removeConnectionFromISStatus(ss, this, prevVV);
    core.addConnectionToISStatus(newss, this);
    this.ss = newss;
  }

 /** 
 *   remove subscriptionset from the incomming connection 
 *     1. update ISStatus: remove this pointer, update lpvv 
 *     2. remove the ss from the subscriptionset 
 **/ 
  public void
  removeSubscriptionSet(SubscriptionSet rss, VV newLpvv){
    if(dbgProgress){
      System.out.println("IncommingInvalConnection.removeSubscriptionSet:( " + rss
          + ", " + newLpvv + ") is called"); 
    }
    if (!ss.isEmpty()){
      core.removeConnectionFromISStatus(rss, this, newLpvv);
      try{
        this.ss = this.ss.remove(rss, true);
      }catch(IllegalRemoveSubscriptionSetException e){
        assert false : ("" + e);
      }
    }
  }

 /** 
 *  update ISStatus lpvv for PTree node overlapping by ss 
 **/ 
  public void
  updateISStatusLpVV(SubscriptionSet uss, VV newLpvv){
    if(dbgProgress){
      System.out.println("IncommingInvalConnection.updateISStatusLpVV:( " + uss
          + ", " + newLpvv + ") is called"); 
    }
    core.removeConnectionFromISStatus(uss, null, newLpvv);
  }

 /** 
 *  Prepare new lpvv, return max of (vv1, vv2-1) 
 **/ 
  public AcceptVV
  prepareNewLpvv(AcceptVV vv1, AcceptVV vv2){
    CounterVV newLpVV = new CounterVV(vv2);
    newLpVV.decrementAllByOne();//(gi.startVV -1)
    newLpVV.advanceTimestamps(vv1);//get component-wise max of prevVV and (gi.startVV-1)
    return newLpVV.cloneAcceptVV();
  }


 /** 
 *  If receive a precise inval from node A, 
 *  advance my outgoing connection to A's cvv 
 *  so that the sender won't receive this invalidate from me again. 
 *  
 *  for more details see designDocs/tbd_after_sosp_2007.txt 
 **/ 
  protected void 
  filterRedundantInvForSender(GeneralInv gi){

    if(USE_II_FILTER){
      if ((gi instanceof PreciseInv)
          ||(gi instanceof MultiObjPreciseInv)
          ||(gi instanceof DeleteInv)){
        core.notifyRecvInval(senderId, gi.getEndVV());
      }
    }
  }


 /** 
 *  return (1) the intersection of gi.invalTarget and targetSS,  
 *             if gi is imprecise  
 *               && the lpvv of the intersection is less than gi.endVV 
 *         
 *         (2) null, otherwise 
 **/ 
  protected SubscriptionSet getKickedSet(GeneralInv gi, SubscriptionSet targetSS){

    if (!gi.isPrecise() && gi.getInvalTarget().intersects(targetSS)){

      assert gi.getInvalTarget() instanceof HierInvalTarget;

      SubscriptionSet kickedSet = targetSS.getIntersection((HierInvalTarget)(gi.getInvalTarget()));
      assert !kickedSet.isEmpty();
      if(!core.getLpVV(kickedSet).includes(gi.getEndVV())){//only kick off if my lpvv is not catchup
        //System.out.println("lpvv: " + core.getLpVV(kickedSet) + " gi.endVV: " + gi.getEndVV());
        return kickedSet;
      }
    }
    return null;
  }


 /** 
 *  Apply an unbindmsg 
 **/ 
  public void
  applyUnbind(UnbindMsg um)
  throws IOException{
    this.core.applyUnbind(um);
  }

 /** 
 *  Apply debargo msg 
 **/ 
  public void
  applyDebargo(DebargoMsg dm) 
  throws IOException{
    this.core.applyDebargo(dm);
  }


 /** 
 *  Apply a general incomming invalidation 
 **/ 
  public void
  applyInsecureGI(GeneralInv gi){
    long start = System.currentTimeMillis();
    long end;
    if(dbgProgress){
      System.out.println(core.getMyNodeId() + 
          " IncommingInvalConnection.applyGI:( " + gi + ") is called"); 
    }

    //
    // advance outgoingConnection cvv to prevent redundant invalidates
    // from sending back to sender again.
    //
    this.filterRedundantInvForSender(gi);
    end = System.currentTimeMillis();
    if(dbgPerformance){
      System.out.println("11111 " + (end-start));
    }
    //
    //note: we must first update the cvv
    //      then update lpvv/prevVV for causal view across objects
    //
    core.applyInval(gi, streamId); 
    start = end;
    end = System.currentTimeMillis();
    if(dbgPerformance){
      System.out.println("22222 " + (end-start));
    }
    if (gi.getEndVV().isAnyPartGreaterThan(this.prevVV)){
      start = end;
      end = System.currentTimeMillis();
      if(dbgPerformance){
        System.out.println("33333 " + (end-start));
      }
      // 
      // from the main stream ==> will advance prevVV
      // 

      assert pendingSet == null: pendingSet;
      assert pendingStartVV == null: pendingStartVV;

      //kick off imprecise subset
      SubscriptionSet kickedSet = this.getKickedSet(gi, this.ss);
      start = end;
      end = System.currentTimeMillis();
      if(dbgPerformance){
        System.out.println("44444 " + (end-start));
      }
      if(kickedSet != null){
        AcceptVV newLpvv = prepareNewLpvv(this.prevVV.cloneAcceptVV(), gi.getStartVV());
        removeSubscriptionSet(kickedSet, newLpvv);
        controller.informSubscribeInvalFailed(senderId, core.getMyNodeId(), 
            kickedSet, newLpvv);
      }
      start = end;
      end = System.currentTimeMillis();
      if(dbgPerformance){
        System.out.println("55555 " + (end-start));
      }
      //advance prevVV
      this.prevVV.advanceTimestamps(gi.getEndVV());
      core.notifyDatastore(this.ss, this.prevVV.cloneAcceptVV());//notify imprecise read thread
      start = end;
      end = System.currentTimeMillis();
      if(dbgPerformance){
        System.out.println("66666 " + (end-start));
      }

    }else if (pendingSet != null){
      //
      // from catch up streams
      //
      applyCatchupGI(gi);

    }
    start = end;
    end = System.currentTimeMillis();
    if(dbgPerformance){
      System.out.println("88888 " + (end-start));
    }
    controller.informReceiveInval(gi, senderId);
  }







 /** 
 *  apply Checkpoint 
 **/ 
  public void
  applyCP(TaggedInputStream s, CPStartMsg cpsm)
  throws IOException, ClassNotFoundException, CausalOrderException{
    if(dbgProgress){
      System.out.println("IncommingInvalConnection.applyCP(" + cpsm + "): is called"); 
    }

    long startTime, end, totalTime;
    if(measureTime){
      startTime = System.currentTimeMillis();
    }
    SubscriptionSet cpSS = cpsm.getSubscriptionSet();
    AcceptVV startVV = cpsm.getStartVV();
    try{
      if(startVV.getStampByServer(core.getMyNodeId()) == RMIClient.ACCEPT_STAMP_TO_FILTER_MY_OWN_UPDATES){
        startVV = startVV.makeVVexcludeNode(core.getMyNodeId());
      }
    }
    catch(NoSuchEntryException nsee){
      ; // OK
    }    



    if(this.checkAndIgnoreInvalidCP(s, cpSS, startVV)){
      controller.informSubscribeInvalFailed(senderId,
          core.getMyNodeId(),
          cpSS,
          startVV);//because of the gap between startVV and minlpvv
          if(measureTime){
            end = System.currentTimeMillis();
            totalTime = (end-startTime);
            LatencyWatcher.put("IncommingInvalConnection.ApplyCatchupStreamStartMsg", 
                totalTime);
            startTime = System.currentTimeMillis();

          }
          return;
    }


    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IncommingInvalConnection.ApplyCatchupStreamStartMsg", 
          totalTime);
      startTime = System.currentTimeMillis();
    }

    //
    // start applying the checkpoint stream
    //
    VV cpEndVV = cpsm.getSenderCVV();
    ImpreciseInv gapInv = null;

    // 1. prepare the big imprecise invalidate for Checkpoint
    if(!core.getCurrentVV().includes(cpEndVV)){
      //the receiver is not safe to apply cp
      gapInv = prepareGapInv(cpEndVV);
      if(gapInv != null){
        this.applyGI(gapInv);
      }else{
        assert core.getCurrentVV().includes(cpEndVV);
      }
    }

    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IncommingInvalConnection.preApplyCP", 
          totalTime);
      startTime = System.currentTimeMillis();
    }

    //2. apply checkpoint to datastore
    core.applyCheckpoint(s, cpsm);

    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IC.core.applyCP", 
          totalTime);
      startTime = System.currentTimeMillis();
    }

    //3. attach if possible
    if(core.getLpVV(cpSS).includes(this.prevVV)){
      this.addSubscriptionSet(cpSS);
      controller.informSubscribeInvalSucceeded(senderId,
          core.getMyNodeId(),
          cpSS,
          startVV);
      if(dbgProgress){
        System.out.println("IncommingInvalConnection::applyCP caught up" + cpSS);
      }

    }else{

      controller.informSubscribeInvalFailed(senderId,
          core.getMyNodeId(),
          cpSS,
          startVV);
      if(dbgProgress){
        System.out.println("IncommingInvalConnection::applyCP NOT caught up");
      }
    }

    if(measureTime){
      end = System.currentTimeMillis();
      totalTime = (end-startTime);
      LatencyWatcher.put("IC.afterApplyCP", 
          totalTime);
      startTime = System.currentTimeMillis();
    }


  }


 /** 
 *  called before applying any incoming checkpoint stream 
 *  
 *  check if my lpVV for all interest sets at or below cpPath are  
 *  at least minLpVV 
 *    -- if this is not satisfied, receive all cp until RAS_DONE 
 *  
 *    -- we need to check this because the RAS will filter out those 
 *       PerObjectStates whose timestamps is between this.minlpvv and startVV 
 *       if the ISStatus for this PerObjectStates is "precise" after applying the cp 
 *       the read will get old value instead of the value between this.minlpvv and startVV 
 **/ 
  protected boolean checkAndIgnoreInvalidCP(TaggedInputStream s, 
      SubscriptionSet cpSS, 
      AcceptVV startVV)
  throws IOException, ClassNotFoundException, CausalOrderException{
    if(!core.checkMinLPVV(cpSS, startVV)){
      if(dbgProgress){
        System.out.println("IncommingInvalConnection.applyCP: aborting b/c minLPVV. cpSS = " 
            + cpSS.toString() + " startVV = " + startVV 
            + " core minLpVV for IS = " + core.getISStatusMinLpVV(cpSS));
      }

      //skip checkpoint data

      Object next = s.readTaggedObject();


      while (!((next instanceof Long)&&(((Long)next).longValue()==RandomAccessState.RAS_SHIP_DONE))){
        next = s.readTaggedObject();
      }


      return true;
    }
    return false;
  }


 /** 
 *  Called by applyCP -- 
 *  
 *    Prepare a legal imprecise invalidate with endVV.includes(startVV) 
 *    that covering the gap between currentVV and cpEndVV 
 *    i.e. only the component where cpEndVV has larger value than  
 *    core.cvv will be kept in the resulted gapInv 
 *     
 *    if no component is larger, then return null; 
 **/ 
  public ImpreciseInv prepareGapInv(VV cpEndVV){
    CounterVV startVV;
    CounterVV newStartVV= new CounterVV();;
    CounterVV endVV;
    CounterVV newEndVV = new CounterVV();

    VVIterator endIter = null;
    NodeId nodeId = null;
    long startStamp = 0;
    long endStamp = 0;
    Object token = null;

    startVV = new CounterVV(core.getCurrentVV());
    endVV = new CounterVV(cpEndVV);

    for(endIter = endVV.getIterator(); endIter.hasMoreElements();){
      token = endIter.getNext();
      nodeId = endVV.getServerByIteratorToken(token);
      endStamp = endVV.getStampByIteratorToken(token);

      try{
        startStamp = startVV.getStampByServer(nodeId);
      }catch(NoSuchEntryException e){
        startStamp = -1;
      }

      if(startStamp < endStamp){//ignore all those component with start >= end
        newStartVV.setStamp(new AcceptStamp(startStamp+1, nodeId));
        newEndVV.setStamp(new AcceptStamp(endStamp, nodeId));
      }
    }
    if(newStartVV.getSize()==0){
      return null;
    }
    return new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(":/*"),
        newStartVV.cloneAcceptVV(),
        newEndVV.cloneAcceptVV());
  }

  //-------------------------------------------------------------------------
  // apply CatchupStreamStartMsg --
  //       put the pending set into pendingSet if startVV match current lpvv
  //-------------------------------------------------------------------------
  public void
  applyCatchupStreamStartMsg(CatchupStreamStartMsg psm){

    if(dbgProgress){
      System.out.println("IncommingInvalConnection.applyCatchupStreamStartMsg:( " + psm
          + ") is called"); 
    }

    SubscriptionSet newSS = psm.getSubscriptionSet();
    VV newStart = psm.getStartVV();

    assert !newSS.isEmpty();
    SubscriptionSet intersection = newSS.getIntersection(this.ss);

    if (intersection.equals(newSS)){//already in the connection
    	if(dbgProgress){
    	      System.out.println("apply catchup stream was useless intersection: "  + intersection + "; newSS: " + newSS + " ss " + ss); 
    	    }

      controller.informSubscribeInvalSucceeded(senderId,
          core.getMyNodeId(),
          newSS,
          newStart);
      return;
    }


    if(core.getLpVV(newSS).includes(this.prevVV)){//already caught up
    	if(dbgProgress){
  	      System.out.println("apply catchup stream was useless lpvv: "  + core.getLpVV(newSS) + "; prevVV: " + prevVV + " ss " + ss); 
  	    }

      this.addSubscriptionSet(newSS);
      controller.informSubscribeInvalSucceeded(senderId,
          core.getMyNodeId(),
          newSS,
          newStart);
    }else{
      //make sure that lpvv has no gap with the newStart
      if(core.getLpVV(newSS).includes(newStart)){//make sure that lpvv catch up startvv 
        //anytime there should be only one pendingSet.
        assert ((pendingSet == null) && (pendingStartVV == null));
        pendingSet = newSS;
        //System.out.println("pendingSet is "+newSS);
        pendingStartVV = new CounterVV(core.getLpVV(newSS));
        if(dbgProgress){
    	      System.out.println("set pendingSet"  + pendingSet); 
    	    }
      }else{//there's a gap between startvv & lpvv --> can't accept it
        controller.informSubscribeInvalFailed(senderId, core.getMyNodeId(), 
            newSS, newStart);
        Env.warn("Subscription of " + newSS.toString() 
            + "failed due to gap between startVV=" + newStart 
            + " and receiver's lpvv.");
        System.out.println("lpvv : " + core.getLpVV(newSS));
      }
    }
  }




  //-------------------------------------------------------------------------
  // apply CatchupStreamEndMsg
  //     -- put pendingSet into subscriptionset if lpvv match prevVV
  //-------------------------------------------------------------------------
  public void
  applyCatchupStreamEndMsg(CatchupStreamStartMsg start){
    if(dbgProgress){
      System.out.println("IncommingInvalConnection.applyCatchupStreamEndMsg:(start=" + start
          + ") is called with pendingSet " + pendingSet); 
    }

    SubscriptionSet newSS = start.getSubscriptionSet();
    VV newStart = start.getStartVV();
    if(pendingSet!=null){

      // anyone survived during the catch up stream 
      // should be able to add
      if(!pendingSet.isEmpty()){

        this.addSubscriptionSet(pendingSet);
        if(dbgProgress){
          Env.dprint(dbgProgress, "IncomingConnection::applyCatchupStreamEndMsg succeeded in adding"
              + newSS + " with newStart " + newStart);
        }
        controller.informSubscribeInvalSucceeded(senderId,
            core.getMyNodeId(),
            newSS,
            newStart);
      }
      pendingSet = null;
      pendingStartVV = null;
    }
    assert pendingSet == null;
    assert pendingStartVV == null;
  }

  //-------------------------------------------------------------------------
  // clean reference putted in ISStatus
  //  -- called when the connection died
  //-------------------------------------------------------------------------
  public void cleanReference(){

    if(dbg){
      Env.inform("DBG IncommingInvalConnection: closeStreamAndSocket " +
          " sender: " + senderId.toString() +
          " ss: " + ss.toString() +
          " vv: " + prevVV.toString());
    }
    if(senderId != null){
      //
      //gc the reference in ISStatus
      //
      SubscriptionSet removedSS = this.ss;
      this.removeSubscriptionSet(ss, prevVV);
      controller.informInvalStreamTerminated(senderId, 
          removedSS, 
          prevVV,
          true,
          streamId);
    }
  }



  //-------------------------------------------------------------------------
  // return streamId
  //-------------------------------------------------------------------------
  public StreamId 
  getStreamId(){
    return this.streamId;
  }

  //-------------------------------------------------------------------------
  // Compare whether "o" is equal to this object used for PTreeNode.liveCons
  //-------------------------------------------------------------------------
  public boolean
  equals(Object o){
    if(o instanceof IncommingInvalConnection){
      if (this.streamId.equals(((IncommingInvalConnection)o).getStreamId())){
        return true;
      }
    }
    return false;
  }


  //----------------------------------------------------------------------
  // main() -- self test code
  //----------------------------------------------------------------------
  public static void 
  main(String[] argv) {
    System.out.println("IncommingInvalConnection unit tests now in junit framework "
        + "-- run IncommingInvalConnectionUnit");
  }

  //----------------------------------------------------------------------
  // unit test helper function
  //----------------------------------------------------------------------
  public void resetSubscriptionSet(SubscriptionSet newss){
    this.ss = newss;
  }

  //----------------------------------------------------------------------
  // unit test helper function
  //----------------------------------------------------------------------
  public void resetPrevVV(VV newVV){
    this.prevVV = new CounterVV(newVV);
  }

  //----------------------------------------------------------------------
  // unit test helper function
  //----------------------------------------------------------------------
  public void resetPendingSet(){
    this.pendingSet = null;
    this.pendingStartVV = null;
  }

  public void applyCatchupGI(GeneralInv gi) {
    //
    // from catch up streams
    //

    //apply to pendingSet
    assert pendingStartVV != null;
    SubscriptionSet kickedPendingSet =  this.getKickedSet(gi, this.pendingSet);

    if(kickedPendingSet != null){
      AcceptVV newLpvv = prepareNewLpvv(pendingStartVV.cloneAcceptVV(), gi.getStartVV());

      //kick off from pendingSet
      this.updateISStatusLpVV(kickedPendingSet, newLpvv);
      try{
        pendingSet = pendingSet.remove(kickedPendingSet, true);
      }catch(IllegalRemoveSubscriptionSetException e){
        assert false : ("" + e);
      }

      if(pendingSet.isEmpty()){
        pendingSet = null;
        pendingStartVV = null;
      }
      controller.informSubscribeInvalFailed(senderId, core.getMyNodeId(), 
          kickedPendingSet, newLpvv);
    }

    //advance pendingStartVV
    if(pendingStartVV != null){
      assert !pendingSet.isEmpty();
      pendingStartVV.advanceTimestamps(gi.getEndVV());
    }


  }


}











//---------------------------------------------------------------------------
/* $Log$ */
//---------------------------------------------------------------------------
