package code;

import java.io.EOFException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InvalidClassException;
import java.io.OptionalDataException;
import java.io.PrintStream;
import java.net.Socket;

 /** 
 **/ 
 /** 
 * Class IncommingInvalConnectionWorker 
 **/ 
 /** 
 **/ 
public class IncommingInvalConnectionWorker extends Thread implements SocketMagicConstant{

  protected Socket underlyingSocket = null;
  protected TaggedInputStream s = null;
  protected IncommingInvalConnection ic = null;

  protected boolean measureTime = IncommingInvalConnection.measureTime; // measure time it takes to apply invals
  protected String outFilePrefix = "IncommingInvalConnection_time_";
  protected PrintStream out;

 /** 
 * Constructor for worker thread 
 **/ 
  public IncommingInvalConnectionWorker(TaggedInputStream tis,
      Socket underlyingSocket,
      IncommingInvalConnection ic){
    this.s = tis;
    this.underlyingSocket = underlyingSocket;
    this.ic = ic;

    openFile();
  }


  protected static boolean tbdPrinted = false;

 /** 
 * handshake for the connection 

 * return false if fail 
 * true if get all information expected 
 **/ 
  protected boolean handshake(){
    try{
      Long magic = (Long)s.readTaggedObject();
      if(IncommingInvalConnection.dbgVerbose){
        Env.inform("DBG IncommingInvalConnection: got magic " + magic);
      }
      if((magic.longValue() != INVAL_SOCKET_MAGIC)&&
          (magic.longValue() != INVAL_SOCKET_RECONNECT_MAGIC)){
        Env.remoteAssert(false, "descriptive error msg");
        throw new Exception("Bad magic number to IncommingInvalConnection");
      }
      if(IncommingInvalConnection.dbgVerbose){
        Env.inform("DBG IncommingInvalConnection: get sender ");
      }
      this.ic.senderId = (NodeId)(s.readTaggedObject());
      if(IncommingInvalConnection.dbgVerbose){
        Env.inform("DBG IncommingInvalConnection: got sender " +
            this.ic.senderId);
      }
    }catch(Exception e){
      // 
      // Attempt to connect failed and we don't even know
      // who it was that was trying. Nothing to do but
      // have this thread die. If this was important,
      // then a controller should timeout and retry. 
      //

      e.printStackTrace();

      return false; // Thread exit.
    }

    AcceptVV streamStartVV = null;
    
    try{
      streamStartVV = (AcceptVV)(s.readTaggedObject());
      //
      //fallback with the original protocol which initialize with sender.cvv
      //and empty streamStartSS
      //
      //streamStartSS = (SubscriptionSet)(s.readTaggedObject());
    }catch(OptionalDataException o){
      o.printStackTrace();
      return false;//Thread exit.
    }catch(IOException i){
      i.printStackTrace();
      return false;//Thread exit.
    }catch(ClassCastException z){
      z.printStackTrace();
      return false; // Thread exit.
    }catch(Exception zz){
      zz.printStackTrace();     
      return false; // Thread exit.
    }
    
    SubscriptionSet streamSS = null;
    
    try{
    	streamSS = (SubscriptionSet)(s.readTaggedObject());
      //
      //fallback with the original protocol which initialize with sender.cvv
      //and empty streamStartSS
      //
      //streamStartSS = (SubscriptionSet)(s.readTaggedObject());
    }catch(OptionalDataException o){
      o.printStackTrace();
      return false;//Thread exit.
    }catch(IOException i){
      i.printStackTrace();
      return false;//Thread exit.
    }catch(ClassCastException z){
      z.printStackTrace();
      return false; // Thread exit.
    }catch(Exception zz){
      zz.printStackTrace();     
      return false; // Thread exit.
    }
    
    return ic.initConnection(s, streamStartVV, streamSS);
    
  }

 /** 
 * Process an incoming stream consisting of 

 * [INVAL_SOCKET_MAGIC, senderId, intial prevVV,  
 * {(CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)| 
 * (GeneralInv|UnbindMsg|DebargoMsg)*| 
 * (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)}* 
 * TerminateMsg] 
 **/ 
  public void
  run(){

    if(IncommingInvalConnection.dbgProgress){
      System.out.println("IncommingInvalConnectionWorker starts");
    }

    //
    // handshake
    //
    if(!handshake()){
      Env.remoteAssert(false);
      this.closeStreamAndSocket("IncommingInvalConnection cannot handshake with sender ");
      closeFile();
      return;
    }

    //tbd from InvalRecvWorker.java
    if(!tbdPrinted){
      Env.tbd("TBD: subclass Socket to be a heartbeat Socket \n"
          + "sender sends a msg every 5 seconds \n "
          + "and receiver expects to recv message every 20 seconds or\n"
          + "throws IO Exception OR USE KEEPALIVE OPTION?");
      tbdPrinted = true;
    }





    long handleInvalStart=0;
    long handleInvalEnd=0;

    //
    // Keep catchupstreamstart so that when a catchupstream
    // finishes, we know what we were working on...
    //
    CatchupStreamStartMsg lastCatchupStreamStart = null;

    //
    // Read the sequence of invalidations/CP from stream
    //
    try{

      Object next = s.readTaggedObject();
      while(! (next instanceof TerminateMsg)){

        if(measureTime){
          handleInvalStart = System.currentTimeMillis();
        }
        if(IncommingInvalConnection.dbg){
          Env.dprintln(IncommingInvalConnection.dbg, "Read next : " + next + " streamCVV " + ic.prevVV);
        }


        if(next instanceof UnbindMsg){
          ic.applyUnbind((UnbindMsg)next);
        }else if(next instanceof DebargoMsg){
          ic.applyDebargo((DebargoMsg)next);



        }else if(next instanceof CatchupStreamStartMsg){
          // 
          // Right now we do not allow nested catchupstreams.
          // If we want to do that. Receiver needs to keep
          // a stack of start messages...
          //
          Env.remoteAssert(lastCatchupStreamStart == null);
          if(lastCatchupStreamStart != null){
            throw new IOException("Nested catchup stream");
          }
          if(IncommingInvalConnection.dbgPerformance){
            long start = System.currentTimeMillis();
            Env.dprintln(IncommingInvalConnection.dbgPerformance, "@ "+ start 
                + " IncommingInvalConnection -- catchupStreamStartMsg arrives");
          }
          lastCatchupStreamStart = (CatchupStreamStartMsg)next;
          ic.applyCatchupStreamStartMsg(lastCatchupStreamStart);

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();

            LatencyWatcher.put("IncommingInvalConnection.ApplyCatchupStreamStartMsg", 
                (handleInvalEnd - handleInvalStart));

          }
        }else if(next instanceof CatchupStreamEndMsg){

          Env.remoteAssert(lastCatchupStreamStart != null);
          if(lastCatchupStreamStart == null){
            throw new IOException("Mismatched catchup stream");
          }
          if(IncommingInvalConnection.dbgPerformance){
            long start = System.currentTimeMillis();
            Env.dprintln(IncommingInvalConnection.dbgPerformance, "@ "+ start 
                + " IncommingInvalConnection -- catchupStreamEndMsg arrives");
          }
          ic.applyCatchupStreamEndMsg( 
              lastCatchupStreamStart);
          lastCatchupStreamStart = null;

          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();

            LatencyWatcher.put("IncommingInvalConnection.ApplyCatchupStreamEndMsg", 
                (handleInvalEnd - handleInvalStart));
          }


        }else if(next instanceof CPStartMsg){
          ic.applyCP(this.s, (CPStartMsg)next);
          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("IncommingInvalConnection.ApplyCP", (handleInvalEnd - handleInvalStart));
          }

        }else {
          if(!(next instanceof GeneralInv)){
            throw(new InvalidClassException("" + next.getClass()));
          }

          if(IncommingInvalConnection.dbgPerformance && (next instanceof PreciseInv)){
            long start = System.currentTimeMillis();
            Env.dprintln(IncommingInvalConnection.dbgPerformance, "IncommingReceive @ "+ start 
                + " " + ((PreciseInv)next).getAcceptStamp().toString());
          }
          ic.applyGI((GeneralInv)next);
          if(measureTime){
            handleInvalEnd = System.currentTimeMillis();
            writeToFile((handleInvalEnd - handleInvalStart));
            LatencyWatcher.put("IncommingInvalConnection.ApplyInval", (handleInvalEnd - handleInvalStart));
            AcceptVV startVV = ((GeneralInv)next).getStartVV();
            AcceptVV endVV = ((GeneralInv)next).getEndVV();
            VVIterator vvi = startVV.getIterator();
            long n = 0;
            while(vvi.hasMoreElements()){
              NodeId nodeId = vvi.getNext();
              long start = startVV.getStampByServer(nodeId);
              long end = endVV.getStampByServer(nodeId);
              n += end-start+1;
            }            
            LatencyWatcher.put("SecureIncomingInvalConnection.AveragedApplyInval",
                (handleInvalEnd-handleInvalStart)/n);
          }
        }



        if(IncommingInvalConnection.dbgPerformance){
          long start = System.currentTimeMillis();
          Env.dprintln(IncommingInvalConnection.dbgPerformance, "@ "+ start 
              + " IncommingInvalConnection -- about to read next message");
        }


        next = s.readTaggedObject();


        if(IncommingInvalConnection.dbgPerformance){
          long start = System.currentTimeMillis();
          Env.dprintln(IncommingInvalConnection.dbgPerformance, "@ "+ start 
              + " IncommingInvalConnection -- done reading next message");
        }
      }// while loop for next object

    }catch(EOFException i){

      this.closeStreamAndSocket("terminated because of EOFException " + i.toString());
      closeFile();

      return; // expected case thread exit
    }catch(IOException io){
      if(IncommingInvalConnection.dbg){
        io.printStackTrace();
      }
      this.closeStreamAndSocket("terminated because of IOException " + io.toString());
      closeFile();
      return; // expected case thread exit
    }catch(Exception i3){
      i3.printStackTrace();
      assert false; // Unexpected cases
      this.closeStreamAndSocket("terminated unexcpectedly because of " + i3.toString());
      closeFile();
      return; // thread exit
    }
    
    this.closeStreamAndSocket("terminated normally");

  }

 /** 
 * Close the stream and socket 
 **/ 
  protected void
  closeStreamAndSocket(String msg){

    try{
      s.close();
      if(this.underlyingSocket != null){
        this.underlyingSocket.close();
      }
    }catch(Exception e){
      // Ignore errors when trying to close
    }


    ic.cleanReference();
    System.out.println("IncommingInvalConnectionWorker closing stream: " + msg);

  }

 /** 
 * openFile 
 **/ 
  public void openFile(){
    try{
      if(measureTime) {
        out = new PrintStream(new FileOutputStream(outFilePrefix + ic.getStreamId() 
            + ".out")); 
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 * closeFile 
 **/ 
  public void closeFile(){
    if(measureTime) {
      out.close();
    }
  }

 /** 
 * writeToFile 
 **/ 
  public void writeToFile(long duration){
    if(measureTime) {
      out.println(duration);
    }
  }

}
