package code;

 /** 
 *  InterestSetRecord: A single record maintained by an interest set 
 **/ 
public class InterestSetRecord{

 /** 
 *  Data members 
 **/ 
  private CounterVV lastPreciseVV;
  private CounterVV currentVV;

 /** 
 *  Constructor 
 **/ 
  public
  InterestSetRecord(){
    this.lastPreciseVV = (CounterVV)CounterVV.makeVVAllNegatives();
    this.currentVV = (CounterVV)CounterVV.makeVVAllNegatives();
  }

 /** 
 *  Private constructor used by clone() 
 **/ 
  private
  InterestSetRecord(CounterVV newLastPreciseVV, CounterVV newCurrentVV){
    this.lastPreciseVV = newLastPreciseVV;
    this.currentVV = newCurrentVV;
  }

 /** 
 *  Return the last precise VV 
 **/ 
  public CounterVV
  getLastPreciseVV(){
    return this.lastPreciseVV;
  }

 /** 
 *  Return the current VV 
 **/ 
  public CounterVV
  getCurrentVV(){
    return currentVV;
  }

 /** 
 *  Set the last precise VV -- there's no reason that 
 *    an interestSet's lpvv will go backward. therefore always use  
 *    advanceLastPreciseVV() instead of this one. 
 **/ 
  public void
  setLastPreciseVV(CounterVV lpvv){
    lastPreciseVV = lpvv;
  }

 /** 
 *  Set the current VV -- there's no reason that 
 *    an interestSet's lpvv will go backward therefore always 
 *    use advanceCurrentVV() instead of this one 
 **/ 
  public void
  setCurrentVV(CounterVV cvv){
    currentVV = cvv;
  }

 /** 
 *  Set the last precise VV -- advance lpVV: there's no reason that 
 *    an interestSet's lpvv will go backward. 
 **/ 
  public void
  advanceLastPreciseVV(CounterVV lpvv){
    lastPreciseVV.advanceTimestamps(lpvv);
  }

 /** 
 *  Set the current VV -- advance currentVV: there's no reason that 
 *    an interestSet's lpvv will go backward. 
 **/ 
  public void
  advanceCurrentVV(CounterVV cvv){
    currentVV.advanceTimestamps(cvv);
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    return(new InterestSetRecord(new CounterVV(this.lastPreciseVV),
                                 new CounterVV(this.currentVV)));
  }

 /** 
 *  Return true if this InterestSetRecord is precise 
 **/ 
  public boolean
  isPrecise(){
    return(this.lastPreciseVV.equals(this.currentVV));
  }

 /** 
 *  Return true if this equals o 
 **/ 
  public final boolean
  equals(Object o){
    boolean result = false;
    InterestSetRecord isr = null;

    if(o instanceof InterestSetRecord){
      isr = (InterestSetRecord)o;
      result = ((lastPreciseVV.equals(isr.lastPreciseVV)) &&
                (currentVV.equals(isr.currentVV)));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Return a hash code 
 **/ 
  public final int
  hashCode(){
    // We assume that this method will never be called.
    assert(false);
    return(0);
  }

 /** 
 *  Return a String representation of this object 
 **/ 
  public String
  toString(){
    return("ISR:(" + this.lastPreciseVV + ", " + this.currentVV + ")");
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    InterestSetRecord isr1 = null;
    InterestSetRecord isr2 = null;
    CounterVV vv1 = null;
    CounterVV vv2 = null;
    AcceptStamp[] as1 = null;
    AcceptStamp[] as2 = null;

    Env.verifyAssertEnabled();
    System.out.println("Testing InterestSetRecord...");

    // Set vv1 to [("1", 10), ("2", 20)]
    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(10, new NodeId(1));
    as1[1] = new AcceptStamp(20, new NodeId(2));
    vv1 = new CounterVV(as1);
    // Set vv2 to [("1", 15), ("2", 25)]
    as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(15, new NodeId(1));
    as1[1] = new AcceptStamp(25, new NodeId(2));
    vv2 = new CounterVV(as1);
    // Make an InterestSetRecord
    isr1 = new InterestSetRecord(vv1, vv2);

    // Test setLastPreciseVV/getLastPreciseVV, setCurrentVV/getCurrentVV,
    // and isPrecise()
    assert(isr1.getLastPreciseVV() == vv1);
    assert(isr1.getCurrentVV() == vv2);
    assert(!isr1.isPrecise());
    isr1.setLastPreciseVV(vv2);
    isr1.setCurrentVV(vv1);
    assert(isr1.getLastPreciseVV() == vv2);
    assert(isr1.getCurrentVV() == vv1);
    assert(!isr1.isPrecise());
    isr1.setLastPreciseVV(vv1);
    assert(isr1.isPrecise());

    // Test clone() and equals()
    isr2 = (InterestSetRecord)isr1.clone();
    assert(isr1.getLastPreciseVV() != isr2.getLastPreciseVV());
    assert(isr1.getCurrentVV() != isr2.getCurrentVV());
    assert(isr1.getLastPreciseVV().equals(isr2.getLastPreciseVV()));
    assert(isr1.getCurrentVV().equals(isr2.getCurrentVV()));
    assert(isr1 != isr2);
    assert(isr1.equals(isr2));

    // Test toString() (tested)
    // System.out.println("" + isr1);

    System.out.println("...Finished");
  }

}

/* $Log: InterestSetRecord.java,v $
/* Revision 1.13  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.12  2005/10/16 04:20:12  zjiandan
/* add "boolean withBody" parameter for Checkpoint exchange.
/*
/* Revision 1.11  2005/02/28 20:37:13  nayate
/* new InterestSetRecord
/*
/* Revision 1.9  2004/05/25 12:59:37  lgao
/* Add chain mechanism to filter the duplicated update forwarding.
/*
/* Revision 1.8  2004/05/25 08:16:41  lgao
/* *** empty log message ***
/*
/* Revision 1.7  2004/05/23 10:28:53  nayate
/* Changed lastPreciseVV and currentVV to start with BEFORE_TIME_BEGAN values
/*
/* Revision 1.6  2004/05/20 00:47:34  dahlin
/* Fixed several bugs to make inval log exchange work (biggest one: ISIterator handling case when an single-writer log exists but has no records in it; also added some debugging tools
/*
/* Revision 1.5  2004/05/18 08:05:18  arun
/* added minor stuff to catch exceptions
/*
/* Revision 1.4  2004/05/10 20:29:32  nayate
/* Made InterestSetRecord compile + added the equals() and hashCode() methods
/*
/* Revision 1.3  2004/05/07 09:14:58  arun
/* added simple functions called by ISStatus.java
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
