package code;

 /** 
 *  Represents a single entry in InvalQueue 
 **/ 

public class InvalQueueEntry implements Comparable{

 /** 
 *  Constants 
 **/ 
  public static final int INV_START = 1;
  public static final int INV_END = 2;
  public static final int UPD = 3;

 /** 
 *  Data member 
 **/ 
  private GeneralInv gi;
  private StreamId streamId;
  private VVMap startVVs;
  private long deadlineMS;
  private int entryType;
  private boolean isProcessed;
  private BodyMsg bodyMsg;

 /** 
 *  Constructor 
 **/ 
  public
  InvalQueueEntry(int newEntryType,
                  GeneralInv newGI,
                  StreamId newStreamId,
                  VVMap newStartVVs,
                  long newDeadlineMS,
                  BodyMsg newBodyMsg,
                  boolean newIsProcessed){
    this.entryType = newEntryType;
    this.gi = newGI;
    this.streamId = newStreamId;
    this.startVVs = newStartVVs;
    this.deadlineMS = newDeadlineMS;
    this.bodyMsg = newBodyMsg;
    this.isProcessed = newIsProcessed;
  }

 /** 
 *  Return the locally-stored entry type 
 **/ 
  public int
  getEntryType(){
    return(this.entryType);
  }

 /** 
 *  Return the locally-stored invalidate 
 **/ 
  public GeneralInv
  getGI(){
    return(this.gi);
  }

 /** 
 *  Return the locally-stored stream ID 
 **/ 
  public StreamId
  getStreamId(){
    return(this.streamId);
  }

 /** 
 *  Return the locally-stored startVV collection 
 **/ 
  public VVMap
  getStartVVs(){
    return(this.startVVs);
  }

 /** 
 *  Return the locally-stored deadline (milliseconds) 
 **/ 
  public long
  getDeadlineMS(){
    return(this.deadlineMS);
  }

 /** 
 *  Return true if this invalidate is to be applied at the start time 
 **/ 
  public boolean
  getIsStartTime(){
    return(this.entryType == InvalQueueEntry.INV_START);
  }

 /** 
 *  Return the locally-stored body stored in this invalQueueEntry 
 **/ 
  public BodyMsg
  getBodyMsg(){
    return(this.bodyMsg);
  }

 /** 
 *  Return true if this invalidate is already processed 
 **/ 
  public boolean
  getIsProcessed(){
    return(this.isProcessed);
  }

 /** 
 *  Set a flag indicating whether this invalidate has been processed 
 **/ 
  public void
  setIsProcessed(boolean handled){
    this.isProcessed = handled;
  }

 /** 
 *  Return -1, 0, or 1 depending on whether the deadline for this 
 *  entry is earlier, equal to, or later than the deadline for 
 *  invalQueueEntry 
 **/ 
  public int
  compareTo(Object o){
    long val = 0;
    int retVal = 0;
    InvalQueueEntry invalQueueEntry = null;

    // Note: "return(this.deadlineMS - invalQueueEntry);" could have worked
    // but might cause long-to-int conversion problems.
    if(o instanceof InvalQueueEntry){
      invalQueueEntry = (InvalQueueEntry)o;
      val = this.deadlineMS - invalQueueEntry.deadlineMS;
      if(val < 0){
        retVal = -1;
      }else if(val > 0){
        retVal = 1;
      }else{
        retVal = 0;
      }
    }else{
      throw new ClassCastException();
    }
    return(retVal);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing InvalQueueEntry...");
    InvalQueueEntry.testAll();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testAll(){
    InvalQueueEntry iqe1 = null;
    InvalQueueEntry iqe2 = null;
    StreamId streamId1 = null;
    StreamId streamId2 = null;
    Object o1 = null;
    Object o2 = null;
    BodyMsg bodyMsg1 = null;

    // Test 1: Check get/set methods
    streamId1 = StreamId.makeNewStreamId();
    iqe1 = new InvalQueueEntry(InvalQueueEntry.INV_START,
                               null, // GeneralInv
                               streamId1, // StreamId
                               null, // startVVs
                               100, // deadlineMS
                               null, // bodies
                               true); // isProcessed
    assert(iqe1.getGI() == null);
    assert(iqe1.getStreamId().equals(streamId1));
    assert(iqe1.getStartVVs() == null);
    assert(iqe1.getDeadlineMS() == 100);
    assert(iqe1.getIsStartTime());
    assert(iqe1.getEntryType() == InvalQueueEntry.INV_START);
    assert(iqe1.getBodyMsg() == null);
    assert(iqe1.getIsProcessed());
    iqe1.setIsProcessed(false);
    assert(!iqe1.getIsProcessed());

    // Test 2: Check get/set methods
    streamId2 = StreamId.makeNewStreamId();
    bodyMsg1 = new BodyMsg(new ObjId("o3"),
                           0,
                           100,
                           new AcceptStamp(100, new NodeId(7)),
                           new ImmutableBytes(new byte[100]),
                           false);
    iqe2 = new InvalQueueEntry(InvalQueueEntry.UPD,
                               null, // GeneralInv
                               streamId2, // StreamId
                               null, // startVVs
                               250, // deadlineMS
                               bodyMsg1, // body
                               false); // isProcessed
    assert(iqe2.getGI() == null);
    assert(iqe2.getStreamId().equals(streamId2));
    assert(iqe2.getStartVVs() == null);
    assert(iqe2.getDeadlineMS() == 250);
    assert(!iqe2.getIsStartTime());
    assert(iqe2.getEntryType() == InvalQueueEntry.UPD);
    assert(iqe2.getBodyMsg() == bodyMsg1);
    assert(!iqe2.getIsProcessed());
    iqe2.setIsProcessed(true);
    assert(iqe2.getIsProcessed());
    iqe2.setIsProcessed(false);
    assert(!iqe2.getIsProcessed());

    // Test 3: Check compareTo
    assert(iqe1.compareTo(iqe2) < 0);
    assert(iqe1.compareTo(iqe1) == 0);
    assert(iqe2.compareTo(iqe2) == 0);
    assert(iqe2.compareTo(iqe1) > 0);
    try{
      o1 = new Integer(100);
      iqe1.compareTo(o1);
      assert(false); // Should not reach here
    }catch(ClassCastException e){
      // This is correct behavior
    }
    try{
      o2 = new Integer(150);
      iqe2.compareTo(o2);
      assert(false); // Should not reach here
    }catch(ClassCastException e){
      // This is correct behavior
    }
  }
}
