package code;

 /** 
 *  InvalSpanningList -- keep a list of nodes to which we are connected 
 *  to as part of spanning trees for different interest sets 
 **/ 

import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.*;

public class InvalSpanningList
{


private HashMap list;

public InvalSpanningList()
{
    list = new HashMap();
}

/*
 *------------------------------------------------------------------
 *
 * insert --
 *
 *          Insert a record for the specified interest set into
 *          the list of things we are subscribed for. Do not duplicate --
 *          keep track of one subscription per interest set.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Return true if this is a "new" subscription -- we did not yet
 *      have a subscription for this interest set. False otherwise.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public boolean
insert(SubscribeInvalRequest r)
{

  	assert(r instanceof SubscribeInvalRequest);
    if(list.containsKey(r.getSubscriptionSet())){
	return false;
    }
    list.put(r.getSubscriptionSet(), r);
    assert(list.containsKey(r.getSubscriptionSet()));

    return true;
}


/*
 *------------------------------------------------------------------
 *
 * remove --
 *
 *          Remove record -- we are not subscribing to anyone
 *          for this IS (either b/c we are root or b/c we don't
 *          care about IS anymore.)
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public void 
remove(SubscribeInvalRequest r)
{
    list.remove(r.getSubscriptionSet());
}

/*
 *------------------------------------------------------------------
 *
 * find --
 *
 *          See if there is a record for this interest set *and* this
 *          sender.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Note -- we hand out a pointer to an object in the table.
 *
 *------------------------------------------------------------------
 */
public SubscribeInvalRequest 
    find(NodeId id, SubscriptionSet ss)
{
    Object o = list.get(ss);
    if(o == null){
	return null;
    }
    SubscribeInvalRequest r = (SubscribeInvalRequest)o;
    if(r.getTarget().equals(id)){
	return r;
    }
    else{
	return null;
    }
}

/*
 *------------------------------------------------------------------
 *
 * find --
 *
 *          Return record for this interest set.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Note -- we hand out a pointer to an object in the table.
 *
 *------------------------------------------------------------------
 */
public SubscribeInvalRequest 
find(SubscriptionSet ss)
{
    Object o = list.get(ss);
    if(o == null){
	return null;
    }
    SubscribeInvalRequest r = (SubscribeInvalRequest)o;
    return r;
}

/*
 *------------------------------------------------------------------
 *
 * find --
 *
 *          Return records that match nodeId (regardless of IS).
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Note -- we hand out a pointer to an object in the table.
 *
 *------------------------------------------------------------------
 */
public List
find(NodeId id)
{
    Set allEntries = list.entrySet();
    Iterator ii = allEntries.iterator();
    List l = new LinkedList();
    while(ii.hasNext()){
		SubscribeInvalRequest r = (SubscribeInvalRequest)(((Map.Entry)ii.next()).getValue());
		if(r.getTarget().equals(id)){
	    	l.add(r);
		}
    }
    return l;
}

/*
 *------------------------------------------------------------------
 *
 * main --
 *
 *          Unit test
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Note -- we hand out a pointer to an object in the table.
 *
 *------------------------------------------------------------------
 */
public static void main(String s[])
{
    InvalSpanningList l = new InvalSpanningList();
    boolean result;
    boolean hideDISBug = false;

    System.err.print("InvalSpanningList unit test....");

    if(hideDISBug){
	System.err.println("WARNING: DirectoryInterestSet bug prevents full testing of InvalSpanningList -- uncomment additional tests when DirectoryInterestSet fixed");
    }

    SubscriptionSet ss1 = SubscriptionSet.makeSubscriptionSet("/foo");
    SubscriptionSet ss2 = SubscriptionSet.makeSubscriptionSet("/bar");
    SubscriptionSet ss3 = SubscriptionSet.makeSubscriptionSet("/foo");
    assert(!ss1.equals(ss2));
    if(!hideDISBug){
        assert(ss1.equals(ss3)); // Need to fix DirectoryInterestSet
    }

    AcceptStamp as = new AcceptStamp(100, new NodeId(100));
    AcceptStamp asa[] = new AcceptStamp[1];
    asa[0] = as;
    VV vv = new AcceptVV(asa);
    NodeId target1 = new NodeId(50);
    NodeId target2 = new NodeId(20);


    SubscribeInvalRequest r1 = new SubscribeInvalRequest(ss1, vv, target1, false);
    SubscribeInvalRequest r2 = new SubscribeInvalRequest(ss2, vv, target1, false);
    SubscribeInvalRequest r3 = new SubscribeInvalRequest(ss3, vv, target1, false);
    SubscribeInvalRequest r4 = new SubscribeInvalRequest(ss3, vv, target2, false);

    
    result = l.insert(r1);
    assert(result);
    result = l.insert(r1);
    assert(!result); // Duplicate
    result = l.insert(r2);
    assert(result);
    if(!hideDISBug){
	result = l.insert(r3);
	assert(!result); // Duplicate
	result = l.insert(r4);
	assert(!result); // Duplicate
    }

    SubscribeInvalRequest r5 = l.find(target1, ss1);
    assert(r5 != null);
    assert(r5.equals(r1));
    
    SubscribeInvalRequest r6 = l.find(target1, ss2);
    assert(r6 != null);
    assert(r6.equals(r2));

    SubscribeInvalRequest r7 = l.find(target2, ss1);
    assert(r7 == null);

    SubscribeInvalRequest r8 = l.find(target2, ss2);
    assert(r8 == null);


    System.err.println("...Succeeds");
    
}

    
}

 /** 
 *  $Log: InvalSpanningList.java,v $ 
 *  Revision 1.11  2005/03/01 10:11:41  nayate 
 *  Modified for new code 
 *  
 *  Revision 1.10  2005/03/01 09:11:47  nayate 
 *  Modified for new code 
 *  
 *  Revision 1.9  2004/05/23 00:25:22  arun 
 *  only fresh writes get inserted in UPQ. Some debugging code related to SDIMS 
 *  
 *  Revision 1.8  2004/05/22 10:10:12  arun 
 *  minor changes to notifyUpq logic. Correction of store behaviour while handling bound invals. 
 *  
 *  Revision 1.7  2004/05/14 20:49:00  dahlin 
 *  SDIMSController logic for Body connections (incoming) 
 *  
 *  Revision 1.6  2004/05/14 19:23:21  dahlin 
 *  Cleanup spanning tree logic; add update bodies 
 *  
 *  Revision 1.5  2004/05/13 18:49:17  dahlin 
 *  Clean SDIMSController by moving SpanningTree directory out as separate class (rather than using raw SDIMSInterface) 
 *  
 *  Revision 1.4  2004/05/09 22:21:11  dahlin 
 *  Unit test and stub implementation for spanningTreeWorker 
 *  
 *  Revision 1.3  2004/05/09 20:09:14  dahlin 
 *  Updated unit tests for WorkQueue, TimeoutQueue, TimeoutQueueWorker, DirectoryInterestSet, SubscribeInvalRequest, InvalSpanningList, AcceptStamp, NodeId, and AcceptVV 
 *  
 *  Revision 1.2  2004/05/09 02:24:16  dahlin 
 *  Complete coding and unit testing of InvalSpanningList and SubscribeInvalRequest 
 *  
 *  Revision 1.1  2004/05/08 22:20:17  dahlin 
 *  Partially complete version of SDIMSController (it should compile w/o error, though) 
 *  
 **/ 
