package code;

 /** 
 *  Used to store invalidate messages waiting to be applied 
 *  
 *  This class uses another helper class, InvalidateBufferWorkerThread 
 *  
 *  Currently, the methods in this class simply apply invalidates in the 
 *  order in which "applyInvalidateAtStartTime" and "applyInvalidateAtEndTime" 
 *  are called. This is valid to do because calls to these methods are already 
 *  serialized by the stream classes. However, we note that because we store 
 *  per-stream information locally, it is actually possible to be more 
 *  intelligent in applying invalidates. 
 *  
 *  We do not implement the optimization where an imprecise invalidate is 
 *  removed out of the buffer sooner if its corresponding precise invalidate 
 *  messages arrive. 
 *  
 *  We currently make a copy of startVVs, although it is possible to save 
 *  space and time by keeping local startVVs in this class instead of in 
 *  InvalRecvWorker. 
 **/ 
import java.util.LinkedList;
import java.util.PriorityQueue;
import java.util.Vector;
import java.util.HashMap;
import java.util.Vector;
import java.util.ListIterator;
import java.io.IOException;

public class InvalidateBuffer{

 /** 
 *  Constants 
 **/ 
  private static final long MAX_UPDATE_BUFFER_BYTES = 5000000;
  private static final boolean PRINT_DEBUG = false;

 /** 
 *  Data members 
 **/ 
  private HashMap pendingUpdates;
  private PriorityQueue heap;
  private UpdateBuffer updateBuffer;
  private Vector pendingReadTokens;
  private long numPendingReads;
  private LinkedList invalQueue;
  private LinkedList bodyQueue;

 /** 
 *  Constructor 
 **/ 
  public
  InvalidateBuffer(DataStore newDataStore, UpdateBuffer newUpdateBuffer){
    this.numPendingReads = 0;
    this.pendingUpdates = new HashMap();
    this.heap = new PriorityQueue();
    this.updateBuffer = newUpdateBuffer;
    this.pendingReadTokens = new Vector(); // Used for debugging only
    this.invalQueue = new LinkedList();
    this.bodyQueue = new LinkedList();
  }

 /** 
 *  Apply the invalidate at its start time 
 **/ 
  public synchronized void
  applyInvalAtStartTime(GeneralInv gi,
                        StreamId streamId,
                        VVMap startVVs,
                        long deadlineMS){
    InvalQueueEntry invalQueueEntry = null;
    InvalQueueEntry lastHeapTopEntry = null;
    InvalQueueEntry newHeapTopEntry = null;

    // Convert bound invalidates into unbound ones
    gi = this.unbindInvalidate(gi, true);
    assert(!gi.isBound());
    invalQueueEntry = new InvalQueueEntry(InvalQueueEntry.INV_START,
                                          gi,
                                          streamId,
                                          (VVMap)startVVs.clone(),
                                          deadlineMS,
                                          null,
                                          false);
    lastHeapTopEntry = this.peekNextInvalQueueEntry();
    if(lastHeapTopEntry != null){
      this.heap.add(invalQueueEntry);
      newHeapTopEntry = this.peekNextInvalQueueEntry();
      assert(newHeapTopEntry != null);
      if(lastHeapTopEntry != newHeapTopEntry){
        // New deadline
        this.notifyAll();
      }
    }else{
      this.heap.add(invalQueueEntry);
      // New deadline
      this.notifyAll();
    }
    this.invalQueue.addLast(invalQueueEntry);
  }

 /** 
 *  Apply the invalidate at its end time 
 **/ 
  public synchronized void
  applyInvalAtEndTime(GeneralInv gi,
                      StreamId streamId,
                      VVMap startVVs,
                      long deadlineMS) throws IOException{
    InvalQueueEntry invalQueueEntry = null;
    InvalQueueEntry newHeapTopEntry = null;
    InvalQueueEntry lastHeapTopEntry = null;

    // Convert bound invalidates into unbound ones
    gi = this.unbindInvalidate(gi, false);
    assert(!gi.isBound());
    invalQueueEntry = new InvalQueueEntry(InvalQueueEntry.INV_END,
                                          gi,
                                          streamId,
                                          (VVMap)startVVs.clone(),
                                          deadlineMS,
                                          null,
                                          false);
    lastHeapTopEntry = this.peekNextInvalQueueEntry();
    if(lastHeapTopEntry != null){
      this.heap.add(invalQueueEntry);
      newHeapTopEntry = this.peekNextInvalQueueEntry();
      assert(newHeapTopEntry != null);
      if(lastHeapTopEntry != newHeapTopEntry){
        // New deadline
        this.notifyAll();
      }
    }else{
      // New deadline
      this.notifyAll();
      this.heap.add(invalQueueEntry);
    }
    this.invalQueue.addLast(invalQueueEntry);
  }

 /** 
 *  Apply a body to the buffer 
 **/ 
  public synchronized long
  applyBody(BodyMsg bodyMsg, boolean waitApply) throws
    InterruptedException, IOException{
    long numBytesWritten = 0;
    PendingUpdateMailBox pendingUpdate = null;

    this.updateBuffer.add(bodyMsg);
    this.bodyQueue.addLast(bodyMsg);
    this.notifyAll();

    if(waitApply){
      pendingUpdate = new PendingUpdateMailBox(false, 0);
      this.pendingUpdates.put(bodyMsg, pendingUpdate);
      while(!pendingUpdate.getValueKnown()){
        try{
          this.wait();
        }catch(InterruptedException e){
          System.err.println("" + e);
        }
        pendingUpdate = (PendingUpdateMailBox)this.pendingUpdates.get(bodyMsg);
      }
      numBytesWritten = pendingUpdate.getNumBytesWritten();
    }
    return(numBytesWritten);
  }

 /** 
 *  Called by the data store when a read is blocked 
 **/ 
  public synchronized void
  notifyReadBlocked(String token){
    // NOTE: token used only for debugging
    this.numPendingReads++;
    assert(this.addPendingRead(token));
    this.notifyAll();
  }

 /** 
 *  Called by the data store when a read is unblocked 
 **/ 
  public synchronized void
  notifyReadUnblocked(String token){
    // NOTE: token used only for debugging
    this.numPendingReads--;
    assert(this.removePendingRead(token));
  }

 /** 
 *  Called by the worker thread to indicate that a buffered update 
 *  has been written 
 **/ 
  public synchronized void
  notifyUpdateWritten(BodyMsg bodyMsg, long numBytesWritten){
    PendingUpdateMailBox pendingUpdate = null;

    pendingUpdate = (PendingUpdateMailBox)this.pendingUpdates.get(bodyMsg);
    if(pendingUpdate != null){
      assert(!pendingUpdate.getValueKnown());
      pendingUpdate = new PendingUpdateMailBox(true, numBytesWritten);
      this.pendingUpdates.put(bodyMsg, pendingUpdate);
      this.notifyAll();
    }
  }

 /** 
 *  Called by the worker thread to figure out what to apply next 
 **/ 
  public synchronized InvalQueueEntry
  dequeue() throws InterruptedException, IOException, CausalOrderException{
    InvalQueueEntry heapTopEntry = null;
    long currentTimeMS = 0;
    long deadlineMS = 0;
    ListIterator li = null;
    long numBytesWritten = 0;
    InvalQueueEntry entry = null;
    InvalQueueEntry possibleEntry = null;
    BodyMsg bodyMsg = null;
    boolean waitCondition = false;

    while(entry == null){
      if(!this.bodyQueue.isEmpty()){
        bodyMsg = (BodyMsg)this.bodyQueue.removeFirst();
        entry = new InvalQueueEntry(InvalQueueEntry.UPD,
                                    null,
                                    null,
                                    null,
                                    0,
                                    bodyMsg,
                                    true);
	if( PRINT_DEBUG ){
	  Env.dprintln(PRINT_DEBUG, "entry = UPD: " + bodyMsg.getObjId());
	}
      }else if(!this.invalQueue.isEmpty()){
        assert(!this.heap.isEmpty());
        heapTopEntry = this.peekNextInvalQueueEntry();
        assert(heapTopEntry != null);
        possibleEntry = (InvalQueueEntry)this.invalQueue.getFirst();
        currentTimeMS = System.currentTimeMillis();
        deadlineMS = heapTopEntry.getDeadlineMS();
        if(!this.canApplyInvalidate(possibleEntry, currentTimeMS, deadlineMS)){
          try{
            this.wait(deadlineMS - currentTimeMS);
          }catch(InterruptedException e){
            System.out.println("" + e);
          }
        }else{
          entry = (InvalQueueEntry)this.invalQueue.removeFirst();
          assert(entry == possibleEntry);
	  if( PRINT_DEBUG ){
	    if(entry.getEntryType() == InvalQueueEntry.INV_START){
	      Env.dprintln(PRINT_DEBUG, "entry = INV_START: " + entry.getGI());
	    }else{
	      Env.dprintln(PRINT_DEBUG, "entry = INV_END: " + entry.getGI());
	    }
	  }
        }
      }else{
        try{
          // Wait until either an invalidate or a body arrives
          this.wait();
        }catch(InterruptedException e){
          System.out.println("" + e);
        }
      }
    }
    entry.setIsProcessed(true);
    return(entry);
  }

 /** 
 *  Unbind the given invalidate (if it is bound) and store 
 *  any corresponding object into the UpdateBuffer 
 **/ 
  private GeneralInv
  unbindInvalidate(GeneralInv gi, boolean addBodies){
    BoundInval bi = null;
    BodyMsg[] bodyMsgArr = null;
    MultiObjPreciseInv mopi = null;
    GeneralInv result = null;

    bodyMsgArr = new BodyMsg[0];
    if(gi instanceof BoundInval){
      // Store bound body into buffer and convert into a PreciseInv
      bi = (BoundInval)gi;
      bodyMsgArr = new BodyMsg[1];
      bodyMsgArr[0] = bi.cloneBodyMsg();
      result = bi.clonePreciseInv();
    }else if(gi instanceof MultiObjPreciseInv){
      // Store bound bodies into buffer and convert into a MultiObjPreciseInv
      mopi = (MultiObjPreciseInv)gi;
      bodyMsgArr = mopi.makeBodyMsgArray();
      result = mopi.cloneUnbound();
    }else{
      result = gi;
    }
    if(addBodies){
      for(int i = 0; i < bodyMsgArr.length; i++){
        this.updateBuffer.add(bodyMsgArr[i]);
        this.bodyQueue.addLast(bodyMsgArr[i]);
      }
      this.notifyAll();
    }
    return(result);
  }

 /** 
 *  Get the next InvalQueueEntry to be processed 
 **/ 
  private InvalQueueEntry
  peekNextInvalQueueEntry(){
    InvalQueueEntry heapTopEntry = null;
    InvalQueueEntry oldHeapTopEntry = null;

    heapTopEntry = (InvalQueueEntry)this.heap.peek();
    while((heapTopEntry != null) && heapTopEntry.getIsProcessed()){
      oldHeapTopEntry = (InvalQueueEntry)this.heap.poll();
      assert(oldHeapTopEntry == heapTopEntry);
      heapTopEntry = (InvalQueueEntry)this.heap.peek();
    }
    return(heapTopEntry);
  }

 /** 
 *  Return true if the invalidate is ready to be applienext invalidate = Pd 
 **/ 
  private boolean
  canApplyInvalidate(InvalQueueEntry invalQueueEntry,
                     long currentTimeMS,
                     long deadlineMS){
    boolean canApply = false;
    GeneralInv gi = null;
    PreciseInv pi = null;
    MultiObjPreciseInv mopi = null;

    assert(!invalQueueEntry.getGI().isBound());
    if(this.numPendingReads > 0){
      canApply = true;
    }else if(!invalQueueEntry.getIsStartTime()){
      canApply = true;
    }else if(deadlineMS <= currentTimeMS){
      canApply = true;
    }else if(invalQueueEntry.getGI().isDelete()){
      canApply = true;
    }else if(invalQueueEntry.getGI().isPrecise()){
      gi = invalQueueEntry.getGI();
      assert((gi instanceof PreciseInv) || (gi instanceof MultiObjPreciseInv));
      if(gi instanceof PreciseInv){
        pi = (PreciseInv)gi;
        canApply = this.updateBuffer.containsAny(pi.getEndVV());
      }else{
        mopi = (MultiObjPreciseInv)gi;
        canApply = this.updateBuffer.containsAll(mopi);
      }
    }else{
      canApply = false;
    }
    return(canApply);
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private boolean
  addPendingRead(String token){
    this.pendingReadTokens.add(token);
    return(this.pendingReadTokens.size() == this.numPendingReads);
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private boolean
  removePendingRead(String token){
    boolean removed = false;

    // Vector.remove() only removes one instance of the token
    removed = this.pendingReadTokens.remove(token);
    return(removed && (this.pendingReadTokens.size() == this.numPendingReads));
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing InvalidateBuffer...");
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(0),
                            "",
                            1,
                            2,
                            3,
                            4,
                            5,
                            "",
                            "",
                            -1,
                            "",
                            6,
                            7,
                            0,
                            10000000,
                            10000000,
                            10000000);

    InvalidateBuffer.testInvalidateApplication();
    InvalidateBuffer.testInvalBodyApplication();
    InvalidateBuffer.testReads();
    InvalidateBuffer.testUnbindInvalidate();
    InvalidateBuffer.testMultipleAdds();
    System.out.println("...Finished");
    // Exit the tests
    System.exit(0);
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private static void
  testInvalidateApplication(){
    InvalidateBuffer ib = null;
    PreciseInv pi1 = null;
    PreciseInv pi2 = null;
    ImpreciseInv ipi1 = null;
    StreamId streamId = null;
    VVMap startVVs = null;
    long deadlineMS1 = 0;
    long deadlineMS2 = 0;
    long currentTimeMS = 0;
    AcceptStamp asArray1[] = null;
    AcceptStamp asArray2[] = null;
    IBFakeDataStore fakeDataStore = null;
    UpdateBuffer updateBuffer = null;

    try{
      // Test 1:
      // Apply an invalidate with a 1000 ms delay
      // Make sure it is not applied after a 200 ms delay
      // Make sure it is applied after 1100 total ms
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      deadlineMS1 = System.currentTimeMillis() + 1000;
      pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(100, new NodeId(1)),
                           new AcceptStamp(100, new NodeId(1)),
                           false);
      ib.applyInvalAtStartTime(pi1, streamId, startVVs, deadlineMS1);
      Thread.sleep(200);
      assert(!fakeDataStore.applied(pi1));
      Thread.sleep(900);
      assert(fakeDataStore.applied(pi1));

      // Test 2:
      // Apply an invalidate with 1000 ms delay
      // Apply a second invalidate with 500 ms delay
      // Make sure both are applied after 600 total ms
      // One Invalidate preempts another
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      currentTimeMS = System.currentTimeMillis();
      pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(100, new NodeId(1)),
                           new AcceptStamp(100, new NodeId(1)),
                           false);
      deadlineMS1 = currentTimeMS + 1000;
      ib.applyInvalAtStartTime(pi1, streamId, startVVs, deadlineMS1);
      ib.applyInvalAtEndTime(pi1, streamId, startVVs, deadlineMS1 + 1);
      pi2 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(110, new NodeId(1)),
                           new AcceptStamp(110, new NodeId(1)),
                           false);
      deadlineMS2 = currentTimeMS + 500;
      ib.applyInvalAtStartTime(pi2, streamId, startVVs, deadlineMS2);
      ib.applyInvalAtEndTime(pi2, streamId, startVVs, deadlineMS2 + 1);
      Thread.sleep(100);
      assert(!fakeDataStore.applied(pi1));
      assert(!fakeDataStore.applied(pi2));
      Thread.sleep(400);
      assert(fakeDataStore.applied(pi1));
      assert(fakeDataStore.applied(pi2));

      // Test 3:
      // Apply an imprecise invalidate with 500 ms delay
      // Make sure it is not applied at 200 ms
      // Make sure it is applied at 600 total ms
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      asArray1 = new AcceptStamp[2];
      asArray1[0] = new AcceptStamp(300, new NodeId(1));
      asArray1[1] = new AcceptStamp(300, new NodeId(2));
      asArray2 = new AcceptStamp[2];
      asArray2[0] = new AcceptStamp(400, new NodeId(1));
      asArray2[1] = new AcceptStamp(400, new NodeId(2));
      ipi1 = new ImpreciseInv(new HierInvalTarget(),
                              new AcceptVV(asArray1),
                              new AcceptVV(asArray2),
                              new AcceptVV(asArray2));
      deadlineMS1 = System.currentTimeMillis() + 500;
      ib.applyInvalAtStartTime(ipi1, streamId, startVVs, deadlineMS1);
      Thread.sleep(200);
      assert(!fakeDataStore.applied(ipi1));
      Thread.sleep(400);
      assert(fakeDataStore.applied(ipi1));
    }catch(InterruptedException e){
      System.err.println("" + e);
      assert(false);
    }catch(IOException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private static void
  testInvalBodyApplication(){
    InvalidateBuffer ib = null;
    PreciseInv pi1 = null;
    PreciseInv pi2 = null;
    StreamId streamId = null;
    VVMap startVVs = null;
    long deadlineMS1 = 0;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    IBFakeDataStore fakeDataStore = null;
    ImpreciseInv ipi1 = null;
    AcceptStamp[] asArray1 = null;
    UpdateBuffer updateBuffer = null;

    try{
      // Test 1:
      // Apply an invalidate in 1000 ms
      // Wait 500 ms
      // Apply the body and make sure it is applicable
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(100, new NodeId(1)),
                           new AcceptStamp(100, new NodeId(1)),
                           false);
      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      deadlineMS1 = System.currentTimeMillis() + 1000;
      ib.applyInvalAtEndTime(pi1, streamId, startVVs, deadlineMS1);
      bodyMsg1 = new BodyMsg(new ObjId("/a"),
                             10,
                             20,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[20]),
                             false);
      assert(!fakeDataStore.applied(bodyMsg1));
      ib.applyBody(bodyMsg1, true);  // Synchronous
      //ib.applyBody(bodyMsg1, false);  // Asynchronous
      Thread.sleep(500);
      assert(fakeDataStore.applied(bodyMsg1));

      // Test 2:
      // Apply a future body; make sure it is not applied
      // Apply the corresponding invalidate with a 1000 ms delay
      // Make sure it is applied immediately
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      bodyMsg1 = new BodyMsg(new ObjId("/b"),
                             10,
                             20,
                             new AcceptStamp(200, new NodeId(1)),
                             new ImmutableBytes(new byte[20]),
                             true);
      ib.applyBody(bodyMsg1, false);  // Asynchronous
      assert(!fakeDataStore.applied(bodyMsg1));
      pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("/b"), 10, 20),
                           new AcceptStamp(200, new NodeId(1)),
                           new AcceptStamp(200, new NodeId(1)),
                           false);
      deadlineMS1 = System.currentTimeMillis() + 1000;
      ib.applyInvalAtEndTime(pi1, streamId, startVVs, deadlineMS1);
      Thread.sleep(200);
      assert(fakeDataStore.applied(bodyMsg1));

      // Test 3:
      // Apply two future bodies; make sure they are not applied
      // Apply the corresponding imprecise invalidate with a 100 ms delay
      // Make sure that the invalidate unlocks both objects at 200 ms
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      bodyMsg1 = new BodyMsg(new ObjId("/b"),
                             10,
                             20,
                             new AcceptStamp(300, new NodeId(1)),
                             new ImmutableBytes(new byte[20]),
                             true);
      bodyMsg2 = new BodyMsg(new ObjId("/c"),
                             10,
                             20,
                             new AcceptStamp(300, new NodeId(2)),
                             new ImmutableBytes(new byte[20]),
                             false);
      ib.applyBody(bodyMsg1, false);  // Asynchronous
      ib.applyBody(bodyMsg2, false);  // Asynchronous
      assert(!fakeDataStore.applied(bodyMsg1));
      assert(!fakeDataStore.applied(bodyMsg2));
      asArray1 = new AcceptStamp[2];
      asArray1[0] = new AcceptStamp(300, new NodeId(1));
      asArray1[1] = new AcceptStamp(300, new NodeId(2));
      ipi1 = new ImpreciseInv(new HierInvalTarget(),
                              new AcceptVV(asArray1),
                              new AcceptVV(asArray1),
                              new AcceptVV(asArray1));
      deadlineMS1 = System.currentTimeMillis() + 100;
      ib.applyInvalAtStartTime(ipi1, streamId, startVVs, deadlineMS1);
      ib.applyInvalAtEndTime(ipi1, streamId, startVVs, deadlineMS1 + 1);
      assert(!fakeDataStore.applied(ipi1));
      Thread.sleep(200);
      assert(fakeDataStore.applied(ipi1));
      assert(fakeDataStore.applied(bodyMsg1));
      assert(fakeDataStore.applied(bodyMsg2));
    }catch(InterruptedException e){
      System.err.println("" + e);
      assert(false);
    }catch(IOException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private static void
  testReads(){
    IBFakeDataStore fakeDataStore = null;
    InvalidateBuffer ib = null;
    StreamId streamId = null;
    VVMap startVVs = null;
    long deadlineMS1 = 0;
    long deadlineMS2 = 0;
    long deadlineMS3 = 0;
    PreciseInv pi1 = null;
    PreciseInv pi2 = null;
    PreciseInv pi3 = null;
    UpdateBuffer updateBuffer = null;

    try{
      // Test 1:
      // Notify that we are blocked for 2 reads
      // Apply an invalidate with a 200 ms delay
      // Make sure the invalidate is applied within 100 ms
      // Unblock one read
      // Apply another invalidate with a 200 ms delay
      // Make sure the invalidate is applied within 100 ms
      // Unblock the other read
      // Apply another invalidate with a 200 ms delay
      // Make sure the invalidate is not applied within 100 ms
      // Make sure it is applied within 300 ms (since scheduling)
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);

      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      ib.notifyReadBlocked("/a");
      ib.notifyReadBlocked("/a");
      pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(100, new NodeId(1)),
                           new AcceptStamp(100, new NodeId(1)),
                           false);
      deadlineMS1 = System.currentTimeMillis() + 200;
      ib.applyInvalAtEndTime(pi1, streamId, startVVs, deadlineMS1);
      Thread.sleep(100);
      assert(fakeDataStore.applied(pi1));
      ib.notifyReadUnblocked("/a");
      pi2 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(200, new NodeId(1)),
                           new AcceptStamp(200, new NodeId(1)),
                           false);
      deadlineMS2 = System.currentTimeMillis() + 200;
      ib.applyInvalAtEndTime(pi2, streamId, startVVs, deadlineMS2);
      Thread.sleep(100);
      assert(fakeDataStore.applied(pi2));
      ib.notifyReadUnblocked("/a");
      pi3 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 40, 20),
                           new AcceptStamp(300, new NodeId(1)),
                           new AcceptStamp(300, new NodeId(1)),
                           false);
      deadlineMS3 = System.currentTimeMillis() + 200;
      ib.applyInvalAtStartTime(pi3, streamId, startVVs, deadlineMS3);
      ib.applyInvalAtEndTime(pi3, streamId, startVVs, deadlineMS3 + 1);
      Thread.sleep(100);
      assert(!fakeDataStore.applied(pi3));
      Thread.sleep(200);
      assert(fakeDataStore.applied(pi3));
    }catch(InterruptedException e){
      System.err.println("" + e);
      assert(false);
    }catch(IOException e){
      System.err.println("" + e);
      assert(false);
    }
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private static void
  testUnbindInvalidate(){
    InvalidateBuffer ib = null;
    IBFakeDataStore fakeDataStore = null;
    BoundInval bi1 = null;
    PreciseInv pi1 = null;
    StreamId streamId = null;
    long deadlineMS1 = 0;
    VVMap startVVs = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;
    MOITBoundEntry[] moitBE = null;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    MultiObjPreciseInv mopi1 = null;
    MultiObjInvalTarget moit1 = null;
    UpdateBuffer updateBuffer = null;

    try{
      // Test 1:
      // Apply a bound invalidate in 1000 ms
      // Wait 200 ms
      // Make sure the invalidate/body is applied
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      bi1 = new BoundInval(new ObjId("/a"),
                           10,
                           20,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[20]),
                           5.7,
                           new AcceptStamp(200, new NodeId(1)),
                           false);
      pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("/a"), 10, 20),
                           new AcceptStamp(100, new NodeId(1)),
                           new AcceptStamp(200, new NodeId(1)),
                           false);
      deadlineMS1 = System.currentTimeMillis() + 1000;
      ib.applyInvalAtStartTime(bi1, streamId, startVVs, deadlineMS1);
      ib.applyInvalAtEndTime(bi1, streamId, startVVs, deadlineMS1 + 1);
      Thread.sleep(200);
      assert(fakeDataStore.applied(pi1));
      bodyMsg1 = new BodyMsg(new ObjId("/a"),
                             10,
                             20,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[20]),
                             false,
                             5.7);
      assert(fakeDataStore.applied(bodyMsg1));

      // Test 2:
      // Apply a partially bound MultiObjPreciseInv in 500 ms
      // Wait 100 ms
      // Make sure the invalidate is not applied
      // Wait 500 ms more
      // Make sure all partially bound objects are applied
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      invs = new ObjInvalTarget[2];
      invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 100);
      invs[1] = new ObjInvalTarget(new ObjId("/b"), 20, 200);
      dels = new ObjInvalTarget[0];
      moit1 = new MultiObjInvalTarget(invs, dels);
      moitBE = new MOITBoundEntry[1];
      moitBE[0] = new MOITBoundEntry(new ObjInvalTarget(new ObjId("/a"),
                                                        10,
                                                        100),
                                     5.8,
                                     new ImmutableBytes(new byte[100]));
      mopi1 = new MultiObjPreciseInv(moit1,
                                     new AcceptStamp(100, new NodeId(1)),
                                     new AcceptStamp(100, new NodeId(1)),
                                     false,
                                     moitBE);
      deadlineMS1 = System.currentTimeMillis() + 500;
      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      ib.applyInvalAtStartTime(mopi1, streamId, startVVs, deadlineMS1);
      ib.applyInvalAtEndTime(mopi1, streamId, startVVs, deadlineMS1 + 1);
      Thread.sleep(100);
      bodyMsg1 = new BodyMsg(new ObjId("/a"),
                             10,
                             100,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[100]),
                             false,
                             5.8);
      bodyMsg2 = new BodyMsg(new ObjId("/b"),
                             20,
                             200,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[200]),
                             false);
      assert(!fakeDataStore.applied(bodyMsg1));
      assert(!fakeDataStore.applied(bodyMsg2));
      Thread.sleep(500);
      assert(fakeDataStore.applied(bodyMsg1));
      assert(!fakeDataStore.applied(bodyMsg2));
    }catch(IOException e){
      assert false : "" + e;
    }catch(InterruptedException e){
      assert false : "" + e;
    }
  }

 /** 
 *  Used for testing/debugging 
 **/ 
  private static void
  testMultipleAdds(){
    IBFakeDataStore fakeDataStore = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;
    InvalidateBuffer ib = null;
    MultiObjInvalTarget moit1 = null;
    MOITBoundEntry[] moitBE = null;
    MultiObjPreciseInv mopi1 = null;
    MultiObjPreciseInv mopi2 = null;
    long deadlineMS1 = 0;
    StreamId streamId = null;
    VVMap startVVs = null;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    BodyMsg bodyMsg3 = null;
    UpdateBuffer updateBuffer = null;

    try{
      // Test 1:
      // Schedule a single-bound invalidate for three objects in 1000ms
      // Make sure that the invalidate is not applied after 100ms
      // Apply one body
      // Make sure that the invalidate is still not applied
      // Apply the remaining bodies
      // Make sure that the invalidate is applied
      fakeDataStore = new IBFakeDataStore();
      updateBuffer = new UpdateBuffer(MAX_UPDATE_BUFFER_BYTES,
                                      UpdateBuffer.STOP_ON_FULL);
      ib = new InvalidateBuffer(fakeDataStore, updateBuffer);
      new InvalidateBufferWorkerThread(ib, fakeDataStore, updateBuffer);
      invs = new ObjInvalTarget[3];
      invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 100);
      invs[1] = new ObjInvalTarget(new ObjId("/b"), 20, 200);
      invs[2] = new ObjInvalTarget(new ObjId("/c"), 30, 300);
      dels = new ObjInvalTarget[0];
      moit1 = new MultiObjInvalTarget(invs, dels);
      moitBE = new MOITBoundEntry[1];
      moitBE[0] = new MOITBoundEntry(new ObjInvalTarget(new ObjId("/a"),
                                                        10,
                                                        100),
                                     5.8,
                                     new ImmutableBytes(new byte[100]));
      mopi1 = new MultiObjPreciseInv(moit1,
                                     new AcceptStamp(100, new NodeId(1)),
                                     new AcceptStamp(100, new NodeId(1)),
                                     false,
                                     moitBE);
      deadlineMS1 = System.currentTimeMillis() + 1000;
      streamId = StreamId.makeNewStreamId();
      startVVs = new VVMap();
      ib.applyInvalAtStartTime(mopi1, streamId, startVVs, deadlineMS1);
      ib.applyInvalAtEndTime(mopi1, streamId, startVVs, deadlineMS1 + 1);
      Thread.sleep(100);
      mopi2 = new MultiObjPreciseInv(moit1,
                                     new AcceptStamp(100, new NodeId(1)),
                                     new AcceptStamp(100, new NodeId(1)),
                                     false,
                                     new MOITBoundEntry[0]);
      bodyMsg1 = new BodyMsg(new ObjId("/a"),
                             10,
                             100,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[100]),
                             false,
                             5.8);
      assert(!fakeDataStore.applied(mopi2));
      assert(!fakeDataStore.applied(bodyMsg1));

      bodyMsg2 = new BodyMsg(new ObjId("/b"),
                             20,
                             200,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[200]),
                             false,
                             6.8);
      ib.applyBody(bodyMsg2, false); // Asynchronous
      assert(!fakeDataStore.applied(mopi2));
      assert(!fakeDataStore.applied(bodyMsg1));
      assert(!fakeDataStore.applied(bodyMsg2));

      bodyMsg3 = new BodyMsg(new ObjId("/c"),
                             30,
                             300,
                             new AcceptStamp(100, new NodeId(1)),
                             new ImmutableBytes(new byte[300]),
                             false,
                             7.8);
      ib.applyBody(bodyMsg3, false); // Asynchronous
      Thread.sleep(100);
      assert(fakeDataStore.applied(mopi2));
      assert(fakeDataStore.applied(bodyMsg1));
      assert(fakeDataStore.applied(bodyMsg2));
      assert(fakeDataStore.applied(bodyMsg3));
    }catch(IOException e){
      assert false : "" + e;
    }catch(InterruptedException e){
      assert false : "" + e;
    }
  }
}

 /** 
 *  Used to represent a single entry for a pending update 
 **/ 
class PendingUpdateMailBox{

 /** 
 *  Data members 
 **/ 
  private boolean valueKnown;
  private long numBytesWritten;

 /** 
 *  Constructor 
 **/ 
  public
  PendingUpdateMailBox(boolean newValueKnown, long newNumBytesWritten){
    this.valueKnown = newValueKnown;
    this.numBytesWritten = newNumBytesWritten;
  }

 /** 
 *  Return true if the value in this mailbox is known 
 **/ 
  public boolean
  getValueKnown(){
    return(this.valueKnown);
  }

 /** 
 *  Return the number of bytes written 
 **/ 
  public long
  getNumBytesWritten(){
    return(this.numBytesWritten);
  }
}
