package code;

 /** 
 *  Worker thread that applies invalidate messages and updates 
 *  Used by InvalidateBuffer 
 **/ 
import java.io.IOException;
import java.util.LinkedList;
import java.util.ListIterator;

public class InvalidateBufferWorkerThread implements Runnable{

 /** 
 *  Data members 
 **/ 
  private InvalidateBuffer invalidateBuffer;
  private DataStore dataStore;
  private UpdateBuffer updateBuffer;

 /** 
 *  Constructor 
 **/ 
  public
  InvalidateBufferWorkerThread(InvalidateBuffer newInvalidateBuffer,
                               DataStore newDataStore,
                               UpdateBuffer newUpdateBuffer){
    this.invalidateBuffer = newInvalidateBuffer;
    this.dataStore = newDataStore;
    this.updateBuffer = newUpdateBuffer;
    (new Thread(this)).start();
  }
  
 /** 
 *  Main thread method 
 **/ 
  public void
  run(){
    InvalQueueEntry invalQueueEntry = null;

    while(true){
      try{
        invalQueueEntry = this.invalidateBuffer.dequeue(); // Blocking call
        //System.out.println("Dequeued: " + invalQueueEntry.getEntryType());
        this.applyEvent(invalQueueEntry);
      }catch(InterruptedException e){
        // Should not be here; print something and continue waiting
        System.err.println("" + e);
      }catch(IOException e){
        // Give up; don't know what would cause this
        System.err.println("" + e);
        assert(false);
      }catch(CausalOrderException e){
        // Give up; don't know what would cause this
        System.err.println("" + e);
        assert(false);
      }
    }
  }
  
 /** 
 *  Apply the next event 
 **/ 
  void
  applyEvent(InvalQueueEntry invalQueueEntry)
    throws IOException, CausalOrderException{

    switch(invalQueueEntry.getEntryType()){
    case InvalQueueEntry.UPD:
      this.applyUpdateEvent(invalQueueEntry);
      break;
    case InvalQueueEntry.INV_START:
      // Fall through
    case InvalQueueEntry.INV_END:
      this.applyInvalidateEvent(invalQueueEntry);
      break;
    }
  }

 /** 
 *  Apply an update to the DataStore (if possible) 
 **/ 
  private void
  applyUpdateEvent(InvalQueueEntry invalQueueEntry) throws IOException{
    LinkedList removedBodies = null;
    AcceptStamp[] acceptStamp = null;
    BodyMsg hintBody = null;
    long numBytesWritten = 0;

    assert(invalQueueEntry.getEntryType() == InvalQueueEntry.UPD);
    hintBody = invalQueueEntry.getBodyMsg();
    assert(hintBody != null);
    acceptStamp = new AcceptStamp[1];
    acceptStamp[0] = hintBody.getAcceptStamp();
    if(this.dataStore.canApplyBody(acceptStamp[0])){
      removedBodies = this.updateBuffer.remove(new AcceptVV(acceptStamp));
      assert(removedBodies.size() == 1);
      assert(hintBody.equals(removedBodies.getFirst()));
      numBytesWritten = this.dataStore.applyBody(hintBody);
      this.invalidateBuffer.notifyUpdateWritten(hintBody, numBytesWritten);
    }
  }

 /** 
 *  Apply an invalidate + possible updates to the DataStore 
 **/ 
  private void
  applyInvalidateEvent(InvalQueueEntry iqEntry)
    throws IOException, CausalOrderException{
    BodyMsg bodyMsg = null;
    long numBytesWritten = 0;
    ListIterator li = null;
    LinkedList bodies = null;

    // Note that since we apply local writes to the DataStore immediately,
    //   by the time we apply an invalidate for a local write it may have
    //   been applied already. However, this just means that the
    //   corresponding invalidate would be re-applied; this causes no harm.
    assert((iqEntry.getEntryType() == InvalQueueEntry.INV_START) ||
           (iqEntry.getEntryType() == InvalQueueEntry.INV_END));
    if(iqEntry.getIsStartTime()){
      this.dataStore.applyOverlappingAtStartTime(iqEntry.getGI(),
                                                 iqEntry.getStartVVs());
    }else{
      this.dataStore.applyNonoverlappingAtEndTime(iqEntry.getGI(),
                                                  iqEntry.getStartVVs());
    }
    if(iqEntry.getGI().isPrecise() || (!iqEntry.getIsStartTime())){
      bodies = this.updateBuffer.remove(iqEntry.getGI().getEndVV());
      li = bodies.listIterator(0);
      while(li.hasNext()){
        bodyMsg = (BodyMsg)li.next();
        assert(this.dataStore.canApplyBody(bodyMsg.getAcceptStamp()));
        numBytesWritten = this.dataStore.applyBody(bodyMsg);
        this.invalidateBuffer.notifyUpdateWritten(bodyMsg, numBytesWritten);
      }
    }
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    IBFakeDataStore fakeDataStore = null;
    IBWTFakeInvalidateBuffer fakeBuffer = null;
    UpdateBuffer updateBuffer = null;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    BodyMsg bodyMsg3 = null;
    PreciseInv pi1 = null;
    PreciseInv pi2 = null;

    Env.verifyAssertEnabled();
    System.out.println("Testing InvalidateBufferWorkerThread...");

    // Initialization stuff
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(0),
                            "",
                            1,
                            2,
                            3,
                            4,
                            5,
                            "",
                            "",
                            -1,
                            "",
                            6,
                            7,
                            0,
                            10000000,
                            10000000,
                            10000000);

    updateBuffer = new UpdateBuffer(UpdateBuffer.INFINITE_BUFFER_BYTES,
                                    UpdateBuffer.FAIL_ON_FULL);
    fakeBuffer = new IBWTFakeInvalidateBuffer(updateBuffer);
    fakeDataStore = new IBFakeDataStore();
    new InvalidateBufferWorkerThread(fakeBuffer, fakeDataStore, updateBuffer);

    // Test 1
    // (1) Put something into the UpdateBuffer
    // (2) Create a message for the worker thread about it
    // (3) Make sure the worker thread does not apply it
    bodyMsg1 = new BodyMsg(new ObjId("o1"),
                           0,
                           100,
                           new AcceptStamp(100, new NodeId(0)),
                           new ImmutableBytes(new byte[100]),
                           false);
    updateBuffer.add(bodyMsg1);
    fakeBuffer.enqueue(new InvalQueueEntry(InvalQueueEntry.UPD,
                                           null,
                                           null,
                                           null,
                                           0,
                                           bodyMsg1,
                                           true));
    try{
      Thread.sleep(100);
    }catch(InterruptedException e){
      System.out.println("" + e);
    }
    assert(!fakeDataStore.applied(bodyMsg1));

    // Test 2
    // (1) Place an invalidate on the queue
    // (2) Place the corresponding update in the UpdateBuffer
    // (3) Create a message to the worker thread about it
    // (4) Make sure the worker thread applies it
    pi1 = new PreciseInv(new ObjInvalTarget(new ObjId("o2"),
                                            0,
                                            100),
                         new AcceptStamp(100, new NodeId(1)),
                         new AcceptStamp(100, new NodeId(1)),
                         false);
    fakeBuffer.enqueue(new InvalQueueEntry(InvalQueueEntry.INV_END,
                                           pi1,
                                           null,
                                           null,
                                           0,
                                           null,
                                           true));
    try{
      // Let the thread dequeue and apply the message
      Thread.sleep(100);
    }catch(InterruptedException e){
      System.out.println("" + e);
    }
    assert(!fakeDataStore.applied(bodyMsg2));
    bodyMsg2 = new BodyMsg(new ObjId("o2"),
                           0,
                           100,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[100]),
                           false);
    updateBuffer.add(bodyMsg2);
    fakeBuffer.enqueue(new InvalQueueEntry(InvalQueueEntry.UPD,
                                           null,
                                           null,
                                           null,
                                           0,
                                           bodyMsg2,
                                           true));
    try{
      Thread.sleep(100);
    }catch(InterruptedException e){
      System.out.println("" + e);
    }
    assert(fakeDataStore.applied(bodyMsg2));

    // Test 3
    // (1) Place an update on the queue
    // (2) Make sure the worker thread does not apply it
    // (3) Place the corresponding invalidate + update on the queue
    // (4) Make sure the worker thread applies it
    bodyMsg3 = new BodyMsg(new ObjId("o3"),
                           0,
                           100,
                           new AcceptStamp(100, new NodeId(2)),
                           new ImmutableBytes(new byte[100]),
                           false);
    updateBuffer.add(bodyMsg3);
    fakeBuffer.enqueue(new InvalQueueEntry(InvalQueueEntry.UPD,
                                           null,
                                           null,
                                           null,
                                           0,
                                           bodyMsg3,
                                           true));
    try{
      Thread.sleep(100);
    }catch(InterruptedException e){
      System.out.println("" + e);
    }
    assert(!fakeDataStore.applied(bodyMsg3));

    pi2 = new PreciseInv(new ObjInvalTarget(new ObjId("o3"),
                                            0,
                                            100),
                         new AcceptStamp(100, new NodeId(2)),
                         new AcceptStamp(100, new NodeId(2)),
                         false);
    fakeBuffer.enqueue(new InvalQueueEntry(InvalQueueEntry.INV_END,
                                           pi2,
                                           null,
                                           null,
                                           0,
                                           null,
                                           true));
    try{
      Thread.sleep(100);
    }catch(InterruptedException e){
      System.out.println("" + e);
    }
    assert(fakeDataStore.applied(pi2));
    assert(fakeDataStore.applied(bodyMsg3));

    /*
    // Test 1: Insert something into the shared queue and see if the
    // right method is called
    invalQueueEntry1 = new InvalQueueEntry(InvalQueueEntry.INV_START,
                                           null,
                                           StreamId.makeNewStreamId(),
                                           null,
                                           100,
                                           null,
                                           false);
    invalQueue.enqueue(invalQueueEntry1);
    try{
      Thread.sleep(500);
    }catch(InterruptedException e){
      assert(false);
    }
    lastEntriesApplied = fakeBuffer.getLastEntriesApplied();
    assert(lastEntriesApplied.removeFirst() == invalQueueEntry1);
    assert(lastEntriesApplied.isEmpty());

    // Test 2: Insert two entries into the shared queue and see if the
    // method is called twice
    invalQueueEntry1 = new InvalQueueEntry(InvalQueueEntry.INV_START,
                                           null,
                                           StreamId.makeNewStreamId(),
                                           null,
                                           100,
                                           null,
                                           false);
    invalQueue.enqueue(invalQueueEntry1);
    invalQueueEntry2 = new InvalQueueEntry(InvalQueueEntry.INV_START,
                                           null,
                                           StreamId.makeNewStreamId(),
                                           null,
                                           200,
                                           null,
                                           false);
    invalQueue.enqueue(invalQueueEntry2);
    try{
      Thread.sleep(500);
    }catch(InterruptedException e){
      assert(false);
    }
    lastEntriesApplied = fakeBuffer.getLastEntriesApplied();
    assert(lastEntriesApplied.size() == 2);
    assert(lastEntriesApplied.removeFirst() == invalQueueEntry1);
    assert(lastEntriesApplied.removeFirst() == invalQueueEntry2);
    */
    System.out.println("...Finished");
    System.exit(0);
  }
}

 /** 
 *  Used for testing 
 **/ 
class IBWTFakeInvalidateBuffer extends InvalidateBuffer{

 /** 
 *  Data members 
 **/ 
  private LinkedList queue;

 /** 
 *  Constructor 
 **/ 
  public
  IBWTFakeInvalidateBuffer(UpdateBuffer updateBuffer){
    super(null, updateBuffer);
    this.queue = new LinkedList();
  }

 /** 
 *  Add an entry to the local pretend queue 
 *  Note: Not a method of InvalidateBuffer; used only for testing 
 **/ 
  public synchronized void
  enqueue(InvalQueueEntry entry){
    this.queue.addLast(entry);
    this.notifyAll();
  }

 /** 
 *  Return the next entry in the invalidate buffer 
 **/ 
  public synchronized InvalQueueEntry
  dequeue(){
    InvalQueueEntry nextEntry = null;

    while(this.queue.isEmpty()){
      try{
        this.wait();
      }catch(InterruptedException e){
        System.err.println("" + e);
      }
    }
    nextEntry = (InvalQueueEntry)this.queue.removeFirst();
    return(nextEntry);
  }

 /** 
 *  Called to handle blocking calls to InvalidateBuffer.applyBody() 
 **/ 
  public synchronized void
  notifyUpdateWritten(BodyMsg bodyMsg, long numBytesWritten){
    // Do nothing
  }
}

 /** 
 *  Represent a fake DataStore class 
 **/ 
class IBFakeDataStore extends DataStore{

 /** 
 *  Data members 
 **/ 
  private LinkedList appliedBodyList;
  private LinkedList appliedInvalList;
  private CounterVV cvv;

 /** 
 *  Constructor 
 **/ 
  public
  IBFakeDataStore(){
    super(".", new NodeId(0));
    this.appliedBodyList = new LinkedList();
    this.appliedInvalList = new LinkedList();
    this.cvv = new CounterVV();
  }

 /** 
 *  Return true if this object is applicable according to the timestamp 
 **/ 
  public boolean
  canApplyBody(AcceptStamp as){
    return(this.cvv.includes(as));
  }

 /** 
 *  Apply this body to the data store 
 *  - Make sure it is applicable 
 **/ 
  public synchronized long
  applyBody(BodyMsg bodyMsg) throws IOException{
    assert(this.cvv.includes(bodyMsg.getAcceptStamp()));
    this.appliedBodyList.addLast(bodyMsg);
    return(bodyMsg.getLength());
  }

 /** 
 *  Apply this invalidate at its start time 
 *  - Increase CVV 
 **/ 
  public synchronized void 
  applyOverlappingAtStartTime(GeneralInv gi, VVMap startVVs) throws
    IOException, CausalOrderException{
    this.cvv.advanceTimestamps(gi.getEndVV());
    this.appliedInvalList.addLast(gi);
  }

 /** 
 *  Apply this invalidate at its end time 
 *  - Increase CVV 
 **/ 
  public synchronized void 
  applyNonoverlappingAtEndTime(GeneralInv gi, VVMap startVVs) throws
    IOException{
    this.cvv.advanceTimestamps(gi.getEndVV());
    this.appliedInvalList.addLast(gi);
  }

 /** 
 *  Return true if this object has been applied 
 **/ 
  public synchronized boolean
  applied(BodyMsg bodyMsg){
    boolean contains = false;
    return(this.appliedBodyList.contains(bodyMsg));
  }

 /** 
 *  Return true if this object has been applied 
 **/ 
  public synchronized boolean
  applied(GeneralInv gi){
    return(this.appliedInvalList.contains(gi));
  }

 /** 
 *  Return a string representation 
 **/ 
  public synchronized String
  toString(){
    String str = null;
    BodyMsg bodyMsg = null;

    str = "";
    str = "Bodies = ";
    for(ListIterator li = this.appliedBodyList.listIterator(0); li.hasNext();){
      bodyMsg = (BodyMsg)li.next();
      str += ("" + bodyMsg.getObjId() +
              ":" + bodyMsg.getObjInvalTarget().getOffset() +
              ":" + bodyMsg.getLength());

    }
    return(str);
  }
}
