package code;
/* LatencyWatcher::
 * 
 * Helper for monitoring the latency breakdown
 * for tuning performance.
 *
 * accumulate latency for certain type, report the avg latency cost
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */




import java.util.*;
import java.io.*;



public class LatencyWatcher{
    
  static Hashtable h = new Hashtable();
  static final int cutInitNoise = 5;

 /** 
 *  reset all records 
 **/ 
  public static void reset(){
      h = new Hashtable();
  }

 /** 
 *  put a new record 
 **/ 
  public static synchronized void put(Object type, long len){
    LatencyRecord newL = null;
    LatencyRecord oldL = (LatencyRecord)h.get(type);
    if(oldL == null){
      newL = new LatencyRecord(len, cutInitNoise);
      h.put(type, newL);
    } else {
      oldL.add(len);
      h.put(type, oldL);
    }
  }
  
  // len is sum of 'count' items
  public static synchronized void put(Object type, long lenSum, int count){
    LatencyRecord newL = null;
    LatencyRecord oldL = (LatencyRecord)h.get(type);
    if(oldL == null){
      newL = new LatencyRecord(lenSum, cutInitNoise, count);
      h.put(type, newL);
    } else {
      oldL.add(lenSum, count);
      h.put(type, oldL);
    }
  }
  

 /** 
 *  get diff of (newLen-lastLen) and put the diff 
 *  this method is used to calculate the latency of one specific item 
 *  in different stage. 
 *  e.g. OutgoingConnection received a request for adding sub for "/obj1" 
 *  we want to track when the request finally get served. 
 *  To calculate the request delay time: 
 *    LatencyWatcher.put("/obj1", start=first receiving the request); 
 *    LatencyWatcher.getDiffAndPut("/obj1", end=sendCP finish for "/obj1"); 
 *  
 **/ 
  public static synchronized void getDiffAndPut(Object type, long len){
    LatencyRecord newL = null;
    LatencyRecord oldL = (LatencyRecord)h.get(type);
    if(oldL == null){
      assert false: "call put() to get a start time first";
    } else {
      oldL.diff(len);
      h.put(type, oldL);
    }
  }
  
 /** 
 *  get the current latency info for a specific type 
 **/ 
  public static synchronized LatencyRecord get(Object type){
    return (LatencyRecord)(h.get(type));
  }

 /** 
 *  get the avg latency for a specific type 
 **/ 
  public static synchronized double getAvg(Object type){
    return ((LatencyRecord)(h.get(type))).getAvg();
  }

 /** 
 *  get the report for the avg of all types 
 **/ 
  public static String getSummary(){
    String ret = "";
    String s = null;
    for(Enumeration e = h.keys(); e.hasMoreElements();){
      s = (String)e.nextElement();
      double l = ((LatencyRecord)h.get(s)).getAvg();
      ret += "(" + s + ":" + l + ") "
	+ " with total count=" + ((LatencyRecord)h.get(s)).getCount()
	+ " with total latency=" + ((LatencyRecord)h.get(s)).getTotal()
	+ "\n";
    }
    return ret; 
  }
	
 /** 
 *  get all records 
 **/ 
  public static Hashtable getAll(){
    return h;
  }
}

 /** 
 *  helping class for log latency segments. 
 **/ 
class LatencyRecord{
  long totalTime;
  int count;
  boolean noiseCutted;
  int maxNoiseCutted;
  public LatencyRecord(long newTime, int maxNoiseCutted){
    this.totalTime = newTime;
    this.count = 1;
    this.noiseCutted = false;
    this.maxNoiseCutted = maxNoiseCutted;
  }
  
  public LatencyRecord(long newTime, int maxNoiseCutted, int _count){
    this.totalTime = newTime;
    this.count = _count;
    this.noiseCutted = false;
    this.maxNoiseCutted = maxNoiseCutted;
  }

  private void cutNoise(){
    this.totalTime = 0;
    this.count = 0;
    this.noiseCutted = true;
  }

  
  public void add(long newTime){
    this.totalTime += newTime;
    this.count ++;
    if((!noiseCutted) && (count == maxNoiseCutted)){
      cutNoise();
    }
  }
  
  // newTimeSum is sum of '_count' items
  public void add(long newTimeSum, int _count){
    this.totalTime += newTimeSum;
    this.count += _count;
    int leftover = count - maxNoiseCutted;
    if((!noiseCutted) && (leftover >= 0)){
      cutNoise();
      this.count = leftover;
      this.totalTime = (long)((double)newTimeSum/(double)_count) * leftover;
    }
  }


  public void diff(long endTime){
    assert count == 1;
    this.totalTime = endTime - this.totalTime;
    assert totalTime >= 0;
  }

  public double getAvg(){
    return (double)totalTime/(double)count;
  }

  public long getTotal(){
    return this.totalTime;
  }

  public int getCount(){
    return this.count;
  }

  
}

//-------------------------------------------------------------------
/* $Log: LatencyWatcher.java,v $
/* Revision 1.1  2007/09/11 18:03:39  zjiandan
/* *** empty log message ***
/* */
//-------------------------------------------------------------------