package code;

import java.io.*;
import java.util.Vector;

public class LocalInterface{
  Core core;
  Controller controller;
  static final boolean dbg=false;
 
  
  boolean logToFile;//for debug to create replay log
  String outputFileName;
  FileOutputStream fos = null;
  ObjectOutputStream oos = null;
  
  static final int READ_1 = 1;
  static final int READ_2 = 2;
  static final int READ_BLOCK_INVALID = 3;
  static final int WRITE_1 = 4;
  
  static final int WRITE_MULTI = 5;
  static final int WRITE_2 = 6;
  static final int DELETE = 7;
  static final int INFORM_DEMAND_READMISS = 8;
  static final int SYNC_STATE_TO_DISK = 9;
  
  static final int SYNC = 10;
  static final int ISVALID = 11;
  static final int ISPRECISE = 12;

  static final boolean PRINT_METHOD = false;
  // 
  // URA local interface: read/write/delete
  //
  // Layer over this interface a full file system interface -- 
  // is the interface described here more or less what we need?
  //
  // Future optimization would be to use file descriptors instead
  // of paths (add open/close). Not clear we need the complexity right
  // now. 
  //
  // Future feature: "big red button" to force propagation of updates
  // and invalidates to other nodes.
  //
  // Future feature: way to specify the priority of a write - present
  //
  // Future feature: way to provide hint on who to propagate write to
  //

  //-----------------------------------------------------------------------
  // Constructor
  //-----------------------------------------------------------------------
  public
  LocalInterface(Controller controller_, Core core_){
    this(controller_, core_, false, null);
    
  }

  //-----------------------------------------------------------------------
  // Constructor
  //-----------------------------------------------------------------------
  public
    LocalInterface(Controller controller_, Core core_, boolean logToFile, 
		   String logFileName){
    core = core_;
    controller = controller_;
    
    
    this.logToFile = logToFile;
    outputFileName = logFileName;
    if(logToFile){
      try{
	File f = new File(outputFileName);
        if (f.exists()){
          f.delete();
        }
	
	fos = new FileOutputStream(outputFileName);
	oos = new ObjectOutputStream(fos);
      }catch(Exception e){
	System.err.println("outputstream  error: "
			   + e.toString());
	e.printStackTrace();
	
      }
    }
    
  }

  //-----------------------------------------------------------------------
  // commit() -- commit the write (targetSet, targetAS)
  // 
  // postCondition: generate a CommitInv with targetSet, targetAS
  // so that any receiver applies this CommitInv will make
  // the object state of targetSet to be committed if the newest acceptStamp
  // in the datastore matching the targetAS.
  //-----------------------------------------------------------------------
  public CommitInv
  commit(InvalTarget targetSet,
        AcceptStamp targetAS)
  throws IOException{
    return core.commit(targetSet, targetAS);
  }
  //-----------------------------------------------------------------------
  // preWrite() -- expose AcceptStamp instead of the length
  //   need cleanup the all the other write methods as well.
  //   needed by Coda  -- rethink the write interface
  //   see case-studies/Coda/tbd.txt
  //-----------------------------------------------------------------------

  public PreciseInv
  preWrite(ObjId objId,
        long offset,
        long length,
        ImmutableBytes buffer, 
        boolean bound,
        long maxBoundHops) throws IOException{
    double priority = Core.DEFAULT_PRIORITY;
    if ( PRINT_METHOD ){
      Env.dprintln(PRINT_METHOD, "write(" + "objId="+objId.toString() 
		 + ", offset=" + offset
		 + ", length=" + length
		 + ", priority=" + priority
		 + ", buffer=" + buffer.toString()
		  + ", bound=" + bound
		 + ", maxBoundHops=" + maxBoundHops);
    }
     
    if(logToFile){
       //Please also UPDATE this.replayAllFromFile() & checkTwoLogs
       //if you update the parameters
     
      oos.writeInt(WRITE_1);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.writeDouble(priority);
      oos.writeObject(buffer);
      oos.writeBoolean(bound);
      oos.writeLong(maxBoundHops);
      oos.flush();
      oos.reset();
    }
    //System.err.println("performing local write to " + objId +
    // " : [" + offset + "," + length + "]");
    PreciseInv resultSummary = null;

    resultSummary = core.write(objId, offset, length, priority, buffer, bound, maxBoundHops);
    controller.informLocalWrite(objId,
                                offset,
                                length,
                                resultSummary.getAcceptStamp(),
			        bound, 
				false); 
    //ObjInvalTarget t = (ObjInvalTarget)resultSummary.getInvalTarget();
    return resultSummary;
  }


  //----------------------------------------------------------------------
  // read() -- read up to <length> bytes starting from <objId, offset>
  // 
  // Returns
  //   If the object exists and there is at least one valid byte at <objId, offset>
  //   we return between 1 and <length> bytes. Notice that we may return fewer
  //   bytes than requested.
  //
  //   If the object does not exist, throw ObjNotFoundException
  //
  //   If there is a low-level IO error or database error, throw IOException
  //
  //   Otherwise
  //
  //   If <blockInvalid> is true, then do not return until the specified range
  //   is valid. (If <blockInvalid> is false and the byte at <objId, offse>
  //   is invalid, throw ReadOfInvalidRangeException.)
  //
  //   If <blockImprecise> is true, then do not return until the specified
  //   object "is precise."
  //
  //   If offset is past the last byte of the file, throw EOFException
  //
  //   Notice that we never return null. 
  //
  //   Note that a read of a "hole" returns a 1-byte result with value 0.
  //   (A hole is created when an offset x is not written but some offset
  //   y>x is written.)
  //
  //----------------------------------------------------------------------
  public BodyMsg
  read(ObjId objId,
       long offset,
       long length,
       boolean blockInvalid,
       boolean blockImprecise)
    throws ObjNotFoundException, IOException,
    EOFException, ReadOfInvalidRangeException, ReadOfHoleException{
    return this.read(objId, offset, length, blockInvalid, blockImprecise, -1);
  }

  //----------------------------------------------------------------------
  // read() -- read up to <length> bytes starting from <objId, offset>
  // same as above execpt set a timeout such that
  // the read won't be blocked for ever if imprecise or invalid
  //----------------------------------------------------------------------
  public BodyMsg
  read(ObjId objId,
       long offset,
       long length,
       boolean blockInvalid,
       boolean blockImprecise,
       long timeout)
    throws ObjNotFoundException, IOException,
    EOFException, ReadOfInvalidRangeException, ReadOfHoleException{

    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "read(" + "objId="+objId.toString() 
                   + ", offset=" + offset
                   + ", length=" + length
                   + ", blockInvalid=" + blockInvalid
                   + ", blockImprecise=" + blockImprecise
                   + ", timeout=" + timeout);
    }

    if(logToFile){
      oos.writeInt(READ_1);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.writeBoolean(blockInvalid);
      oos.writeBoolean(blockImprecise);
      oos.writeLong(timeout);
      oos.flush();
      oos.reset();
    }
    
    return core.read(objId,
                     offset,
                     length,
                     blockInvalid,
                     blockImprecise,
                     true,
                     controller,
		     timeout);
  }
    //---------------------------------------------------
    // add max Temporal Error
    //---------------------------------------------------
  public BodyMsg
    read(ObjId objId,
	 long offset,
	 long length,
	 boolean blockInvalid,
	 boolean blockImprecise,
	 long maxTE,
	 long timeout)
    throws ObjNotFoundException, IOException,
	   EOFException, ReadOfInvalidRangeException, ReadOfHoleException{
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "read(" + "objId="+objId.toString() 
                   + ", offset=" + offset
                   + ", length=" + length
                   + ", blockInvalid=" + blockInvalid
                   + ", blockImprecise=" + blockImprecise
                   + ", maxTE=" + maxTE
                   + ", timeout=" + timeout);
    }
    if(logToFile){
      oos.writeInt(READ_2);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.writeBoolean(blockInvalid);
      oos.writeBoolean(blockImprecise);
      oos.writeLong(maxTE);
      oos.writeLong(timeout);
      oos.flush();
      oos.reset();
    }
    return core.read(objId,
                     offset,
                     length,
                     blockInvalid,
                     blockImprecise,
                     true,
		     maxTE,
                     controller,
		     timeout);
  }

  //----------------------------------------------------------------------
  // read() -- read up to <length> bytes starting from <objId, offset>
  //
  // This version of read() always blocks if the range is invalid
  // so it never throws ReadOfInvalidRangeException.
  //----------------------------------------------------------------------
  public BodyMsg
  readBlockInvalid(ObjId objId,
                   long offset,
                   long length,
                   boolean blockImprecise)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "read(" + "objId="+objId.toString() 
                   + ", offset=" + offset
                   + ", length=" + length
                   + ", blockImprecise=" + blockImprecise);
    }
    if(logToFile){
      oos.writeInt(READ_BLOCK_INVALID);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.writeBoolean(blockImprecise);
      oos.flush();
      oos.reset();
    }
    return core.readBlockInvalid(objId, offset, length, 
                                 blockImprecise, true, controller, -1);
  }

  //-----------------------------------------------------------------------
  // write() -- caller promises not to ever change byte array, so
  //  we don't need to make a copy. 
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
  public long
  write(ObjId objId,
        long offset,
        long length,
        ImmutableBytes buffer, 
        boolean bound) throws IOException{
    // Pass the call onto another version of this method
    return(this.write(objId,
                      offset,
                      length,
                      Core.DEFAULT_PRIORITY,
                      buffer,
                      bound));
  }

  //-----------------------------------------------------------------------
  // write() -- caller promises not to ever change byte array, so
  //  we don't need to make a copy.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------

  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        ImmutableBytes buffer, 
        boolean bound) throws IOException{
  return(this.write(objId,
                    offset,
                    length,
                    priority,
                    buffer,
                    bound,
                    Long.MAX_VALUE));
  }

 //-----------------------------------------------------------------------
  // write() -- caller promises not to ever change byte array, so
  //  we don't need to make a copy.
  //  
  //  This interface supports the maxHop for bound writes
  //
  //-----------------------------------------------------------------------

  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        ImmutableBytes buffer, 
        boolean bound,
        long maxBoundHops) throws IOException{
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "write(" + "objId="+objId.toString() 
                   + ", offset=" + offset
                   + ", length=" + length
                   + ", priority=" + priority
                   + ", buffer=" + buffer.toString()
                   + ", bound=" + bound
                   + ", maxBoundHops=" + maxBoundHops);
    }

     
     if(logToFile){
       //Please also UPDATE this.replayAllFromFile() & checkTwoLogs
       //if you update the parameters
     
      oos.writeInt(WRITE_1);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.writeDouble(priority);
      oos.writeObject(buffer);
      oos.writeBoolean(bound);
      oos.writeLong(maxBoundHops);
      oos.flush();
      oos.reset();
    }
    //System.err.println("performing local write to " + objId +
    // " : [" + offset + "," + length + "]");
    PreciseInv resultSummary = null;

    resultSummary = core.write(objId, offset, length, priority, buffer, bound, maxBoundHops);
    controller.informLocalWrite(objId,
                                offset,
                                length,
                                resultSummary.getAcceptStamp(),
			        bound, 
				false); 
    ObjInvalTarget t = (ObjInvalTarget)resultSummary.getInvalTarget();
    return t.getLength();
  }

  
  //-----------------------------------------------------------------------
  // copyBytesAndwrite() -- Caller might change byte array, so we need
  //  to make a copy that is immutable before returing.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
  public long
  write(ObjId objId, long offset, long length, byte buffer[], boolean bound) 
    throws IOException {
    // Pass the call onto the other version of this method
    return(this.write(objId,
                      offset,
                      length,
                      Core.DEFAULT_PRIORITY,
                      buffer,
                      bound));
  }

  //-----------------------------------------------------------------------
  // copyBytesAndwrite() -- Caller might change byte array, so we need
  //  to make a copy that is immutable before returing.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
//-----------------------------------------------------------------------
  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        byte buffer[],
        boolean bound) 
    throws IOException {
    return this.write(objId,
                      offset,
                      length,
                      priority,
                      buffer,
                      bound,
                      Long.MAX_VALUE);
  }
     

 public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        byte buffer[],
        boolean bound,
        long maxBoundHops) 
    throws IOException {
    
    
    ImmutableBytes b = new ImmutableBytes(buffer);
    buffer = null; // Don't use this pointer any more in here;
                   // caller still welcome to use it
    // Pass the call onto another version of this method
    return this.write(objId,
		      offset,
		      length,
		      priority,
		      b,
		      bound,
                      maxBoundHops);
  }

  //-----------------------------------------------------------------------
  // Version of the write() method that allows writing to multiple
  // object atomically
  //-----------------------------------------------------------------------
  public long[]
  writeMulti(MultiWriteEntry[] mwe, boolean embargoed)
    throws IOException{
    
    if(PRINT_METHOD){
      String paraStr ="writeMulti(";
      for(int i = 0; i < mwe.length; i++){
	paraStr = paraStr+ "mwe["+i+"]="+mwe[i].toString();
      }
      paraStr = paraStr + "embargoed="+embargoed;
      if ( PRINT_METHOD ){
	Env.dprintln(PRINT_METHOD, paraStr);
      }
    }
    if(logToFile){
      oos.writeInt(WRITE_MULTI);
      oos.writeInt(mwe.length);
      for(int i = 0; i < mwe.length; i++){
	oos.writeObject(mwe[i]);
      }
      oos.writeBoolean(embargoed);
      oos.flush();
      oos.reset();
    }
    
    //Env.dprintln(dbg, "LocalInterface::writeMulti"+mwe[0].getInvalTarget());
    Vector vec = null;
    long[] results = null;
    MultiObjInvalTarget moit = null;
    MultiObjPreciseInv mopi = null;
    ObjInvalTarget oit = null;

    //System.err.println("performing local write to " + objId + " : [" +
    //offset + "," + length + "]");
    mopi = this.core.writeMulti(mwe, embargoed);
    moit = (MultiObjInvalTarget)mopi.getInvalTarget();
    vec = new Vector();
    for(MOITIterator iter = moit.getIterator(true); iter.hasNext();){
      vec.add(iter.getNext());
    }
    results = new long[vec.size()];
    for(int i = 0; i < vec.size(); i++){
      results[i] = ((ObjInvalTarget)vec.get(i)).getLength();
    }

    //Env.dprintln(dbg, "return LocalInterface::writeMulti"+mwe[0].getInvalTarget());
    return(results);
  }

  //-----------------------------------------------------------------------
  //-----------------------------------------------------------------------
  // The following write interfaces are the same write interfaces as above 
  // but with Order Error Constrains and Embargoed option
  //-----------------------------------------------------------------------
  //-----------------------------------------------------------------------
  
  //-----------------------------------------------------------------------
  // write() -- caller promises not to ever change byte array, so
  //  we don't need to make a copy.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
  public long
  write(ObjId objId,
        long offset,
        long length,
        ImmutableBytes buffer, 
        boolean bound,
        long targetOE,
	boolean embargoed) throws IOException{
    // Pass the call onto another version of this method
    return(this.write(objId,
                      offset,
                      length,
                      Core.DEFAULT_PRIORITY,
                      buffer,
                      bound,
                      targetOE,
		      embargoed));
  }

  //-----------------------------------------------------------------------
  // write() -- caller promises not to ever change byte array, so
  //  we don't need to make a copy.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        ImmutableBytes buffer, 
        boolean bound,
        long targetOE,
	boolean embargoed) throws IOException{
    return this.write(objId,
                      offset,
                      length,
                      priority,
                      buffer,
                      bound,
                      Long.MAX_VALUE,
                      targetOE,
                      embargoed);
  }

  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        ImmutableBytes buffer, 
        boolean bound,
        long maxBoundHops,
        long targetOE,
	boolean embargoed) throws IOException{
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "write(" + "objId="+objId.toString() 
                   + ", offset=" + offset
                   + ", length=" + length
                   + ", priority=" + priority
                   + ", buffer=" + buffer.toString()
                   + ", bound=" + bound
                   + ", maxBoundHopes=" + maxBoundHops
                   + ", embargoed=" + embargoed);
    }
    if(logToFile){
      //Please also UPDATE this.replayAllFromFile() & checkTwoLogs
      //if you update the parameters
      oos.writeInt(WRITE_2);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.writeDouble(priority);
      oos.writeObject(buffer);
      oos.writeBoolean(bound);
      oos.writeLong(maxBoundHops);
      oos.writeLong(targetOE);
      oos.writeBoolean(embargoed);
      oos.flush();
      oos.reset();
    }
     
    //System.err.println("performing local write to " + objId +
    // " : [" + offset + "," + length + "]");
    PreciseInv resultSummary = null;

    resultSummary = core.write(objId, offset, length, priority, buffer, bound,
                               maxBoundHops, targetOE, embargoed);
    controller.informLocalWrite(objId,
                                offset,
                                length,
                                resultSummary.getAcceptStamp(),
                                bound,
                                embargoed); 
    ObjInvalTarget t = (ObjInvalTarget)resultSummary.getInvalTarget();
    return t.getLength();
  }

  //-----------------------------------------------------------------------
  // copyBytesAndwrite() -- Caller might change byte array, so we need
  //  to make a copy that is immutable before returing.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
  public long
  write(ObjId objId, 
        long offset, 
        long length, 
        byte buffer[], 
        boolean bound,
        long targetOE,
	boolean embargoed) 
    throws IOException {
    // Pass the call onto the other version of this method
    return(this.write(objId,
                      offset,
                      length,
                      Core.DEFAULT_PRIORITY,
                      buffer,
                      bound,
                      targetOE,
		      embargoed));
  }

  //-----------------------------------------------------------------------
  // copyBytesAndwrite() -- Caller might change byte array, so we need
  //  to make a copy that is immutable before returing.
  //
  // WARNING -- if write is not bound, then body is not in redo 
  // log --> data may be lost if we crash before next sync()
  // (The bound parameter was put in for debugging -- leave
  // it TRUE and unbind after sync unless you know what you
  // are doing!)
  //-----------------------------------------------------------------------
  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        byte buffer[],
        boolean bound,
        long targetOE,
	boolean embargoed) 
    throws IOException {
    return this.write(objId,
                      offset,
                      length,
                      priority,
                      buffer,
                      bound,
                      Long.MAX_VALUE,
                      targetOE,
                      embargoed);
                      
  }
  public long
  write(ObjId objId,
        long offset,
        long length,
        double priority,
        byte buffer[],
        boolean bound,
        long maxBoundHops,
        long targetOE,
	boolean embargoed) 
    throws IOException {
    //System.err.println("performing local write to " + objId + " : [" +
    //offset + "," + length + "]");
    ImmutableBytes b = new ImmutableBytes(buffer);
    buffer = null; // Don't use this pointer any more in here;
                   // caller still welcome to use it
    //Pass the call onto the other version of this method
    return this.write(objId, offset, length, priority, b, bound, maxBoundHops, targetOE, embargoed);
  }

/*
  //-----------------------------------------------------------------------
  // Version of the write() method that allows writing to multiple
  // object atomically
  //-----------------------------------------------------------------------
  public long[]
  writeMulti(MultiWriteEntry[] mwe, long targetOE, boolean embargoed)
    throws IOException{
    Vector vec = null;
    long[] results = null;
    MultiObjInvalTarget moit = null;
    MultiObjPreciseInv mopi = null;
    ObjInvalTarget oit = null;

    //System.err.println("performing local write to " + objId + " : [" +
    //offset + "," + length + "]");
    mopi = this.core.writeMulti(mwe, targetOE, embargoed);
    moit = (MultiObjInvalTarget)mopi.getInvalTarget();
    vec = new Vector();
    for(MOITIterator iter = moit.getIterator(true); iter.hasNext();){
      vec.add(iter.getNext());
    }
    results = new long[vec.size()];
    for(int i = 0; i < vec.size(); i++){
      results[i] = ((ObjInvalTarget)vec.get(i)).getLength();
    }
    // AMOL NOTE: Need to fix and uncomment folowing line!
    //controller.informLocalWrite(mwe, resultSummary.getAcceptStamp());
    return(results);
  }
*/

  //-----------------------------------------------------------------------
  // Delete this object
  //-----------------------------------------------------------------------
  public void
  delete(ObjId path)
  throws IOException{
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "delete(" + "path="+path.toString());
    }
    if(logToFile){
      oos.writeInt(DELETE);
      oos.writeObject(path);
      oos.flush();
      oos.reset();
    }
    controller.informLocalDelete(path);
    core.delete(path);
  }

  //-----------------------------------------------------------------------
  // Inform the controller about a read demand miss
  //-----------------------------------------------------------------------
  public void
  informDemandReadMiss(ObjId obj, long offset, long length){
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "informDemandReadMiss(" + "obj="+obj.toString() 
                   + ", offset=" + offset
                   + ", length=" + length);
    }
    if(logToFile){
      try{
	oos.writeInt(INFORM_DEMAND_READMISS);
	oos.writeObject(obj);
	oos.writeLong(offset);
	oos.writeLong(length);
	oos.flush();
	oos.reset();
      }catch(IOException e){
	 Env.dprintln(true, e.toString());
	 assert false;
       }

    }
    controller.informDemandReadMiss(obj, offset, length);
  }

  //-----------------------------------------------------------------------
  // Sync current system state to disk
  //-----------------------------------------------------------------------
  public void
  syncStateToDisk(){
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "syncStateToDisk");
    }

    if(logToFile){
      try{
	oos.writeInt(SYNC_STATE_TO_DISK);
	oos.flush();
	oos.reset();
       }catch(IOException e){
	 Env.dprintln(true, e.toString());
	 assert false;
       }

    }
    core.syncStateToDisk();
  }

  //-----------------------------------------------------------------------
  // Sync current system state until logical time "a" to disk
  //-----------------------------------------------------------------------
  public void sync(AcceptStamp a)
    throws java.lang.InterruptedException{
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "sync("+ a.toString());
    }
     if(logToFile){
       try{
	 oos.writeInt(SYNC);
	 oos.writeObject(a);
	 oos.flush();
	 oos.reset();
       }catch(IOException e){
	 Env.dprintln(true, e.toString());
	 assert false;
       }

    }
    core.syncCheck(a);
  }

  //----------------------------------------------------------------
  // checks if the object is valid
  //----------------------------------------------------------------  
  public boolean
  isValid(ObjId objId, long offset, long length)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException {
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "isValid(" + "objId="+objId.toString() 
                   + ", offset=" + offset
                   + ", length=" + length);
    }
    if(logToFile){
      oos.writeInt(ISVALID);
      oos.writeObject(objId);
      oos.writeLong(offset);
      oos.writeLong(length);
      oos.flush();
      oos.reset();
    }
    return core.isValid(objId, offset, length);
  }

  //----------------------------------------------------------------
  // Checks if the object is precise
  //----------------------------------------------------------------  
  public boolean
  isPrecise(ObjId objId){
    if(PRINT_METHOD){
      Env.dprintln(PRINT_METHOD, "isPrecise(" + "objId="+objId.toString());
    }
     if(logToFile){
       try{
	 oos.writeInt(ISPRECISE);
	 oos.writeObject(objId);
	 oos.flush();
	 oos.reset();
       }catch(IOException e){
	 Env.dprintln(true, e.toString());
	 assert false;
       }
    }
    return core.isPrecise(objId);
  }
  public void closeLogFile()
    throws IOException{
    if(oos !=null){
      oos.close();
    }
    if(fos != null){
      fos.close();
    }

  }

  public void
    replayAllFromFile(String fileName)
    {

    final boolean verbose = true;
    FileInputStream fis = null;
    ObjectInputStream ois = null;
    try{
      new File(fileName).createNewFile();
      fis = new FileInputStream(fileName);
      ois = new ObjectInputStream(fis);
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
    
    int methodID = -1;
    ObjId objId = null;
    long offset, length, maxE;
    boolean blockInvalid, blockImprecise, bound, embargoed;
    double priority;
    BodyMsg rResult;
    boolean bResult;
    long lResult;
    long[] lResults;
    ImmutableBytes buffer;
    int arrSize;
    long maxHop;
    long start = System.currentTimeMillis();
    MultiWriteEntry [] mwe;
    AcceptStamp a;
    long timeout;
    try{
      methodID = ois.readInt();
    }catch(Exception e){
      assert false;
    } 
    try{
      while(true){

	switch(methodID){
	case READ_1:
          
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  blockInvalid = ois.readBoolean();
	  blockImprecise = ois.readBoolean();
	  timeout = ois.readLong();
	  if ( verbose ){
	    Env.dprintln(verbose, "start read " + objId.toString());
	  }
	  try{
	    rResult=this.read(objId, offset, length, blockInvalid, blockImprecise, timeout);
	  }catch(Exception e){
	      System.err.println("ignore read Exception: "+e.toString()); 
          }
	  if ( verbose ){
	    Env.dprintln(verbose, "read " + objId.toString());
	  }
	  break;

	case READ_2:
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  blockInvalid = ois.readBoolean();
	  blockImprecise = ois.readBoolean();
	  maxE = ois.readLong();
	  timeout = ois.readLong();
	  if ( verbose ){
	    Env.dprintln(verbose, "start read " + objId.toString());
	  }
	  try{
	    rResult=this.read(objId, offset, length, blockInvalid, blockImprecise, maxE, timeout);
	  }catch(Exception e){
              System.err.println("ignore read Exception: "+e.toString());
          }
	  if ( verbose ){
	    Env.dprintln(verbose, "read " + objId.toString());
	  }
	  break;

	case READ_BLOCK_INVALID:
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  blockImprecise = ois.readBoolean();
	  if ( verbose ){
	    Env.dprintln(verbose, "start read " + objId.toString());
	  }
	  try{
	  rResult=this.readBlockInvalid(objId, offset, length, blockImprecise);
	  }catch(Exception e){
              System.err.println("ignore read Exception: "+e.toString());
          }
	  if ( verbose ){
	    Env.dprintln(verbose, "read " + objId.toString());
	  }
	  break;

	case WRITE_1:
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  priority = ois.readDouble();
	  buffer = (ImmutableBytes)(ois.readObject());
	  bound = ois.readBoolean();
	  if ( verbose ){
	    Env.dprintln(verbose, "start write " + objId.toString());
	  }
	  maxHop = ois.readLong();
	  lResult=this.write(objId, offset, length, priority, buffer, bound, maxHop);
	  if ( verbose ){
	    Env.dprintln(verbose, "write " + objId.toString());
	  }
	  break;

	case WRITE_MULTI:
	  arrSize = ois.readInt();
	  mwe = new MultiWriteEntry[arrSize];
	  for(int i = 0; i < arrSize; i++){
	    mwe[i] = (MultiWriteEntry)(ois.readObject());
	  }
	  
	  embargoed = ois.readBoolean();
	  if ( verbose ){
	    Env.dprintln(verbose, "start write " + mwe[0].getObjInvalTarget().toString());
	  }
	  lResults=this.writeMulti(mwe, embargoed);
	  if ( verbose ){
	    Env.dprintln(verbose, "write " + mwe[0].getObjInvalTarget().toString());
	  }

	  break;

	case WRITE_2:
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  priority = ois.readDouble();
	  buffer = (ImmutableBytes)(ois.readObject());
	  bound = ois.readBoolean();
	  maxHop = ois.readLong();
	  maxE = ois.readLong();
	  embargoed = ois.readBoolean();
	  if ( verbose ){
	    Env.dprintln(verbose, "start write " + objId.toString());
	  }

	  lResult=this.write(objId, offset, length, priority, buffer, bound, maxHop, maxE, embargoed);
	  if ( verbose ){
	    Env.dprintln(verbose, "write " + objId.toString());
	  }
	  break;

	case DELETE:
	  objId = (ObjId)(ois.readObject());
	  if ( verbose ){
	    Env.dprintln(verbose, "start delete " + objId.toString());
	  }
	  this.delete(objId);
	  if ( verbose ){
	    Env.dprintln(verbose, "delete " + objId.toString());
	  }
	  break;

	case INFORM_DEMAND_READMISS:
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  this.informDemandReadMiss(objId, offset, length);
	  break;

	case SYNC_STATE_TO_DISK:
	  this.syncStateToDisk();
	  break;

	case SYNC:
	  a = (AcceptStamp) (ois.readObject());
	  this.sync(a);
	  break;

	case ISVALID:
	  objId = (ObjId)(ois.readObject());
	  offset = ois.readLong();
	  length = ois.readLong();
	  bResult = this.isValid(objId, offset, length);
	  break;

	case ISPRECISE:
	  objId = (ObjId)(ois.readObject());
	  bResult=this.isPrecise(objId);
	  break;

	default:
	  System.err.println("corrupted file");
	  assert false;
	  System.exit(-1);
	}//switch
	try{
	  methodID = ois.readInt();
	}catch(EOFException e){
	  if(ois != null){
	    ois.close();
	  }
	  if(ois !=null){
	    ois.close();
	  }
	  long end = System.currentTimeMillis();
	  System.out.println("total replay time: " + (end-start));  
	  return;
	}
   }//while
   
   }catch(Exception ie){
     ie.printStackTrace();
	assert false;  
   }
   
  }

  static public void
    checkTwoLogs(String fileName1, String fileName2)
    throws IOException{
    
    FileInputStream fis1 = null;
    ObjectInputStream ois1 = null;
    FileInputStream fis2 = null;
    ObjectInputStream ois2 = null;
    try{
      
      fis1 = new FileInputStream(fileName1);
      ois1 = new ObjectInputStream(fis1);
      
      fis2 = new FileInputStream(fileName2);
      ois2 = new ObjectInputStream(fis2);
    }catch(Exception e){
      e.printStackTrace();
      assert false;
    }
    int methodID = -1;
    ObjId objId = null;
    long offset, length, maxE;
    boolean blockInvalid, blockImprecise, bound, embargoed;
    double priority;
    BodyMsg rResult;
    boolean bResult;
    long lResult;
    long[] lResults;
    ImmutableBytes buffer;
    int arrSize;
    byte[] data1;
    byte[] data2;

    MultiWriteEntry [] mwe1, mwe2;
    AcceptStamp a;

    System.out.println("11111");
    int count = 0;
    try{
      methodID = ois1.readInt();
      assert methodID == ois2.readInt();
      while(true){
	switch(methodID){
	case READ_1:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
	  
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readBoolean() == ois1.readBoolean();
	  assert ois2.readBoolean() == ois1.readBoolean();
	  assert ois2.readLong() == ois1.readLong();//timeout

	  break;

	case READ_2:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readBoolean() == ois1.readBoolean();
	  assert ois2.readBoolean() == ois1.readBoolean();

	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readLong() == ois1.readLong();//timeout
	  //rResult=this.read(objId, offset, length, blockInvalid, blockImprecise, maxE);
	  break;

	case READ_BLOCK_INVALID:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
          assert ois2.readLong() == ois1.readLong();
          assert ois2.readLong() == ois1.readLong();
          assert ois2.readBoolean() == ois1.readBoolean();

//	  offset = ois1.readLong();
//	  length = ois1.readLong();
//	  blockImprecise = ois1.readBoolean();
//	  rResult=this.readBlockInvalid(objId, offset, length, blockImprecise);
	  break;

	case WRITE_1:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readLong() == ois1.readLong();
	  
	  assert ois2.readDouble() == ois1.readDouble();
	  
	  buffer = (ImmutableBytes)(ois1.readObject());
	  data1 = buffer.dangerousGetReferenceToInternalByteArray();

	  buffer = (ImmutableBytes)(ois2.readObject());
	  data2 =  buffer.dangerousGetReferenceToInternalByteArray();
	  assert data1.length == data2.length;
	  for(int i = 0; i < data1.length; i++){
	    assert data1[i] == data2[i];
	  }

	  assert ois2.readBoolean() == ois1.readBoolean();

	  assert ois2.readLong() == ois1.readLong();//maxHop
	  //lResult=this.write(objId, offset, length, priority, buffer, bound);
	  break;

	case WRITE_MULTI:
	  arrSize = ois1.readInt();
	  assert ois2.readInt() == arrSize;
	  mwe1 = new MultiWriteEntry[arrSize];
	  mwe2 = new MultiWriteEntry[arrSize];
	  for(int i = 0; i < arrSize; i++){
	    mwe1[i] = (MultiWriteEntry)(ois1.readObject());
	    mwe2[i] = (MultiWriteEntry)(ois2.readObject());
	    assert mwe1[i].equals(mwe2[i]);
	  }
	  assert ois2.readBoolean() == ois1.readBoolean();
	  //embargoed = ois1.readBoolean();
	  //lResults=this.writeMulti(mwe, embargoed);
	  break;

	case WRITE_2:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
          assert ois2.readLong() == ois1.readLong();
          assert ois2.readLong() == ois1.readLong();
          assert ois2.readDouble() == ois1.readDouble();

//	  offset = ois1.readLong();
//	  length = ois1.readLong();
//	  priority = ois1.readDouble();
	  buffer = (ImmutableBytes)(ois1.readObject());
          data1 = buffer.dangerousGetReferenceToInternalByteArray();

          buffer = (ImmutableBytes)(ois2.readObject());
          data2 =  buffer.dangerousGetReferenceToInternalByteArray();
          assert data1.length == data2.length;
          for(int i = 0; i < data1.length; i++){
            assert data1[i] == data2[i];
          }

          assert ois2.readBoolean() == ois1.readBoolean();
	  assert ois2.readLong() == ois1.readLong();//maxhop
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readBoolean() == ois1.readBoolean();

	  //bound = ois1.readBoolean();
	  //maxE = ois1.readLong();
	  //embargoed = ois1.readBoolean();
	  //lResult=this.write(objId, offset, length, priority, buffer, bound, maxE, embargoed);
	  break;

	case DELETE:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
	  //this.delete(objId);
	  break;

	case INFORM_DEMAND_READMISS:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
	  assert ois2.readLong() == ois1.readLong();
	  assert ois2.readLong() == ois1.readLong();

	  //offset = ois1.readLong();
	  //length = ois1.readLong();
	  //this.informDemandReadMiss(objId, offset, length);
	  break;

	case SYNC_STATE_TO_DISK:
	  //this.syncStateToDisk();
	  break;

	case SYNC:
	  a = (AcceptStamp) (ois1.readObject());
	  ((AcceptStamp)(ois2.readObject())).equals(a);
	  //this.sync(a);
	  break;

	case ISVALID:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
          assert ois2.readLong() == ois1.readLong();
          assert ois2.readLong() == ois1.readLong();

	  //offset = ois1.readLong();
	  //length = ois1.readLong();

	  //bResult = this.isValid(objId, offset, length);
	  break;

	case ISPRECISE:
	  objId = (ObjId)(ois1.readObject());
	  assert ((ObjId)(ois2.readObject())).equals(objId);
	  //bResult=this.isPrecise(objId);
	  break;

	default:
	  System.err.println("corrupted file");
	  assert false;
	  System.exit(-1);
	}//switch
	try{
	  methodID = ois1.readInt();
	}catch(EOFException e){
	  if(ois1 != null){
	    ois1.close();
	  }
	  try{
	    assert methodID == ois2.readInt();
	  }catch(EOFException ee){
	    if(ois2 != null){
	      ois2.close();
	    }
	    System.out.println("the two files are identical");
	    //expected
	    return;
	  }
	  assert false;
	}
	assert methodID == ois2.readInt();
	if ( dbg ){
	  Env.dprintln(dbg, "round:"+count);
	}
	count++;
      }//while
   
    }catch(Exception e){
      if(ois1 != null){
	ois1.close();
      }
      if(ois1 !=null){
	ois1.close();
      }
      if(ois2 != null){
	ois2.close();
      }
      if(ois2 !=null){
	ois2.close();
      }
      e.printStackTrace();
      assert false;
    }
    
  }

  static void unitTest() throws Exception{
    
    String configFileName = "test/LIUnit.config";
    NodeId myId = new NodeId(1);
    boolean cleanDb = true;
    boolean noSyncLog = true;
    boolean del = true;
    boolean logLITOFile = true;
    String logLIFileName = "test/LIUnit.log";
    URANode uraNode = new URANode(configFileName,
				  myId,
				  Controller.NULL_CONTROLLER,
				  cleanDb, noSyncLog,
				  logLITOFile, logLIFileName);
    LocalInterface li = uraNode.getLocalInterface();
    
    ObjId objId1, objId2, objId3, objId4;
    long offset, length, maxE;
    boolean blockInvalid, blockImprecise, bound, embargoed;
    double priority;
    BodyMsg rResult;
    boolean bResult;
    long lResult;
    long[] lResults;
    ImmutableBytes buffer;
    int arrSize;
    int bodyLen=5;
    byte[] bdy1 = new byte[bodyLen];
    bdy1[0] = 'o';
    bdy1[1] = 'b';
    bdy1[2] = 'j';
    bdy1[3] = '0';
    bdy1[4] = '1';

    byte[] bdy2 = new byte[bodyLen];
    bdy2[0] = 'o';
    bdy2[1] = 'b';
    bdy2[2] = 'j';
    bdy2[3] = '0';
    bdy2[4] = '2';

    byte[] bdy3 = new byte[bodyLen];
    bdy3[0] = 'o';
    bdy3[1] = 'b';
    bdy3[2] = 'j';
    bdy3[3] = '0';
    bdy3[4] = '3';

    byte[] bdy4 = new byte[bodyLen];
    bdy4[0] = 'o';
    bdy4[1] = 'b';
    bdy4[2] = 'j';
    bdy4[3] = '0';
    bdy4[4] = '4';

    //WRITE_1
    objId1 = new ObjId("/obj1");
    offset = 0;
    length = bodyLen;
    buffer = new ImmutableBytes(bdy1);
    bound = true;
    lResult = li.write(objId1, offset, length, buffer, bound);
    assert lResult == bodyLen;

    //WRITE_MULTI
    priority = 0.0;
    bound = true;
    del = false;
    MultiWriteEntry[] mwe = new MultiWriteEntry[2];

    objId2 = new ObjId("/obj2");
    buffer = new ImmutableBytes(bdy2);    
    mwe[0]= new MultiWriteEntry(objId2, offset, length, priority, buffer, bound, del);
    
    objId3 = new ObjId("/obj3");
    buffer = new ImmutableBytes(bdy3);
    mwe[1]=  new MultiWriteEntry(objId3, offset, length, priority, buffer, bound, del);
   
    objId4 = new ObjId("/obj4"); 
    embargoed = false;
    lResults=li.writeMulti(mwe, embargoed);
    assert lResults.length == 2;
    assert lResults[0] == bodyLen;
    assert lResults[1] == bodyLen;
    
    //WRITE_2
    buffer = new ImmutableBytes(bdy4);
    maxE = 10000;
    lResult=li.write(objId4, offset, length, priority, buffer, bound, maxE, embargoed);
    assert lResult == bodyLen;

    //read1
    blockInvalid = true;
    blockImprecise = true;
    rResult = li.read(objId1, offset, length, blockInvalid, blockImprecise);
    assert rResult.getBody().dangerousGetReferenceToInternalByteArray()[4] == '1';

    //read2
    rResult = li.read(objId2, offset, length, blockInvalid, blockImprecise, Long.MAX_VALUE);
    assert rResult.getBody().dangerousGetReferenceToInternalByteArray()[4] == '2';

    li.delete(objId1);
    li.informDemandReadMiss(objId1, offset, length);
    li.syncStateToDisk();
    
    //li.sync(new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, new NodeId(1)));
    assert li.isValid(objId3, offset, length);

    assert li.isPrecise(objId4);
    
    li.closeLogFile();
   
    Thread.sleep(1000);
    String logLIFileName2 = "test/LIUnit.log2";
    URANode uraNode2 = new URANode(configFileName,
				   new NodeId(2),
				  Controller.NULL_CONTROLLER,
				  cleanDb, noSyncLog,
				  logLITOFile, logLIFileName2);
    LocalInterface li2 = uraNode2.getLocalInterface();
    li2.replayAllFromFile(logLIFileName);
    li2.closeLogFile();
    System.out.println("start checking");
    //now check if the two output files are identical
    LocalInterface.checkTwoLogs(logLIFileName, logLIFileName2);
    uraNode.shutdown();
    uraNode2.shutdown();
    

  }
  
  //---------------------------------------------------------------------------
  //self test
  //---------------------------------------------------------------------------
  public static void main(String[] argv)
  throws Exception{
    String configFileName = "test/pfs.config";
    NodeId myId = new NodeId(1);
    boolean cleanDb = true;
    boolean noSyncLog = true;
    boolean del = true;
    boolean logLITOFile = false;
    String logLIFileName = "test/LIOp.log";
    URANode uraNode = new URANode(configFileName,
                                  myId,
                                  Controller.NULL_CONTROLLER,
                                  cleanDb, noSyncLog,
                                  logLITOFile, logLIFileName);
    LocalInterface li = uraNode.getLocalInterface();
    li.replayAllFromFile(logLIFileName);
    li.closeLogFile();
    System.out.println("replay finish!!!!!!!!!!");
  }
}

//---------------------------------------------------------------------------
/* $Log: LocalInterface.java,v $
/* Revision 1.41  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
/* Revision 1.40  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.39  2007/03/15 21:58:31  dahlin
/* Added experimetn to test BW for subscribe for SOSP paper
/*
/* Revision 1.38  2007/03/11 04:16:54  zjiandan
/* Add timeout into read interface and expose acceptstamp in LocalInterface.write.
/*
/* Revision 1.37  2007/03/01 08:12:37  nalini
/* added maxBoundHop support for MultiObjWrites
/*
/* Revision 1.36  2007/03/01 06:39:44  nalini
/* added support for maxBoundHop at Local Interface
/*
/* Revision 1.35  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.34  2007/02/23 20:54:28  zjiandan
/* Fixed mem leak problems in NFS2Interface and some other bugs.
/* Andrew benchmark passed, performance still needs to tune.
/*
/* Revision 1.33  2006/12/08 23:13:18  nalini
/* reverted to old read interface
/*
/* Revision 1.26  2006/09/01 21:32:24  dahlin
/* PicSharReader test now works for case when writer dies and restarts
/*
/* Revision 1.25  2006/07/21 05:42:25  nayate
/* Added a version of writeMulti() without targetOE
/*
/* Revision 1.24  2006/06/22 18:21:39  nalini
/* added isPrecise and isValid to LocalInterface
/*
/* Revision 1.23  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.22  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.21  2005/07/18 05:10:23  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.20  2005/03/14 09:15:27  nayate
/* Added a priority parameter
/*
/* Revision 1.19  2005/03/10 19:38:55  lgao
/* *** empty log message ***
/*
/* Revision 1.18  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.17  2004/10/22 20:46:55  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.16  2004/07/28 20:18:25  dahlin
/* encapsulated byte[] bodies in ImmutableBytes for safety
/*
/* Revision 1.15  2004/05/26 12:28:11  arun
/* sdims debugging
/*
/* Revision 1.14  2004/05/25 07:19:35  nayate
/* Removed a System.out.println
/*
/* Revision 1.13  2004/05/24 21:16:23  arun
/* minor corrections to make unbind messages with sdims work
/*
/* Revision 1.12  2004/05/23 21:15:35  arun
/* informLocalWrite added to URANode, LocalInterface on local write.
/*
/* Revision 1.11  2004/05/22 10:10:12  arun
/* minor changes to notifyUpq logic. Correction of store behaviour while handling bound invals.
/*
/* Revision 1.10  2004/05/20 01:06:11  arun
/* added some code for recovery support in store. Added maxVV definition in COunterVV etc.
/*
/* Revision 1.9  2004/05/19 04:34:55  dahlin
/* SDIMSController test compiles and runs but deadlocks
/*
/* Revision 1.8  2004/05/15 00:23:17  lgao
/* Add flag to distingurish demand fetch from prefetch data.
/*
/* Revision 1.7  2004/05/13 21:58:14  dahlin
/* simple unbind implemented
/*
/* Revision 1.6  2004/05/11 03:49:45  zjiandan
/* Made some changes to get the first version to work.
/*
/* Revision 1.5  2004/05/10 21:00:22  lgao
/* Just to make things compile.
/*
/* Revision 1.4  2004/04/30 19:22:55  nayate
/* Added a constructor
/*
/* Revision 1.3  2004/04/29 21:44:12  lgao
/* For Compilation purposes
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
