package code;
 /** 
 *  MOITBoundEntry: Used to represent entries that are bound in a 
 *  MultiObjPreciseInv. 
 **/ 
import java.io.Serializable;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.io.ObjectOutputStream;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class MOITBoundEntry implements Serializable, Immutable
{

 /** 
 *  Constants 
 **/ 
  private final static short MAGIC_NUM = 0x1F2F;

 /** 
 *  Data members 
 **/ 
  private final ObjInvalTarget oit;
  private final double priority;
  private final ImmutableBytes ib;

 /** 
 *  Constructor 
 **/ 
  public
  MOITBoundEntry(final ObjInvalTarget newOIT,
                 final double newPriority,
                 final ImmutableBytes newIB){
    this.oit = newOIT;
    this.priority = newPriority;
    this.ib = newIB;
  }

 /** 
 *  Constructor: Read this object from an input stream 
 **/ 
  public
  MOITBoundEntry(InputStream is) throws IOException{
    ObjectInputStream ois = null;
    short magicNum = 0;

    ois = new ObjectInputStream(is);
    magicNum = ois.readShort();
    assert(magicNum == MOITBoundEntry.MAGIC_NUM);
    this.oit = new ObjInvalTarget(ois);
    this.priority = ois.readDouble();
    this.ib = this.readImmutableBytes(ois);
  }

 /** 
 *  Write this object to the given object output stream 
 **/ 
  public void
  writeSelf(OutputStream os) throws IOException{
    ObjectOutputStream oos = null;

    oos = new ObjectOutputStream(os);
    oos.writeShort(MOITBoundEntry.MAGIC_NUM);
    this.oit.copySelfOntoOS(oos);
    oos.writeDouble(this.priority);
    this.writeImmutableBytes(oos, this.ib);
  }

 /** 
 *  Return the ObjInvalTarget 
 **/ 
  public final ObjInvalTarget
  getObjInvalTarget(){
    return(this.oit);
  }

 /** 
 *  Return the priority 
 **/ 
  public final double
  getPriority(){
    return(this.priority);
  }

 /** 
 *  Return the data 
 **/ 
  public final ImmutableBytes
  getImmutableBytes(){
    return(this.ib);
  }

 /** 
 *  Return a hash code for this object (note: not implemented) 
 **/ 
  public final int
  hashCode(){
    System.err.println("MOITBoundEntry.hashCode() not implemented");
    assert(false);
    return(0);
  }

 /** 
 *  Return true if this equals "obj" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean result = false;
    MOITBoundEntry moitBE = null;

    // 2 Java-provided fields, 4 declared fields
    assert(this.getClass().getDeclaredFields().length == 6);
    if(obj instanceof MOITBoundEntry){
      moitBE = (MOITBoundEntry)obj;
      result = ((this.oit.equals(moitBE.oit)) &&
                (this.priority == moitBE.priority) &&
                (this.ib.equals(moitBE.ib)));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Write immutable bytes to the ObjectOutputStream 
 **/ 
  public static void
  writeImmutableBytes(ObjectOutputStream oos, ImmutableBytes immBytes)
    throws IOException{
    byte[] readOnlyBuffer = null;

    readOnlyBuffer = immBytes.dangerousGetReferenceToInternalByteArray();
    oos.writeObject(readOnlyBuffer);
    readOnlyBuffer = null;
  }

 /** 
 *  Read immutable bytes from the ObjectInputStream 
 **/ 
  public static ImmutableBytes
  readImmutableBytes(ObjectInputStream ois) throws IOException{
    byte[] readOnlyBuffer = null;
    ImmutableBytes immBytes = null;

    try{
      readOnlyBuffer = (byte[])ois.readObject();
      immBytes = ImmutableBytes.dangerousCreateImmutableBytes(readOnlyBuffer);
      readOnlyBuffer = null;
    }catch(ClassNotFoundException e){
      assert false : ("" + e);
    }
    return(immBytes);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing MOITBoundEntry.java...");
    MOITBoundEntry.test1();
    MOITBoundEntry.test2();
    System.out.println("...Finished");
  }

 /** 
 *  Test 
 **/ 
  private static void
  test1(){
    MOITBoundEntry moitBE1 = null;
    MOITBoundEntry moitBE2 = null;
    ObjInvalTarget oit1 = null;

    // Test 1
    moitBE1 = new MOITBoundEntry(new ObjInvalTarget(new ObjId("/a"), 10, 10),
                                 5.5,
                                 new ImmutableBytes(new byte[7]));
    assert(moitBE1.getObjInvalTarget().equals(new
                                              ObjInvalTarget(new ObjId("/a"),
                                                             10,
                                                             10)));
    assert(moitBE1.getPriority() == 5.5);
    assert(moitBE1.getImmutableBytes().getLength() == 7);
    assert(moitBE1.equals(moitBE1));

    // Test 2
    moitBE2 = new MOITBoundEntry(new ObjInvalTarget(new ObjId("/b"), 20, 30),
                                 3.14,
                                 new ImmutableBytes(new byte[3]));
    assert(moitBE2.getObjInvalTarget().equals(new
                                              ObjInvalTarget(new ObjId("/b"),
                                                             20,
                                                             30)));
    assert(moitBE2.getPriority() == 3.14);
    assert(moitBE2.getImmutableBytes().getLength() == 3);
    assert(!moitBE1.equals(moitBE2));
    assert(moitBE2.equals(moitBE2));

    // Test 3
    moitBE2 = new MOITBoundEntry(new ObjInvalTarget(new ObjId("/a"), 10, 10),
                                 5.5,
                                 new ImmutableBytes(new byte[7]));
    assert(moitBE1.equals(moitBE2));
    assert(moitBE2.equals(moitBE1));
  }

 /** 
 *  Test serialization/deserialization 
 **/ 
  private static void
  test2(){
    MOITBoundEntry moitBE1 = null;
    MOITBoundEntry moitBE2 = null;
    byte[] buffer = null;
    ByteArrayInputStream bais = null;
    ByteArrayOutputStream baos = null;

    try{
      baos = new ByteArrayOutputStream();
      moitBE1 = new MOITBoundEntry(new ObjInvalTarget(new ObjId("/a"), 10, 10),
                                   5.5,
                                   new ImmutableBytes(new byte[7]));
      moitBE1.writeSelf(baos);
      baos.flush();
      buffer = baos.toByteArray();
      bais = new ByteArrayInputStream(buffer);
      moitBE2 = new MOITBoundEntry(bais);
      assert(moitBE1 != moitBE2);
      assert(moitBE1.equals(moitBE2));
    }catch(IOException e){
      assert false : ("" + e);
    }
  }
}

 /** 
 *  $Log: MOITBoundEntry.java,v $ 
 *  Revision 1.2  2006/07/19 22:34:38  nayate 
 *  Added serialization 
 *  
 *  Revision 1.1  2006/04/04 15:59:59  nayate 
 *  Added the ability to (1) delay invalidates, and (2) support transactional updates. 
 *  
 **/ 
