package code;
 /** 
 *  MultiObjInvalTarget: Used to represent an invalidate/delete message 
 *  that atomically invalidates/deletes multiple objects simultaneously 
 **/ 
import java.io.Serializable;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.util.HashMap;
import java.util.Set;
import java.util.Iterator;
import java.util.Collection;
import java.util.Map;
import java.util.Vector;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

public final class MultiObjInvalTarget
  implements InvalTarget, Serializable, Cloneable, Immutable{

 /** 
 *  Private data 
 **/ 
  private final ObjInvalTarget[] invalidated;
  private final ObjInvalTarget[] deleted;

  private static final short MULTI_OBJ_INVAL_TARGET_MAGIC = 0x5B;

 /** 
 *  Constructor 
 **/ 
  public
  MultiObjInvalTarget(ObjectInputStream ois) throws IOException{
    short magicNum = 0;

    // Read the magic number
    magicNum = ois.readShort();
    assert(magicNum == MultiObjInvalTarget.MULTI_OBJ_INVAL_TARGET_MAGIC);

    // Read the list of invalidated objects
    this.invalidated = new ObjInvalTarget[ois.readInt()];
    for(int i = 0; i < this.invalidated.length; i++){
      this.invalidated[i] = new ObjInvalTarget(ois);
    }

    // Read the list of deleted objects
    this.deleted = new ObjInvalTarget[ois.readInt()];
    for(int i = 0; i < this.deleted.length; i++){
      this.deleted[i] = new ObjInvalTarget(ois);
    }
    assert(this.isValid());
}

 /** 
 *  Constructor 
 **/ 
  public
  MultiObjInvalTarget(ObjInvalTarget[] newInvalidated,
                      ObjInvalTarget[] newDeleted){
    this.invalidated = new ObjInvalTarget[newInvalidated.length];
    this.deleted = new ObjInvalTarget[newDeleted.length];
    for(int i = 0; i < newInvalidated.length; i++){
      this.invalidated[i] = newInvalidated[i];
    }
    for(int i = 0; i < newDeleted.length; i++){
      this.deleted[i] = newDeleted[i];
    }
    assert(this.isValid());
  }

 /** 
 *  Send a copy of myself onto an ObjectOutputStream 
 **/ 
  public void
  copySelfOntoOOS(ObjectOutputStream oos) throws IOException{
    int numDeletedObjects = 0;
    int numInvalidatedObjects = 0;

    // Send the magic number
    oos.writeShort(MultiObjInvalTarget.MULTI_OBJ_INVAL_TARGET_MAGIC);

    // Send the invalidated object list
    oos.writeInt(this.invalidated.length);
    for(int i = 0; i < this.invalidated.length; i++){
      this.invalidated[i].copySelfOntoOS(oos);
    }

    // Send the deleted object list
    oos.writeInt(this.deleted.length);
    for(int i = 0; i < this.deleted.length; i++){
      this.deleted[i].copySelfOntoOS(oos);
    }
  }

 /** 
 *  Make a copy of this object (make use of immutability) 
 **/ 
  public Object
  clone(){
    return(this);
  }

 /** 
 *  Return true if this MultiObjInvalTarget intersects the passed-in one 
 **/ 
  public boolean
  intersects(SubscriptionSet ss){
    boolean intersection = false;

    for(int i = 0; (!intersection) && (i < this.invalidated.length); i++){
      if(this.invalidated[i].intersects(ss)){
        intersection = true;
      }
    }
    for(int i = 0; (!intersection) && (i < this.deleted.length); i++){
      if(this.deleted[i].intersects(ss)){
        intersection = true;
      }
    }
    return(intersection);
  }

 /** 
 *  Return the intersection of this object and "it" 
 **/ 
  public InvalTarget
  getIntersection(InvalTarget it){
    InvalTarget result = null;
    InvalTarget current = null;
    SubscriptionSet allSS = null;

    if(this.equals(it)){
      result = this;
    }else if(it instanceof HierInvalTarget){
      allSS = SubscriptionSet.makeSubscriptionSet(":/*");
      for(int i = 0; i < this.invalidated.length; i++){
        current = this.invalidated[i].getIntersection(it);
        if(!current.isEmpty()){
          if(result == null){
            result = current;
          }else{
            result = current.getUnion(result, allSS);
          }
        }
      }
      for(int i = 0; i < this.deleted.length; i++){
        current = this.deleted[i].getIntersection(it);
        if(!current.isEmpty()){
          if(result == null){
            result = current;
          }else{
            result = current.getUnion(result, allSS);
          }
        }
      }
      if(result == null){
        // No intersection; return an empty invalidate
        result = new HierInvalTarget();
      }
    }else{
      System.err.println("MultiObjInvalIterator.getIntersection: Class " +
                         it.getClass().getName());
      assert(false);
    }
    return(result);
  }

  //-------------------------------------------------------------------------
  // Return the union of this and the passed-in InvalTarget
  //-------------------------------------------------------------------------
  public InvalTarget
  getUnion(InvalTarget it, SubscriptionSet ss){
    InvalTarget result = null;
    SubscriptionSet allSS = null;

    if(this.equals(it)){
      return(this);
    }else{
      allSS = SubscriptionSet.makeSubscriptionSet(":/*");
      for(int i = 0; i < this.invalidated.length; i++){
        if(result == null){
          result = this.invalidated[i];
        }else{
          result = result.getUnion(this.invalidated[i], allSS);
        }
      }
      for(int i = 0; i < this.deleted.length; i++){
        if(result == null){
          result = this.deleted[i];
        }else{
          result = result.getUnion(this.deleted[i], allSS);
        }
      }
      assert(result != null);
    }
    return(result.getUnion(it, ss));
  }

  //-------------------------------------------------------------------------
  // Return true if this instance of InvalTarget is empty (which it isn't!)
  //-------------------------------------------------------------------------
  public boolean
  isEmpty(){
    return(false);
  }

  //-------------------------------------------------------------------------
  // Return true if the passed-in object is equal to this
  //-------------------------------------------------------------------------
  public boolean
  equals(Object obj){
    Vector moitIO = null;
    Vector moitDO = null;
    boolean eq = true;
    MultiObjInvalTarget moit = null;

    // Two java-provided fields, three local variables
    assert(this.getClass().getDeclaredFields().length == 5);

    if(obj instanceof MultiObjInvalTarget){
      moit = (MultiObjInvalTarget)obj;
      moitIO = new Vector();
      moitDO = new Vector();
      for(int i = 0; i < moit.invalidated.length; i++){
        moitIO.add(moit.invalidated[i]);
      }
      for(int i = 0; i < moit.deleted.length; i++){
        moitDO.add(moit.deleted[i]);
      }
      if((moitIO.size() == this.invalidated.length) &&
         (moitDO.size() == this.deleted.length)){
        for(int i = 0; eq && (i < this.invalidated.length); i++){
          eq = moitIO.contains(this.invalidated[i]);
        }
        for(int i = 0; eq && (i < this.deleted.length); i++){
          eq = moitDO.contains(this.deleted[i]);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

  //-------------------------------------------------------------------------
  // Return a hashCode for this object; currently should not be called
  //-------------------------------------------------------------------------
  public int
  hashCode(){
    // Should not be called!
    assert(false);
    return(0);
  }

  //-------------------------------------------------------------------------
  // Make a HierInvalTarget representation of this MultiObjInvalTarget
  //-------------------------------------------------------------------------
  public HierInvalTarget
  makeHierInvalTarget(){
    SubscriptionSet allSS = null;
    InvalTarget it = null;
    HierInvalTarget result = null;
    String path = null;

    // Note: We assume that HierInvalTarget.getUnion(anything) returns
    // another HierInvalTarget
    allSS = SubscriptionSet.makeSubscriptionSet(":/*");
    for(int i = 0; i < this.invalidated.length; i++){
      if(result == null){
        path = this.invalidated[i].getObjId().getPath();
        result = HierInvalTarget.makeHierInvalTarget(path);
      }else{
        it = result.getUnion(this.invalidated[i], allSS);
        assert(it instanceof HierInvalTarget);
        result = (HierInvalTarget)it;
      }
    }
    for(int i = 0; i < this.deleted.length; i++){
      if(result == null){
        path = this.deleted[i].getObjId().getPath();
        result = HierInvalTarget.makeHierInvalTarget(path);
      }else{
        it = result.getUnion(this.deleted[i], allSS);
        assert(it instanceof HierInvalTarget);
        result = (HierInvalTarget)it;
      }
    }
    assert(result != null);
    return(result);
  }

  //-------------------------------------------------------------------------
  // Get an iterator over elements invalidated/deleted by this MOIT
  //-------------------------------------------------------------------------
  public MOITIterator
  getIterator(boolean invals){
    MOITIterator moitIter = null;

    if(invals){
      moitIter = new MOITIterator(this.invalidated);
    }else{
      moitIter = new MOITIterator(this.deleted);
    }
    return(moitIter);
  }

  //-------------------------------------------------------------------------
  // Convert this object to a string representation
  //-------------------------------------------------------------------------
  public String
  toString(){
    String str = null;

    str = "I:";
    if(this.invalidated.length > 0){
      str += "(" + this.invalidated[0];
      for(int i = 1; i < this.invalidated.length; i++){
        str += ", " + this.invalidated[i];
      }
      str += ")";
    }
    str += ", D:";
    if(this.deleted.length > 0){
      str += "(" + this.deleted[0];
      for(int i = 1; i < this.deleted.length; i++){
        str += ", " + this.deleted[i];
      }
      str += ")";
    }
    return(str);
  }

  //-------------------------------------------------------------------------
  // Return true if this data structure is "valid" - i.e., no internal
  // invalidates/deletes intersect with each other
  //-------------------------------------------------------------------------
  private boolean
  isValid(){
    boolean valid = true;
    ObjId myObjId = null;
    ObjId otherObjId = null;
    long myOffset = 0;
    long myLength = 0;
    long otherOffset = 0;
    long otherLength = 0;

    for(int i = 0; valid && (i < this.invalidated.length); i++){
      myObjId = this.invalidated[i].getObjId();
      myOffset = this.invalidated[i].getOffset();
      myLength = this.invalidated[i].getLength();
      for(int j = 0; valid && (j < this.invalidated.length); j++){
        otherObjId = this.invalidated[j].getObjId();
        otherOffset = this.invalidated[j].getOffset();
        otherLength = this.invalidated[j].getLength();
        if((i != j) && (myObjId.equals(otherObjId))){
          valid = (((myOffset + myLength) <= otherOffset) ||
                   ((otherOffset + otherLength) <= myOffset));
        }
      }
      for(int j = 0; valid && (j < this.deleted.length); j++){
        otherObjId = this.deleted[j].getObjId();
        otherOffset = this.deleted[j].getOffset();
        otherLength = this.deleted[j].getLength();
        valid = ((!myObjId.equals(otherObjId)) ||
                 ((myOffset + myLength) <= otherOffset) ||
                 ((otherOffset + otherLength) <= myOffset));
      }
    }
    for(int i = 0; valid && (i < this.deleted.length); i++){
      myObjId = this.deleted[i].getObjId();
      myOffset = this.deleted[i].getOffset();
      myLength = this.deleted[i].getLength();
      for(int j = 0; valid && (j < this.deleted.length); j++){
        otherObjId = this.deleted[j].getObjId();
        otherOffset = this.deleted[j].getOffset();
        otherLength = this.deleted[j].getLength();
        if((i != j) && (myObjId.equals(otherObjId))){
          valid = (((myOffset + myLength) <= otherOffset) ||
                   ((otherOffset + otherLength) <= myOffset));
        }
      }
      for(int j = 0; valid && (j < this.invalidated.length); j++){
        otherObjId = this.invalidated[j].getObjId();
        otherOffset = this.invalidated[j].getOffset();
        otherLength = this.invalidated[j].getLength();
        valid = ((!myObjId.equals(otherObjId)) ||
                 ((myOffset + myLength) <= otherOffset) ||
                 ((otherOffset + otherLength) <= myOffset));
      }
    }
    if(!valid){
      System.out.println("" + this);
    }
    return(valid);
  }

  public int onDiskSize(){
    return invalidated.length + deleted.length;
  }
  //-------------------------------------------------------------------------
  // Used for testing
  //-------------------------------------------------------------------------
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing MultiObjInvalTarget...");
    MultiObjInvalTarget.test1();
    MultiObjInvalTarget.test2();
    MultiObjInvalTarget.test3();
    MultiObjInvalTarget.test4();
    MultiObjInvalTarget.test5();
    MultiObjInvalTarget.test6();
    MultiObjInvalTarget.test7();
    System.out.println("...Finished");
  }

  //-------------------------------------------------------------------------
  // Test equals() and isValid()
  //-------------------------------------------------------------------------
  private static void
  test1(){
    MultiObjInvalTarget moit1 = null;
    MultiObjInvalTarget moit2 = null;
    MultiObjInvalTarget moit3 = null;
    ObjInvalTarget[] invArr1 = null;
    ObjInvalTarget[] delArr1 = null;
    ObjInvalTarget[] invArr2 = null;
    ObjInvalTarget[] delArr2 = null;
    ObjInvalTarget[] invArr3 = null;
    ObjInvalTarget[] delArr3 = null;

    // Test 1: X.equals(X)
    invArr1 = new ObjInvalTarget[1];
    delArr1 = new ObjInvalTarget[2];
    invArr1[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    delArr1[0] = new ObjInvalTarget(new ObjId("/b"), 20, 30);
    delArr1[1] = new ObjInvalTarget(new ObjId("/c"), 20, 30);
    moit1 = new MultiObjInvalTarget(invArr1, delArr1);
    assert(moit1.equals(moit1));

    // Test 2: !X.equals(Y)
    invArr2 = new ObjInvalTarget[2];
    delArr2 = new ObjInvalTarget[1];
    invArr2[0] = new ObjInvalTarget(new ObjId("/b"), 20, 30);
    invArr2[1] = new ObjInvalTarget(new ObjId("/c"), 20, 30);
    delArr2[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    moit2 = new MultiObjInvalTarget(invArr2, delArr2);
    assert(moit2.equals(moit2));
    assert(!moit1.equals(moit2));
    assert(!moit2.equals(moit1));
    // Already tested the next lines
    // System.out.println("" + moit1);
    // System.out.println("" + moit2);

    // Test 3: isValid() should be true
    assert(moit1.isValid());
    assert(moit2.isValid());
    invArr3 = new ObjInvalTarget[1];
    delArr3 = new ObjInvalTarget[1];
    invArr3[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    delArr3[0] = new ObjInvalTarget(new ObjId("/a"), 30, 20);
    moit3 = new MultiObjInvalTarget(invArr3, delArr3);
    assert(moit3.isValid());

    // Test 4: isValid() should not be true
    invArr3 = new ObjInvalTarget[1];
    delArr3 = new ObjInvalTarget[1];
    invArr3[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    delArr3[0] = new ObjInvalTarget(new ObjId("/a"), 15, 10);
    // The next line correctly fails
    //moit3 = new MultiObjInvalTarget(invArr3, delArr3);

    invArr3 = new ObjInvalTarget[2];
    delArr3 = new ObjInvalTarget[0];
    invArr3[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    invArr3[1] = new ObjInvalTarget(new ObjId("/a"), 15, 1);
    // The next line correctly fails
    //moit3 = new MultiObjInvalTarget(invArr3, delArr3);
  }

  //-------------------------------------------------------------------------
  // Test putting this invalidate onto a stream and removing it
  //-------------------------------------------------------------------------
  private static void
  test2(){
    MultiObjInvalTarget moit1 = null;
    MultiObjInvalTarget moit2 = null;
    ObjInvalTarget[] invArr1 = null;
    ObjInvalTarget[] delArr1 = null;
    ByteArrayInputStream bais = null;
    ByteArrayOutputStream baos = null;
    ObjectInputStream ois = null;
    ObjectOutputStream oos = null;

    try{
      // Test 1
      invArr1 = new ObjInvalTarget[2];
      invArr1[0] = new ObjInvalTarget(new ObjId("/a"), 5, 7);
      invArr1[1] = new ObjInvalTarget(new ObjId("/b"), 7, 5);
      delArr1 = new ObjInvalTarget[0];

      moit1 = new MultiObjInvalTarget(invArr1, delArr1);
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      moit1.copySelfOntoOOS(oos);
      oos.flush();
      baos.flush();
      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      moit2 = new MultiObjInvalTarget(ois);
      assert(moit1 != moit2);
      assert(moit1.equals(moit2));
    }catch(IOException e){
      System.err.println("" + e);
      assert(false);
    }
  }

  //-------------------------------------------------------------------------
  // Test intersects()
  //-------------------------------------------------------------------------
  private static void
  test3(){
    SubscriptionSet ss = null;
    ObjInvalTarget[] invalidated = null;
    ObjInvalTarget[] deleted = null;
    MultiObjInvalTarget moit = null;

    // Check intersection with everything
    ss = SubscriptionSet.makeSubscriptionSet(":/*");
    invalidated = new ObjInvalTarget[1];
    deleted = new ObjInvalTarget[1];
    invalidated[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    deleted[0] = new ObjInvalTarget(new ObjId("/b"), 30, 40);
    moit = new MultiObjInvalTarget(invalidated, deleted);
    assert(moit.intersects(ss));

    // Check intersection one place or no place
    ss = SubscriptionSet.makeSubscriptionSet("/a");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    assert(!moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/b");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/b/*");
    assert(!moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a/*:/b");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a:/b/*");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a/*:/b/*");
    assert(!moit.intersects(ss));

    // Check if intersects() works with subdirectories
    ss = SubscriptionSet.makeSubscriptionSet("/a/*");
    invalidated = new ObjInvalTarget[1];
    deleted = new ObjInvalTarget[1];
    invalidated[0] = new ObjInvalTarget(new ObjId("/a/b/c/d"), 10, 30);
    deleted[0] = new ObjInvalTarget(new ObjId("/b/c/d"), 30, 30);
    moit = new MultiObjInvalTarget(invalidated, deleted);
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/b/*");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a:/b");
    assert(!moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a/*:/b");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a:/b/*");
    assert(moit.intersects(ss));
    ss = SubscriptionSet.makeSubscriptionSet("/a/*:/b/*");
    assert(moit.intersects(ss));
  }

  //-------------------------------------------------------------------------
  // Test getIntersection()
  //-------------------------------------------------------------------------
  private static void
  test4(){
    MultiObjInvalTarget moit1 = null;
    MultiObjInvalTarget moit2 = null;
    MultiObjInvalTarget moit3 = null;
    MultiObjInvalTarget moit4 = null;
    MultiObjInvalTarget moit5 = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;
    InvalTarget result1 = null;
    InvalTarget result2 = null;
    InvalTarget result3 = null;
    InvalTarget result4 = null;
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    HierInvalTarget hit3 = null;
    HierInvalTarget hit4 = null;
    InvalTarget it5 = null;
    InvalTarget it6 = null;

    // Intersection of a MultiObjInvalTarget and itself
    invs = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    dels = new ObjInvalTarget[0];
    moit1 = new MultiObjInvalTarget(invs, dels);
    result1 = moit1.getIntersection(moit1);
    assert(result1.equals(moit1));

    // Intersection of a MOIT and equivalent HierInvalTarget
    invs = new ObjInvalTarget[0];
    dels = new ObjInvalTarget[1];
    dels[0] = new ObjInvalTarget(new ObjId("/a"), 20, 30);
    moit2 = new MultiObjInvalTarget(invs, dels);
    hit1 = HierInvalTarget.makeHierInvalTarget("/a");
    result2 = moit2.getIntersection(hit1);
    it6 = new ObjInvalTarget(new ObjId("/a"), 20, 30);
    assert(result2.equals(it6));

    // Intersection of a MOIT and non-overlapping HierInvalTarget
    invs = new ObjInvalTarget[1];
    dels = new ObjInvalTarget[0];
    invs[0] = new ObjInvalTarget(new ObjId("/b/*"), 30, 40);
    moit3 = new MultiObjInvalTarget(invs, dels);
    hit2 = HierInvalTarget.makeHierInvalTarget("/c");
    result3 = moit3.getIntersection(hit2);
    hit3 = new HierInvalTarget();
    assert(result3.equals(hit3));

    // Intersection of a MOIT and partially-overlapping HierInvalTarget
    invs = new ObjInvalTarget[1];
    dels = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a/b/c"), 10, 20);
    dels[0] = new ObjInvalTarget(new ObjId("/c/d/e"), 50, 60);
    moit4 = new MultiObjInvalTarget(invs, dels);
    hit4 = HierInvalTarget.makeHierInvalTarget("/c/*");
    result4 = moit4.getIntersection(hit4);
    it5 = new ObjInvalTarget(new ObjId("/c/d/e"), 50, 60);
    assert(it5.equals(result4));
  }

  //-------------------------------------------------------------------------
  // Test getUnion()
  //-------------------------------------------------------------------------
  private static void
  test5(){
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;
    MultiObjInvalTarget moit1 = null;
    SubscriptionSet ss1 = null;
    HierInvalTarget hit1 = null;
    HierInvalTarget hit2 = null;
    InvalTarget it1 = null;

    // Union of a MOIT and an all-encompassing InvalTarget and SS
    invs = new ObjInvalTarget[0];
    dels = new ObjInvalTarget[1];
    dels[0] = new ObjInvalTarget(new ObjId("/a"), 10, 10);
    ss1 = SubscriptionSet.makeSubscriptionSet(":/*");
    hit1 = HierInvalTarget.makeHierInvalTarget(":/*");
    moit1 = new MultiObjInvalTarget(invs, dels);
    it1 = moit1.getUnion(hit1, ss1);
    assert(it1 instanceof HierInvalTarget);
    assert(it1.equals(hit1));

    // Union of a MOIT and a partially-overlapping InvalTarget and SS
    invs = new ObjInvalTarget[1];
    dels = new ObjInvalTarget[0];
    invs[0] = new ObjInvalTarget(new ObjId("/a/b"), 10, 20);
    moit1 = new MultiObjInvalTarget(invs, dels);
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/*");
    ss1 = SubscriptionSet.makeSubscriptionSet("/a/*");
    it1 = moit1.getUnion(hit1, ss1);
    assert(it1 instanceof HierInvalTarget);
    assert(it1.equals(hit1));

    // Union of a MOIT and a partially-overlapping InvalTarget and SS
    invs = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a/b"), 10, 20);
    dels = new ObjInvalTarget[1];
    dels[0] = new ObjInvalTarget(new ObjId("/c/d"), 10, 20);
    moit1 = new MultiObjInvalTarget(invs, dels);
    hit1 = HierInvalTarget.makeHierInvalTarget("/a/*:/c/*");
    ss1 = SubscriptionSet.makeSubscriptionSet(":/*");
    it1 = moit1.getUnion(hit1, ss1);
    assert(it1 instanceof HierInvalTarget);
    assert(it1.equals(hit1));

    // Union of a MOIT and a partially-overlapping InvalTarget and SS
    invs = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a/b"), 10, 20);
    dels = new ObjInvalTarget[1];
    dels[0] = new ObjInvalTarget(new ObjId("/c/d"), 10, 20);
    moit1 = new MultiObjInvalTarget(invs, dels);
    hit1 = HierInvalTarget.makeHierInvalTarget("/e/*");
    ss1 = SubscriptionSet.makeSubscriptionSet(":/*");
    it1 = moit1.getUnion(hit1, ss1);
    hit2 = HierInvalTarget.makeHierInvalTarget("/a/b:/c/d:/e/*");
    assert(it1 instanceof HierInvalTarget);
    assert(it1.equals(hit2));
  }

  //-------------------------------------------------------------------------
  // Test makeHierInvalTarget
  //-------------------------------------------------------------------------
  private static void
  test6(){
    MultiObjInvalTarget moit1 = null;
    HierInvalTarget hit1 = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;

    // Test 1
    invs = new ObjInvalTarget[1];
    dels = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    dels[0] = new ObjInvalTarget(new ObjId("/b"), 20, 30);
    moit1 = new MultiObjInvalTarget(invs, dels);
    hit1 = moit1.makeHierInvalTarget();
    assert(hit1.getChildren().size() == 2);
    assert(hit1.getChild("a") != null);
    assert(hit1.getChild("b") != null);

    // Test 2
    invs = new ObjInvalTarget[0];
    dels = new ObjInvalTarget[1];
    dels[0] = new ObjInvalTarget(new ObjId("/c"), 10, 10);
    moit1 = new MultiObjInvalTarget(invs, dels);
    hit1 = moit1.makeHierInvalTarget();
    assert(hit1.getChildren().size() == 1);
    assert(hit1.getChild("c") != null);
  }

  //-------------------------------------------------------------------------
  // Test getIterator
  //-------------------------------------------------------------------------
  private static void
  test7(){
    MultiObjInvalTarget moit1 = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;
    MOITIterator moitIter = null;
    ObjInvalTarget oit = null;

    // Test 1
    invs = new ObjInvalTarget[1];
    dels = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 20);
    dels[0] = new ObjInvalTarget(new ObjId("/b"), 20, 30);
    moit1 = new MultiObjInvalTarget(invs, dels);

    moitIter = moit1.getIterator(true);
    assert(moitIter.hasNext());
    oit = moitIter.getNext();
    assert(oit.equals(new ObjInvalTarget(new ObjId("/a"), 10, 20)));
    assert(!moitIter.hasNext());

    moitIter = moit1.getIterator(false);
    assert(moitIter.hasNext());
    oit = moitIter.getNext();
    assert(oit.equals(new ObjInvalTarget(new ObjId("/b"), 20, 30)));
    assert(!moitIter.hasNext());

    // Test 2
    invs = new ObjInvalTarget[0];
    dels = new ObjInvalTarget[1];
    dels[0] = new ObjInvalTarget(new ObjId("/c"), 10, 10);
    moit1 = new MultiObjInvalTarget(invs, dels);

    moitIter = moit1.getIterator(true);
    assert(!moitIter.hasNext());

    moitIter = moit1.getIterator(false);
    assert(moitIter.hasNext());
    oit = moitIter.getNext();
    assert(oit.equals(new ObjInvalTarget(new ObjId("/c"), 10, 10)));
    assert(!moitIter.hasNext());

    // Test 3
    invs = new ObjInvalTarget[3];
    dels = new ObjInvalTarget[0];
    invs[0] = new ObjInvalTarget(new ObjId("/d"), 10, 10);
    invs[1] = new ObjInvalTarget(new ObjId("/e"), 30, 10);
    invs[2] = new ObjInvalTarget(new ObjId("/f"), 40, 50);
    moit1 = new MultiObjInvalTarget(invs, dels);

    moitIter = moit1.getIterator(true);
    assert(moitIter.hasNext());
    oit = moitIter.getNext();
    assert(oit.equals(new ObjInvalTarget(new ObjId("/d"), 10, 10)));
    assert(moitIter.hasNext());
    oit = moitIter.getNext();
    assert(oit.equals(new ObjInvalTarget(new ObjId("/e"), 30, 10)));
    assert(moitIter.hasNext());
    oit = moitIter.getNext();
    assert(oit.equals(new ObjInvalTarget(new ObjId("/f"), 40, 50)));
    assert(!moitIter.hasNext());

    moitIter = moit1.getIterator(false);
    assert(!moitIter.hasNext());
  }
}
