package code;
 /** 
 *  Immutable data structure that stores fields of a precise invalidate message 
 **/ 
import java.util.Vector;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

public class MultiObjPreciseInv
  implements GeneralInv, SingleWriterInval, Immutable
{

 /** 
 *  Constructor 
 **/ 
  private final static short MAGIC_NUM = (short)0xFABC;

 /** 
 *  Data members 
 **/ 
  private final MultiObjInvalTarget moit;
  private final AcceptStamp acceptStamp;
  private final AcceptStamp rtAcceptStamp;
  private final boolean embargoed;
  private final long maxBoundHops;
  private final MOITBoundEntry[] boundEntries;


 /** 
 *  Constructor 
 **/ 

   public
  MultiObjPreciseInv(final MultiObjInvalTarget moit_,
                     final AcceptStamp acceptStamp_,
                     final AcceptStamp rtAcceptStamp_,
                     boolean embargoed_,
                     final MOITBoundEntry[] newBoundEntries){
     this(moit_, acceptStamp_, rtAcceptStamp_, embargoed_, Long.MAX_VALUE, newBoundEntries);
   }

 /** 
 *  Constructor 
 **/ 
  public
  MultiObjPreciseInv(final MultiObjInvalTarget moit_,
                     final AcceptStamp acceptStamp_,
                     final AcceptStamp rtAcceptStamp_,
                     boolean embargoed_,
                     long maxBoundHops_, 
                     final MOITBoundEntry[] newBoundEntries) {
 
    assert(moit_ instanceof Immutable);
    this.moit = moit_;  // moit is immutable
    assert(acceptStamp_ instanceof Immutable);
    this.acceptStamp = acceptStamp_; // acceptStamp is immutable
    assert rtAcceptStamp_ != null;
    assert(rtAcceptStamp_ instanceof Immutable);
    this.rtAcceptStamp = rtAcceptStamp_;
    assert (rtAcceptStamp_.getNodeId().equals(acceptStamp_.getNodeId()));
    this.embargoed = embargoed_;
    assert(newBoundEntries != null);
    this.maxBoundHops = maxBoundHops_;
    this.boundEntries = newBoundEntries;

  }

 /** 
 *  Constructor: Read this object from an input stream 
 *  Note: Used for the BerkeleyDB TupleInput stream 
 **/ 
  public
  MultiObjPreciseInv(InputStream is) throws IOException{
    short readMagicNum = 0;
    ObjectInputStream ois = null;
    MOITBoundEntry[] newBoundEntries = null;

    ois = new ObjectInputStream(is);
    readMagicNum = ois.readShort();
    assert(readMagicNum == MultiObjPreciseInv.MAGIC_NUM);
    this.moit = new MultiObjInvalTarget(ois);
    this.acceptStamp = this.readAcceptStamp(ois);
    this.rtAcceptStamp = this.readAcceptStamp(ois);
    this.embargoed = ois.readBoolean();
    this.maxBoundHops = ois.readLong();
    newBoundEntries = new MOITBoundEntry[ois.readInt()];
    for(int i = 0; i < newBoundEntries.length; i++){
      newBoundEntries[i] = new MOITBoundEntry(ois);
    }
    this.boundEntries = newBoundEntries;

  }

 /** 
 *  Constructor 
 **/ 
  public
  MultiObjPreciseInv(final MultiWriteEntry[] mwe,
                     final AcceptStamp acceptStamp_,
                     final AcceptStamp rtAcceptStamp_,
                     final boolean embargoed_){
    Vector boundEntryVec = null;
    Vector invVec = null;
    Vector delVec = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;

    long maxHop = -1;

    // Extract an InvalTarget and bound object array from mwe
    boundEntryVec = new Vector();
    invVec = new Vector();
    delVec = new Vector();
    for(int i = 0; i < mwe.length; i++){
      if(mwe[i].getDelete()){
        assert(!mwe[i].getBound());
        delVec.add(mwe[i].getObjInvalTarget());
      }else{
        invVec.add(mwe[i].getObjInvalTarget());
        if(mwe[i].getBound()){
          boundEntryVec.add(new MOITBoundEntry(mwe[i].getObjInvalTarget(),
                                               mwe[i].getPriority(),
                                               mwe[i].getImmutableBytes()));
          // we look at maxBoundHops for each entry, and set it to the
          // maximum one
          if(maxHop < mwe[i].getMaxBoundHops()){
            maxHop = mwe[i].getMaxBoundHops();
          }
        }
      }
    }
    invs = (ObjInvalTarget[])invVec.toArray(new ObjInvalTarget[0]);
    dels = (ObjInvalTarget[])delVec.toArray(new ObjInvalTarget[0]);
    this.moit = new MultiObjInvalTarget(invs, dels);
    this.maxBoundHops = maxHop;
    this.boundEntries =
      (MOITBoundEntry[])boundEntryVec.toArray(new MOITBoundEntry[0]);
    assert(acceptStamp_ instanceof Immutable);
    this.acceptStamp = acceptStamp_; // acceptStamp is immutable
    assert rtAcceptStamp_ != null;
    assert(rtAcceptStamp_ instanceof Immutable);
    this.rtAcceptStamp = rtAcceptStamp_;
    assert (rtAcceptStamp_.getNodeId().equals(acceptStamp_.getNodeId()));
    this.embargoed = embargoed_;
  }

 /** 
 *  Write this object to an output stream 
 *  Note: Used by the BerkeleyDB TupleOutput class 
 **/ 
  public void
  writeSelf(OutputStream os) throws IOException{
    ObjectOutputStream oos = null;

    oos = new ObjectOutputStream(os);
    oos.writeShort(MultiObjPreciseInv.MAGIC_NUM);
    this.moit.copySelfOntoOOS(oos);
    this.writeAcceptStamp(oos, this.acceptStamp);
    this.writeAcceptStamp(oos, this.rtAcceptStamp);
    oos.writeBoolean(this.embargoed);
    oos.writeLong(this.maxBoundHops);
    oos.writeInt(this.boundEntries.length);
    for(int i = 0; i < this.boundEntries.length; i++){
      this.boundEntries[i].writeSelf(oos);
    }
    oos.flush();
  }

 /** 
 *  Return true if this invalidate is embargoed 
 **/ 
  public boolean
  isEmbargoed(){
    return(this.embargoed);
  }

 /** 
 *  Return the accept stamp 
 **/ 
  public final AcceptStamp
  getAcceptStamp(){
    assert(this.acceptStamp instanceof Immutable);
    return(this.acceptStamp);
  }

 /** 
 *  Return the real time accept stamp 
 **/ 
  public final AcceptStamp
  getRTAcceptStamp(){
    assert(this.rtAcceptStamp != null);
    assert(this.rtAcceptStamp instanceof Immutable);
    return(this.rtAcceptStamp);
  }

 /** 
 *  Return the stored MultiObjInvalTarget 
 **/ 
  public final MultiObjInvalTarget
  getMultiObjInvalTarget(){
    assert(this.moit instanceof Immutable);
    return(this.moit);
  }


 /** 
 *  Return the stored maxBoundHops 
 **/ 
  public final long
  getMaxBoundHops(){
    return(this.maxBoundHops);
  }

 /** 
 *  Return the local copy of boundEntries 
 *  Note: Dangerous! Caller should make a copy if it intends to modify 
 *    the array 
 **/ 
  public final MOITBoundEntry[]
  getBoundEntriesDangerous(){
    return(this.boundEntries);
  }

 /** 
 *  Return true if obj is equivalent to this object 
 *  Note: Not implemented! 
 **/ 
  public boolean
  equals(Object obj){
    boolean eq = false;
    boolean found = false;
    MultiObjPreciseInv mopi = null;

    // Two java-provided fields, nine local variables
    // nalini add one more field.
    assert(this.getClass().getDeclaredFields().length == 13): "expected length" + this.getClass().getDeclaredFields().length;

    if(obj instanceof MultiObjPreciseInv){
      mopi = (MultiObjPreciseInv)obj;
      eq = (this.moit.equals(mopi.moit) &&
            this.acceptStamp.equals(mopi.acceptStamp) &&
            this.rtAcceptStamp.equals(mopi.rtAcceptStamp) &&
            (this.embargoed == mopi.embargoed) &&
            (this.maxBoundHops == mopi.maxBoundHops) &&
            (this.boundEntries.length == mopi.boundEntries.length));
      // Compare boundEntries
      for(int i = 0; (i < this.boundEntries.length) && eq; i++){
        found = false;
        for(int j = 0; (j < this.boundEntries.length) && (!found); j++){
          found = this.boundEntries[i].equals(mopi.boundEntries[j]);
        }
        eq = found;
      }
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 *  Note: Not implemented! 
 **/ 
  public final int
  hashCode(){
    System.err.println("MultiObjPreciseInv.hashCode() unimplemented!");
    assert(false);
    return(-1);
  }

 /** 
 *  Return true if there is any bound entry 
 **/ 
  public final boolean
  hasBound(){
    if(this.boundEntries.length > 0 ) {
      return true;
    }
    return false;
  }


 /** 
 *  Return true if there is a bound entry for oit in this invalidate 
 **/ 
  public final boolean
  hasBoundEntry(ObjInvalTarget oit){
    boolean found = false;

    for(int i = 0; (i < this.boundEntries.length) && (!found); i++){
      found = this.boundEntries[i].getObjInvalTarget().equals(oit);
    }
    return(found);
  }

 /** 
 *  Return the stored MultiObjInvalTarget 
 **/ 
  public final BodyMsg[]
  makeBodyMsgArray(){
    Vector vec = null;
    BodyMsg bodyMsg = null;
    ObjInvalTarget oit = null;
    BodyMsg[] bodyMessages = null;

    vec = new Vector();
    for(int i = 0; i < this.boundEntries.length; i++){
      oit = this.boundEntries[i].getObjInvalTarget();
      bodyMsg = new BodyMsg(oit.getObjId(),
                            oit.getOffset(),
                            oit.getLength(),
                            this.acceptStamp,
                            this.boundEntries[i].getImmutableBytes(),
                            false,
                            this.boundEntries[i].getPriority());
      vec.add(bodyMsg);
    }
    bodyMessages = (BodyMsg[])vec.toArray(new BodyMsg[0]);
    return(bodyMessages);
  }

 /** 
 *  METHODS INHERITED FROM GENERALINV: 
 **/ 

 /** 
 *  Return the InvalTarget for this invalidate 
 **/ 
  public final InvalTarget
  getInvalTarget(){
    return(this.moit);
  }

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
  public final AcceptVV
  getStartVV(){
    AcceptVV vv = null;
    AcceptStamp[] uniStamp = null;

    uniStamp = new AcceptStamp[1];
    uniStamp[0] = this.acceptStamp;
    vv = new AcceptVV(uniStamp);
    return(vv);
  }

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
  public final AcceptVV
  getEndVV(){
    return(this.getStartVV()); // For a precise inv, startVV == endVV
  }

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
  public final AcceptVV
  getRTVV(){
    AcceptVV vv = null;
    AcceptStamp[] uniStamp = null;

    assert(this.rtAcceptStamp != null);

    uniStamp = new AcceptStamp[1];
    uniStamp[0] = this.rtAcceptStamp;
    vv = new AcceptVV(uniStamp);
    return(vv);
  }

 /** 
 *  Clone this object 
 *  
 *  If we *know* that this object and all objects we reference 
 *  are immutable, then just return this; otherwise perform 
 *      newOne = super.clone();  
 *      now replace any fields in newOne that point to mutable  
 *          objects with a clone of that field 
 *      return newOne 
 *  
 **/ 
  public Object
  clone(){
    assert(this instanceof Immutable);
    return this;
  }

 /** 
 *  Return a cloned version of this MultiObjPreciseInv with certain 
 *  bound entries removed 
 **/ 
  public MultiObjPreciseInv
  cloneRemoveBoundEntry(ObjInvalTarget oit){
    int index = 0;
    MOITBoundEntry moitBE[] = null;
    MultiObjPreciseInv mopi = null;

    assert(this.hasBoundEntry(oit));
    // Make a clone of the array of bound entries
    moitBE = new MOITBoundEntry[this.boundEntries.length - 1];
    for(int i = 0; i < this.boundEntries.length; i++){
      if(!this.boundEntries[i].getObjInvalTarget().equals(oit)){
        moitBE[index] = this.boundEntries[i];
        index++;
      }
    }
    assert(index == (this.boundEntries.length - 1));
    mopi = new MultiObjPreciseInv(this.moit,
                                  this.acceptStamp,
                                  this.rtAcceptStamp,
                                  this.embargoed,
                                  moitBE);
    return(mopi);
  }

 /** 
 *  Return a cloned version of this MultiObjPreciseInv without any 
 *  bound entries 
 **/ 
  public MultiObjPreciseInv
  cloneUnbound(){
    MultiObjPreciseInv mopi = null;

    mopi = new MultiObjPreciseInv(this.moit,
                                  this.acceptStamp,
                                  this.rtAcceptStamp,
                                  this.embargoed,
                                  new MOITBoundEntry[0]);
    return(mopi);
  }


  private static boolean warned = false;
  private static boolean warned2 = false;

 /** 
 *  Return a new invalidate with mutiple operations performed 
 **/ 
  public GeneralInv
  cloneIntersectInvaltargetChopStartEnd(GeneralInv inv){
    InvalTarget invTarget = null;
    GeneralInv intersectInv = null;
    MOITBoundEntry[] newBEs = null;
    MultiObjPreciseInv mopi = null;

    if(!warned2){
      Env.tbd("Check code for cloneIntersectInvaltargetChopStartEnd()" +
              " for all subclasses of GeneralInv." +
              " I don't trust it (see TODO.txt)");
      warned2 = true;
    }

    invTarget = this.moit.getIntersection(inv.getInvalTarget());
    assert(invTarget instanceof MultiObjInvalTarget);
    assert invTarget.equals(this.moit);
    assert !inv.isCommitted();//at any point of time: either a commit or a precise inv
    return this;
    /* commented by zjd
       a MultipleObjectPreciseInval should always be the most precise information
       for the space in in-mem log
     
    // This call should be made for uncommitted invalidates only
    invTarget = this.moit.getIntersection(inv.getInvalTarget());
    assert(invTarget != null);
    if(invTarget instanceof ObjInvalTarget){
      intersectInv = new PreciseInv((ObjInvalTarget)invTarget,
                                    this.acceptStamp,
                                    this.rtAcceptStamp,
                                    (this.embargoed && inv.isEmbargoed()));
    }else if(invTarget instanceof HierInvalTarget){//dummy
      assert (!this.embargoed) || (!inv.isEmbargoed());
      intersectInv = new SingleWriterImpreciseInv((HierInvalTarget)invTarget,
                                                  this.acceptStamp,
                                                  this.acceptStamp,
                                                  this.rtAcceptStamp);
    }else if(invTarget instanceof MultiObjInvalTarget){
      // Note: Our code is currently written so that the return type of the
      // intersection of a MOIT and anything other than a MOIT is a
      // HierInvalTarget. Therefore, if the result is a MOIT, it MUST have
      // resulted from the interesection of two MultiObjPreciseInvs.
      assert(inv instanceof MultiObjPreciseInv);
      mopi = (MultiObjPreciseInv)inv;
      newBEs = this.getBEIntersection(this.boundEntries, mopi.boundEntries);
      intersectInv = new
        MultiObjPreciseInv((MultiObjInvalTarget)invTarget,
                           this.acceptStamp,
                           this.rtAcceptStamp,
                           (this.embargoed && inv.isEmbargoed()),
                           newBEs);
    }else{
      System.err.println("Invalid inval target type");
      System.err.println("this: " + this);
      System.err.println("inv: " + inv);
      System.err.println("invTarget: " + invTarget);
      assert(false);
    }
    return(intersectInv);
    */
  }

  private static boolean warned3 = false;

 /** 
 *  Return the portion of an invalidate for one writer whose 
 *  starting stamp is potentially chopped. 
 **/ 
  public GeneralInv
  getOneWriterSubsetFrom(NodeId node, long startStampInclusive){
    /*
      return an invalidation (possibly this) with 
      interestSet = this.interestSet
      nodeId = node // assert node was in this.VV
      startVV{node} == startStampInclusive
        // assert this.startStamp{node} >= startSTampInclusive
        endVV{node} == this.endVV{node}
        (Note: OK to just return this if we meet all of the other criteria)
    */
    GeneralInv subsetInv = null;

    if(!warned3){
      Env.tbd("Check code for getOneWriterSubsetFrom()" +
              "==  for startTime seems wrong!");
      warned3 = true;
    }

    assert(node.equals(this.acceptStamp.getNodeId()));
    if(this.acceptStamp.getLocalClock() == startStampInclusive){
      subsetInv = this;
    }
    return(subsetInv);
  }

 /** 
 *  Return true if the entire invalidation is already accounted 
 *  for by the specified vv 
 **/ 
  public boolean
  isIncludedBy(VV vv){
    //if we are precise, return vv.includes(this.timestamp)
    //if we are imprecise, return vv.includes(this.*endVV*)
    return(vv.includes(this.acceptStamp));
  }

 /** 
 *  Return true if any component of this invalidation is already 
 *  accounted for by the specified vv 
 **/ 
  public boolean
  anyComponentIncludedBy(VV vv){
    //if we are precise, return vv.includes(this.timestamp)
    //if we are imprecise, true if for any entry e in *startVV*,
    //  vv.includes(e.timestamp)
    return(vv.includes(this.acceptStamp));
  }

 /** 
 *  Return true if this invalidation is committed 
 **/ 
  public boolean
  isCommitted(){
    return(false);
  }

 /** 
 *  Return true if this invalidate is precise (which it is!) 
 **/ 
  public boolean
  isPrecise(){
    return(true);
  }

 /** 
 *  Return true if any objects were invalidated locally 
 **/ 
  public boolean
  isBound(){
    return(this.boundEntries.length > 0);
  }

 /** 
 *  isDelete() 
 **/ 
  public final boolean
  isDelete(){
    return false;
  }

 /** 
 *  Create a new generalInv that is the union of this and i 
 **/ 
  public GeneralInv
  newUnion(GeneralInv i, SubscriptionSet ss){
    assert (!this.embargoed);
    assert (!i.isEmbargoed());
    /*
      each element of startVV is min of corresponding
                      end        max
      interest set is union of corresponding
      assert(either both are uncommitted or both are committed)
      if both committed 
         assert no gap between csnEnd of one and csnStart of
           next (OK to overlap)
         csnstart = min of both csnStart
         csnEnd = max of both csnMax
    */
    InvalTarget it = null;
    ImpreciseInv impreciseInv = null;
    GeneralInv gi = null;
    AcceptVV newStartVV = null;
    AcceptVV newEndVV = null;
    AcceptVV newRTVV = null;

    it = (i.getInvalTarget()).getUnion(this.moit, ss);
    // NOTE: We currently assume that the union operation ALWAYS
    // returns an InterestSet. We might want to change this in
    // the future.

    // We assume that the union of a MultiObjInvalTarget and anything
    // cannot be an ObjInvalTarget
    assert(!(it instanceof ObjInvalTarget));
    if((it instanceof MultiObjInvalTarget) && (!i.equals(this))){
      // Union of this and another MultiObjPreciseInv; convert the local
      // MultiObjInvalTarget into a HierInvalTarget to build an imprecise
      // invalidate.
      it = ((MultiObjInvalTarget)it).makeHierInvalTarget();
    }
    assert(it instanceof HierInvalTarget);

    // Make an imprecise inval out of this
    newStartVV = this.getStartVV().cloneMinVV(i.getStartVV());
    newEndVV = this.getEndVV().cloneMaxVV(i.getEndVV());
    assert this.getRTVV() != null;
    newRTVV = this.getRTVV().cloneMaxVV(i.getRTVV());
    impreciseInv = new ImpreciseInv((HierInvalTarget)it,
                                    newStartVV,
                                    newEndVV,
                                    newRTVV);
    return(impreciseInv);
  }

 /** 
 *  Return -1, 0, or 1 if this is smaller, equal, or larger than 
 *  gi2, respectively, based on mechanism to provide total orders. 
 **/ 
  public int
  totalOrderCompare(GeneralInv gi2){
    /*
      return -1 if this < gi2 in the total order of the log
      return +1 if this > gi2 in the total order of the log
      return 0 if this and gi2 are equivalent and can be ordered either
      way (both start at the same time and end at the same time)

      we sort by CSN then by min acceptStamp then by serverId

      (1) CSN

      If this.CSN < gi2.CSN, then return -1

      Note that an unassigned CSN has an infinite value)

      For a bulk invalidation, consider the *minimum* CSN included
      in the invalidation.

      If this.CSN == gi2.CSN != CSN_UNASSIGNED, either order 
      is legal. If either is a bulk invalidation, the one with 
      the lower maxCSN goes first. If both have the same maxCSN
      then return 0

      (2) startTime acceptStamp

      assert this.CSN == gi2.CSN == CSN_UNASSIGNED

      if the invalidation is precise, consider its acceptStamp
      if the invalidation is imprecise, consider the startVV and
      look at the minimum entry across all nodeId's in startVV

      if they differ, return -1 or 1

      (3) nodeId
      if both minimum accept stamps in step (2) are equal, 
      then compare the node Ids

      if they differ, return -1 or 1

      (4) endtime acceptStamp

      assert csn, startTime, and nodeId are all the same

      send the "more precise" one first --> look at the endTime
      and the one with the smaller max endtime comes first

      if they differ, return -1 or 1
      if they are the same, return 0 (note that if the invals are
      imprecise, they may still differ in elements other than
      their max or min...but that's ok)
    */
    int result = 0;
    long stamp = Long.MAX_VALUE;
    long minStamp = Long.MAX_VALUE;
    Object gi2StampToken = null;
    Object minStampToken = null;
    AcceptVV gi2AcceptVV = null;
    NodeId nodeId = null;
    VVIterator i = null;

    gi2AcceptVV = gi2.getStartVV();
    for(i = gi2AcceptVV.getIterator(); i.hasMoreElements();){
      gi2StampToken = i.getNext();
      stamp = gi2AcceptVV.getStampByIteratorToken(gi2StampToken);
      if(stamp < minStamp){
        minStamp = stamp;
        minStampToken = gi2StampToken;
      }
    }
    assert(minStampToken != null);
    if(this.acceptStamp.getLocalClock() < minStamp){
      result = -1;
    }else if(minStamp < this.acceptStamp.getLocalClock()){
      result = 1;
    }else{
      nodeId = gi2AcceptVV.getServerByIteratorToken(minStampToken);
      // NOTE: NodeId.compareTo must return {-1,0,1}
      result = this.acceptStamp.getNodeId().compareTo(nodeId);
      if(result == 0){
        // NodeId's are the same; try end stamps
        assert(this.acceptStamp.getNodeId().equals(nodeId));
        if(this.acceptStamp.getLocalClock() <
           gi2.getEndVV().getMaxTimeStamp()){
          result = -1;
        }else if(gi2.getEndVV().getMaxTimeStamp() <
                 this.acceptStamp.getLocalClock()){
          result = 1;
        }else{
          result = 0;
        }
      }
    }
    return(result);
  }

 /** 
 *  METHODS INHERITED FROM SINGLEWRITERINVAL 
 **/ 

 /** 
 *  Return the start vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getStart(){
    return(this.acceptStamp.getLocalClock());
  }

 /** 
 *  Return the end vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getEnd(){
    return(this.acceptStamp.getLocalClock());
  }

 /** 
 *  Return the real vv entry for this writer 
 **/ 
  public long
  getReal(){
    return this.rtAcceptStamp.getLocalClock();
  }

 /** 
 *  Return the ID of the node that did this write 
 **/ 
  public NodeId
  getNodeId(){
    return(this.acceptStamp.getNodeId());
  }

 /** 
 *  Clone this object while chopping off a prefix of the startTime 
 **/ 
  public SingleWriterInval
  cloneChopStartTime(long startTimeOfNew){
    assert(this.acceptStamp.getLocalClock() == startTimeOfNew);
    return((SingleWriterInval)this.clone());
  }

 /** 
 *  Clone this object while chopping off a suffix of the startTime 
 **/ 
  public SingleWriterInval
  cloneChopEndTime(long endTimeOfNew){
    assert(this.acceptStamp.getLocalClock() == endTimeOfNew);
    return((SingleWriterInval)this.clone());
  }

 /** 
 *  Convert to a string 
 **/ 
  public String
  toString(){
    String str = null;

    str = ("MultiObjPreciseInval : <<" + this.moit + ">, " + this.acceptStamp +
           ", " + this.rtAcceptStamp + ", " + this.maxBoundHops + ", " + this.embargoed + ">");
    return(str);
  }

 /** 
 *  memSize() -- 
 *    Estimate the memory footprint of this object in a cheap while not 
 *    very accurate way. 
 *  
 *  Use SizeOf and TestClass to get the number of overheads: 
 *  if delete or add new members, recalculate the number of bytes 
 *  refering to ./test/FootPrintSizeOfStandardObject.txt 
 **/ 
  public long
  memSize(){
    Env.performanceWarning("We still need to figure out a way to calculate " +
                           "how much space a MultiObjInvalTarget will take");
    return 300+this.getLength();
  }

 /** 
 *  get the total data size  
 *    -- used to estimate the in-mem finger print 
 **/ 
  public long 
    getLength(){
    long totalLen = 0;
    ImmutableBytes ib = null;
    byte[] data = null;
    for(int i = 0; i < this.boundEntries.length; i++){
      ib = this.boundEntries[i].getImmutableBytes();
      data = ib.dangerousGetReferenceToInternalByteArray();
      totalLen += data.length;
    }
    data = null;
    ib = null;
    return totalLen;
  }

  private static boolean warnedMemSize = false;

 /** 
 *  Return the intersection of the two passed-in BoundEntry arrays 
 **/ 
  public static MOITBoundEntry[]
  getBEIntersection(MOITBoundEntry[] arr1, MOITBoundEntry[] arr2){
    Vector vec = null;
    MOITBoundEntry[] moitBEs = null;

    vec = new Vector();
    for(int i = 0; i < arr1.length; i++){
      for(int j = 0; j < arr2.length; j++){
        if(arr1[i].getObjInvalTarget().equals(arr2[j].getObjInvalTarget())){
          // Part of the intersection
          vec.add(arr1[i]);
        }
      }
    }
    moitBEs = (MOITBoundEntry[])vec.toArray(new MOITBoundEntry[0]);
    return(moitBEs);
  }

 /** 
 *  Write an accept stamp to an object output stream 
 **/ 
  private static void
  writeAcceptStamp(ObjectOutputStream oos, AcceptStamp as) throws IOException{
    oos.writeLong(as.getLocalClock());
    if(SangminConfig.forkjoin){
      oos.writeObject(as.getNodeId()); 
    } else {
      oos.writeLong(as.getNodeId().getIDint());
    }
  }

 /** 
 *  Read an accept stamp from an object input stream 
 **/ 
  private static AcceptStamp
  readAcceptStamp(ObjectInputStream ois) throws IOException{
    AcceptStamp as = null;
    long localClock = 0;
    //long nodeIDint = 0;
    NodeId nodeId = null;
    localClock = ois.readLong();
    if(SangminConfig.forkjoin){
      try{
        nodeId = (BranchID)ois.readObject();
      }catch(ClassNotFoundException e){
        e.printStackTrace();
        throw new IOException(e.getLocalizedMessage());        
      }
    } else {
      nodeId = new NodeId(ois.readLong());
    }
    //nodeIDint = ois.readLong();
    as = new AcceptStamp(localClock, nodeId);
    return(as);
  }

  

  
}

 /** 
 *  $Log: MultiObjPreciseInv.java,v $ 
 *  Revision 1.7  2007/03/06 18:25:51  zjiandan 
 *  Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget 
 *  and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint. 
 *  
 *  Revision 1.6  2007/03/02 09:43:46  zjiandan 
 *  Modified in-mem zip operation for BoundInval to adapt to the new semantics 
 *  of BoundInval. 
 *  
 *  Revision 1.5  2007/03/01 08:12:37  nalini 
 *  added maxBoundHop support for MultiObjWrites 
 *  
 *  Revision 1.4  2007/02/23 20:54:28  zjiandan 
 *  Fixed mem leak problems in NFS2Interface and some other bugs. 
 *  Andrew benchmark passed, performance still needs to tune. 
 *  
 *  Revision 1.3  2006/07/19 22:34:58  nayate 
 *  Added serialization 
 *  
 *  Revision 1.2  2006/04/20 03:52:53  zjiandan 
 *  Callbacks merged with runTime. 
 *  
 *  Revision 1.1  2006/04/04 15:59:59  nayate 
 *  Added the ability to (1) delay invalidates, and (2) support transactional updates. 
 *  
 **/ 
