package code;
 /** 
 *  MultiWriteEntry: Used to represent a single write event that is part 
 *  of a transaction of writes 
 **/ 
import java.io.Serializable;

public class MultiWriteEntry implements Immutable, Serializable
{
  final ObjInvalTarget oit;
  final double priority;
  final ImmutableBytes ib;
  final boolean bound;
  final long maxBoundHops;
  final boolean delete;

 /** 
 *  Constructor 
 **/ 
  public
  MultiWriteEntry(final ObjId objId,
                  final long offset,
                  final long length,
                  final double newPriority,
                  final ImmutableBytes newIB,
                  final boolean newBound,
                  final long newMaxBoundHops,
                  final boolean newDelete){
    this.oit = new ObjInvalTarget(objId, offset, length);
    this.priority = newPriority;
    this.ib = newIB;
    this.bound = newBound;
    this.maxBoundHops = newMaxBoundHops;
    this.delete = newDelete;
    assert((!this.bound) || (!this.delete));
  }

 /** 
 *  Constructor 
 **/ 
  public
  MultiWriteEntry(final ObjInvalTarget newOIT,
                  final double newPriority,
                  final ImmutableBytes newIB,
                  final boolean newBound,
                  final long newMaxBoundHops,
                  final boolean newDelete){
    this.oit = newOIT;
    this.priority = newPriority;
    this.ib = newIB;
    this.bound = newBound;
    this.maxBoundHops = newMaxBoundHops;
    this.delete = newDelete;
    assert((!this.bound) || (!this.delete));
  }


 /** 
 *  Constructor with maxBoundHops default to Long.MAX_VALUE 
 **/ 
  public
  MultiWriteEntry(final ObjId objId,
                  final long offset,
                  final long length,
                  final double newPriority,
                  final ImmutableBytes newIB,
                  final boolean newBound,
                  final boolean newDelete){
    this(objId, offset, length, newPriority, newIB, 
         newBound, Long.MAX_VALUE, newDelete);
  }
  
  public
  MultiWriteEntry(final ObjInvalTarget newOIT,
                  final double newPriority,
                  final ImmutableBytes newIB,
                  final boolean newBound,
                  final boolean newDelete){
    this(newOIT, newPriority, newIB, newBound, Long.MAX_VALUE, newDelete);
  }

 /** 
 *  Return the ObjInvalTarget 
 **/ 
  public final ObjInvalTarget
  getObjInvalTarget(){
    return(this.oit);
  }

 /** 
 *  Return the priority 
 **/ 
  public final double
  getPriority(){
    return(this.priority);
  }

 /** 
 *  Return the data 
 **/ 
  public final ImmutableBytes
  getImmutableBytes(){
    return(this.ib);
  }

 /** 
 *  Return true if this entry is for a bound write 
 **/ 
  public final boolean
  getBound(){
    return(this.bound);
  }

 /** 
 *  Return the maxBoundHops 
 **/ 
  public final long
  getMaxBoundHops(){
    return(this.maxBoundHops);
  }

 /** 
 *  Return true if this entry is for a delete 
 **/ 
  public final boolean
  getDelete(){
    return(this.delete);
  }

 /** 
 *  Return a hash code for this object (note: not implemented) 
 **/ 
  public final int
  hashCode(){
    System.err.println("MultiWriteEntry.hashCode() not implemented");
    assert(false);
    return(0);
  }

 /** 
 *  Return true if this equals "obj" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean result = false;
    MultiWriteEntry mwe = null;

    // 2 Java-provided fields, 6 declared fields
    assert(this.getClass().getDeclaredFields().length == 8);
    if(obj instanceof MultiWriteEntry){
      mwe = (MultiWriteEntry)obj;
      result = ((this.oit.equals(mwe.oit)) &&
                (this.priority == mwe.priority) &&
                (this.ib.equals(mwe.ib)) &&
                (this.bound == mwe.bound) &&
                (this.maxBoundHops == mwe.maxBoundHops) &&
                (this.delete == mwe.delete));
    }else{
      result = false;
    }
    return(result);
  }
  
 /** 
 *  Return String representation of this object 
 **/ 
  public String toString(){
    return oit.toString()+priority+ib.toString()+bound+maxBoundHops+delete;
  }
 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing MultiWriteEntry.java...");
    MultiWriteEntry.test1();
    System.out.println("...Finished");
  }

 /** 
 *  Test 
 **/ 
  private static void
  test1(){
    MultiWriteEntry mwe1 = null;
    MultiWriteEntry mwe2 = null;
    ObjInvalTarget oit1 = null;

    // Test 1
    mwe1 = new MultiWriteEntry(new ObjInvalTarget(new ObjId("/a"), 10, 10),
                               5.5,
                               new ImmutableBytes(new byte[7]),
                               true,
                               5,
                               false);
    assert(mwe1.getObjInvalTarget().equals(new
                                           ObjInvalTarget(new ObjId("/a"),
                                                          10,
                                                          10)));
    assert(mwe1.getPriority() == 5.5);
    assert(mwe1.getImmutableBytes().getLength() == 7);
    assert(mwe1.equals(mwe1));
    assert(mwe1.getBound());
    assert(mwe1.getMaxBoundHops() == 5);
    assert(!mwe1.getDelete());

    // Test 2
    mwe2 = new MultiWriteEntry(new ObjInvalTarget(new ObjId("/b"), 20, 30),
                               3.14,
                               new ImmutableBytes(new byte[3]),
                               false,
                               5,
                               true);
    assert(mwe2.getObjInvalTarget().equals(new
                                           ObjInvalTarget(new ObjId("/b"),
                                                          20,
                                                          30)));
    assert(mwe2.getPriority() == 3.14);
    assert(mwe2.getImmutableBytes().getLength() == 3);
    assert(!mwe2.getBound());
    assert(mwe2.getDelete());
    assert(!mwe1.equals(mwe2));
    assert(mwe2.equals(mwe2));

    // Test 3
    mwe2 = new MultiWriteEntry(new ObjInvalTarget(new ObjId("/a"), 10, 10),
                               5.5,
                               new ImmutableBytes(new byte[7]),
                               true,
                               5,
                               false);
    assert(mwe1.equals(mwe2));
    assert(mwe2.equals(mwe1));

    // Test 4
    mwe2 = new MultiWriteEntry(new ObjId("/a"),
                               10,
                               10,
                               5.5,
                               new ImmutableBytes(new byte[7]),
                               true,
                               5,
                               false);
    assert(mwe1.equals(mwe2));
    assert(mwe2.equals(mwe1));
  }
}

 /** 
 *  $Log: MultiWriteEntry.java,v $ 
 *  Revision 1.3  2007/03/01 08:12:37  nalini 
 *  added maxBoundHop support for MultiObjWrites 
 *  
 *  Revision 1.2  2007/02/23 20:54:28  zjiandan 
 *  Fixed mem leak problems in NFS2Interface and some other bugs. 
 *  Andrew benchmark passed, performance still needs to tune. 
 *  
 *  Revision 1.1  2006/04/04 15:59:59  nayate 
 *  Added the ability to (1) delay invalidates, and (2) support transactional updates. 
 *  
 **/ 
