package code;
import java.io.*;
import java.net.*;
import java.util.*;

class MountdHandler extends rpcHandler 
  implements RPCConsts, MountdConsts, NFSConsts{

 /** 
 *  Private data members 
 **/ 
  private Handle fileHandles = null;
  private PathMapper mapper = null;
  private Exports exports = null;
    
 /** 
 *  Constructor 
 **/ 
  public
  MountdHandler(Handle handles, Exports exp, PathMapper pm){
    // tell parent class about me
    super(MD_PROG, MD_VERS);

    this.fileHandles = handles;
    this.exports = exp;
    this.mapper = pm;
  }

 /** 
 *  Handle the request 
 *  
 *  Each function defined in http: 
 **/ 
  public void
  Run(UDPPacketPort port, long xid, long procedure, XDRPacket packet){
    System.out.print("Mountd run method called\n");

    switch((int) procedure) {
    case (int) MD_NULL:
      System.out.print("MD_NULL called\n");
      MDNull(port, xid, procedure, packet);
      break;
    case (int) MD_MNT:
      System.out.print("MD_MNT called\n");
      MDMount(port, xid, procedure, packet);
      break;
    case (int) MD_DUMP:
      System.err.print("Unsupported mount procedure MD_DUMP called\n");
      break;
    case (int) MD_UMNT:
      System.out.print("MD_UMNT called\n");
      MDUnMount(port, xid, procedure, packet);
      break;
    case (int) MD_UMNTALL:
      System.out.print("MD_UMNTALL called\n");
      MDUnMount(port, xid, procedure, packet);
      break;
    case (int) MD_EXPORT:
      System.out.print("MD_EXPORT called\n");
      MDExport(port, xid, procedure, packet);
      break;
    default:
      System.err.print("Unsupported mountd procedure called ("
                       + procedure + ")\n");
      break;
    }
  }

 /** 
 *  Handle the NULL method 
 **/ 
  public void
  MDNull(UDPPacketPort port, long xid, long procedure, XDRPacket packet){
    // Put together an XDR reply packet
    XDRPacket result = new XDRPacket(128);
    result.AddReplyHeader(xid);

    // send the reply back
    System.out.print("Sending reply back to address " +
                     packet.Source().getHostAddress() + " port " +
                     packet.Port() + "\n");
    port.SendPacket(packet.Source(), packet.Port(), result);
  }
    



 /** 
 *  Handle the mount method 
 **/ 
  public void
  MDMount(UDPPacketPort port, long xid, long procedure, XDRPacket packet){
    // Keep track of whether any errors have been encountered
    long err = NFS_OK;

    // skip the authentication records
    packet.ReadAuthentication();
    packet.ReadAuthentication();

    // next should be a dirpath, which is a string.  Replace unix 
    //   style path with local style path
    String path = packet.GetString();
    if (path == null) {
      err = NFSERR_STALE; // what should go here?
    }
    else {
      String old = path;
      path = mapper.Convert(old);
      path = mapper.Canonicalize(path);
      System.out.print("Mount request for " + path 
                       + "was " + old + "\n");
    }

    XDRPacket result = new XDRPacket(128);
    result.AddReplyHeader(xid);

    //
    // Try to validate this mount, if there is an error make an error
    //   packet, otherwise send back the handle.
    //
    if (err != NFS_OK) {
      result.AddLong(err);
    }
    else if (exports.Matches(packet.Source(), path) == false) {
      // No permission for this mount in the exports file
      result.AddLong(NFSERR_PERM);
      System.err.print("!!! Mount request for " + path + 
                       "from " + packet.Source() + " denied.\n");
    }
    else {
      // put together a file handle
      /*
        long handle = fileHandles.Allocate(path);
        fhandle fh = new fhandle();
        fh.Set(handle, handle, 0);
      */
      fhandle fh = fhandle.makeAllZeroHandle();

      /*
        System.out.print(">>> emiting fhandle root " + fh.Root()
        + " file " + fh.Handle() + "\n");
      */
      System.out.println(">>> emiting fhandle root " + fh);

      // Put together the reply packet.
      result.AddLong(NFS_OK);
      this.emitFHandle(result, fh);
    }

    // Send the result
    port.SendPacket(packet.Source(), packet.Port(), result);
  }
    
 /** 
 *  Handle the unmount method 
 **/ 
  public void
  MDUnMount(UDPPacketPort port, long xid, long procedure, XDRPacket packet){
    // skip the authentication records
    packet.ReadAuthentication();
    packet.ReadAuthentication();

    // next should be a dirpath, which is a string
    String path = mapper.Convert(packet.GetString());
    System.out.print("Mount request for " + path + "\n");

    // put together a file handle
    long handle = fileHandles.Allocate(path);

    XDRPacket result = new XDRPacket(128);
    result.AddReplyHeader(xid);

    result.AddLong(NFS_OK);

    // Send the result
    System.out.print("Sending reply handle " + handle 
                     + " address " + packet.Source().getHostAddress()
                     + " port " + packet.Port() + "\n");
    port.SendPacket(packet.Source(), packet.Port(), result);
  }

 /** 
 *  Handle the EXPORT method 
 **/ 
  public void
  MDExport(UDPPacketPort port, long xid, long procedure, XDRPacket packet){
    // Put together an XDR reply packet
    XDRPacket result = new XDRPacket(128);
    result.AddReplyHeader(xid);
    //    result.AddLong(NFS_OK); NO! Don't send this!

    //
    // According to http://www.faqs.org/rfcs/rfc1094.html
    // our reply must be
    //        typedef string dirpath<MNTPATHLEN>;
    //        typedef string name<MNTNAMLEN>;
    // 
    //            struct *groups {
    //               name grname;
    //                groups grnext;
    //       };
    //
    //       struct *exportlist {
    //               dirpath filesys;
    //               groups groups;
    //               exportlist next;
    //       };
    //
    //      exportlist
    //       MNTPROC_EXPORT(void) = 5;
    //
    // We send the list of exports with no group list record
    // (which seems to mean "anyone can mount this")
    //
    
    List li = exports.getListOfExports();
    Iterator ii = li.iterator();
    while(ii.hasNext()){
      String s = (String)ii.next();
      System.out.println("Exports item: " + s);
      result.AddLong(NFS_TRUE); // There is a directory list record here
      result.AddString(s);
      // Now list of groups
      result.AddLong(NFS_FALSE); // There no group list record here
    }
    result.AddLong(NFS_FALSE); // No more exports
                         
    

    // send the reply back
    System.out.print("Sending reply back to address " +
                     packet.Source().getHostAddress() + " port " +
                     packet.Port() + "\n");
    port.SendPacket(packet.Source(), packet.Port(), result);
  }
    


 /** 
 *  Read the fhandle from the packet 
 **/ 
  private void
  emitFHandle(XDRPacket packet, fhandle fh){
    byte[] handleBytes = null;

    handleBytes = fh.getData();
    packet.AddNextBytes(handleBytes);
  }
}
