package code;
import java.io.*;
import java.util.*;

public class NFSDir implements NFSConsts{
  
  static private boolean dbg = false;
  
 /** 
 *  Data members 
 **/ 
  private Handle handles;
  private PathMapper pm;
  private TimeMapper tm;
  private FileSystemInfo fsinfo;
  private NFS2FS pfs;
    
 /** 
 *  Constructor 
 **/ 
  public
  NFSDir(NFS2FS newPFS,
         Handle h,
         PathMapper p,
         TimeMapper t,
         FileSystemInfo f){
    this.handles = h;
    this.pm = p;
    this.tm = t;
    this.fsinfo = f;
    this.pfs = newPFS;
  }

 /** 
 *  Return the operation attributes 
 **/ 
  public XDRPacket
  GetAttr(long xid, XDRPacket packet) throws NFSException{
    fhandle fh = null;
    attrstat result = null;
    fattr attr = null;
    XDRPacket replyPacket = null;

    fh = this.readFHandle(packet);
    result = this.pfs.getAttributes(fh);

    if(result.getStatus() != stat.NFS_OK){
      throw new NFSException(xid, result.getStatus());
    }

    // make the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    attr = result.getAttributes();
    this.emitAttributes(replyPacket, attr);

    return(replyPacket);
  }

 /** 
 *  Set attributes 
 **/ 
  public XDRPacket
  SetAttr(long xid, XDRPacket packet) throws NFSException{
    sattrargs args = null;
    attrstat result = null;
    fattr attr = null;
    XDRPacket replyPacket = null;

    // get info out of the packet
    args = this.readSattrArgs(packet);
    result = this.pfs.setAttributes(args);
    if(result.getStatus() != stat.NFS_OK){
      throw new NFSException(xid, result.getStatus());
    }

    // make the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    attr = result.getAttributes();
    this.emitAttributes(replyPacket, attr);

    return(replyPacket);
  }

 /** 
 *  Perform a directory lookup 
 **/ 
  public XDRPacket
  Lookup(long xid, XDRPacket packet) throws NFSException{
    diropargs args = null;
    diropres result = null;
    XDRPacket replyPacket = null;

    args = this.readDirOpArgs(packet);
    result = this.pfs.lookup(args);
    if(result.getStatus() != stat.NFS_OK){
      throw new NFSException(xid, result.getStatus());
    }

    // make the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    this.emitFHandle(replyPacket, result.getFile());
    this.emitAttributes(replyPacket, result.getAttributes());

    return(replyPacket);
  }

  // keep these between calls so subsequent calls getting the rest of the 
  //   contents of a directory are fast.
  private fhandle cachedDirFHandle;
  private Vector cachedFiles;

 /** 
 *  Perform a directory lookup 
 **/ 
  public XDRPacket
  Readdir(long xid, XDRPacket packet) throws NFSException{
    readdirargs args = null;
    readdirres result = null;
    fhandle fh = null;
    long cookie = 0;
    long count = 0;
    entry dirEntry = null;

    args = this.readReadDirArgs(packet);
    fh = args.getDir();
    cookie = args.getCookie();
    count = args.getCount();

    // if this is a new call to readdir (cookie=0) or it is a new
    //   directory to read, replace the cache.
    if((cookie == 0) || (!fh.equals(this.cachedDirFHandle))){
      // Read the entries
      result = this.pfs.readdir(args);

      if(result.getStatus() != stat.NFS_OK){
        throw new NFSException(xid, result.getStatus());
      }
      dirEntry = result.getEntries();
      this.cachedFiles = new Vector();
      while(dirEntry != null){
        this.cachedFiles.add(dirEntry);
        dirEntry = dirEntry.getNextEntry();
      }
      this.cachedDirFHandle = fh;

      /*
        System.out.println("Readdir:");
        for(int i = 0; i < this.cachedFiles.size(); i++){
        System.out.println("" + this.cachedFiles.get(i));
        }
        System.out.println();
      */
    }

    // prepare the reply packet.
    XDRPacket reply = new XDRPacket((int) count);
    reply.AddReplyHeader(xid);
    reply.AddLong(NFS_OK);

    // Add files to the list until there are no more files or all of
    //   the count bytes have been used.
    int current = reply.Length();
    boolean more = false; // are there more files to get
    // if there are any files to add
    if((this.cachedFiles != null) && (this.cachedFiles.size() > 0)){
      for(int i = (int)cookie; i < this.cachedFiles.size(); i++) {
        // see if there is enough room for another file - 3 longs of id,
        //   the name (rounded up to 4 bytes) and a trailing long 
        //   indicating whether there are more files to get
        dirEntry = (entry)this.cachedFiles.get(i);
        int needed = 3 * 4 + (dirEntry.getName().length() + 3) + 8;
        if (needed + current >= count) {
          more = true;
          break;
        }

        // add an entry to the packet for this file
        reply.AddLong(NFS_TRUE);
        reply.AddLong(dirEntry.getFileId());
        reply.AddString(dirEntry.getName());
        reply.AddLong(i + 1); // this is the cookie
        current = reply.Length();
      }
    }
    reply.AddLong(NFS_FALSE); // no more entries in this packet

    // tell the client if this packet has returned the last of the files
    if (more) 
      reply.AddLong(NFS_FALSE);
    else
      reply.AddLong(NFS_TRUE);
    return reply;
  }

 /** 
 *  Create a new file 
 **/ 
  public XDRPacket
  Create(long xid, XDRPacket packet) throws NFSException{
    createargs args = null;
    diropres result = null;
    XDRPacket replyPacket = null;

    args = this.readCreateArgs(packet);
    result = this.pfs.create(args);

    if(result.getStatus() != stat.NFS_OK){
      throw new NFSException(xid, result.getStatus());
    }

    // create the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    this.emitFHandle(replyPacket, result.getFile());
    this.emitAttributes(replyPacket, result.getAttributes());
    return(replyPacket);
  }

 /** 
 *  Delete a new file 
 **/ 
  public XDRPacket
  Remove(long xid, XDRPacket packet) throws NFSException{
    diropargs args = null;
    int status = 0;
    XDRPacket replyPacket = null;

    args = this.readDirOpArgs(packet);
    status = this.pfs.remove(args);

    if(status != stat.NFS_OK){
      throw new NFSException(xid, status);
    }

    // create the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    return(replyPacket);
  }

 /** 
 *  Create a new directory 
 **/ 
  public XDRPacket
  Mkdir(long xid, XDRPacket packet) throws NFSException{
    createargs args = null;
    diropres result = null;
    XDRPacket replyPacket = null;

    args = this.readCreateArgs(packet);
    result = this.pfs.mkdir(args);

    if(result.getStatus() != stat.NFS_OK){
      throw new NFSException(xid, result.getStatus());
    }

    // create the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    this.emitFHandle(replyPacket, result.getFile());
    this.emitAttributes(replyPacket, result.getAttributes());
    return(replyPacket);
  }

 /** 
 *  Remove a new directory 
 **/ 
  public XDRPacket
  Rmdir(long xid, XDRPacket packet) throws NFSException{
    diropargs args = null;
    int status = 0;
    XDRPacket replyPacket = null;

    args = this.readDirOpArgs(packet);
    status = this.pfs.rmdir(args);

    if(status != stat.NFS_OK){
      throw new NFSException(xid, status);
    }

    // create the reply packet
    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    return(replyPacket);
  }

 /** 
 *  Return the file systems information 
 **/ 
  public XDRPacket
  StatFS(long xid, XDRPacket packet) throws NFSException{
    statfsres result = null;
    fhandle fh = null;
    XDRPacket replyPacket = null;

    fh = this.readFHandle(packet);
    result = this.pfs.statfs(fh);

    if(result.getStatus() != stat.NFS_OK){
      throw new NFSException(xid, result.getStatus());
    }

    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    replyPacket.AddLong(result.getTSize());
    replyPacket.AddLong(result.getBSize());
    replyPacket.AddLong(result.getBlocks());
    replyPacket.AddLong(result.getBFree());
    replyPacket.AddLong(result.getBAvail());
    return(replyPacket);
  }

 /** 
 *  Rename a file 
 **/ 
  public XDRPacket
  Rename(long xid, XDRPacket packet) throws NFSException{
    int status = 0;
    renameargs args = null;
    XDRPacket replyPacket = null;

    args = this.readRenameArgs(packet);
    status = this.pfs.rename(args);

    if(status != stat.NFS_OK){
      throw new NFSException(xid, status);
    }

    replyPacket = new XDRPacket(128);
    replyPacket.AddReplyHeader(xid);
    replyPacket.AddLong(NFS_OK);
    return(replyPacket);
  }

/*    
    // local procedure to get the associated with a handle, throws an exception
    //   if there is a problem.
    private String GetNameFromHandle(long handle, long xid) throws NFSException{
	String result = handles.Lookup(handle);
	if (result == null)
	    throw new NFSException(xid, NFSERR_STALE);
	return result;
    }
*/

 /** 
 *  Write out attributes to the packet 
 **/ 
  private void
  emitAttributes(XDRPacket packet, fattr attr){
    packet.AddLong(attr.getType());
    packet.AddLong(attr.getMode());
    packet.AddLong(attr.getNLink());
    packet.AddLong(attr.getUID());
    packet.AddLong(attr.getGID());
    packet.AddLong(attr.getSize());
    packet.AddLong(attr.getBlockSize());
    packet.AddLong(attr.getRDev());
    packet.AddLong(attr.getBlocks());
    packet.AddLong(attr.getFSID());
    packet.AddLong(attr.getFileId());
    this.emitTimeval(packet, attr.getATime());
    this.emitTimeval(packet, attr.getMTime());
    this.emitTimeval(packet, attr.getCTime());
  }

 /** 
 *  Write out the time to the packet 
 **/ 
  private void
  emitTimeval(XDRPacket packet, timeval time){
    packet.AddLong(time.getSeconds());
    packet.AddLong(time.getUSeconds());
  }

 /** 
 *  Read the fhandle from the packet 
 **/ 
  private void
  emitFHandle(XDRPacket packet, fhandle fh){
    byte[] handleBytes = null;

    handleBytes = fh.getData();
    packet.AddNextBytes(handleBytes);
  }

 /** 
 *  Read the sattr from the packet 
 **/ 
  private sattr
  readSattr(XDRPacket packet){
    long mode = 0;
    long uid = 0;
    long gid = 0;
    long size = 0;
    timeval atime = null;
    timeval mtime = null;
    sattr attr = null;

    mode = packet.GetLong();
    uid = packet.GetLong();
    gid = packet.GetLong();
    size = packet.GetLong();
    atime = this.readTimeval(packet);
    mtime = this.readTimeval(packet);
    attr = new sattr(mode, uid, gid, size, atime, mtime);
    return(attr);
  }

 /** 
 *  Read the timeval from the packet 
 **/ 
  private timeval
  readTimeval(XDRPacket packet){
    long seconds = 0;
    long useconds = 0;
    timeval tv = null;

    seconds = packet.GetLong();
    useconds = packet.GetLong();
    tv = new timeval(seconds, useconds);
    return(tv);
  }

 /** 
 *  Read the sattrargs from the packet 
 **/ 
  private sattrargs
  readSattrArgs(XDRPacket packet){
    fhandle fh = null;
    sattrargs args = null;
    sattr attr = null;

    fh = this.readFHandle(packet);
    attr = this.readSattr(packet);
    args = new sattrargs(fh, attr);
    return(args);
  }

 /** 
 *  Read the fhandle from the packet 
 **/ 
  private fhandle
  readFHandle(XDRPacket packet){
    fhandle fh = null;
    byte[] handleBytes = null;

    handleBytes = new byte[fhandle.FHSIZE];
    packet.GetNextBytes(handleBytes);
    fh = fhandle.makeHandleDangerous(handleBytes);
    handleBytes = null; // Release the pointer to avoid trouble
    return(fh);
  }

 /** 
 *  Read the diropargs from the packet 
 **/ 
  private diropargs
  readDirOpArgs(XDRPacket packet){
    fhandle fh = null;
    String name = null;
    diropargs args = null;
    
    Env.dprintln(dbg, "before readFHandle: currentPosition=" + packet.CurrentPosition());
    fh = this.readFHandle(packet);
    Env.dprintln(dbg, "fh: " + fh);
    name = packet.GetString();
    Env.dprintln(dbg, "name: " + name);
    args = new diropargs(fh, name);
    return(args);
  }

 /** 
 *  Read the readdirargs from the packet 
 **/ 
  private readdirargs
  readReadDirArgs(XDRPacket packet){
    fhandle fh = null;
    long cookie = 0;
    long count = 0;
    readdirargs args = null;

    fh = this.readFHandle(packet);
    cookie = packet.GetLong();
    count = packet.GetLong();
    args = new readdirargs(fh, cookie, count);
    return(args);
  }

 /** 
 *  Read the createargs from the packet 
 **/ 
  private createargs
  readCreateArgs(XDRPacket packet){
    createargs args = null;
    diropargs dirOpArgs = null;
    sattr attr = null;
    
    dirOpArgs = this.readDirOpArgs(packet);
    attr = this.readSattr(packet);
    args = new createargs(dirOpArgs, attr);

    return(args);
  }


 /** 
 *  Read the renameargs from the packet 
 **/ 
  private renameargs
  readRenameArgs(XDRPacket packet){
    diropargs fromArgs = null;
    diropargs toArgs = null;
    renameargs args = null;

    fromArgs = this.readDirOpArgs(packet);
    toArgs = this.readDirOpArgs(packet);
    args = new renameargs(fromArgs, toArgs);
    return(args);
  }
}
