package code;
import java.io.*;
import java.net.*;
import java.util.*;

public class NFSHandler implements RPCConsts, NFSConsts{

 /** 
 *  Constants 
 **/ 
  private static boolean PRINT_ERROR = false;
  private static boolean PRINT_PERIODIC_RT = false;
  private static boolean PRINT_PACKET = false;
  private static boolean PRINT_PROCEDURE = false;
  private static boolean dbg = false;
  private static boolean verbose = false;
 /** 
 *  Data members 
 **/ 
  // state variables passed from parent, to deal with platform local issues
  private Handle fileHandles;
  private PathMapper pathmapper;
  private FileSystemInfo fsinfo;
  private TimeMapper timemapper;
  //  private PRACTIFSLocalInterface li;

  // Queue of received packets, used to avoid redoing retransmissions
  // XXX Remove this module from the system - I don't think it is useful
  XIDCache packetQueue;

  // cache of packets that were allocated so we don't have to allocate
  //   memory buffers a lot, that is slow in java
  private XDRPacketCache packetCache;

  // classes that carry out various nfs operations
  private NFSDir dirService;
  private NFSIO ioService;

  private final boolean useXIDCache = true;
  private final boolean usePacketCache = true;
 /** 
 *  Constructor 
 **/ 
  public
  NFSHandler(NFS2FS fs,
             Handle handles,
             PathMapper pm,
             FileSystemInfo fsi,
             TimeMapper tm){
    // tell parent about me
    //super(NFS_PROG, NFS_VERS);

    // assign state variables
    fileHandles = handles; // file handle to file name mapping class
    pathmapper = pm; // maps network paths to local paths
    fsinfo = fsi; // gets info from file system, eg. free blocks, used blocks
    timemapper = tm;  // converts java returned file mod times to UNIX times

    // directory helper, does readdir
    dirService = new NFSDir(fs, fileHandles, pathmapper, timemapper, fsinfo);

    // io helper, does file read and write
    ioService = new NFSIO(fs, fileHandles, timemapper);

    // XXX the xid cache has been removed
    if(useXIDCache){
      packetQueue = new XIDCache();
    }else{
      packetQueue = null;
    }
    if(usePacketCache){
      packetCache = new XDRPacketCache();
    }else{
      packetCache = null;
    }
  }
    
 /** 
 *  process a packet and send a reply packet. 
 **/ 
  int iteration = 0;
  public void
  Run(UDPPacketPort port,
      long xid,
      long procedure,
      XDRPacket packet,
      int threadNum){

    
    //
    // added by zjd
    //
    // problem to solve: <"xid", "procedure"> might not match with
    //                   the handed "packet", which might lead to 
    //                   out of mem.
    //
    // solution: let the thread ignore the parameters "xid" and "procedure"
    //           instead getting them directly from the packet.
    //
    packet.Reset();
    RPCCallHeader rHead = packet.advanceRPCCallHeader();
    Env.dprintln(verbose, "old xid = " + xid + " new xid = " + rHead.getXid()
		 + " old proc= " + procedure + " new proc= " + rHead.getProc());  
    xid = rHead.getXid();
    procedure = rHead.getProc();
    
    iteration++;
    Date begin = new Date();

    XDRPacket result = null;
    
    try{
      //
      // see if this packet has already been received 
      //
      // XXX Turn off the XIDCache - is this thing really helpful?
      //
      if(useXIDCache){
	assert packetQueue != null;
	XIDCache.XIDCacheItem qi = packetQueue.Find(xid);
        if (qi != null) {
          Env.dprintln(dbg, "handling duplicate request " + xid + "\n");
	  if (qi.inprogress == false){
            port.SendPacket(packet.Source(), packet.Port(), qi.packet);
	  }
	  return;
	}
	//called inside find(xid)
	//packetQueue.Start(xid);
      }
      
      // get rid of authentication recorde in packet, I don't use them
      packet.ReadAuthentication();
      packet.ReadAuthentication();
      if(dbg){
	packet.verifyHead(xid, RPCCall);
      }

      boolean resultFromPacketCache = false;
      // run the requested procedure
      try {
        switch((int) procedure) {
        case (int) NFS_NULL:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_Null");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_Null for packet[" 
		       + packet.toRequestString((int)procedure));
	  
          result = NFSNull(port, xid, procedure, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_Null request [ " + xid + "]" 
		       + result.toResultString((int)procedure) + "\n-------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_Null \n");
          break;

        case (int) NFS_GETATTR:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_GETATTR");
          Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_GETATTR for packet[" 
		       + packet.toRequestString((int)procedure));
	  result = dirService.GetAttr(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_GETATTR request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+ "\n-------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_GETATTR \n");
          break;

        case (int) NFS_SETATTR:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_SETATTR");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_SETATTR for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.SetAttr(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_SETATTR request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_SETATTR \n");
          break;

        case (int) NFS_LOOKUP:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_LOOKUP");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_LOOKUP for packet[" 
	  	       + packet.toRequestString((int)procedure));
          result = dirService.Lookup(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_LOOKUP request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return  request [ " + xid + " ] procedure : NFS_LOOKUP \n");
          break;

        case (int) NFS_READ:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_READ");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_READ for packet[" 
		       + packet.toRequestString((int)procedure));
          result = ioService.Read(xid, packet, packetCache);
	  resultFromPacketCache = true;

	  Env.dprintln(PRINT_PACKET, "result for NFS_READ request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "processing request [ " + xid + " ] procedure : NFS_READ \n");
	  
          break;

        case (int) NFS_WRITE:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_WRITE");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_WRITE for packet[" 
		       + packet.toRequestString((int)procedure));
          result = ioService.Write(xid, packet, packetCache, threadNum);
	  resultFromPacketCache = true;

	  Env.dprintln(PRINT_PACKET, "result for NFS_WRITE request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_WRITE\n");
          break;

        case (int) NFS_CREATE:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_CREATE");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_CREATE for packet[" 
	  	       + packet.toRequestString((int)procedure));
          result = dirService.Create(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_CREATE request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_CREATE \n");
          break;

        case (int) NFS_REMOVE:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_REMOVE");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_REMOVE for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.Remove(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_REMOVE request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_REMOVE \n");
          break;

        case (int) NFS_RENAME:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_RENAME");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_RENAME for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.Rename(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_RENAME request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_RENAME \n");
          break;

        case (int) NFS_MKDIR:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_MKDIR");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_MKDIR for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.Mkdir(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_MKDIR request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_MKDIR \n");
          break;

        case (int) NFS_RMDIR:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_RMDIR");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_RMDIR for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.Rmdir(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_RMDIR request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_RMDIR \n");
          break;

        case (int) NFS_READDIR:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_READDIR");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_READDIR for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.Readdir(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_READDIR request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_READDIR \n");
          break;

        case (int) NFS_STATFS:
	  Env.dprintln(PRINT_PROCEDURE, "\n processing request [ " + xid + " ] procedure : NFS_STATFS");
	  Env.dprintln(PRINT_PACKET, "processing request [ " + xid + " ] procedure : NFS_STATFS for packet[" 
		       + packet.toRequestString((int)procedure));
          result = dirService.StatFS(xid, packet);
	  Env.dprintln(PRINT_PACKET, "result for NFS_STATFS request [ " + xid + "]" 
		       + result.toResultString((int)procedure)+"\n--------------------\n");
	  Env.dprintln(PRINT_PROCEDURE, "return request [ " + xid + " ] procedure : NFS_STATFS \n");
          break;

        default:
          System.err.print("Unsupported NFS procedure called (" 
                           + procedure + ") from "
                           + packet.Source().getHostAddress() + "\n");
          throw new NFSException(xid, NFSERR_IO);
        }
      } catch(NFSException e) {
        // make a reply packet that includes the error
        result = new XDRPacket(64);
        result.AddReplyHeader(e.GetXID());
        result.AddLong(e.GetError());
	assert xid == e.GetXID();
	Env.dprintln(PRINT_PACKET||PRINT_PROCEDURE, "FAILURE!!! Request [ " + xid + "] failed with error code=" 
		     + e.GetError());
      }
      Date end = new Date();
      
      if (iteration % 10 == 0) {
        Env.dprintln(PRINT_PERIODIC_RT,
                     "NFSHandler: Call took " +
                     (end.getTime() - begin.getTime()) +
                     "ms");
        /*
          System.out.println(" memory is at "
          + Runtime.getRuntime().freeMemory());
        */
      }

      //add by zjd
      if(dbg){
	result.verifyHead(xid, RPCReply);
      }

      // XXX disabled the xid queue
      if(useXIDCache){
	assert packetQueue != null;
	packetQueue.SetPacket(xid, result);
      }
      
     
      port.SendPacket(packet.Source(), packet.Port(), result);

      // put this packet back into the cache
      if((usePacketCache) && (resultFromPacketCache)){
	assert packetCache != null;
	packetCache.add(result);
      }else{
	result = null;//gc this packet
      }
    }catch(NegativeArraySizeException e){
      Env.dprinterrln(PRINT_ERROR, "Discarding packet");
    }
    packet = null;//gc this packet
  }

 /** 
 *  The trivial RPC, does nothing but send back an OK message. 
 **/ 
  public XDRPacket
  NFSNull(UDPPacketPort port, long xid, long procedure, XDRPacket packet){
    // Put together an XDR reply packet
    XDRPacket result = new XDRPacket(128);
    result.AddReplyHeader(xid);
    return result;
  }

 /** 
 *  Report to the client that the requested proc is not supported 
 **/ 
  public XDRPacket
  NFSProtoNotSupported(UDPPacketPort port,
                       long xid,
                       long procedure,
                       XDRPacket packet){
    // Put together an XDR reply packet
    XDRPacket result = new XDRPacket(128);
    result.AddReplyHeader(xid);
    result.AddLong(NFSERR_STALE);

    return result;
  }
}
	    
