 /** 
 *  Contains the data structures used by the NFS ACCESS3 call 
 **/ 

 /** 
 *  Represents the arguments to the ACCESS NFS call 
 **/ 
class ACCESS3args{

 /** 
 *  Data members 
 **/ 
  private final NFS_nfs_fh3 object;
  private final int access;

 /** 
 *  Constructor 
 **/ 
  public
  ACCESS3args(NFS_nfs_fh3 newObject, int newAccess){
    this.object = newObject;
    this.access = newAccess;
  }

 /** 
 *  Return the object 
 **/ 
  public NFS_nfs_fh3
  getObject(){
    return(this.object);
  }

 /** 
 *  Return the access type 
 **/ 
  public int
  getAccess(){
    return(this.access);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    ACCESS3args args = null;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof ACCESS3args){
      args = (ACCESS3args)obj;
      eq = this.object.equals(args.object) && (this.access == args.access);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = this.object.hashCode() ^ (new Integer(this.access)).hashCode();
    return(code);
  }
}

 /** 
 *  Represent the result of the ACCESS NFS call when successful 
 **/ 
class ACCESS3resok{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_attr objAttributes;
  private final int access;

 /** 
 *  Constructor 
 **/ 
  public
  ACCESS3resok(NFS_post_op_attr newObjAttributes, int newAccess){
    this.objAttributes = newObjAttributes;
    this.access = newAccess;
  }

 /** 
 *  Return the object attributes 
 **/ 
  public NFS_post_op_attr
  getObjAttributes(){
    return(this.objAttributes);
  }

 /** 
 *  Return the access value 
 **/ 
  public int
  getAccess(){
    return(this.access);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    ACCESS3resok resok = null;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof ACCESS3resok){
      resok = (ACCESS3resok)obj;
      eq = (this.objAttributes.equals(resok.objAttributes) &&
            (this.access == resok.access));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = (this.objAttributes.hashCode() ^
            (new Integer(this.access)).hashCode());
    return(code);
  }
}

 /** 
 *  Represent the result of the ACCESS NFS call when unsuccessful 
 **/ 
class ACCESS3resfail{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_attr objAttributes;

 /** 
 *  Constructor 
 **/ 
  public
  ACCESS3resfail(NFS_post_op_attr newObjAttributes){
    this.objAttributes = newObjAttributes;
  }

 /** 
 *  Return the object attributes 
 **/ 
  public NFS_post_op_attr
  getObjAttributes(){
    return(this.objAttributes);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    ACCESS3resfail resfail = null;

    // 2 Java-declared fields, 1 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 3);
    if(obj instanceof ACCESS3resfail){
      resfail = (ACCESS3resfail)obj;
      eq = this.objAttributes.equals(resfail.objAttributes);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 1 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 3);
    code = this.objAttributes.hashCode();
    return(code);
  }
}

 /** 
 *  Represent the result of the ACCESS NFS call 
 **/ 
class ACCESS3res{

 /** 
 *  Data members 
 **/ 
  private final int status;
  private final ACCESS3resok resok;
  private final ACCESS3resfail resfail;

 /** 
 *  Constructor 
 **/ 
  public
  ACCESS3res(int newStatus, ACCESS3resok newResOk, ACCESS3resfail newResFail){
    this.status = newStatus;
    this.resok = newResOk;
    this.resfail = newResFail;
  }

 /** 
 *  Return the status 
 **/ 
  public int
  getStatus(){
    return(this.status);
  }

 /** 
 *  Return the result of the operation if successful 
 **/ 
  public ACCESS3resok
  getResOk(){
    assert(this.status == NFS_nfsstat3.NFS3_OK);
    return(this.resok);
  }

 /** 
 *  Return the result of the operation if unsuccessful 
 **/ 
  public ACCESS3resfail
  getResFail(){
    assert(this.status != NFS_nfsstat3.NFS3_OK);
    return(this.resfail);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    ACCESS3res res = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof ACCESS3res){
      res = (ACCESS3res)obj;
      if(this.status == res.status){
        if(this.status == NFS_nfsstat3.NFS3_OK){
          eq = this.resok.equals(res.resok);
        }else{
          eq = this.resfail.equals(res.resfail);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(this.status == NFS_nfsstat3.NFS3_OK){
      code = (new Integer(this.status)).hashCode() ^ this.resok.hashCode();
    }else{
      code = (new Integer(this.status)).hashCode() ^ this.resfail.hashCode();
    }
    return(code);
  }
}

 /** 
 *  Stores constant values 
 **/ 
public class ACCESS3{

 /** 
 *  Constants 
 **/ 
  public static final int ACCESS3_READ =    0x0001;
  public static final int ACCESS3_LOOKUP =  0x0002;
  public static final int ACCESS3_MODIFY =  0x0004;
  public static final int ACCESS3_EXTEND =  0x0008;
  public static final int ACCESS3_DELETE =  0x0010;
  public static final int ACCESS3_EXECUTE = 0x0020;

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing ACCESS3.java...");
    ACCESS3.testACCESS3args();
    ACCESS3.testACCESS3resok();
    ACCESS3.testACCESS3resfail();
    ACCESS3.testACCESS3res();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testACCESS3args(){
    ACCESS3args args1 = null;
    ACCESS3args args2 = null;
    ACCESS3args args3 = null;

    // Test 1
    args1 = new ACCESS3args(NFS_nfs_fh3.makeAllZeroHandle(),
                            ACCESS3.ACCESS3_READ);
    assert(args1.getObject().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args1.getAccess() == ACCESS3.ACCESS3_READ);
    assert(args1.equals(args1));

    // Test 2
    args2 = new ACCESS3args(NFS_nfs_fh3.makeAllZeroHandle(),
                            ACCESS3.ACCESS3_LOOKUP);
    assert(args2.getObject().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args2.getAccess() == ACCESS3.ACCESS3_LOOKUP);
    assert(args2.equals(args2));
    assert(!args2.equals(args1));
    assert(!args1.equals(args2));
    assert(args2.hashCode() != args1.hashCode());

    // Test 3
    args3 = new ACCESS3args(NFS_nfs_fh3.makeAllZeroHandle(),
                            ACCESS3.ACCESS3_READ);
    assert(args3.getObject().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args3.getAccess() == ACCESS3.ACCESS3_READ);
    assert(args3.equals(args3));
    assert(args3.equals(args1));
    assert(args1.equals(args3));
    assert(!args3.equals(args2));
    assert(!args2.equals(args3));
    assert(args3.hashCode() == args1.hashCode());
    assert(args3.hashCode() != args2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testACCESS3resok(){
    ACCESS3resok resok1 = null;
    ACCESS3resok resok2 = null;
    ACCESS3resok resok3 = null;

    // Test 1
    resok1 = new ACCESS3resok(new NFS_post_op_attr(false, null),
                              ACCESS3.ACCESS3_READ);
    assert(resok1.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok1.getAccess() == ACCESS3.ACCESS3_READ);
    assert(resok1.equals(resok1));

    // Test 2
    resok2 = new ACCESS3resok(new NFS_post_op_attr(false, null),
                              ACCESS3.ACCESS3_MODIFY);
    assert(resok2.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok2.getAccess() == ACCESS3.ACCESS3_MODIFY);
    assert(resok2.equals(resok2));
    assert(!resok2.equals(resok1));
    assert(!resok1.equals(resok2));
    assert(resok2.hashCode() != resok1.hashCode());

    // Test 3
    resok3 = new ACCESS3resok(new NFS_post_op_attr(false, null),
                              ACCESS3.ACCESS3_READ);
    assert(resok3.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok3.getAccess() == ACCESS3.ACCESS3_READ);
    assert(resok3.equals(resok3));
    assert(resok3.equals(resok1));
    assert(resok1.equals(resok3));
    assert(!resok3.equals(resok2));
    assert(!resok2.equals(resok3));
    assert(resok3.hashCode() == resok1.hashCode());
    assert(resok3.hashCode() != resok2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testACCESS3resfail(){
    ACCESS3resfail resfail1 = null;
    ACCESS3resfail resfail2 = null;
    ACCESS3resfail resfail3 = null;
    NFS_fattr3 attr = null;

    // Test 1
    attr = new NFS_fattr3(NFS_ftype3.NF3REG,
                          36,
                          63,
                          71,
                          85,
                          (72L << 32) + 83L,
                          (81L << 32) + 66L,
                          new NFS_specdata3(55, 49),
                          (22L << 32) + 68L,
                          (99L << 32) + 163L,
                          new NFS_nfstime3(43, 7),
                          new NFS_nfstime3(31, 48),
                          new NFS_nfstime3(12, 8));
    resfail1 = new ACCESS3resfail(new NFS_post_op_attr(true, attr));
    assert(resfail1.getObjAttributes().equals(new NFS_post_op_attr(true,
                                                                   attr)));
    assert(resfail1.equals(resfail1));

    // Test 2
    resfail2 = new ACCESS3resfail(new NFS_post_op_attr(false, null));
    assert(resfail2.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                   null)));
    assert(resfail2.equals(resfail2));
    assert(!resfail2.equals(resfail1));
    assert(!resfail1.equals(resfail2));
    assert(resfail2.hashCode() != resfail1.hashCode());

    // Test 3
    resfail3 = new ACCESS3resfail(new NFS_post_op_attr(true, attr));
    assert(resfail3.getObjAttributes().equals(new NFS_post_op_attr(true,
                                                                   attr)));
    assert(resfail3.equals(resfail3));
    assert(resfail3.equals(resfail1));
    assert(resfail1.equals(resfail3));
    assert(!resfail3.equals(resfail2));
    assert(!resfail2.equals(resfail3));
    assert(resfail3.hashCode() == resfail1.hashCode());
    assert(resfail3.hashCode() != resfail2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testACCESS3res(){
    ACCESS3res res1 = null;
    ACCESS3res res2 = null;
    ACCESS3res res3 = null;

    // Test 1
    res1 = new ACCESS3res(NFS_nfsstat3.NFS3_OK,
                          new ACCESS3resok(new NFS_post_op_attr(false, null),
                                           ACCESS3.ACCESS3_READ),
                          null);
    assert(res1.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res1.getResOk().equals(new
                                  ACCESS3resok(new
                                               NFS_post_op_attr(false, null),
                                               ACCESS3.ACCESS3_READ)));
    assert(res1.equals(res1));

    // Test 2
    res2 = new ACCESS3res(NFS_nfsstat3.NFS3ERR_IO,
                          null,
                          new ACCESS3resfail(new NFS_post_op_attr(false,
                                                                  null)));
    assert(res2.getStatus() == NFS_nfsstat3.NFS3ERR_IO);
    assert(res2.getResFail().equals(new
                                    ACCESS3resfail(new
                                                   NFS_post_op_attr(false,
                                                                    null))));
    assert(res2.equals(res2));
    assert(!res2.equals(res1));
    assert(!res1.equals(res2));
    assert(res2.hashCode() != res1.hashCode());

    // Test 3
    res3 = new ACCESS3res(NFS_nfsstat3.NFS3_OK,
                          new ACCESS3resok(new NFS_post_op_attr(false, null),
                                           ACCESS3.ACCESS3_READ),
                          null);
    assert(res3.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res3.getResOk().equals(new
                                  ACCESS3resok(new
                                               NFS_post_op_attr(false, null),
                                               ACCESS3.ACCESS3_READ)));
    assert(res3.equals(res3));
    assert(res3.equals(res1));
    assert(res1.equals(res3));
    assert(!res3.equals(res2));
    assert(!res2.equals(res3));
    assert(res3.hashCode() == res1.hashCode());
    assert(res3.hashCode() != res2.hashCode());
  }
}
