 /** 
 *  Represent data structures used by the CREATE NFS call 
 **/ 
import java.util.Arrays;

 /** 
 *  Store some constants used by the create operation 
 **/ 
class NFS_createmode3 implements Immutable{

 /** 
 *  Data members 
 **/ 
  public final static int UNCHECKED = 0;
  public final static int GUARDED = 1;
  public final static int EXCLUSIVE = 2;
}

 /** 
 *  Represent the NFS3 data structure used to verify creation of files 
 **/ 
class NFS_createverf3 implements Immutable{

 /** 
 *  Constants 
 **/ 
  public static final int NFS3_CREATEVERFSIZE = 8;

 /** 
 *  Local data members 
 **/ 
  private final byte[] data;

 /** 
 *  Constructor 
 **/ 
  public
  NFS_createverf3(byte[] newData){
    byte[] dataCopy = null;

    assert(newData.length == NFS3_CREATEVERFSIZE);
    dataCopy = new byte[NFS3_CREATEVERFSIZE];
    for(int i = 0; i < NFS3_CREATEVERFSIZE; i++){
      dataCopy[i] = newData[i];
    }
    this.data = dataCopy;
  }

 /** 
 *  Return the data 
 **/ 
  public byte[]
  getData(){
    byte[] dataCopy = null;

    dataCopy = new byte[NFS3_CREATEVERFSIZE];
    for(int i = 0; i < NFS3_CREATEVERFSIZE; i++){
      dataCopy[i] = this.data[i];
    }
    return(dataCopy);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    NFS_createverf3 verf = null;
    boolean eq = false;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof NFS_createverf3){
      verf = (NFS_createverf3)obj;
      if(verf.data.length == NFS3_CREATEVERFSIZE){
        eq = true;
        for(int i = 0; (i < NFS3_CREATEVERFSIZE) && eq; i++){
          eq = (this.data[i] == verf.data[i]);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = Arrays.hashCode(this.data);
    return(code);
  }
}

 /** 
 *  Store different values depending on the mode of creation 
 **/ 
class NFS_createhow3 implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final int mode;
  private final NFS_sattr3 objAttributes;
  private final NFS_createverf3 verf;

 /** 
 *  Constructor 
 **/ 
  public
  NFS_createhow3(int newMode,
                 NFS_sattr3 newObjAttributes,
                 NFS_createverf3 newVerf){
    this.mode = newMode;
    this.objAttributes = newObjAttributes;
    this.verf = newVerf;
  }

 /** 
 *  Return the mode 
 **/ 
  public int
  getMode(){
    return(this.mode);
  }

 /** 
 *  Return the object attributes 
 **/ 
  public NFS_sattr3
  getObjAttributes(){
    assert(this.mode == NFS_createmode3.GUARDED);
    return(this.objAttributes);
  }

 /** 
 *  Return the verification data 
 **/ 
  public NFS_createverf3
  getVerf(){
    assert(this.mode == NFS_createmode3.EXCLUSIVE);
    return(this.verf);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    NFS_createhow3 createHow = null;
    boolean eq = false;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof NFS_createhow3){
      createHow = (NFS_createhow3)obj;
      if(this.mode == createHow.mode){
        if(this.mode == NFS_createmode3.GUARDED){
          eq = this.objAttributes.equals(createHow.objAttributes);
        }else if(this.mode == NFS_createmode3.EXCLUSIVE){
          eq = this.verf.equals(createHow.verf);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    code = (new Integer(this.mode)).hashCode();
    if(this.mode == NFS_createmode3.GUARDED){
      code = (code ^ this.objAttributes.hashCode());
    }else if(this.mode == NFS_createmode3.EXCLUSIVE){
      code = (code ^ this.verf.hashCode());
    }
    return(code);
  }
}

 /** 
 *  Represent the arguments to the CREATE NFS call 
 **/ 
class CREATE3args implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_diropargs3 where;
  private final NFS_createhow3 how;

 /** 
 *  Constructor 
 **/ 
  public
  CREATE3args(NFS_diropargs3 newWhere, NFS_createhow3 newHow){
    this.where = newWhere;
    this.how = newHow;
  }

 /** 
 *  Return the directory operation arguments 
 **/ 
  public NFS_diropargs3
  getWhere(){
    return(this.where);
  }

 /** 
 *  Return the creation data 
 **/ 
  public NFS_createhow3
  getHow(){
    return(this.how);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    CREATE3args args = null;
    boolean eq = false;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof CREATE3args){
      args = (CREATE3args)obj;
      eq = this.where.equals(args.where) && this.how.equals(args.how);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = this.where.hashCode() ^ this.how.hashCode();
    return(code);
  }
}

 /** 
 *  Represent the result of the NFS CREATE3 call if successful 
 **/ 
class CREATE3resok implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_fh3 obj;
  private final NFS_post_op_attr objAttributes;
  private final NFS_wcc_data dirWCC;

 /** 
 *  Constructor 
 **/ 
  public
  CREATE3resok(NFS_post_op_fh3 newObj,
               NFS_post_op_attr newObjAttributes,
               NFS_wcc_data newDirWCC){
    this.obj = newObj;
    this.objAttributes = newObjAttributes;
    this.dirWCC = newDirWCC;
  }

 /** 
 *  Return the file handle 
 **/ 
  public NFS_post_op_fh3
  getObj(){
    return(this.obj);
  }

 /** 
 *  Return the object attributes 
 **/ 
  public NFS_post_op_attr
  getObjAttributes(){
    return(this.objAttributes);
  }

 /** 
 *  Return the directory weak-cache-consistency data 
 **/ 
  public NFS_wcc_data
  getDirWCC(){
    return(this.dirWCC);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    CREATE3resok resok = null;
    boolean eq = false;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof CREATE3resok){
      resok = (CREATE3resok)obj;
      eq = (this.obj.equals(resok.obj) &&
            this.objAttributes.equals(resok.objAttributes) &&
            this.dirWCC.equals(resok.dirWCC));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    code = (this.obj.hashCode() ^
            this.objAttributes.hashCode() ^
            this.dirWCC.hashCode());
    return(code);
  }
}

 /** 
 *  Represent the result of the NFS CREATE3 call if unsuccessful 
 **/ 
class CREATE3resfail implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_wcc_data dirWCC;

 /** 
 *  Constructor 
 **/ 
  public
  CREATE3resfail(NFS_wcc_data newDirWCC){
    this.dirWCC = newDirWCC;
  }

 /** 
 *  Get the weak-cache-consistency data 
 **/ 
  public NFS_wcc_data
  getDirWCC(){
    return(this.dirWCC);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    CREATE3resfail resfail = null;
    boolean eq = false;

    // 2 Java-declared fields, 1 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 3);
    if(obj instanceof CREATE3resfail){
      resfail = (CREATE3resfail)obj;
      eq = this.dirWCC.equals(resfail.dirWCC);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 1 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 3);
    code = this.dirWCC.hashCode();
    return(code);
  }
}

 /** 
 *  Store the result of the CREATE NFS call 
 **/ 
class CREATE3res{

 /** 
 *  Data members 
 **/ 
  private final int status;
  private final CREATE3resok resok;
  private final CREATE3resfail resfail;

 /** 
 *  Constructor 
 **/ 
  public
  CREATE3res(int newStatus, CREATE3resok newResOk, CREATE3resfail newResFail){
    this.status = newStatus;
    this.resok = newResOk;
    this.resfail = newResFail;
  }

 /** 
 *  Return the status 
 **/ 
  public int
  getStatus(){
    return(this.status);
  }

 /** 
 *  Return the result of the operation if successful 
 **/ 
  public CREATE3resok
  getResOk(){
    assert(this.status == NFS_nfsstat3.NFS3_OK);
    return(this.resok);
  }

 /** 
 *  Return the result of the operation if unsuccessful 
 **/ 
  public CREATE3resfail
  getResFail(){
    assert(this.status != NFS_nfsstat3.NFS3_OK);
    return(this.resfail);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public boolean
  equals(Object obj){
    boolean eq = false;
    CREATE3res res = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof CREATE3res){
      res = (CREATE3res)obj;
      if(this.status == res.status){
        if(this.status == NFS_nfsstat3.NFS3_OK){
          eq = this.resok.equals(res.resok);
        }else{
          eq = this.resfail.equals(resfail);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(this.status == NFS_nfsstat3.NFS3_OK){
      code = (new Integer(this.status)).hashCode() ^ this.resok.hashCode();
    }else{
      code = (new Integer(this.status)).hashCode() ^ this.resfail.hashCode();
    }
    return(code);
  }
}

 /** 
 *  Used for testing 
 **/ 
class CREATE3{

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("CREATE3 self test...");
    CREATE3.testNFS_createverf3();
    CREATE3.testNFS_createhow3();
    CREATE3.testCREATE3args();
    CREATE3.testCREATE3resok();
    CREATE3.testCREATE3resfail();
    CREATE3.testCREATE3res();
    System.err.println("...CREATE3 self test succeeds");
    System.exit(0);
  }

 /** 
 *  Test the NFS_createverf3 class 
 **/ 
  private static void
  testNFS_createverf3(){
    byte[] data1 = null;
    byte[] data2 = null;
    byte[] data3 = null;
    NFS_createverf3 cv1 = null;    
    NFS_createverf3 cv2 = null;    
    NFS_createverf3 cv3 = null;    

    // Test 1
    data1 = new byte[NFS_createverf3.NFS3_CREATEVERFSIZE];
    for(int i = 0; i < NFS_createverf3.NFS3_CREATEVERFSIZE; i++){
      data1[i] = (byte)i;
    }
    cv1 = new NFS_createverf3(data1);
    assert(cv1.equals(cv1));

    // Test 2
    data2 = new byte[NFS_createverf3.NFS3_CREATEVERFSIZE];
    for(int i = 0; i < NFS_createverf3.NFS3_CREATEVERFSIZE; i++){
      data2[i] = (byte)i;
    }
    cv2 = new NFS_createverf3(data2);
    assert(cv2.equals(cv2));
    assert(cv1.equals(cv2));
    assert(cv2.equals(cv1));
    assert(cv1.hashCode() == cv2.hashCode());

    // Test 3
    data3 = new byte[NFS_createverf3.NFS3_CREATEVERFSIZE];
    for(int i = 0; i < NFS_createverf3.NFS3_CREATEVERFSIZE; i++){
      data3[i] = (byte)(NFS_createverf3.NFS3_CREATEVERFSIZE - i);
    }
    data3[5] = 10; // Random change
    cv3 = new NFS_createverf3(data3);
    assert(cv3.equals(cv3));
    assert(!cv1.equals(cv3));
    assert(!cv3.equals(cv1));
    assert(!cv2.equals(cv3));
    assert(!cv3.equals(cv2));
    assert(cv1.hashCode() != cv3.hashCode());
    assert(cv2.hashCode() != cv3.hashCode());
  }

 /** 
 *  Test the NFS_createhow3 class 
 **/ 
  private static void
  testNFS_createhow3(){
    NFS_createhow3 how1 = null;
    NFS_createhow3 how2 = null;
    NFS_createhow3 how3 = null;
    NFS_sattr3 sattr = null;

    // Test 1
    sattr = new NFS_sattr3(new NFS_set_mode3(true, 53),
                           new NFS_set_uid3(false, 2),
                           new NFS_set_gid3(true, 11),
                           new NFS_set_size3(true, (1L << 32) + 58L),
                           new NFS_set_atime(NFS_time_how.DONT_CHANGE,
                                             new NFS_nfstime3(66, 67)),
                           new NFS_set_mtime(NFS_time_how.SET_TO_CLIENT_TIME,
                                             new NFS_nfstime3(5, 15)));
    how1 = new NFS_createhow3(NFS_createmode3.GUARDED, sattr, null);
    assert(how1.getMode() == NFS_createmode3.GUARDED);
    assert(how1.getObjAttributes().equals(sattr));
    assert(how1.equals(how1));

    // Test 2
    how2 = new NFS_createhow3(NFS_createmode3.EXCLUSIVE,
                              null,
                              new NFS_createverf3(new byte[8]));
    assert(how2.getMode() == NFS_createmode3.EXCLUSIVE);
    assert(how2.getVerf().equals(new NFS_createverf3(new byte[8])));
    assert(how2.equals(how2));
    assert(!how2.equals(how1));
    assert(!how1.equals(how2));
    assert(how2.hashCode() != how1.hashCode());

    // Test 3
    how3 = new NFS_createhow3(NFS_createmode3.GUARDED, sattr, null);
    assert(how3.getMode() == NFS_createmode3.GUARDED);
    assert(how3.getObjAttributes().equals(sattr));
    assert(how3.equals(how3));
    assert(how3.equals(how1));
    assert(how1.equals(how3));
    assert(!how3.equals(how2));
    assert(!how2.equals(how3));
    assert(how3.hashCode() == how1.hashCode());
    assert(how3.hashCode() != how2.hashCode());
  }

 /** 
 *  Test the CREATE3args class 
 **/ 
  private static void
  testCREATE3args(){
    CREATE3args args1 = null;
    CREATE3args args2 = null;
    CREATE3args args3 = null;
    NFS_createhow3 how = null;
    NFS_diropargs3 dirOpArgs1 = null;
    NFS_diropargs3 dirOpArgs2 = null;

    // Test 1
    dirOpArgs1 = new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(), "a");
    how = new NFS_createhow3(NFS_createmode3.EXCLUSIVE,
                             null,
                             new NFS_createverf3(new byte[8]));
    args1 = new CREATE3args(dirOpArgs1, how);
    assert(args1.getWhere().equals(dirOpArgs1));
    assert(args1.getHow().equals(how));
    assert(args1.equals(args1));

    // Test 2
    dirOpArgs2 = new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(), "b");
    args2 = new CREATE3args(dirOpArgs2, how);
    assert(args2.getWhere().equals(dirOpArgs2));
    assert(args2.getHow().equals(how));
    assert(args2.equals(args2));
    assert(!args2.equals(args1));
    assert(!args1.equals(args2));
    assert(args2.hashCode() != args1.hashCode());

    // Test 3
    args3 = new CREATE3args(dirOpArgs1, how);
    assert(args3.getWhere().equals(dirOpArgs1));
    assert(args3.getHow().equals(how));
    assert(args3.equals(args3));
    assert(args3.equals(args1));
    assert(args1.equals(args3));
    assert(!args3.equals(args2));
    assert(!args2.equals(args3));
    assert(args3.hashCode() == args1.hashCode());
    assert(args3.hashCode() != args2.hashCode());
  }

 /** 
 *  Test the CREATE3resok class 
 **/ 
  private static void
  testCREATE3resok(){
    CREATE3resok resok1 = null;
    CREATE3resok resok2 = null;
    CREATE3resok resok3 = null;
    NFS_wcc_data wccData = null;
    NFS_pre_op_attr preAttr1 = null;
    NFS_pre_op_attr preAttr2 = null;

    // Test 1
    preAttr1 = new NFS_pre_op_attr(true,
                                   new NFS_wcc_attr(10,
                                                    new NFS_nfstime3(1, 2),
                                                    new NFS_nfstime3(3, 4)));
    resok1 = new CREATE3resok(new NFS_post_op_fh3(false, null),
                              new NFS_post_op_attr(false, null),
                              new NFS_wcc_data(preAttr1,
                                               new NFS_post_op_attr(false,
                                                                    null)));
    assert(resok1.getObj().equals(new NFS_post_op_fh3(false, null)));
    assert(resok1.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok1.getDirWCC().equals(new
                                     NFS_wcc_data(preAttr1,
                                                  new
                                                  NFS_post_op_attr(false,
                                                                   null))));
    assert(resok1.equals(resok1));

    // Test 2
    preAttr2 = new NFS_pre_op_attr(true,
                                   new NFS_wcc_attr(20,
                                                    new NFS_nfstime3(1, 2),
                                                    new NFS_nfstime3(3, 4)));
    resok2 = new CREATE3resok(new NFS_post_op_fh3(false, null),
                              new NFS_post_op_attr(false, null),
                              new NFS_wcc_data(preAttr2,
                                               new NFS_post_op_attr(false,
                                                                    null)));
    assert(resok2.getObj().equals(new NFS_post_op_fh3(false, null)));
    assert(resok2.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok2.getDirWCC().equals(new
                                     NFS_wcc_data(preAttr2,
                                                  new
                                                  NFS_post_op_attr(false,
                                                                   null))));
    assert(resok2.equals(resok2));
    assert(!resok2.equals(resok1));
    assert(!resok1.equals(resok2));
    assert(resok2.hashCode() != resok1.hashCode());

    // Test 3
    resok3 = new CREATE3resok(new NFS_post_op_fh3(false, null),
                              new NFS_post_op_attr(false, null),
                              new NFS_wcc_data(preAttr1,
                                               new NFS_post_op_attr(false,
                                                                    null)));
    assert(resok3.getObj().equals(new NFS_post_op_fh3(false, null)));
    assert(resok3.getObjAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok3.getDirWCC().equals(new
                                     NFS_wcc_data(preAttr1,
                                                  new
                                                  NFS_post_op_attr(false,
                                                                   null))));
    assert(resok3.equals(resok3));
    assert(resok3.equals(resok1));
    assert(resok1.equals(resok3));
    assert(!resok3.equals(resok2));
    assert(!resok2.equals(resok3));
    assert(resok3.hashCode() == resok1.hashCode());
    assert(resok3.hashCode() != resok2.hashCode());
  }

 /** 
 *  Test the CREATE3resfail class 
 **/ 
  private static void
  testCREATE3resfail(){
    CREATE3resfail resfail1 = null;
    CREATE3resfail resfail2 = null;
    CREATE3resfail resfail3 = null;
    NFS_pre_op_attr preAttr1 = null;
    NFS_pre_op_attr preAttr2 = null;
    NFS_post_op_attr postAttr = null;

    // Test 1
    preAttr1 = new NFS_pre_op_attr(false, null);
    postAttr = new NFS_post_op_attr(false, null);
    resfail1 = new CREATE3resfail(new NFS_wcc_data(preAttr1, postAttr));
    assert(resfail1.getDirWCC().equals(new NFS_wcc_data(preAttr1, postAttr)));
    assert(resfail1.equals(resfail1));

    // Test 2
    preAttr2 = new NFS_pre_op_attr(true,
                                   new
                                   NFS_wcc_attr(5000,
                                                new NFS_nfstime3(78, 80),
                                                new NFS_nfstime3(23, 12)));
    resfail2 = new CREATE3resfail(new NFS_wcc_data(preAttr2, postAttr));
    assert(resfail2.getDirWCC().equals(new NFS_wcc_data(preAttr2, postAttr)));
    assert(resfail2.equals(resfail2));
    assert(!resfail2.equals(resfail1));
    assert(!resfail1.equals(resfail2));
    assert(resfail2.hashCode() != resfail1.hashCode());

    // Test 3
    preAttr1 = new NFS_pre_op_attr(false, null);
    postAttr = new NFS_post_op_attr(false, null);
    resfail3 = new CREATE3resfail(new NFS_wcc_data(preAttr1, postAttr));
    assert(resfail3.getDirWCC().equals(new NFS_wcc_data(preAttr1, postAttr)));
    assert(resfail3.equals(resfail3));
    assert(resfail3.equals(resfail1));
    assert(resfail1.equals(resfail3));
    assert(!resfail3.equals(resfail2));
    assert(!resfail2.equals(resfail3));
    assert(resfail3.hashCode() == resfail1.hashCode());
    assert(resfail3.hashCode() != resfail2.hashCode());
  }

 /** 
 *  Test the CREATE3resfail class 
 **/ 
  private static void
  testCREATE3res(){
    CREATE3res res1 = null;
    CREATE3res res2 = null;
    CREATE3res res3 = null;
    CREATE3resok resok = null;
    CREATE3resfail resfail = null;
    NFS_pre_op_attr preAttr = null;

    // Test 1
    preAttr = new NFS_pre_op_attr(true,
                                  new NFS_wcc_attr(10,
                                                   new NFS_nfstime3(1, 2),
                                                   new NFS_nfstime3(3, 4)));
    resok = new CREATE3resok(new NFS_post_op_fh3(false, null),
                             new NFS_post_op_attr(false, null),
                             new NFS_wcc_data(preAttr,
                                              new NFS_post_op_attr(false,
                                                                   null)));
    res1 = new CREATE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res1.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res1.getResOk().equals(resok));
    assert(res1.equals(res1));

    // Test 2
    resfail = new
      CREATE3resfail(new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                                      new NFS_post_op_attr(false, null)));
    res2 = new CREATE3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    assert(res2.getStatus() == NFS_nfsstat3.NFS3ERR_IO);
    assert(res2.getResFail().equals(resfail));
    assert(res2.equals(res2));
    assert(!res2.equals(res1));
    assert(!res1.equals(res2));
    assert(res2.hashCode() != res1.hashCode());

    // Test 3
    res3 = new CREATE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res3.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res3.getResOk().equals(resok));
    assert(res3.equals(res3));
    assert(res3.equals(res1));
    assert(res1.equals(res3));
    assert(!res3.equals(res2));
    assert(!res2.equals(res3));
    assert(res3.hashCode() == res1.hashCode());
    assert(res3.hashCode() != res2.hashCode());
  }
}

 /** 
 *  $Log: CREATE3.java,v $ 
 *  Revision 1.1  2006/06/12 20:07:19  nayate 
 *  First version of the Java NFS library 
 *  
 **/ 
