 /** 
 *  A data structure that maps ObjectIDs and FileHandles in both directions 
 *  
 *  Note that we do not currently bound the size of this data structure 
 *  though a real implementation would normally do so. 
 **/ 
import java.util.HashMap;

public class FHCache{

 /** 
 *  Data members 
 **/ 
  private HashMap objIdHandleTable;
  private HashMap handleObjIdTable;
  private NFS_nfs_fh3 lastHandle;

 /** 
 *  Constructor 
 **/ 
  public
  FHCache(){
    this.objIdHandleTable = new HashMap();
    this.handleObjIdTable = new HashMap();
    // Make an all-zero handle
    this.lastHandle = NFS_nfs_fh3.makeAllZeroHandle();
    // Add an entry for this handle (root directory)
    this.objIdHandleTable.put(new ObjId(""), this.lastHandle);
    this.handleObjIdTable.put(this.lastHandle, new ObjId(""));
  }

 /** 
 *  Return a file handle from an objId 
 *  Note: We assume that objId is valid 
 **/ 
  public NFS_nfs_fh3
  getFileHandle(ObjId objIdPrefix){
    NFS_nfs_fh3 handle = null;

    handle = (NFS_nfs_fh3)this.objIdHandleTable.get(objIdPrefix);
    if(handle == null){
      handle = this.lastHandle.cloneIncOne();
      this.lastHandle = handle;
      this.objIdHandleTable.put(objIdPrefix, handle);
      this.handleObjIdTable.put(handle, objIdPrefix);
    }else{
      assert(this.handleObjIdTable.get(handle) != null);
      assert(this.handleObjIdTable.get(handle).equals(objIdPrefix));
    }
    return(handle);
  }

 /** 
 *  Return an objId from a file handle 
 **/ 
  public ObjId
  getObjIdPrefix(NFS_nfs_fh3 handle) throws BadFileHandleException{
    ObjId objId = null;

    objId = (ObjId)this.handleObjIdTable.get(handle);
    if(objId == null){
      throw new BadFileHandleException("Bad handle: " + handle);
    }
    return(objId);
  }

 /** 
 *  Remove an entry from the table (if present) 
 **/ 
  public void
  removeObjIdPrefix(ObjId objIdPrefix){
    NFS_nfs_fh3 handle = null;
    NFS_nfs_fh3 removedHandle = null;
    ObjId removedObjIdPrefix = null;

    handle = (NFS_nfs_fh3)this.objIdHandleTable.get(objIdPrefix);
    if(handle != null){
      removedHandle = (NFS_nfs_fh3)this.objIdHandleTable.remove(objIdPrefix);
      removedObjIdPrefix = (ObjId)this.handleObjIdTable.remove(handle);
      assert(removedHandle.equals(handle));
      assert(removedObjIdPrefix.equals(objIdPrefix));
    }
  }

 /** 
 *  Return true if "this" equals "obj" (note: currently unimplemented) 
 **/ 
  public boolean
  equals(Object obj){
    assert false:"Unimplemented";
    return(false);
  }
 /** 
 *  Return a hash code for "this" (note: currently unimplemented) 
 **/ 
  public int
  hashCode(Object obj){
    assert false:"Unimplemented";
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("FHCache self test...");
    FHCache.testSimple();
    System.err.println("...FHCache self test succeeds");
    System.exit(0);
  }

 /** 
 *  Test the simple methods of the class 
 **/ 
  private static void
  testSimple(){
    FHCache cache1 = null;
    FHCache cache2 = null;
    NFS_nfs_fh3 handle1 = null;
    NFS_nfs_fh3 handle2 = null;
    NFS_nfs_fh3 handle3 = null;
    NFS_nfs_fh3 handle4 = null;
    NFS_nfs_fh3 handle5 = null;
    NFS_nfs_fh3 handle6 = null;
    NFS_nfs_fh3 compHandle1 = null;
    NFS_nfs_fh3 compHandle2 = null;
    NFS_nfs_fh3 compHandle3 = null;
    byte[] b = null;

    cache1 = new FHCache();
    handle1 = cache1.getFileHandle(new ObjId("a"));
    handle2 = cache1.getFileHandle(new ObjId("b"));
    try{
      assert(cache1.getObjIdPrefix(handle1).equals(new ObjId("a")));
    }catch(BadFileHandleException e){
      assert(false);
    }
    try{
      assert(cache1.getObjIdPrefix(handle2).equals(new ObjId("b")));
    }catch(BadFileHandleException e){
      assert(false);
    }

    // Make sure the handles we got back follow what we expect
    b = new byte[NFS_nfs_fh3.NFS3_FHSIZE];
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 1;
    compHandle1 = NFS_nfs_fh3.makeHandle(b);
    assert(compHandle1.equals(handle1));
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 2;
    compHandle2 = NFS_nfs_fh3.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Try to pass in a bad file name
    try{
      b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 3;
      cache1.getObjIdPrefix(NFS_nfs_fh3.makeHandle(b));
      assert(false);
    }catch(BadFileHandleException e){
      // We should be here
    }

    // Try to overflow the file handle
    for(int i = 0; i < NFS_nfs_fh3.NFS3_FHSIZE; i++){
      b[i] = -1;
    }
    handle3 = NFS_nfs_fh3.makeHandle(b);
    handle4 = handle3.cloneIncOne();
    for(int i = 0; i < NFS_nfs_fh3.NFS3_FHSIZE; i++){
      b[i] = 0;
    }
    compHandle3 = NFS_nfs_fh3.makeHandle(b);
    assert(handle4.equals(compHandle3));

    // Try to remove entries
    cache2 = new FHCache();
    handle5 = cache2.getFileHandle(new ObjId("a"));
    handle6 = cache2.getFileHandle(new ObjId("b"));
    try{
      assert(cache2.getObjIdPrefix(handle5).equals(new ObjId("a")));
      assert(cache2.getObjIdPrefix(handle6).equals(new ObjId("b")));
      cache2.removeObjIdPrefix(new ObjId("a"));
      assert(cache2.getObjIdPrefix(handle6).equals(new ObjId("b")));
      try{
        cache2.getObjIdPrefix(handle5);
        assert false;
      }catch(BadFileHandleException e){
        // This was expected
      }
    }catch(BadFileHandleException e2){
      assert false : ("" + e2);
    }
  }
}