 /** 
 *  Represent the data structures used by the the LINK NFS 3 method 
 **/ 

 /** 
 *  Represent the arguments to the LINK NFS 3 call 
 **/ 
class LINK3args implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_nfs_fh3 file;
  private final NFS_diropargs3 link;

 /** 
 *  Constructor 
 **/ 
  public
  LINK3args(NFS_nfs_fh3 newFile, NFS_diropargs3 newLink){
    this.file = newFile;
    this.link = newLink;
  }

 /** 
 *  Return the file handle 
 **/ 
  public NFS_nfs_fh3
  getFile(){
    return(this.file);
  }

 /** 
 *  Return the directory operations information 
 **/ 
  public NFS_diropargs3
  getLink(){
    return(this.link);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    LINK3args args = null;

    // 2 Java-declared fields, 2 self-declared field
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof LINK3args){
      args = (LINK3args)obj;
      eq = (this.file.equals(args.file) && this.link.equals(args.link));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared field
    assert(this.getClass().getDeclaredFields().length == 4);
    code = (this.file.hashCode() ^ this.link.hashCode());

    return(code);
  }
}

 /** 
 *  Represent the results to the NFS 3 LINK call when successful 
 **/ 
class LINK3resok implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_attr fileAttributes;
  private final NFS_wcc_data linkDirWCC;

 /** 
 *  Constructor 
 **/ 
  public
  LINK3resok(NFS_post_op_attr newFileAttributes, NFS_wcc_data newLinkDirWCC){
    this.fileAttributes = newFileAttributes;
    this.linkDirWCC = newLinkDirWCC;
  }

 /** 
 *  Return the file attributes 
 **/ 
  public NFS_post_op_attr
  getFileAttributes(){
    return(this.fileAttributes);
  }

 /** 
 *  Return the new link directory weak-cache-consistency data 
 **/ 
  public NFS_wcc_data
  getLinkDirWCC(){
    return(this.linkDirWCC);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    LINK3resok resok = null;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof LINK3resok){
      resok = (LINK3resok)obj;
      eq = (this.fileAttributes.equals(resok.fileAttributes) &&
            this.linkDirWCC.equals(resok.linkDirWCC));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = this.fileAttributes.hashCode() ^ this.linkDirWCC.hashCode();
    return(code);
  }
}

 /** 
 *  Represent the results to the NFS 3 LINK call when it fails 
 **/ 
class LINK3resfail implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_attr fileAttributes;
  private final NFS_wcc_data linkDirWCC;

 /** 
 *  Constructor 
 **/ 
  public
  LINK3resfail(NFS_post_op_attr newFileAttributes, NFS_wcc_data newLinkDirWCC){
    this.fileAttributes = newFileAttributes;
    this.linkDirWCC = newLinkDirWCC;
  }

 /** 
 *  Return the file attributes 
 **/ 
  public NFS_post_op_attr
  getFileAttributes(){
    return(this.fileAttributes);
  }

 /** 
 *  Return the new link directory weak-cache-consistency data 
 **/ 
  public NFS_wcc_data
  getLinkDirWCC(){
    return(this.linkDirWCC);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    LINK3resfail resfail = null;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof LINK3resfail){
      resfail = (LINK3resfail)obj;
      eq = (this.fileAttributes.equals(resfail.fileAttributes) &&
            this.linkDirWCC.equals(resfail.linkDirWCC));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = this.fileAttributes.hashCode() ^ this.linkDirWCC.hashCode();
    return(code);
  }
}

 /** 
 *  Represent the results to the NFS 3 LINK call 
 **/ 
class LINK3res implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final int status;
  private final LINK3resok resok;
  private final LINK3resfail resfail;

 /** 
 *  Constructor 
 **/ 
  public
  LINK3res(int newStatus,
           LINK3resok newResOk,
           LINK3resfail newResFail){
    this.status = newStatus;
    this.resok = newResOk;
    this.resfail = newResFail;
    assert(((this.status == NFS_nfsstat3.NFS3_OK) && (this.resok != null)) ||
           ((this.status != NFS_nfsstat3.NFS3_OK) && (this.resfail != null)));
  }

 /** 
 *  Return the status 
 **/ 
  public int
  getStatus(){
    return(this.status);
  }

 /** 
 *  Return the result data if successful 
 **/ 
  public
  LINK3resok getResOk(){
    assert(this.status == NFS_nfsstat3.NFS3_OK);
    return(this.resok);
  }

 /** 
 *  Return the result data if the call failed 
 **/ 
  public
  LINK3resfail getResFail(){
    assert(this.status != NFS_nfsstat3.NFS3_OK);
    return(this.resfail);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    LINK3res res = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof LINK3res){
      res = (LINK3res)obj;
      if(this.status == res.status){
        if(this.status == NFS_nfsstat3.NFS3_OK){
          eq = this.resok.equals(res.resok);
        }else{
          eq = this.resfail.equals(res.resfail);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(this.status == NFS_nfsstat3.NFS3_OK){
      code = (new Integer(this.status)).hashCode() ^ this.resok.hashCode();
    }else{
      code = (new Integer(this.status)).hashCode() ^ this.resfail.hashCode();
    }
    return(code);
  }
}

 /** 
 *  Used for testing 
 **/ 
public class LINK3{

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("LINK3 self test...");
    LINK3.testLINK3args();
    LINK3.testLINK3resok();
    LINK3.testLINK3resfail();
    LINK3.testLINK3res();
    System.err.println("...LINK3 self test succeeds");
    System.exit(0);
  }

 /** 
 *  Test the LINK3args 
 **/ 
  private static void
  testLINK3args(){
    LINK3args args1 = null;
    LINK3args args2 = null;
    LINK3args args3 = null;

    // Test 1
    args1 = new LINK3args(NFS_nfs_fh3.makeAllZeroHandle(),
                          new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(),
                                             "a"));
    assert(args1.getFile().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args1.getLink().equals(new
                                  NFS_diropargs3(
                                    NFS_nfs_fh3.makeAllZeroHandle(), "a")));
    assert(args1.equals(args1));

    // Test 2
    args2 = new LINK3args(NFS_nfs_fh3.makeAllZeroHandle(),
                          new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(),
                                             "b"));
    assert(args2.getFile().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args2.getLink().equals(new
                                  NFS_diropargs3(
                                    NFS_nfs_fh3.makeAllZeroHandle(), "b")));
    assert(args2.equals(args2));
    assert(!args2.equals(args1));
    assert(!args1.equals(args2));
    assert(args2.hashCode() != args1.hashCode());

    // Test 3
    args3 = new LINK3args(NFS_nfs_fh3.makeAllZeroHandle(),
                          new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(),
                                             "a"));
    assert(args3.getFile().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args3.getLink().equals(new
                                  NFS_diropargs3(
                                    NFS_nfs_fh3.makeAllZeroHandle(), "a")));
    assert(args3.equals(args3));
    assert(args3.equals(args1));
    assert(args1.equals(args3));
    assert(!args3.equals(args2));
    assert(!args2.equals(args3));
    assert(args3.hashCode() == args1.hashCode());
    assert(args3.hashCode() != args2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testLINK3resok(){
    LINK3resok resok1 = null;
    LINK3resok resok2 = null;
    LINK3resok resok3 = null;
    NFS_pre_op_attr pre_attr1 = null;
    NFS_pre_op_attr pre_attr2 = null;
    NFS_post_op_attr post_attr1 = null;
    NFS_wcc_data data1 = null;
    NFS_wcc_data data2 = null;

    // Test 1
    pre_attr1 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(54,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    post_attr1 = new NFS_post_op_attr(true,
                                      new NFS_fattr3(NFS_ftype3.NF3REG,
                                                     36,
                                                     63,
                                                     71,
                                                     85,
                                                     (72L << 32) + 83L,
                                                     (81L << 32) + 66L,
                                                     new NFS_specdata3(55, 49),
                                                     (22L << 32) + 68L,
                                                     (99L << 32) + 163L,
                                                     new NFS_nfstime3(43, 7),
                                                     new NFS_nfstime3(31, 48),
                                                     new NFS_nfstime3(12, 8)));
    data1 = new NFS_wcc_data(pre_attr1, post_attr1);
    pre_attr2 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(4563,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    data2 = new NFS_wcc_data(pre_attr2, post_attr1);
    resok1 = new LINK3resok(new NFS_post_op_attr(false, null), data1);
    assert(resok1.getFileAttributes().equals(new NFS_post_op_attr(false,
                                                                  null)));
    assert(resok1.getLinkDirWCC().equals(data1));
    assert(resok1.equals(resok1));

    // Test 2
    resok2 = new LINK3resok(new NFS_post_op_attr(false, null), data2);
    assert(resok2.getFileAttributes().equals(new NFS_post_op_attr(false,
                                                                  null)));
    assert(resok2.getLinkDirWCC().equals(data2));
    assert(resok2.equals(resok2));
    assert(!resok2.equals(resok1));
    assert(!resok1.equals(resok2));
    assert(resok2.hashCode() != resok1.hashCode());

    // Test 3
    resok3 = new LINK3resok(new NFS_post_op_attr(false, null), data1);
    assert(resok3.getFileAttributes().equals(new NFS_post_op_attr(false,
                                                                  null)));
    assert(resok3.getLinkDirWCC().equals(data1));
    assert(resok3.equals(resok3));
    assert(resok3.equals(resok1));
    assert(resok1.equals(resok3));
    assert(!resok3.equals(resok2));
    assert(!resok2.equals(resok3));
    assert(resok3.hashCode() == resok1.hashCode());
    assert(resok3.hashCode() != resok2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testLINK3resfail(){
    LINK3resfail resfail1 = null;
    LINK3resfail resfail2 = null;
    LINK3resfail resfail3 = null;
    NFS_pre_op_attr pre_attr1 = null;
    NFS_pre_op_attr pre_attr2 = null;
    NFS_post_op_attr post_attr1 = null;
    NFS_wcc_data data1 = null;
    NFS_wcc_data data2 = null;

    // Test 1
    pre_attr1 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(54,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    post_attr1 = new NFS_post_op_attr(true,
                                      new NFS_fattr3(NFS_ftype3.NF3REG,
                                                     36,
                                                     63,
                                                     71,
                                                     85,
                                                     (72L << 32) + 83L,
                                                     (81L << 32) + 66L,
                                                     new NFS_specdata3(55, 49),
                                                     (22L << 32) + 68L,
                                                     (99L << 32) + 163L,
                                                     new NFS_nfstime3(43, 7),
                                                     new NFS_nfstime3(31, 48),
                                                     new NFS_nfstime3(12, 8)));
    data1 = new NFS_wcc_data(pre_attr1, post_attr1);
    pre_attr2 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(4563,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    data2 = new NFS_wcc_data(pre_attr2, post_attr1);
    resfail1 = new LINK3resfail(new NFS_post_op_attr(false, null), data1);
    assert(resfail1.getFileAttributes().equals(new NFS_post_op_attr(false,
                                                                    null)));
    assert(resfail1.getLinkDirWCC().equals(data1));
    assert(resfail1.equals(resfail1));

    // Test 2
    resfail2 = new LINK3resfail(new NFS_post_op_attr(false, null), data2);
    assert(resfail2.getFileAttributes().equals(new NFS_post_op_attr(false,
                                                                    null)));
    assert(resfail2.getLinkDirWCC().equals(data2));
    assert(resfail2.equals(resfail2));
    assert(!resfail2.equals(resfail1));
    assert(!resfail1.equals(resfail2));
    assert(resfail2.hashCode() != resfail1.hashCode());

    // Test 3
    resfail3 = new LINK3resfail(new NFS_post_op_attr(false, null), data1);
    assert(resfail3.getFileAttributes().equals(new NFS_post_op_attr(false,
                                                                    null)));
    assert(resfail3.getLinkDirWCC().equals(data1));
    assert(resfail3.equals(resfail3));
    assert(resfail3.equals(resfail1));
    assert(resfail1.equals(resfail3));
    assert(!resfail3.equals(resfail2));
    assert(!resfail2.equals(resfail3));
    assert(resfail3.hashCode() == resfail1.hashCode());
    assert(resfail3.hashCode() != resfail2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testLINK3res(){
    LINK3res res1 = null;
    LINK3res res2 = null;
    LINK3res res3 = null;
    LINK3resok resok = null;
    LINK3resfail resfail = null;
    NFS_pre_op_attr pre_attr1 = null;
    NFS_post_op_attr post_attr1 = null;
    NFS_wcc_data data1 = null;

    // Test 1
    pre_attr1 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(54,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    post_attr1 = new NFS_post_op_attr(true,
                                      new NFS_fattr3(NFS_ftype3.NF3REG,
                                                     36,
                                                     63,
                                                     71,
                                                     85,
                                                     (72L << 32) + 83L,
                                                     (81L << 32) + 66L,
                                                     new NFS_specdata3(55, 49),
                                                     (22L << 32) + 68L,
                                                     (99L << 32) + 163L,
                                                     new NFS_nfstime3(43, 7),
                                                     new NFS_nfstime3(31, 48),
                                                     new NFS_nfstime3(12, 8)));
    data1 = new NFS_wcc_data(pre_attr1, post_attr1);
    resok = new LINK3resok(new NFS_post_op_attr(false, null), data1);
    res1 = new LINK3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res1.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res1.getResOk().equals(resok));
    assert(res1.equals(res1));

    // Test 2
    resfail = new LINK3resfail(new NFS_post_op_attr(false, null), data1);
    res2 = new LINK3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    assert(res2.getStatus() == NFS_nfsstat3.NFS3ERR_IO);
    assert(res2.getResFail().equals(resfail));
    assert(res2.equals(res2));
    assert(!res2.equals(res1));
    assert(!res1.equals(res2));
    assert(res2.hashCode() != res1.hashCode());

    // Test 3
    res3 = new LINK3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res3.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res3.getResOk().equals(resok));
    assert(res3.equals(res3));
    assert(res3.equals(res1));
    assert(res1.equals(res3));
    assert(!res3.equals(res2));
    assert(!res2.equals(res3));
    assert(res3.hashCode() == res1.hashCode());
    assert(res3.hashCode() != res2.hashCode());
  }
}

 /** 
 *  $Log: LINK3.java,v $ 
 *  Revision 1.1  2006/06/12 20:07:19  nayate 
 *  First version of the Java NFS library 
 *  
 **/ 
