 /** 
 *  Represent an NFS3 file handle 
 **/ 
public class NFS_nfs_fh3 implements Immutable{

 /** 
 *  Constants 
 **/ 
  public static final int NFS3_FHSIZE = 64;

 /** 
 *  Local data members 
 **/ 
  private final byte[] data;

 /** 
 *  Constructor 
 **/ 
  private
  NFS_nfs_fh3(byte[] newData){
    this.data = newData;
  }

 /** 
 *  Create a new NFS_nfs_fh3 object with all zeros 
 **/ 
  public static NFS_nfs_fh3
  makeAllZeroHandle(){
    byte[] data = null;
    NFS_nfs_fh3 handle = null;

    data = new byte[NFS_nfs_fh3.NFS3_FHSIZE];
    handle = new NFS_nfs_fh3(data);
    return(handle);
  }

 /** 
 *  Create a new NFS_nfs_fh3 object without copying the array 
 **/ 
  public static NFS_nfs_fh3
  makeHandleDangerous(byte[] data){
    NFS_nfs_fh3 handle = null;

    assert(data.length == NFS_nfs_fh3.NFS3_FHSIZE);
    handle = new NFS_nfs_fh3(data);
    return(handle);
  }

 /** 
 *  Create a new NFS_nfs_fh3 object without copying the array 
 **/ 
  public static NFS_nfs_fh3
  makeHandle(byte[] data){
    byte[] dataCopy = null;
    NFS_nfs_fh3 handle = null;

    assert(data.length == NFS_nfs_fh3.NFS3_FHSIZE);
    dataCopy = new byte[NFS_nfs_fh3.NFS3_FHSIZE];
    for(int i = 0; i < NFS_nfs_fh3.NFS3_FHSIZE; i++){
      dataCopy[i] = data[i];
    }
    handle = new NFS_nfs_fh3(dataCopy);
    return(handle);
  }

 /** 
 *  Make a handle whose binary value is one larger than this one 
 **/ 
  public NFS_nfs_fh3
  cloneIncOne(){
    NFS_nfs_fh3 newHandle = null;
    byte[] dataCopy = null;
    byte carry = 0;
    int val = 0;

    // Note: We can play around with dataCopy because getData() returns a copy
    dataCopy = this.getData();
    carry = 1;
    for(int i = (NFS_nfs_fh3.NFS3_FHSIZE - 1); (carry > 0) && (i >= 0); i--){
      val = ((int)dataCopy[i]) & 0xFF;
      val += carry;
      carry = (byte)((val & 0xFFF0) >> 8);
      dataCopy[i] = (byte)(val & 0xFF);
    }
    newHandle = NFS_nfs_fh3.makeHandleDangerous(dataCopy);
    return(newHandle);
  }

 /** 
 *  Return the data 
 **/ 
  public byte[]
  getDataDangerous(){
    return(this.data);
  }

 /** 
 *  Return a copy of the data 
 **/ 
  public byte[]
  getData(){
    byte[] dataCopy = null;
    NFS_nfs_fh3 handle = null;

    dataCopy = new byte[NFS_nfs_fh3.NFS3_FHSIZE];
    for(int i = 0; i < NFS_nfs_fh3.NFS3_FHSIZE; i++){
      dataCopy[i] = this.data[i];
    }
    return(dataCopy);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    NFS_nfs_fh3 fh = null;
    boolean eq = false;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof NFS_nfs_fh3){
      fh = (NFS_nfs_fh3)obj;
      if(fh.data.length == NFS3_FHSIZE){
        eq = true;
        for(int i = 0; (i < NFS3_FHSIZE) && eq; i++){
          eq = (this.data[i] == fh.data[i]);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;
    int newCode = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    assert((NFS3_FHSIZE % 4) == 0);

    for(int i = 0; i < (NFS3_FHSIZE / 4); i++){
      newCode = ((((int)this.data[i * 4]) << 24) |
                 (((int)this.data[i * 4 + 1]) << 16) |
                 (((int)this.data[i * 4 + 2]) << 8) |
                 ((int)this.data[i * 4 + 3]));
      code = code ^ newCode;
    }
    return(code);
  }

 /** 
 *  Create and return a String representation for this object 
 **/ 
  public String
  toString(){
    String str = null;

    str = "[";
    for(int i = 0; i < NFS3_FHSIZE; i++){
      str += this.data[i];
    }
    str += "]";
    return(str);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("NFS_nfs_fh3 self test...");
    NFS_nfs_fh3.testSimple();
    NFS_nfs_fh3.testCloneIncOne();
    System.err.println("...NFS_nfs_fh3 self test succeeds");
    System.exit(0);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    byte[] data1 = null;
    byte[] data2 = null;
    byte[] data3 = null;
    NFS_nfs_fh3 fh1 = null;    
    NFS_nfs_fh3 fh2 = null;    
    NFS_nfs_fh3 fh3 = null;
    NFS_nfs_fh3 fh4 = null;
    NFS_nfs_fh3 fh5 = null;

    // Test 1
    data1 = new byte[NFS3_FHSIZE];
    for(int i = 0; i < NFS3_FHSIZE; i++){
      data1[i] = (byte)i;
    }
    fh1 = NFS_nfs_fh3.makeHandle(data1);
    assert(fh1.equals(fh1));
    fh4 = NFS_nfs_fh3.makeHandleDangerous(data1);
    assert(fh4.equals(fh4));
    assert(fh4.equals(fh1));
    assert(fh1.equals(fh4));

    // Test 2
    data2 = new byte[NFS3_FHSIZE];
    for(int i = 0; i < NFS3_FHSIZE; i++){
      data2[i] = (byte)i;
    }
    fh2 = NFS_nfs_fh3.makeHandle(data2);
    assert(fh2.equals(fh2));
    assert(fh1.equals(fh2));
    assert(fh2.equals(fh1));
    assert(fh1.hashCode() == fh2.hashCode());

    // Test 3
    data3 = new byte[NFS3_FHSIZE];
    for(int i = 0; i < NFS3_FHSIZE; i++){
      data3[i] = (byte)(NFS3_FHSIZE - i);
    }
    data3[5] = 10; // Random change
    fh3 = NFS_nfs_fh3.makeHandle(data3);
    assert(fh3.equals(fh3));
    assert(!fh1.equals(fh3));
    assert(!fh3.equals(fh1));
    assert(!fh2.equals(fh3));
    assert(!fh3.equals(fh2));
    assert(fh1.hashCode() != fh3.hashCode());
    assert(fh2.hashCode() != fh3.hashCode());
    fh5 = NFS_nfs_fh3.makeHandleDangerous(data3);
    assert(fh5.equals(fh5));
    assert(fh5.equals(fh3));
    assert(fh3.equals(fh5));
    assert(!fh5.equals(fh2));
    assert(!fh2.equals(fh5));
    assert(fh5.hashCode() != fh1.hashCode());
    assert(fh5.hashCode() != fh2.hashCode());
    assert(fh5.hashCode() == fh3.hashCode());
  }

 /** 
 *  Test the cloneIncOne method 
 **/ 
  private static void
  testCloneIncOne(){
    NFS_nfs_fh3 handle1 = null;
    NFS_nfs_fh3 handle2 = null;
    NFS_nfs_fh3 compHandle2 = null;
    byte[] b = null;

    // Test 1
    b = new byte[NFS_nfs_fh3.NFS3_FHSIZE];
    handle1 = NFS_nfs_fh3.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 1;
    compHandle2 = NFS_nfs_fh3.makeHandle(b);
    assert(handle2.equals(compHandle2));

    // Test 2
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = (byte)-1;
    handle1 = NFS_nfs_fh3.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 0;
    b[NFS_nfs_fh3.NFS3_FHSIZE - 2] = 1;
    compHandle2 = NFS_nfs_fh3.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Test 3
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 0;
    b[NFS_nfs_fh3.NFS3_FHSIZE - 2] = 10;
    handle1 = NFS_nfs_fh3.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    b[NFS_nfs_fh3.NFS3_FHSIZE - 1] = 1;
    compHandle2 = NFS_nfs_fh3.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Test 4
    for(int i = 0; i < 10; i++){
      b[NFS_nfs_fh3.NFS3_FHSIZE - i - 1] = -1;
    }
    handle1 = NFS_nfs_fh3.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    for(int i = 0; i < 10; i++){
      b[NFS_nfs_fh3.NFS3_FHSIZE - i - 1] = 0;
    }
    b[NFS_nfs_fh3.NFS3_FHSIZE - 11] = 1;
    compHandle2 = NFS_nfs_fh3.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Test 5
    for(int i = 0; i < NFS_nfs_fh3.NFS3_FHSIZE; i++){
      b[i] = (byte)-1;
    }
    handle1 = NFS_nfs_fh3.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    assert(handle2.equals(NFS_nfs_fh3.makeAllZeroHandle()));
  }
}

 /** 
 *  $Log: NFS_nfs_fh3.java,v $ 
 *  Revision 1.1  2006/06/12 20:07:20  nayate 
 *  First version of the Java NFS library 
 *  
 **/ 
