 /** 
 *  Store attributes before and after operations 
 *  Note: NFS gives only weak cache consistency so the attributes 
 *    are only as accurate as they can be 
 **/ 

 /** 
 *  Store attributes returned for some NFS operations 
 **/ 
class NFS_wcc_attr{

 /** 
 *  Data members 
 **/ 
  private final long size;
  private final NFS_nfstime3 mtime;
  private final NFS_nfstime3 ctime;

 /** 
 *  Constructor 
 **/ 
  public
  NFS_wcc_attr(long newSize, NFS_nfstime3 newMTime, NFS_nfstime3 newCTime){
    this.size = newSize;
    this.mtime = newMTime;
    this.ctime = newCTime;
  }

 /** 
 *  Return the size 
 **/ 
  public long
  getSize(){
    return(this.size);
  }

 /** 
 *  Return the mtime 
 **/ 
  public NFS_nfstime3
  getMTime(){
    return(this.mtime);
  }

 /** 
 *  Return the ctime 
 **/ 
  public NFS_nfstime3
  getCTime(){
    return(this.ctime);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    NFS_wcc_attr attr = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof NFS_wcc_attr){
      attr = (NFS_wcc_attr)obj;
      eq = ((this.size == attr.size) &&
            (this.mtime.equals(attr.mtime)) &&
            (this.ctime.equals(attr.ctime)));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    code = ((new Long(this.size)).hashCode() ^
            this.mtime.hashCode() ^
            this.ctime.hashCode());
    return(code);
  }
}

 /** 
 *  (Possibly) store attributes 
 **/ 
class NFS_pre_op_attr{

 /** 
 *  Data members 
 **/ 
  private final boolean attributesFollow;
  private final NFS_wcc_attr attributes;

 /** 
 *  Constructor 
 **/ 
  public
  NFS_pre_op_attr(boolean newAttributesFollow, NFS_wcc_attr newAttributes){
    this.attributesFollow = newAttributesFollow;
    this.attributes = newAttributes;
  }

 /** 
 *  Return the attributesFollow flag 
 **/ 
  public boolean
  getAttributesFollow(){
    return(this.attributesFollow);
  }

 /** 
 *  Return the stored attributes flag 
 **/ 
  public NFS_wcc_attr
  getAttributes(){
    return(this.attributes);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    NFS_pre_op_attr attr = null;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof NFS_pre_op_attr){
      attr = (NFS_pre_op_attr)obj;
      if(this.attributes != null){
        eq = ((this.attributesFollow == attr.attributesFollow) &&
              this.attributes.equals(attr.attributes));
      }else{
        eq = ((this.attributesFollow == attr.attributesFollow) &&
              (attr.attributes == null));
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(this.attributes != null){
      code = ((new Boolean(this.attributesFollow)).hashCode() ^
              this.attributes.hashCode());
    }else{
      code = (new Boolean(this.attributesFollow)).hashCode() ^ 0;
    }
    return(code);
  }
}

 /** 
 *  Implement the NFS3 wcc_data data structure 
 **/ 
public class NFS_wcc_data{

 /** 
 *  Data members 
 **/ 
  private final NFS_pre_op_attr before;
  private final NFS_post_op_attr after;

 /** 
 *  Data members 
 **/ 
  public
  NFS_wcc_data(NFS_pre_op_attr newBefore, NFS_post_op_attr newAfter){
    this.before = newBefore;
    this.after = newAfter;
  }

 /** 
 *  Return the attributes before the operation 
 **/ 
  public NFS_pre_op_attr
  getBefore(){
    return(this.before);
  }

 /** 
 *  Return the attributes before the operation 
 **/ 
  public NFS_post_op_attr
  getAfter(){
    return(this.after);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    NFS_wcc_data data = null;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof NFS_wcc_data){
      data = (NFS_wcc_data)obj;
      eq = this.before.equals(data.before) && this.after.equals(data.after);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    code = this.before.hashCode() ^ this.after.hashCode();
    return(code);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing NFS_wcc_data.java...");
    NFS_wcc_data.testNFS_wcc_attr();
    NFS_wcc_data.testNFS_pre_op_attr();
    NFS_wcc_data.testNFS_wcc_data();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testNFS_wcc_attr(){
    NFS_wcc_attr attr1 = null;
    NFS_wcc_attr attr2 = null;
    NFS_wcc_attr attr3 = null;

    // Test 1
    attr1 = new NFS_wcc_attr(100,
                             new NFS_nfstime3(10, 20),
                             new NFS_nfstime3(20, 10));
    assert(attr1.getSize() == 100);
    assert(attr1.getMTime().equals(new NFS_nfstime3(10, 20)));
    assert(attr1.getCTime().equals(new NFS_nfstime3(20, 10)));
    assert(attr1.equals(attr1));

    // Test 2
    attr2 = new NFS_wcc_attr(200,
                             new NFS_nfstime3(35, 47),
                             new NFS_nfstime3(48, 22));
    assert(attr2.getSize() == 200);
    assert(attr2.getMTime().equals(new NFS_nfstime3(35, 47)));
    assert(attr2.getCTime().equals(new NFS_nfstime3(48, 22)));
    assert(attr2.equals(attr2));
    assert(!attr1.equals(attr2));
    assert(!attr2.equals(attr1));
    assert(attr1.hashCode() != attr2.hashCode());

    // Test 3
    attr3 = new NFS_wcc_attr(100,
                             new NFS_nfstime3(10, 20),
                             new NFS_nfstime3(20, 10));
    assert(attr3.getSize() == 100);
    assert(attr3.getMTime().equals(new NFS_nfstime3(10, 20)));
    assert(attr3.getCTime().equals(new NFS_nfstime3(20, 10)));
    assert(attr3.equals(attr3));
    assert(!attr2.equals(attr3));
    assert(!attr3.equals(attr2));
    assert(attr1.equals(attr3));
    assert(attr3.equals(attr1));
    assert(attr1.hashCode() == attr3.hashCode());
    assert(attr2.hashCode() != attr3.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testNFS_pre_op_attr(){
    NFS_pre_op_attr attr1 = null;
    NFS_pre_op_attr attr2 = null;
    NFS_pre_op_attr attr3 = null;

    // Test 1
    attr1 = new NFS_pre_op_attr(true,
                                new NFS_wcc_attr(100,
                                                 new NFS_nfstime3(1, 3),
                                                 new NFS_nfstime3(4, 5)));
    assert(attr1.getAttributesFollow());
    assert(attr1.getAttributes().equals(new
                                        NFS_wcc_attr(100,
                                                     new NFS_nfstime3(1, 3),
                                                     new NFS_nfstime3(4, 5))));
    assert(attr1.equals(attr1));

    // Test 2
    attr2 = new NFS_pre_op_attr(false, null);
    assert(!attr2.getAttributesFollow());
    assert(attr2.getAttributes() == null);
    assert(attr2.equals(attr2));
    assert(!attr1.equals(attr2));
    assert(!attr2.equals(attr1));
    assert(attr1.hashCode() != attr2.hashCode());

    // Test 3
    attr3 = new NFS_pre_op_attr(true,
                                new NFS_wcc_attr(100,
                                                 new NFS_nfstime3(1, 3),
                                                 new NFS_nfstime3(4, 5)));
    assert(attr3.getAttributesFollow());
    assert(attr3.getAttributes().equals(new
                                        NFS_wcc_attr(100,
                                                     new NFS_nfstime3(1, 3),
                                                     new NFS_nfstime3(4, 5))));
    assert(attr3.equals(attr3));
    assert(attr3.equals(attr1));
    assert(attr1.equals(attr3));
    assert(!attr2.equals(attr3));
    assert(!attr3.equals(attr2));
    assert(attr3.hashCode() != attr2.hashCode());
    assert(attr3.hashCode() == attr1.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testNFS_wcc_data(){
    NFS_pre_op_attr pre_attr1 = null;
    NFS_pre_op_attr pre_attr2 = null;
    NFS_pre_op_attr pre_attr3 = null;
    NFS_post_op_attr post_attr1 = null;
    NFS_post_op_attr post_attr2 = null;
    NFS_post_op_attr post_attr3 = null;
    NFS_wcc_data data1 = null;
    NFS_wcc_data data2 = null;
    NFS_wcc_data data3 = null;

    // Test 1
    pre_attr1 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(54,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    post_attr1 = new NFS_post_op_attr(true,
                                      new NFS_fattr3(NFS_ftype3.NF3REG,
                                                     36,
                                                     63,
                                                     71,
                                                     85,
                                                     (72L << 32) + 83L,
                                                     (81L << 32) + 66L,
                                                     new NFS_specdata3(55, 49),
                                                     (22L << 32) + 68L,
                                                     (99L << 32) + 163L,
                                                     new NFS_nfstime3(43, 7),
                                                     new NFS_nfstime3(31, 48),
                                                     new NFS_nfstime3(12, 8)));
    data1 = new NFS_wcc_data(pre_attr1, post_attr1);
    assert(data1.getBefore().equals(pre_attr1));
    assert(data1.getAfter().equals(post_attr1));
    assert(data1.equals(data1));

    // Test 2
    pre_attr2 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(54,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    post_attr2 = new NFS_post_op_attr(false, null);
    data2 = new NFS_wcc_data(pre_attr2, post_attr2);

    assert(data2.getBefore().equals(pre_attr2));
    assert(data2.getAfter().equals(post_attr2));
    assert(data2.equals(data2));
    assert(!data2.equals(data1));
    assert(!data1.equals(data2));
    assert(data1.hashCode() != data2.hashCode());

    // Test 3
    pre_attr3 = new NFS_pre_op_attr(true,
                                    new
                                    NFS_wcc_attr(54,
                                                 new NFS_nfstime3(78, 80),
                                                 new NFS_nfstime3(23, 12)));
    post_attr3 = new NFS_post_op_attr(true,
                                      new NFS_fattr3(NFS_ftype3.NF3REG,
                                                     36,
                                                     63,
                                                     71,
                                                     85,
                                                     (72L << 32) + 83L,
                                                     (81L << 32) + 66L,
                                                     new NFS_specdata3(55, 49),
                                                     (22L << 32) + 68L,
                                                     (99L << 32) + 163L,
                                                     new NFS_nfstime3(43, 7),
                                                     new NFS_nfstime3(31, 48),
                                                     new NFS_nfstime3(12, 8)));
    data3 = new NFS_wcc_data(pre_attr3, post_attr3);

    assert(data3.getBefore().equals(pre_attr3));
    assert(data3.getAfter().equals(post_attr3));
    assert(data3.equals(data3));
    assert(data3.equals(data1));
    assert(data1.equals(data3));
    assert(!data3.equals(data2));
    assert(!data2.equals(data3));
    assert(data3.hashCode() != data2.hashCode());
    assert(data3.hashCode() == data1.hashCode());
  }
}