 /** 
 *  Implement the Java version of the NFS loopback server 
 **/ 
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.ByteArrayOutputStream;

public class PRACTIFS{

 /** 
 *  Constants 
 **/ 
  public static String META_SUFFIX = ".meta";
  public static String DATA_SUFFIX = ".data";
  public static int DEFAULT_READ_BLOCK_SIZE = 1000;
  public static long FSID = 1;

 /** 
 *  Data members 
 **/ 
  private PRACTIFSLocalInterface localInterface;
  private FHCache fhCache;
  private NFS_writeverf3 writeVerf;
  private long fileid;

 /** 
 *  Constructor 
 **/ 
  public
  PRACTIFS(PRACTIFSLocalInterface newLocalInterface){
    byte[] b = null;

    this.localInterface = newLocalInterface;
    this.fhCache = new FHCache();
    b = new byte[NFS_writeverf3.NFS3_WRITEVERFSIZE];
    b[1] = (byte)9;
    this.writeVerf = new NFS_writeverf3(b);
    this.fileid = 0;

    this.makeRootDirectory();
  }

 /** 
 *  Implement the GETATTR NFS method 
 **/ 
  public GETATTR3res
  getAttributes(GETATTR3args args){
    ObjId prefix = null;
    ObjId metaDataObjId = null;
    PRACTIFSReadLockToken token = null;
    NFS_fattr3 fattr = null;
    GETATTR3res result = null;
    GETATTR3resok resok = null;

    try{
      prefix = this.fhCache.getObjIdPrefix(args.getObject());
      metaDataObjId = new ObjId(prefix.getPath() + META_SUFFIX);
      token = this.localInterface.acquireReadLock(metaDataObjId);
      fattr = this.readAttributes(token, metaDataObjId);
      resok = new GETATTR3resok(fattr);
      result = new GETATTR3res(NFS_nfsstat3.NFS3_OK, resok);
    }catch(BadFileHandleException e){
      // The handle was bad
      result = new GETATTR3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null);
    }catch(FileNotFoundException e){
      // The file has already been deleted
      result = new GETATTR3res(NFS_nfsstat3.NFS3ERR_STALE, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new GETATTR3res(NFS_nfsstat3.NFS3ERR_IO, null);
    }/*catch(Exception e){
      result = new GETATTR3res(NFS_nfsstat3.NFS3ERR_SERVERFAULT, null);
      }*/finally{
      if(token != null){
        assert(prefix != null);
        this.localInterface.releaseReadLock(token, prefix);
      }
    }
    return(result);
  }

 /** 
 *  Implement the SETATTR NFS method 
 *  NOTE: This method does not currently allow increasing file sizes. 
 *  NOTE: We always set the time to the server time. Need to change this in 
 *  the future 
 *  Note: This method currently ignores the guard parameter. In the future 
 *  it should not do so. 
 *  Note: This method does not return attributes at all. Might want to 
 *  change this in the future. 
 **/ 
  public SETATTR3res
  setAttributes(SETATTR3args args){
    SETATTR3res result = null;
    ObjId prefix = null;
    ObjId metaDataObjId = null;
    PRACTIFSWriteLockToken token = null;
    NFS_wcc_data objWCC = null;
    NFS_pre_op_attr preOpAttr = null;
    NFS_post_op_attr postOpAttr = null;
    SETATTR3resok resok = null;
    SETATTR3resfail resfail = null;

    preOpAttr = new NFS_pre_op_attr(false, null);
    postOpAttr = new NFS_post_op_attr(false, null);
    objWCC = new NFS_wcc_data(preOpAttr, postOpAttr);
    try{
      prefix = this.fhCache.getObjIdPrefix(args.getObject());
      metaDataObjId = new ObjId(prefix.getPath() + META_SUFFIX);
      token = this.localInterface.acquireWriteLock(metaDataObjId);
      this.writeAttributes(token, metaDataObjId, args.getNewAttributes());
      resok = new SETATTR3resok(objWCC);
      result = new SETATTR3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // The file handle was bad
      resfail = new SETATTR3resfail(objWCC);
      result = new SETATTR3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new SETATTR3resfail(objWCC);
      result = new SETATTR3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      resfail = new SETATTR3resfail(objWCC);
      result = new SETATTR3res(NFS_nfsstat3.NFS3ERR_SERVERFAULT,
                               null,
                               resfail);
                               }*/finally{
      if(token != null){
        assert(prefix != null);
        this.localInterface.releaseWriteLock(token, prefix);
      }
    }
    return(result);
  }

 /** 
 *  Implement the LOOKUP NFS method 
 *  
 *  NOTE: This method does not currently check whether the file being 
 *    opened is really a directory (by checking its attributes). It only 
 *    does a simple test of trying to open it as a directory and 
 *    complaining if it fails. 
 **/ 
  public LOOKUP3res
  lookup(LOOKUP3args args){
    ObjId dirPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    PRACTIFSReadLockToken dirToken = null;
    PRACTIFSDirectory dir = null;
    ObjId filePrefix = null;
    NFS_post_op_attr objAttr = null;
    NFS_post_op_attr dirAttr = null;
    LOOKUP3resok resok = null;
    LOOKUP3resfail resfail = null;
    NFS_nfs_fh3 fileHandle = null;
    LOOKUP3res result = null;

    objAttr = new NFS_post_op_attr(false, null);
    dirAttr = new NFS_post_op_attr(false, null);
    try{
      dirPrefix = this.fhCache.getObjIdPrefix(args.getWhat().getDir());
      dirDataObjId = new ObjId(dirPrefix.getPath() + DATA_SUFFIX);
      dirMetaObjId = new ObjId(dirPrefix.getPath() + META_SUFFIX);
      dirToken = this.localInterface.acquireReadLock(dirDataObjId);
      dir = this.readDir(dirToken, dirDataObjId);
      filePrefix = dir.getObjIdPrefix(args.getWhat().getName());
      fileHandle = this.fhCache.getFileHandle(filePrefix);
      resok = new LOOKUP3resok(fileHandle, objAttr, dirAttr);
      result = new LOOKUP3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Bad directory handle
      resfail = new LOOKUP3resfail(dirAttr);
      result = new LOOKUP3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // This object is not a directory
      resfail = new LOOKUP3resfail(dirAttr);
      result = new LOOKUP3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      resfail = new LOOKUP3resfail(dirAttr);
      result = new LOOKUP3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(ObjNotInDirectoryException e){
      // Directory has no problem; the file just doesn't exist
      resfail = new LOOKUP3resfail(dirAttr);
      result = new LOOKUP3res(NFS_nfsstat3.NFS3ERR_NOENT, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new LOOKUP3resfail(dirAttr);
      result = new LOOKUP3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
     // TODO: Do something (e.g.: NFSERR3_STALE_HANDLE)
     
     }*/finally{
      if(dirToken != null){
        assert(dirPrefix != null);
        this.localInterface.releaseReadLock(dirToken, dirPrefix);
      }
    }
    return(result);
  }

 /** 
 *  Implement the ACCESS NFS method 
 *  
 *  NOTE: This method is currently simply a bypass; it allows any user 
 *    to perform any operation 
 **/ 
  public ACCESS3res
  access(ACCESS3args args){
    NFS_post_op_attr objAttr = null;
    ACCESS3resok resok = null;
    ACCESS3resfail resfail = null;
    ACCESS3res result = null;
    ObjId objPrefix = null;
    int val = 0;

    objAttr = new NFS_post_op_attr(false, null);
    val = (ACCESS3.ACCESS3_READ |
           ACCESS3.ACCESS3_LOOKUP |
           ACCESS3.ACCESS3_MODIFY |
           ACCESS3.ACCESS3_EXTEND |
           ACCESS3.ACCESS3_DELETE |
           ACCESS3.ACCESS3_EXECUTE);
    try{
      objPrefix = this.fhCache.getObjIdPrefix(args.getObject());
      resok = new ACCESS3resok(objAttr, val);
      result = new ACCESS3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      resfail = new ACCESS3resfail(objAttr);
      result = new ACCESS3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }
    return(result);
  }

 /** 
 *  Implement the READLINK NFS method 
 *  
 *  NOTE: This method does not currently correctly check whether the object 
 *    is in fact a symbolic link. It should check with the file attributes. 
 **/ 
  public READLINK3res
  readlink(READLINK3args args){
    ObjId symPrefix = null;
    ObjId symlinkDataObjId = null;
    NFS_post_op_attr symlinkAttr = null;
    READLINK3resok resok = null;
    READLINK3resfail resfail = null;
    READLINK3res result = null;
    String symlinkDataStr = null;
    PRACTIFSReadLockToken symlinkToken = null;

    symlinkAttr = new NFS_post_op_attr(false, null);
    try{
      symPrefix = this.fhCache.getObjIdPrefix(args.getSymLink());
      symlinkDataObjId = new ObjId(symPrefix.getPath() + DATA_SUFFIX);
      symlinkToken = this.localInterface.acquireReadLock(symlinkDataObjId);
      symlinkDataStr = this.readSymlinkData(symlinkToken, symlinkDataObjId);
      resok = new READLINK3resok(symlinkAttr, symlinkDataStr);
      result = new READLINK3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new READLINK3resfail(symlinkAttr);
      result = new READLINK3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new READLINK3resfail(symlinkAttr);
      result = new READLINK3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*
       catch(Exception e){...
       }
      */finally{
      if(symlinkToken != null){
        assert(symPrefix != null);
        this.localInterface.releaseReadLock(symlinkToken, symPrefix);
      }
    }
    return(result);
  }

 /** 
 *  Implement the READ NFS method 
 *  
 *  NOTE: This method does not check whether the read is to a real file 
 *    or not. Need to change this in the future. 
 *  Note: This method makes a copy of the data bytes. This could be made  
 *    more efficient in the future (though it requires handling with care). 
 **/ 
  public READ3res
  read(READ3args args){
    BodyMsg bodyMsg = null;
    ObjId objIdPrefix = null;
    ObjId objDataObjId = null;
    ObjId objMetaObjId = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken objDataToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    int length = 0;
    long size = 0;
    byte[] data = null;
    NFS_fattr3 objAttr = null;
    boolean eof = false;
    READ3res result = null;
    READ3resok resok = null;
    READ3resfail resfail = null;

    postAttr = new NFS_post_op_attr(false, null);
    try{
      objIdPrefix = this.fhCache.getObjIdPrefix(args.getFile());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      objMetaToken = this.localInterface.acquireReadLock(objMetaObjId);
      objDataToken = this.localInterface.acquireReadLock(objDataObjId);
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      size = objAttr.getSize();
      if(args.getOffset() < size){
        try{
          bodyMsg = this.localInterface.read(objDataObjId,
                                             args.getOffset(),
                                             (long)args.getCount(),
                                             true,
                                             true);
        }catch(ReadOfInvalidRangeException e){
          // This should never happen because of the flags we set
          assert false : "" + e;
        }
        length = (int)bodyMsg.getLength();
        eof = (args.getOffset() + (long)length) >= size;
        data = bodyMsg.getBody().getCopyBytes();
        resok = new READ3resok(postAttr, length, eof, data);
      }else{
        // Tried to read outside byte range; allow this but specify 0 bytes
        // read (as per NFS specification)
        resok = new READ3resok(postAttr, 0, true, new byte[0]);
      }
      result = new READ3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new READ3resfail(postAttr);
      result = new READ3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(ObjNotFoundException e){
      // File has already been deleted
      resfail = new READ3resfail(postAttr);
      result = new READ3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new READ3resfail(postAttr);
      result = new READ3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*
       catch(Exception e){
       // TODO: Do something
       }*/finally{
      // Release the data lock
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseReadLock(objDataToken, objDataObjId);
      }
      // Release the metadata lock
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseReadLock(objMetaToken, objMetaObjId);
      }
    }
    return(result);
  }

 /** 
 *  Implement the WRITE NFS method 
 *  
 *  NOTE: This method does not change any attributes except the size; 
 *    need to change this 
 *  NOTE: This method does not check whether the write is to a real file 
 *    or not. Need to change this in the future. 
 *  NOTE: This method currently ignores the stability flag and makes all 
 *    writes DATA_SYNC stable 
 *  Note: This method makes a copy of the data bytes. This could be made  
 *    more efficient in the future (though it requires handling with care). 
 **/ 
  public WRITE3res
  write(WRITE3args args){
    ObjId objIdPrefix = null;
    ObjId objDataObjId = null;
    ObjId objMetaObjId = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken objDataToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    NFS_wcc_data objWCC = null;
    MultiWriteEntry[] mwArr = null;
    byte[] metaBytes = null;
    long[] writeLengths = null;
    WRITE3resok resok = null;
    WRITE3resfail resfail = null;
    NFS_fattr3 objAttr = null;
    WRITE3res result = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    try{
      objIdPrefix = this.fhCache.getObjIdPrefix(args.getFile());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);
      mwArr = new MultiWriteEntry[2];

      // Set new attributes
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      if((args.getOffset() + (long)args.getCount()) > objAttr.getSize()){
        objAttr.setSize(args.getOffset() + (long)args.getCount());
      }
      metaBytes = this.getAttributeBytes(objAttr);
      mwArr[0] = new MultiWriteEntry(objMetaObjId,
                                     0,
                                     metaBytes.length,
                                     0.0,
                                     new ImmutableBytes(metaBytes),
                                     false,
                                     false);

      // Write the actual data
      mwArr[1] = new MultiWriteEntry(objDataObjId,
                                     args.getOffset(),
                                     (long)args.getCount(),
                                     0.0,
                                     new ImmutableBytes(args.getData()),
                                     false,
                                     false);

      // Perform the write
      writeLengths = this.localInterface.writeMulti(mwArr, false);
      resok = new WRITE3resok(objWCC,
                              (int)writeLengths[1],
                              NFS_stable_how.DATA_SYNC,
                              this.writeVerf);
      result = new WRITE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new WRITE3resfail(objWCC);
      result = new WRITE3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new WRITE3resfail(objWCC);
      result = new WRITE3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the CREATE NFS method 
 *  
 *  NOTE: This method does not check whether the write is to an existing 
 *    file or not. Need to change this in the future. 
 *  NOTE: This method does not work for the EXCLUSIVE createhow switch 
 *  NOTE: This method does not change the directory attributes timestamp 
 *  NOTE: The newly created file attributes give everyone permission to 
 *    do everything 
 **/ 
  public CREATE3res
  create(CREATE3args args){
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    PRACTIFSDirectory dir = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    NFS_wcc_data objWCC = null;
    MultiWriteEntry[] mwArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    CREATE3resok resok = null;
    CREATE3resfail resfail = null;
    NFS_fattr3 objAttr = null;
    CREATE3res result = null;
    NFS_fattr3 dirAttr = null;
    long now = 0;
    ImmutableBytes dirIB = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getWhere().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      objIdPrefix = new ObjId(dirDataObjId.getPath() +
                              "/" +
                              args.getWhere().getName());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);

      mwArr = new MultiWriteEntry[3];

      // Read the directory, add the entry, and prepare to write it again
      dir = this.readDir(dirDataToken, dirDataObjId);
      try{
        dir.addObjIdPrefix(args.getWhere().getName(), objIdPrefix);
      }catch(ObjAlreadyPresentException e){
        // If we are creating this file unchecked, we ignore this exception
        if(args.getHow().getMode() != NFS_createmode3.UNCHECKED){
          throw e;
        }
      }
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      mwArr[0] = new MultiWriteEntry(dirDataObjId,
                                     0,
                                     dir.getSize(),
                                     0.0,
                                     dirIB,
                                     false,
                                     false);

      // Prepare to write the new directory attributes
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      mwArr[1] = new MultiWriteEntry(dirMetaObjId,
                                     0,
                                     dirMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(dirMetaBytes),
                                     false,
                                     false);

      // Write the new file attributes
      now = System.currentTimeMillis();
      this.fileid++;
      objAttr = new NFS_fattr3(NFS_ftype3.NF3REG,
                               0x00FFF, // Give all full permission
                               0, // No links to this file yet (nlink)
                               0, // uid
                               0, // gid
                               0, // size
                               0, // used
                               new NFS_specdata3(0, 0), // rdev
                               PRACTIFS.FSID, // fsid
                               this.fileid, // fileid
                               new NFS_nfstime3(now),  // atime
                               new NFS_nfstime3(now),  // mtime
                               new NFS_nfstime3(now)); // ctime;
      objMetaBytes = this.getAttributeBytes(objAttr);
      mwArr[2] = new MultiWriteEntry(objMetaObjId,
                                     0,
                                     objMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(objMetaBytes),
                                     false,
                                     false);

      // Perform the writes
      writeLengths = this.localInterface.writeMulti(mwArr, false);
      resok = new CREATE3resok(new NFS_post_op_fh3(false, null),
                               new NFS_post_op_attr(false, null),
                               objWCC);
      result = new CREATE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new CREATE3resfail(objWCC);
      result = new CREATE3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      resfail = new CREATE3resfail(objWCC);
      result = new CREATE3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(ObjAlreadyPresentException e){
      // Adding an existing file to a directory
      assert(args.getHow().getMode() != NFS_createmode3.UNCHECKED);
      resfail = new CREATE3resfail(objWCC);
      result = new CREATE3res(NFS_nfsstat3.NFS3ERR_EXIST, null, resfail);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      resfail = new CREATE3resfail(objWCC);
      result = new CREATE3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new CREATE3resfail(objWCC);
      result = new CREATE3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the MKDIR NFS method 
 *  
 *  NOTE: This method does not check whether the write is to an existing 
 *    file or not. Need to change this in the future. 
 *  NOTE: This method does not change the directory attributes timestamp 
 *  NOTE: The newly created directory attributes give everyone permission to 
 *    do everything 
 **/ 
  public MKDIR3res
  mkdir(MKDIR3args args){
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory obj = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    NFS_wcc_data objWCC = null;
    MultiWriteEntry[] mwArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    MKDIR3resok resok = null;
    MKDIR3resfail resfail = null;
    NFS_fattr3 objAttr = null;
    MKDIR3res result = null;
    NFS_fattr3 dirAttr = null;
    long now = 0;
    ImmutableBytes dirIB = null;
    ImmutableBytes objIB = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getWhere().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      objIdPrefix = new ObjId(dirDataObjId.getPath() +
                              "/" +
                              args.getWhere().getName());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);

      mwArr = new MultiWriteEntry[4];

      // Read the outer directory, add the entry, and prepare to write it again
      dir = this.readDir(dirDataToken, dirDataObjId);
      dir.addObjIdPrefix(args.getWhere().getName(), objIdPrefix);
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      mwArr[0] = new MultiWriteEntry(dirDataObjId,
                                     0,
                                     dir.getSize(),
                                     0.0,
                                     dirIB,
                                     false,
                                     false);

      // Prepare to write the new outer directory attributes
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      mwArr[1] = new MultiWriteEntry(dirMetaObjId,
                                     0,
                                     dirMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(dirMetaBytes),
                                     false,
                                     false);

      // Write the inner directory data
      obj = new PRACTIFSDirectory();
      objIB = new ImmutableBytes(obj.dangerousGetBytes());
      mwArr[2] = new MultiWriteEntry(objDataObjId,
                                     0,
                                     obj.getSize(),
                                     0.0,
                                     objIB,
                                     false,
                                     false);

      // Write the new inner directory attributes
      now = System.currentTimeMillis();
      this.fileid++;
      objAttr = new NFS_fattr3(NFS_ftype3.NF3DIR,
                               0x00FFF, // Give all full permission
                               0, // No links to this file yet (nlink)
                               0, // uid
                               0, // gid
                               obj.getSize(), // size
                               obj.getSize(), // used
                               new NFS_specdata3(0, 0), // rdev
                               PRACTIFS.FSID, // fsid
                               this.fileid, // fileid
                               new NFS_nfstime3(now),  // atime
                               new NFS_nfstime3(now),  // mtime
                               new NFS_nfstime3(now)); // ctime;
      objMetaBytes = this.getAttributeBytes(objAttr);
      mwArr[3] = new MultiWriteEntry(objMetaObjId,
                                     0,
                                     objMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(objMetaBytes),
                                     false,
                                     false);

      // Perform the writes
      writeLengths = this.localInterface.writeMulti(mwArr, false);
      resok = new MKDIR3resok(new NFS_post_op_fh3(false, null),
                              new NFS_post_op_attr(false, null),
                              objWCC);
      result = new MKDIR3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new MKDIR3resfail(objWCC);
      result = new MKDIR3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      resfail = new MKDIR3resfail(objWCC);
      result = new MKDIR3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(ObjAlreadyPresentException e){
      // Adding an existing file to a directory
      resfail = new MKDIR3resfail(objWCC);
      result = new MKDIR3res(NFS_nfsstat3.NFS3ERR_EXIST, null, resfail);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      resfail = new MKDIR3resfail(objWCC);
      result = new MKDIR3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new MKDIR3resfail(objWCC);
      result = new MKDIR3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the SYMLINK NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public SYMLINK3res
  symlink(SYMLINK3args args){
    NFS_wcc_data objWCC = null;
    SYMLINK3resfail resfail = null;
    SYMLINK3res result = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    resfail = new SYMLINK3resfail(objWCC);
    result = new SYMLINK3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Implement the MKNOD NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public MKNOD3res
  mknod(MKNOD3args args){
    NFS_wcc_data objWCC = null;
    MKNOD3resfail resfail = null;
    MKNOD3res result = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    resfail = new MKNOD3resfail(objWCC);
    result = new MKNOD3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Implement the REMOVE NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public REMOVE3res
  remove(REMOVE3args args){
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory obj = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    NFS_wcc_data objWCC = null;
    MultiWriteEntry[] mwArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    REMOVE3resok resok = null;
    REMOVE3resfail resfail = null;
    NFS_fattr3 objAttr = null;
    REMOVE3res result = null;
    NFS_fattr3 dirAttr = null;
    long now = 0;
    ImmutableBytes dirIB = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getObject().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      objIdPrefix = new ObjId(dirDataObjId.getPath() +
                              "/" +
                              args.getObject().getName());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);

      mwArr = new MultiWriteEntry[4];

      // Read the outer directory, remove the file, and prepare to write
      // it again
      dir = this.readDir(dirDataToken, dirDataObjId);
      dir.removeObjIdPrefix(args.getObject().getName());
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      mwArr[0] = new MultiWriteEntry(dirDataObjId,
                                     0,
                                     dir.getSize(),
                                     0.0,
                                     dirIB,
                                     false,
                                     false);

      // Prepare to write the new directory attributes
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      mwArr[1] = new MultiWriteEntry(dirMetaObjId,
                                     0,
                                     dirMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(dirMetaBytes),
                                     false,
                                     false);

      // Read the file metadata to know how many bytes to delete, and then
      // delete the file
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      mwArr[2] = new MultiWriteEntry(objDataObjId,
                                     0,
                                     objAttr.getSize(),
                                     0.0,
                                     null,
                                     false,
                                     true);

      // Delete the file metadata
      objMetaBytes = this.getAttributeBytes(objAttr);
      mwArr[3] = new MultiWriteEntry(objMetaObjId,
                                     0,
                                     objMetaBytes.length,
                                     0.0,
                                     null,
                                     false,
                                     true);

      // Perform the writes
      writeLengths = this.localInterface.writeMulti(mwArr, false);
      resok = new REMOVE3resok(objWCC);
      result = new REMOVE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new REMOVE3resfail(objWCC);
      result = new REMOVE3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      resfail = new REMOVE3resfail(objWCC);
      result = new REMOVE3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      resfail = new REMOVE3resfail(objWCC);
      result = new REMOVE3res(NFS_nfsstat3.NFS3ERR_NOENT, null, resfail);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      resfail = new REMOVE3resfail(objWCC);
      result = new REMOVE3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new REMOVE3resfail(objWCC);
      result = new REMOVE3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the RMDIR NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public RMDIR3res
  rmdir(RMDIR3args args){
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory obj = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    NFS_wcc_data objWCC = null;
    MultiWriteEntry[] mwArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    RMDIR3resok resok = null;
    RMDIR3resfail resfail = null;
    NFS_fattr3 objAttr = null;
    RMDIR3res result = null;
    NFS_fattr3 dirAttr = null;
    ImmutableBytes dirIB = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getObject().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      objIdPrefix = new ObjId(dirDataObjId.getPath() +
                              "/" +
                              args.getObject().getName());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);

      mwArr = new MultiWriteEntry[4];

      // Read the outer directory, remove the entry, and prepare to write
      // it again
      dir = this.readDir(dirDataToken, dirDataObjId);
      dir.removeObjIdPrefix(args.getObject().getName());
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      mwArr[0] = new MultiWriteEntry(dirDataObjId,
                                     0,
                                     dir.getSize(),
                                     0.0,
                                     dirIB,
                                     false,
                                     false);

      // Prepare to write the new outer directory attributes
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      mwArr[1] = new MultiWriteEntry(dirMetaObjId,
                                     0,
                                     dirMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(dirMetaBytes),
                                     false,
                                     false);

      // Read the inner directory metadata to know how many bytes to
      // delete, and then delete the directory
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      mwArr[2] = new MultiWriteEntry(objDataObjId,
                                     0,
                                     objAttr.getSize(),
                                     0.0,
                                     null,
                                     false,
                                     true);

      // Delete the file metadata
      objMetaBytes = this.getAttributeBytes(objAttr);
      mwArr[3] = new MultiWriteEntry(objMetaObjId,
                                     0,
                                     objMetaBytes.length,
                                     0.0,
                                     null,
                                     false,
                                     true);

      // Perform the writes
      writeLengths = this.localInterface.writeMulti(mwArr, false);
      resok = new RMDIR3resok(objWCC);
      result = new RMDIR3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new RMDIR3resfail(objWCC);
      result = new RMDIR3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      resfail = new RMDIR3resfail(objWCC);
      result = new RMDIR3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      resfail = new RMDIR3resfail(objWCC);
      result = new RMDIR3res(NFS_nfsstat3.NFS3ERR_NOENT, null, resfail);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      resfail = new RMDIR3resfail(objWCC);
      result = new RMDIR3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new RMDIR3resfail(objWCC);
      result = new RMDIR3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the RENAME NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public RENAME3res
  rename(RENAME3args args){
    /*
      ObjId objMetaObjId = null;
      ObjId objDataObjId = null;
      NFS_post_op_attr postAttr = null;
      byte[] objMetaBytes = null;
      NFS_fattr3 objAttr = null;
    */
    NFS_wcc_data fromWCC = null;
    NFS_wcc_data toWCC = null;
    ObjId fromDirIdPrefix = null;
    ObjId fromDirDataObjId = null;
    ObjId fromDirMetaObjId = null;
    ObjId toDirIdPrefix = null;
    ObjId toDirDataObjId = null;
    ObjId toDirMetaObjId = null;
    PRACTIFSReadLockToken fromDirDataToken = null;
    PRACTIFSReadLockToken fromDirMetaToken = null;
    PRACTIFSReadLockToken toDirMetaToken = null;
    PRACTIFSReadLockToken toDirDataToken = null;
    PRACTIFSDirectory fromDir = null;
    PRACTIFSDirectory toDir = null;
    ImmutableBytes fromDirIB = null;
    ImmutableBytes toDirIB = null;
    NFS_fattr3 fromDirAttr = null;
    NFS_fattr3 toDirAttr = null;
    byte[] fromDirMetaBytes = null;
    byte[] toDirMetaBytes = null;
    ObjId objIdPrefix = null;
    MultiWriteEntry[] mwArr = null;
    long[] writeLengths = null;
    RENAME3resok resok = null;
    RENAME3resfail resfail = null;
    RENAME3res result = null;

    fromWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                               new NFS_post_op_attr(false, null));
    toWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                             new NFS_post_op_attr(false, null));
    try{
      fromDirIdPrefix = this.fhCache.getObjIdPrefix(args.getFrom().getDir());
      fromDirMetaObjId = new ObjId(fromDirIdPrefix.getPath() + META_SUFFIX);
      fromDirDataObjId = new ObjId(fromDirIdPrefix.getPath() + DATA_SUFFIX);
      toDirIdPrefix = this.fhCache.getObjIdPrefix(args.getTo().getDir());
      toDirMetaObjId = new ObjId(toDirIdPrefix.getPath() + META_SUFFIX);
      toDirDataObjId = new ObjId(toDirIdPrefix.getPath() + DATA_SUFFIX);

      fromDirMetaToken =
        this.localInterface.acquireWriteLock(fromDirMetaObjId);
      fromDirDataToken =
        this.localInterface.acquireWriteLock(fromDirDataObjId);
      toDirMetaToken = this.localInterface.acquireWriteLock(toDirMetaObjId);
      toDirDataToken = this.localInterface.acquireWriteLock(toDirDataObjId);

      mwArr = new MultiWriteEntry[4];

      // Read the source directory, remove the entry, and prepare to write
      // it again
      fromDir = this.readDir(fromDirDataToken, fromDirDataObjId);
      objIdPrefix = fromDir.getObjIdPrefix(args.getFrom().getName());
      fromDir.removeObjIdPrefix(args.getFrom().getName());
      fromDirIB = new ImmutableBytes(fromDir.dangerousGetBytes());
      mwArr[0] = new MultiWriteEntry(fromDirDataObjId,
                                     0,
                                     fromDir.getSize(),
                                     0.0,
                                     fromDirIB,
                                     false,
                                     false);

      // Prepare to write the source directory attributes
      fromDirAttr = this.readAttributes(fromDirMetaToken, fromDirMetaObjId);
      fromDirAttr.setSize(fromDir.getSize());
      fromDirMetaBytes = this.getAttributeBytes(fromDirAttr);
      mwArr[1] = new MultiWriteEntry(fromDirMetaObjId,
                                     0,
                                     fromDirMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(fromDirMetaBytes),
                                     false,
                                     false);

      // Read the target directory, add the entry, and prepare to write
      // it again
      toDir = this.readDir(toDirDataToken, toDirDataObjId);
      toDir.addObjIdPrefix(args.getTo().getName(), objIdPrefix);
      toDirIB = new ImmutableBytes(toDir.dangerousGetBytes());
      mwArr[2] = new MultiWriteEntry(toDirDataObjId,
                                     0,
                                     toDir.getSize(),
                                     0.0,
                                     toDirIB,
                                     false,
                                     false);

      // Prepare to write the target directory attributes
      toDirAttr = this.readAttributes(toDirMetaToken, toDirMetaObjId);
      toDirAttr.setSize(toDir.getSize());
      toDirMetaBytes = this.getAttributeBytes(toDirAttr);
      mwArr[3] = new MultiWriteEntry(toDirMetaObjId,
                                     0,
                                     toDirMetaBytes.length,
                                     0.0,
                                     null,
                                     false,
                                     false);

      // Perform the writes
      writeLengths = this.localInterface.writeMulti(mwArr, false);
      resok = new RENAME3resok(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3_OK, resok, null);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      resfail = new RENAME3resfail(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      resfail = new RENAME3resfail(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      resfail = new RENAME3resfail(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3ERR_NOENT, null, resfail);
    }catch(ObjAlreadyPresentException e){
      // Moving a file to a location where it already exists
      resfail = new RENAME3resfail(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3ERR_EXIST, null, resfail);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      resfail = new RENAME3resfail(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3ERR_STALE, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      resfail = new RENAME3resfail(fromWCC, toWCC);
      result = new RENAME3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((toDirDataObjId != null) && (toDirDataToken != null)){
        this.localInterface.releaseWriteLock(toDirDataToken, toDirDataObjId);
      }
      if((toDirMetaObjId != null) && (toDirMetaToken != null)){
        this.localInterface.releaseWriteLock(toDirMetaToken, toDirMetaObjId);
      }
      if((fromDirDataObjId != null) && (fromDirDataToken != null)){
        this.localInterface.releaseWriteLock(fromDirDataToken,
                                             fromDirDataObjId);
      }
      if((fromDirMetaObjId != null) && (fromDirMetaToken != null)){
        this.localInterface.releaseWriteLock(fromDirMetaToken,
                                             fromDirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the LINK NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public LINK3res
  link(LINK3args args){
    NFS_wcc_data objWCC = null;
    LINK3resfail resfail = null;
    LINK3res result = null;

    objWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                              new NFS_post_op_attr(false, null));
    resfail = new LINK3resfail(new NFS_post_op_attr(false, null), objWCC);
    result = new LINK3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Implement the READDIR NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 *  NOTE: We currently ignore the cookieverf token 
 *  NOTE: We currently ignore the usage of cookies 
 **/ 
  public READDIR3res
  readdir(READDIR3args args){
    READDIR3resfail resfail = null;
    READDIR3resok resok = null;
    READDIR3res result = null;
    ObjId dirObjIdPrefix = null;
    ObjId dirMetaObjId = null;
    ObjId dirDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory.DirIterator it = null;
    NFS_entry3 entry = null;
    String name = null;
    NFS_dirlist3 dirlist = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    byte[] cookieBytes = null;

    try{
      dirObjIdPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirObjIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirObjIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      dir = this.readDir(dirDataToken, dirDataObjId);
      it = dir.getIterator();
      entry = null;
      while(it.hasNext()){
        name = it.getNext();
        this.fileid++;
        entry = new NFS_entry3(this.fileid, name, 0, entry);
      }
      dirlist = new NFS_dirlist3(entry, true);
      cookieBytes = new byte[NFS_cookieverf3.NFS3_COOKIEVERFSIZE];
      resok = new READDIR3resok(new NFS_post_op_attr(false, null),
                                new NFS_cookieverf3(cookieBytes),
                                dirlist);
    }catch(BadFileHandleException e){
      // Got a bad file handle for the directory
      resfail = new READDIR3resfail(new NFS_post_op_attr(false, null));
      result = new READDIR3res(NFS_nfsstat3.NFS3ERR_BADHANDLE, null, resfail);
    }catch(IsNotDirectoryException e){
      // Got a file handle that wasn't for a directory
      resfail = new READDIR3resfail(new NFS_post_op_attr(false, null));
      result = new READDIR3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IOException
      resfail = new READDIR3resfail(new NFS_post_op_attr(false, null));
      result = new READDIR3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }finally{
      /*
      if((toDirDataObjId != null) && (toDirDataToken != null)){
        this.localInterface.releaseWriteLock(toDirDataToken, toDirDataObjId);
      }
      */
    }
    return(result);
  }

 /** 
 *  Implement the READDIRPLUS NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 *  NOTE: We currently ignore the cookieverf token 
 *  NOTE: We currently ignore the usage of cookies 
 *  NOTE: We completely ignore the maxcount and dircount parameters 
 **/ 
  public READDIRPLUS3res
  readdirplus(READDIRPLUS3args args){
    READDIRPLUS3resfail resfail = null;
    READDIRPLUS3resok resok = null;
    READDIRPLUS3res result = null;
    ObjId dirObjIdPrefix = null;
    ObjId dirMetaObjId = null;
    ObjId dirDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory.DirIterator it = null;
    NFS_entryplus3 entry = null;
    String name = null;
    NFS_dirlistplus3 dirlist = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    byte[] cookieBytes = null;

    try{
      dirObjIdPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirObjIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirObjIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      dir = this.readDir(dirDataToken, dirDataObjId);
      it = dir.getIterator();
      entry = null;
      while(it.hasNext()){
        name = it.getNext();
        this.fileid++;
        entry = new NFS_entryplus3(this.fileid,
                                   name,
                                   0,
                                   new NFS_post_op_attr(false, null),
                                   new NFS_post_op_fh3(false, null),
                                   entry);
      }
      dirlist = new NFS_dirlistplus3(entry, true);
      cookieBytes = new byte[NFS_cookieverf3.NFS3_COOKIEVERFSIZE];
      resok = new READDIRPLUS3resok(new NFS_post_op_attr(false, null),
                                    new NFS_cookieverf3(cookieBytes),
                                    dirlist);
    }catch(BadFileHandleException e){
      // Got a bad file handle for the directory
      resfail = new READDIRPLUS3resfail(new NFS_post_op_attr(false, null));
      result = new READDIRPLUS3res(NFS_nfsstat3.NFS3ERR_BADHANDLE,
                                   null,
                                   resfail);
    }catch(IsNotDirectoryException e){
      // Got a file handle that wasn't for a directory
      resfail = new READDIRPLUS3resfail(new NFS_post_op_attr(false, null));
      result = new READDIRPLUS3res(NFS_nfsstat3.NFS3ERR_NOTDIR, null, resfail);
    }catch(IOException e){
      // Don't know what caused this; generic IOException
      resfail = new READDIRPLUS3resfail(new NFS_post_op_attr(false, null));
      result = new READDIRPLUS3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    }finally{
      /*
      if((toDirDataObjId != null) && (toDirDataToken != null)){
        this.localInterface.releaseWriteLock(toDirDataToken, toDirDataObjId);
      }
      */
    }
    return(result);
  }

  /*
  //-------------------------------------------------------------------------
  // Implement the READDIRPLUS NFS method
  //
  // NOTE: This method isn't currently implemented
  //-------------------------------------------------------------------------
  public READDIRPLUS3res
  readdirplus(READDIRPLUS3args args){
    READDIRPLUS3resfail resfail = null;
    READDIRPLUS3res result = null;

    resfail = new READDIRPLUS3resfail(new NFS_post_op_attr(false, null));
    result = new READDIRPLUS3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }
  */

 /** 
 *  Implement the FSSTAT NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public FSSTAT3res
  fsstat(FSSTAT3args args){
    FSSTAT3resfail resfail = null;
    FSSTAT3res result = null;

    resfail = new FSSTAT3resfail(new NFS_post_op_attr(false, null));
    result = new FSSTAT3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Implement the FSINFO NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public FSINFO3res
  fsinfo(FSINFO3args args){
    FSINFO3resfail resfail = null;
    FSINFO3res result = null;

    resfail = new FSINFO3resfail(new NFS_post_op_attr(false, null));
    result = new FSINFO3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Implement the PATHCONF NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public PATHCONF3res
  pathconf(PATHCONF3args args){
    PATHCONF3resfail resfail = null;
    PATHCONF3res result = null;

    resfail = new PATHCONF3resfail(new NFS_post_op_attr(false, null));
    result = new PATHCONF3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Implement the COMMIT NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public COMMIT3res
  commit(COMMIT3args args){
    COMMIT3resfail resfail = null;
    COMMIT3res result = null;
    NFS_wcc_data fileWCC = null;

    fileWCC = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                               new NFS_post_op_attr(false, null));
    resfail = new COMMIT3resfail(fileWCC);
    result = new COMMIT3res(NFS_nfsstat3.NFS3ERR_NOTSUPP, null, resfail);

    return(result);
  }

 /** 
 *  Return the attributes for this object 
 **/ 
  private NFS_fattr3
  readAttributes(PRACTIFSReadLockToken token, ObjId metaDataObjId)
    throws IOException{
    NFS_fattr3 fattr = null;
    PRACTIFileInputStream pfis = null;

    assert(token.isAcquired());
    try{
      pfis = new PRACTIFileInputStream(this.localInterface,
                                       metaDataObjId,
                                       PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      fattr = new NFS_fattr3(pfis);
    }finally{
      if(pfis != null){
        // Close the stream
        pfis.close();
      }
    }
    return(fattr);
  }

 /** 
 *  Write the attributes for this object 
 **/ 
  private void
  writeAttributes(PRACTIFSWriteLockToken token,
                  ObjId metaDataObjId,
                  NFS_sattr3 newAttributes) throws IOException{
    NFS_fattr3 fattr = null;
    NFS_set_mode3 mode = null;
    NFS_set_uid3 uid = null;
    NFS_set_gid3 gid = null;
    NFS_set_size3 size = null;
    NFS_set_atime atime = null;
    NFS_set_mtime mtime = null;
    PRACTIFileOutputStream pfos = null;

    assert(token.isAcquired());
    try{
      fattr = this.readAttributes(token, metaDataObjId);
      mode = newAttributes.getMode();
      uid = newAttributes.getUID();
      gid = newAttributes.getGID();
      size = newAttributes.getSize();
      atime = newAttributes.getATime();
      mtime = newAttributes.getMTime();

      if(mode.getSetIt()){
        fattr.setMode(mode.getMode());
      }
      if(uid.getSetIt()){
        fattr.setUID(uid.getUID());
      }
      if(gid.getSetIt()){
        fattr.setGID(gid.getGID());
      }
      if(size.getSetIt()){
        fattr.setSize(size.getSize());
      }
      if(atime.getSetIt() != NFS_time_how.DONT_CHANGE){
        fattr.setATime(atime.getATime());
      }
      if(mtime.getSetIt() != NFS_time_how.DONT_CHANGE){
        fattr.setMTime(mtime.getMTime());
      }
      pfos = new PRACTIFileOutputStream(this.localInterface, metaDataObjId);
      fattr.toOutputStream(pfos);
    }finally{
      if(pfos != null){
        pfos.close();
      }
    }
  }

 /** 
 *  Make a PRACTIFSDirectory for the root directory 
 **/ 
  private void
  makeRootDirectory(){
    ObjId dirIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    PRACTIFSDirectory rootDir = null;
    NFS_post_op_attr postAttr = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    MultiWriteEntry[] mwArr = null;
    byte[] dirMetaBytes = null;
    long[] writeLengths = null;
    NFS_fattr3 dirAttr = null;
    long now = 0;
    ImmutableBytes dirIB = null;

    try{
      rootDir = new PRACTIFSDirectory();
      dirIdPrefix = this.fhCache.getObjIdPrefix(NFS_nfs_fh3.makeAllZeroHandle());
      assert(dirIdPrefix.equals(new ObjId("")));
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);

      mwArr = new MultiWriteEntry[2];

      // Prepare to write the root directory data
      dirIB = new ImmutableBytes(rootDir.dangerousGetBytes());
      mwArr[0] = new MultiWriteEntry(dirDataObjId,
                                     0,
                                     rootDir.getSize(),
                                     0.0,
                                     dirIB,
                                     false,
                                     false);

      // Write the new inner directory attributes
      now = System.currentTimeMillis();
      this.fileid++;
      dirAttr = new NFS_fattr3(NFS_ftype3.NF3DIR,
                               0x00FFF, // Give all full permission
                               0, // No links to this file yet (nlink)
                               0, // uid
                               0, // gid
                               dirIB.getLength(), // size
                               dirIB.getLength(), // used
                               new NFS_specdata3(0, 0), // rdev
                               PRACTIFS.FSID, // fsid
                               this.fileid, // fileid
                               new NFS_nfstime3(now),  // atime
                               new NFS_nfstime3(now),  // mtime
                               new NFS_nfstime3(now)); // ctime;
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      mwArr[1] = new MultiWriteEntry(dirMetaObjId,
                                     0,
                                     dirMetaBytes.length,
                                     0.0,
                                     new ImmutableBytes(dirMetaBytes),
                                     false,
                                     false);

      // Perform the writes
      writeLengths = this.localInterface.writeMulti(mwArr, false);
    }catch(BadFileHandleException e){
      // Did not find all-zero-file-handle in cache; this is an error!
      assert(false);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      assert(false);
      }/*catch(Exception e){
      // TODO: Something
      }*/finally{
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
  }

 /** 
 *  Return the attributes for this object 
 **/ 
  private PRACTIFSDirectory
  readDir(PRACTIFSReadLockToken token, ObjId dirDataObjId)
    throws IOException, IsNotDirectoryException{
    PRACTIFSDirectory dir = null;
    PRACTIFileInputStream pfis = null;

    assert(token.isAcquired());
    try{
      pfis = new PRACTIFileInputStream(this.localInterface,
                                       dirDataObjId,
                                       PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      dir = new PRACTIFSDirectory(pfis);
    }finally{
      if(pfis != null){
        // Close the stream
        pfis.close();
      }
    }
    return(dir);
  }

 /** 
 *  Read symbolic link data 
 **/ 
  private String
  readSymlinkData(PRACTIFSReadLockToken token, ObjId symlinkDataObjId)
    throws IOException{
    SymlinkData symlinkData = null;
    PRACTIFileInputStream pfis = null;
    String symlinkDataStr = null;

    assert(token.isAcquired());
    try{
      pfis = new PRACTIFileInputStream(this.localInterface,
                                       symlinkDataObjId,
                                       PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      symlinkData = new SymlinkData(pfis);
      symlinkDataStr = symlinkData.getData();
    }finally{
      if(pfis != null){
        // Close the stream
        pfis.close();
      }
    }
    return(symlinkDataStr);
  }

 /** 
 *  Serialize a given set of attributes into an array 
 **/ 
  private byte[]
  getAttributeBytes(NFS_fattr3 attr){
    ByteArrayOutputStream baos = null;
    byte[] barr = null;

    try{
      baos = new ByteArrayOutputStream();
      attr.toOutputStream(baos);
      baos.flush();
      barr = baos.toByteArray();
    }catch(IOException e){
      assert false : ("" + e);
    }finally{
      if(baos != null){
        try{
          baos.close();
        }catch(IOException e){
          // Do nothing; not much we can do anyway
        }
      }
    }
    return(barr);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PRACTIFS.java...");
    PRACTIFS.testSimple();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    PRACTIFSLocalInterface li = null;
    PRACTIFS pfs = null;
    NFS_nfs_fh3 handle = null;

    li = new FakePRACTIFSLocalInterface();
    pfs = new PRACTIFS(li);
    //System.out.println("" + li);
    PRACTIFS.testCreate(li, pfs, "a");
    handle = PRACTIFS.testLookup(li, pfs, "a");
    PRACTIFS.testGetAttributes(li, pfs, handle);
    PRACTIFS.testWrite(li, pfs, handle);
    PRACTIFS.testGetAttributes(li, pfs, handle);
    PRACTIFS.testRead(li, pfs, handle);
  }

 /** 
 *  Test the create method 
 **/ 
  private static void
  testCreate(PRACTIFSLocalInterface li, PRACTIFS pfs, String name){
    CREATE3args args = null;
    CREATE3res result = null;
    NFS_createhow3 how = null;
    NFS_diropargs3 dirOpArgs = null;
    GETATTR3 getAttrArgs = null;

    dirOpArgs = new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(), name);
    how = new NFS_createhow3(NFS_createmode3.EXCLUSIVE,
                             null,
                             new NFS_createverf3(new byte[8]));
    args = new CREATE3args(dirOpArgs, how);
    result = pfs.create(args);

    // Make sure creation succeeded
    assert(result.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(result.getResOk() != null);

    // Check to see what our fake local interface has
    //System.out.println("" + li);
  }

 /** 
 *  Test the lookup method 
 **/ 
  private static NFS_nfs_fh3
  testLookup(PRACTIFSLocalInterface li, PRACTIFS pfs, String name){
    LOOKUP3args args = null;
    LOOKUP3res result = null;
    LOOKUP3resok resok = null;
    NFS_nfs_fh3 handle = null;

    args = new LOOKUP3args(new NFS_diropargs3(NFS_nfs_fh3.makeAllZeroHandle(),
                                              name));
    result = pfs.lookup(args);
    assert(result.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(result.getResOk() != null);

    resok = result.getResOk();
    handle = resok.getObject();

    return(handle);
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  private static void
  testGetAttributes(PRACTIFSLocalInterface li,
                    PRACTIFS pfs,
                    NFS_nfs_fh3 handle){
    GETATTR3args args = null;
    GETATTR3res result = null;
    GETATTR3resok resok = null;
    NFS_fattr3 attr = null;

    args = new GETATTR3args(handle);
    result = pfs.getAttributes(args);
    assert(result.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(result.getResOk() != null);

    resok = result.getResOk();
    attr = resok.getObjAttributes();
    //System.out.println("Size = " + attr.getSize());
  }

 /** 
 *  Test the write method 
 **/ 
  private static void
  testWrite(PRACTIFSLocalInterface li,
            PRACTIFS pfs,
            NFS_nfs_fh3 handle){
    WRITE3args args = null;
    WRITE3res result = null;
    WRITE3resok resok = null;
    byte[] data = null;

    data = new byte[100];
    data[0] = 7;
    data[52] = 3;
    args = new WRITE3args(handle,
                          0,
                          100,
                          NFS_stable_how.DATA_SYNC,
                          data);
    result = pfs.write(args);
    assert(result.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(result.getResOk() != null);

    // See what our fake local interface has
    //System.out.println("" + li);
  }


 /** 
 *  Test the read method 
 **/ 
  private static void
  testRead(PRACTIFSLocalInterface li,
           PRACTIFS pfs,
           NFS_nfs_fh3 handle){
    READ3args args = null;
    READ3res result = null;
    READ3resok resok = null;
    byte[] data = null;

    // Read the first 50 bytes
    args = new READ3args(handle, 0, 50);
    result = pfs.read(args);
    assert(result.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(result.getResOk() != null);

    resok = result.getResOk();
    assert(resok.getCount() == 50);
    data = resok.getData();
    assert(data[0] == 7);

    // Read the next 50 bytes
    args = new READ3args(handle, 50, 50);
    result = pfs.read(args);
    assert(result.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(result.getResOk() != null);

    resok = result.getResOk();
    assert(resok.getCount() == 50);
    data = resok.getData();
    assert(data[2] == 3);
  }
}
