 /** 
 *  Implement an NFS directory 
 *  
 *  Note: The current routines read/write the entire directory. A more 
 *  efficient implementation could read/write modified chunks only. 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class PRACTIFSDirectory{

 /** 
 *  Constants 
 **/ 
  public static byte MAGIC_NUM = 5;

 /** 
 *  Data members 
 **/ 
  private HashMap handleTable;
  private byte[] serializedData;

 /** 
 *  Constructor 
 **/ 
  public
  PRACTIFSDirectory(){
    try{
      this.handleTable = new HashMap();
      this.serializedData = this.getTableBytes();
    }catch(IOException e){
      assert false : ("" + e);
    }
  }

 /** 
 *  Constructor 
 *  
 *  Read this object from an InputStream 
 **/ 
  public
  PRACTIFSDirectory(InputStream is)
    throws IOException, IsNotDirectoryException{
    DataInputStream dis = null;
    ObjectInputStream ois = null;
    byte magicNum = (byte)0;
    ByteArrayInputStream bais = null;

    try{
      dis = new DataInputStream(is);
      magicNum = dis.readByte();
      if(magicNum != PRACTIFSDirectory.MAGIC_NUM){
        throw new IsNotDirectoryException("Not a directory");
      }
      ois = new ObjectInputStream(dis);
      this.handleTable = (HashMap)ois.readObject();
      this.serializedData = this.getTableBytes();
    }catch(ClassNotFoundException e){
      assert false : ("" + e);
    }
  }

 /** 
 *  Write this object to an OutputStream 
 *  
 *  Note: Writing a directory to the output stream isn't enough. We need 
 *  to change the size attribute for the directory metadata as well. 
 *  
 *  Note: We write data to a byte array first and then send the byte array 
 *  to the output stream both (1) for efficiency and (2) to get the size of 
 *  data written. 
 **/ 
  public void
  toOutputStream(OutputStream os) throws IOException{
    os.write(this.serializedData);
  }

 /** 
 *  Retrieve a directory entry 
 **/ 
  public ObjId
  getObjIdPrefix(String fileName) throws ObjNotInDirectoryException{
    ObjId objId = null;

    objId = (ObjId)this.handleTable.get(fileName);
    if(objId == null){
      throw new ObjNotInDirectoryException(fileName + " not found");
    }
    return(objId);
  }

 /** 
 *  Add an entry 
 **/ 
  public void
  addObjIdPrefix(String fileName, ObjId objId)
    throws ObjAlreadyPresentException{

    if(this.handleTable.containsKey(fileName)){
      throw new ObjAlreadyPresentException(fileName + " already present");
    }else{
      this.handleTable.put(fileName, objId);
      try{
        this.serializedData = this.getTableBytes();
      }catch(IOException e){
        assert false : ("" + e);
      }
    }
  }

 /** 
 *  Remove an entry 
 **/ 
  public void
  removeObjIdPrefix(String fileName) throws ObjNotInDirectoryException{
    if(!this.handleTable.containsKey(fileName)){
      throw new ObjNotInDirectoryException(fileName + " not found");
    }else{
      this.handleTable.remove(fileName);
      try{
        this.serializedData = this.getTableBytes();
      }catch(IOException e){
        assert false : ("" + e);
      }
    }
  }

 /** 
 *  Return the amount of space needed to serialized this directory 
 **/ 
  public long
  getSize(){
    return(this.serializedData.length);
  }

 /** 
 *  Get a pointer to the serialized data. 
 *  
 *  Note that the caller MUST not change the data and must throw away the 
 *  pointer as soon as possible 
 **/ 
  public byte[]
  dangerousGetBytes(){
    return(this.serializedData);
  }

 /** 
 *  Return an iterator 
 **/ 
  public PRACTIFSDirectory.DirIterator
  getIterator(){
    return(new PRACTIFSDirectory.DirIterator(this.handleTable));
  }

 /** 
 *  Return true if this object equals "obj" 
 **/ 
  public boolean
  equals(Object obj){
    PRACTIFSDirectory dir = null;
    boolean eq = false;
    Map.Entry entry = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof PRACTIFSDirectory){
      dir = (PRACTIFSDirectory)obj;
      // Iterate through all elements to make sure that they match
      if(dir.handleTable.size() == this.handleTable.size()){
        eq = true;
        for(Iterator i = this.handleTable.entrySet().iterator();
            (i.hasNext() && eq);){
          entry = (Map.Entry)i.next();
          eq = (dir.handleTable.containsKey(entry.getKey()) &&
                entry.getValue().equals(dir.handleTable.get(entry.getKey())));
        }
      }else{
        // Different number of elements in both tables
        eq = false;
      }
    }else{
      // Different object types
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;
    Map.Entry entry = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    for(Iterator i = this.handleTable.entrySet().iterator(); i.hasNext();){
      entry = (Map.Entry)i.next();
      code = code ^ entry.getKey().hashCode();
      code = code ^ entry.getValue().hashCode();
    }
    return(code);
  }

 /** 
 *  Fill a byte array with this object serialized 
 **/ 
  private byte[]
  getTableBytes() throws IOException{
    byte[] dataBytes = null;
    ByteArrayOutputStream baos = null;
    ObjectOutputStream oos = null;
    DataOutputStream dos = null;

    try{
      baos = new ByteArrayOutputStream();
      dos = new DataOutputStream(baos);
      dos.writeByte(PRACTIFSDirectory.MAGIC_NUM);
      oos = new ObjectOutputStream(dos);
      oos.writeObject(this.handleTable);
      oos.flush();
      dos.flush();
      baos.flush();
      dataBytes = baos.toByteArray();
    }finally{
      if(oos != null){
        try{
          oos.close();
        }catch(IOException e){
          // Do nothing; not much that we can do
        }
      }
      if(dos != null){
        try{
          dos.close();
        }catch(IOException e){
          // Do nothing; not much that we can do
        }
      }
      if(baos != null){
        try{
          baos.close();
        }catch(IOException e){
          // Do nothing; not much that we can do
        }
      }
    }
    return(dataBytes);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PRACTIFSDirectory.java...");
    PRACTIFSDirectory.testSimple();
    PRACTIFSDirectory.testSerialize();
    PRACTIFSDirectory.testIterator();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    PRACTIFSDirectory dir1 = null;
    PRACTIFSDirectory dir2 = null;
    PRACTIFSDirectory dir3 = null;
    PRACTIFSDirectory dir4 = null;

    // Test 1
    dir1 = new PRACTIFSDirectory();
    dir2 = new PRACTIFSDirectory();
    assert(dir1.equals(dir2));
    assert(dir2.equals(dir1));

    // Test 2
    try{
      dir1 = new PRACTIFSDirectory();
      dir1.addObjIdPrefix("a", new ObjId("a"));
      dir1.addObjIdPrefix("b", new ObjId("b"));
      assert(dir1.equals(dir1));
      assert(dir1.getObjIdPrefix("a").equals(new ObjId("a")));
      assert(dir1.getObjIdPrefix("b").equals(new ObjId("b")));
      assert(dir1.dangerousGetBytes() == dir1.serializedData);
    }catch(ObjNotInDirectoryException e){
      assert false : ("" + e);
    }catch(ObjAlreadyPresentException e){
      assert false : ("" + e);
    }

    // Test 3
    try{
      dir2 = new PRACTIFSDirectory();
      dir2.addObjIdPrefix("a", new ObjId("a1"));
      dir2.addObjIdPrefix("b", new ObjId("b2"));
      assert(dir2.equals(dir2));
      assert(dir2.getObjIdPrefix("a").equals(new ObjId("a1")));
      assert(dir2.getObjIdPrefix("b").equals(new ObjId("b2")));
      assert(dir2.dangerousGetBytes() == dir2.serializedData);
      assert(!dir1.equals(dir2));
      assert(!dir2.equals(dir1));
      assert(dir1.hashCode() != dir2.hashCode());
    }catch(ObjNotInDirectoryException e){
      assert false : ("" + e);
    }catch(ObjAlreadyPresentException e){
      assert false : ("" + e);
    }

    // Test 4
    try{
      dir3 = new PRACTIFSDirectory();
      dir3.addObjIdPrefix("a", new ObjId("a"));
      dir3.addObjIdPrefix("b", new ObjId("b"));
      assert(dir3.getObjIdPrefix("a").equals(new ObjId("a")));
      assert(dir3.getObjIdPrefix("b").equals(new ObjId("b")));
      assert(dir3.dangerousGetBytes() == dir3.serializedData);
      assert(dir3.equals(dir3));
      assert(dir3.equals(dir1));
      assert(dir1.equals(dir3));
      assert(!dir3.equals(dir2));
      assert(!dir2.equals(dir3));
      assert(dir3.hashCode() != dir2.hashCode());
      assert(dir3.hashCode() == dir1.hashCode());
    }catch(ObjNotInDirectoryException e){
      assert false : ("" + e);
    }catch(ObjAlreadyPresentException e){
      assert false : ("" + e);
    }

    // Test 5
    try{
      dir4 = new PRACTIFSDirectory();
      dir4.addObjIdPrefix("a", new ObjId("a"));
      dir4.addObjIdPrefix("b", new ObjId("b"));
      assert(dir4.getObjIdPrefix("a").equals(new ObjId("a")));
      assert(dir4.getObjIdPrefix("b").equals(new ObjId("b")));
      dir4.removeObjIdPrefix("a");
      try{
        dir4.getObjIdPrefix("a");
        assert false;
      }catch(ObjNotInDirectoryException e){
        // This was expected
      }
      assert(dir4.getObjIdPrefix("b").equals(new ObjId("b")));
    }catch(ObjNotInDirectoryException e2){
      assert false : ("" + e2);
    }catch(ObjAlreadyPresentException e2){
      assert false : ("" + e2);
    }
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSerialize(){
    ByteArrayInputStream bais = null;
    ByteArrayOutputStream baos = null;
    PRACTIFSDirectory dir1 = null;
    PRACTIFSDirectory dir2 = null;
    byte[] bArray = null;

    // Test 1
    try{
      baos = new ByteArrayOutputStream();
      dir1 = new PRACTIFSDirectory();
      dir1.toOutputStream(baos);
      baos.flush();
      bArray = baos.toByteArray();
      bais = new ByteArrayInputStream(bArray);
      dir2 = new PRACTIFSDirectory(bais);
      assert(dir1.equals(dir2));
      assert(dir2.equals(dir1));
    }catch(IOException e){
      assert false : ("" + e);
    }catch(IsNotDirectoryException e){
      assert false : ("" + e);
    }

    // Test 2
    try{
      baos = new ByteArrayOutputStream();
      dir1 = new PRACTIFSDirectory();
      dir1.addObjIdPrefix("a", new ObjId("a1"));
      dir1.addObjIdPrefix("b", new ObjId("b1"));
      dir1.toOutputStream(baos);
      baos.flush();
      bArray = baos.toByteArray();
      bais = new ByteArrayInputStream(bArray);
      dir2 = new PRACTIFSDirectory(bais);
      assert(dir1.equals(dir2));
      assert(dir2.equals(dir1));
      assert(dir2.getObjIdPrefix("a").equals(new ObjId("a1")));
      assert(dir2.getObjIdPrefix("b").equals(new ObjId("b1")));
    }catch(IOException e){
      assert false : ("" + e);
    }catch(ObjAlreadyPresentException e){
      assert false : ("" + e);
    }catch(ObjNotInDirectoryException e){
      assert false : ("" + e);
    }catch(IsNotDirectoryException e){
      assert false : ("" + e);
    }
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testIterator(){
    PRACTIFSDirectory dir1 = null;
    PRACTIFSDirectory.DirIterator iter = null;
    String name1 = null;
    String name2 = null;

    // Test 1
    try{
      dir1 = new PRACTIFSDirectory();
      dir1.addObjIdPrefix("a", new ObjId("a1"));
      dir1.addObjIdPrefix("b", new ObjId("b1"));
      assert(dir1.getObjIdPrefix("a").equals(new ObjId("a1")));
      assert(dir1.getObjIdPrefix("b").equals(new ObjId("b1")));
      iter = dir1.getIterator();
      assert(iter.hasNext());
      name1 = iter.getNext();
      assert(iter.hasNext());
      name2 = iter.getNext();
      assert(!iter.hasNext());
      assert((name1.equals("a") && name2.equals("b")) ||
             (name1.equals("b") && name2.equals("a")));
      assert((dir1.getObjIdPrefix(name1).equals(new ObjId("a1")) &&
              dir1.getObjIdPrefix(name2).equals(new ObjId("b1"))) ||
             (dir1.getObjIdPrefix(name1).equals(new ObjId("b1")) &&
              dir1.getObjIdPrefix(name2).equals(new ObjId("a1"))));
    }catch(ObjAlreadyPresentException e){
      assert false : ("" + e);
    }catch(ObjNotInDirectoryException e){
      assert false : ("" + e);
    }
  }

 /** 
 *  An iterator that iterates through file names 
 **/ 
  public static class DirIterator{

 /** 
 *  Data members 
 **/ 
    private Iterator iter;

 /** 
 *  Constructor 
 **/ 
    public
    DirIterator(HashMap table){
      this.iter = table.keySet().iterator();
    }

 /** 
 *  Return true if this iterator has more items 
 **/ 
    public boolean
    hasNext(){
      return(this.iter.hasNext());
    }

 /** 
 *  Return the next item 
 **/ 
    public String
    getNext(){
      return((String)this.iter.next());
    }
  }
}