 /** 
 *  Represent the data structures used by the the READDIR NFS 3 method 
 **/ 
import java.util.Arrays;

 /** 
 *  Represent the arguments to the READDIR NFS 3 call 
 **/ 
class READDIR3args implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_nfs_fh3 dir;
  private final long cookie;
  private final NFS_cookieverf3 cookieverf;
  private final int count;

 /** 
 *  Constructor 
 **/ 
  public
  READDIR3args(NFS_nfs_fh3 newDir,
               long newCookie,
               NFS_cookieverf3 newCookieVerf,
               int newCount){
    this.dir = newDir;
    this.cookie = newCookie;
    this.cookieverf = newCookieVerf;
    this.count = newCount;
  }

 /** 
 *  Return the directory handle 
 **/ 
  public NFS_nfs_fh3
  getDir(){
    return(this.dir);
  }

 /** 
 *  Return the cookie 
 **/ 
  public long
  getCookie(){
    return(this.cookie);
  }

 /** 
 *  Return the cookie verification token 
 **/ 
  public NFS_cookieverf3
  getCookieverf(){
    return(this.cookieverf);
  }

 /** 
 *  Return the count 
 **/ 
  public int
  getCount(){
    return(this.count);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    READDIR3args args = null;

    // 2 Java-declared fields, 4 self-declared field
    assert(this.getClass().getDeclaredFields().length == 6);
    if(obj instanceof READDIR3args){
      args = (READDIR3args)obj;
      eq = (this.dir.equals(args.dir) &&
            (this.cookie == args.cookie) &&
            this.cookieverf.equals(args.cookieverf) &&
            (this.count == args.count));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 4 self-declared field
    assert(this.getClass().getDeclaredFields().length == 6);
    code = (this.dir.hashCode() ^
            (new Long(this.cookie)).hashCode() ^
            this.cookieverf.hashCode() ^
            (new Integer(this.count)).hashCode());

    return(code);
  }
}

 /** 
 *  Represent a single entry from the directory 
 **/ 
class NFS_entry3{

 /** 
 *  Data members 
 **/ 
  private final long fileId;
  private final String name;
  private final long cookie;
  private final NFS_entry3 nextEntry;

 /** 
 *  Constructor 
 **/ 
  public
  NFS_entry3(long newFileId,
             String newName,
             long newCookie,
             NFS_entry3 newNextEntry){
    this.fileId = newFileId;
    this.name = newName;
    this.cookie = newCookie;
    this.nextEntry = newNextEntry;
  }

 /** 
 *  Return the fileId 
 **/ 
  public long
  getFileId(){
    return(this.fileId);
  }

 /** 
 *  Return the name 
 **/ 
  public String
  getName(){
    return(this.name);
  }

 /** 
 *  Return the cookie 
 **/ 
  public long
  getCookie(){
    return(this.cookie);
  }

 /** 
 *  Return the next entry 
 **/ 
  public NFS_entry3
  getNextEntry(){
    return(this.nextEntry);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    NFS_entry3 entry = null;

    // 2 Java-declared fields, 4 self-declared field
    assert(this.getClass().getDeclaredFields().length == 6);
    if(obj instanceof NFS_entry3){
      entry = (NFS_entry3)obj;
      eq = ((this.fileId == entry.fileId) &&
            this.name.equals(entry.name) &&
            (this.cookie == entry.cookie));
      if(this.nextEntry != null){
        // Recursively compare the rest of the list
        eq = this.nextEntry.equals(entry.nextEntry);
      }else{
        eq = (entry.nextEntry == null);
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 4 self-declared field
    assert(this.getClass().getDeclaredFields().length == 6);
    code = ((new Long(this.fileId)).hashCode() ^
            this.name.hashCode() ^
            (new Long(this.cookie)).hashCode());
    if(this.nextEntry != null){
      // Recursively calculate the hashCode of the list
      code = code ^ this.nextEntry.hashCode();
    }

    return(code);
  }
}

 /** 
 *  Represent the list of entries in the directory 
 **/ 
class NFS_dirlist3{

 /** 
 *  Data members 
 **/ 
  private final NFS_entry3 entries;
  private final boolean eof;

 /** 
 *  Constructor 
 **/ 
  public
  NFS_dirlist3(NFS_entry3 newEntries, boolean newEOF){
    this.entries = newEntries;
    this.eof = newEOF;
  }

 /** 
 *  Return the list 
 **/ 
  public NFS_entry3
  getEntries(){
    return(this.entries);
  }

 /** 
 *  Return true if this set of entries marks the end of entries 
 **/ 
  public boolean
  getEOF(){
    return(this.eof);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    NFS_dirlist3 dirlist = null;

    // 2 Java-declared fields, 2 self-declared field
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof NFS_dirlist3){
      dirlist = (NFS_dirlist3)obj;
      if(this.entries != null){
        eq = this.entries.equals(dirlist.entries) && (this.eof == dirlist.eof);
      }else{
        eq = (dirlist.entries == null) && (this.eof == dirlist.eof);
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 2 self-declared field
    assert(this.getClass().getDeclaredFields().length == 4);
    if(this.entries != null){
      code = this.entries.hashCode() ^ (new Boolean(this.eof)).hashCode();
    }else{
      code = (new Boolean(this.eof)).hashCode();
    }
    return(code);
  }
}

 /** 
 *  Represent the results to the NFS 3 READDIR call when successful 
 **/ 
class READDIR3resok implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_attr dirAttributes;
  private final NFS_cookieverf3 cookieVerf;
  private final NFS_dirlist3 reply;

 /** 
 *  Constructor 
 **/ 
  public
  READDIR3resok(NFS_post_op_attr newDirAttributes,
                NFS_cookieverf3 newCookieVerf,
                NFS_dirlist3 newReply){
    this.dirAttributes = newDirAttributes;
    this.cookieVerf = newCookieVerf;
    this.reply = newReply;
  }

 /** 
 *  Return the directory attributes 
 **/ 
  public NFS_post_op_attr
  getDirAttributes(){
    return(this.dirAttributes);
  }

 /** 
 *  Return the cookie verification token 
 **/ 
  public NFS_cookieverf3
  getCookieverf(){
    return(this.cookieVerf);
  }

 /** 
 *  Return the list of entries 
 **/ 
  public NFS_dirlist3
  getReply(){
    return(this.reply);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    READDIR3resok resok = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof READDIR3resok){
      resok = (READDIR3resok)obj;
      eq = (this.dirAttributes.equals(resok.dirAttributes) &&
            this.cookieVerf.equals(resok.cookieVerf) &&
            this.reply.equals(resok.reply));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    code = (this.dirAttributes.hashCode() ^
            this.cookieVerf.hashCode() ^
            this.reply.hashCode());

    return(code);
  }
}

 /** 
 *  Represent the results to the NFS 3 READDIR call when it fails 
 **/ 
class READDIR3resfail implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final NFS_post_op_attr dirAttributes;

 /** 
 *  Constructor 
 **/ 
  public
  READDIR3resfail(NFS_post_op_attr newDirAttributes){
    this.dirAttributes = newDirAttributes;
  }

 /** 
 *  Return the file attributes 
 **/ 
  public NFS_post_op_attr
  getDirAttributes(){
    return(this.dirAttributes);
  }

 /** 
 *  Return true if "obj" equals "this" (note: currently unimplemented) 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    READDIR3resfail resfail = null;

    // 2 Java-declared fields, 1 self-declared field
    assert(this.getClass().getDeclaredFields().length == 3);
    if(obj instanceof READDIR3resfail){
      resfail = (READDIR3resfail)obj;
      eq = this.dirAttributes.equals(resfail.dirAttributes);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 1 self-declared field
    assert(this.getClass().getDeclaredFields().length == 3);
    code = this.dirAttributes.hashCode();
    return(code);
  }
}

 /** 
 *  Represent the results to the NFS 3 READDIR call 
 **/ 
class READDIR3res implements Immutable{

 /** 
 *  Data members 
 **/ 
  private final int status;
  private final READDIR3resok resok;
  private final READDIR3resfail resfail;

 /** 
 *  Constructor 
 **/ 
  public
  READDIR3res(int newStatus,
              READDIR3resok newResOk,
              READDIR3resfail newResFail){
    this.status = newStatus;
    this.resok = newResOk;
    this.resfail = newResFail;
    assert(((this.status == NFS_nfsstat3.NFS3_OK) && (this.resok != null)) ||
           ((this.status != NFS_nfsstat3.NFS3_OK) && (this.resfail != null)));
  }

 /** 
 *  Return the status 
 **/ 
  public int
  getStatus(){
    return(this.status);
  }

 /** 
 *  Return the result data if successful 
 **/ 
  public
  READDIR3resok getResOk(){
    assert(this.status == NFS_nfsstat3.NFS3_OK);
    return(this.resok);
  }

 /** 
 *  Return the result data if the call failed 
 **/ 
  public
  READDIR3resfail getResFail(){
    assert(this.status != NFS_nfsstat3.NFS3_OK);
    return(this.resfail);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    boolean eq = false;
    READDIR3res res = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof READDIR3res){
      res = (READDIR3res)obj;
      if(this.status == res.status){
        if(this.status == NFS_nfsstat3.NFS3_OK){
          eq = this.resok.equals(res.resok);
        }else{
          eq = this.resfail.equals(res.resfail);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(this.status == NFS_nfsstat3.NFS3_OK){
      code = (new Integer(this.status)).hashCode() ^ this.resok.hashCode();
    }else{
      code = (new Integer(this.status)).hashCode() ^ this.resfail.hashCode();
    }
    return(code);
  }
}

 /** 
 *  Used for testing 
 **/ 
public class READDIR3{

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("READDIR3 self test...");
    READDIR3.testREADDIR3args();
    READDIR3.testNFS_entry3();
    READDIR3.testNFS_dirlist3();
    READDIR3.testREADDIR3resok();
    READDIR3.testREADDIR3resfail();
    READDIR3.testREADDIR3res();
    System.err.println("...READDIR3 self test succeeds");
    System.exit(0);
  }

 /** 
 *  Test the READDIR3args 
 **/ 
  private static void
  testREADDIR3args(){
    READDIR3args args1 = null;
    READDIR3args args2 = null;
    READDIR3args args3 = null;

    // Test 1
    args1 = new READDIR3args(NFS_nfs_fh3.makeAllZeroHandle(),
                             20,
                             new NFS_cookieverf3(new byte[8]),
                             10);
    assert(args1.getDir().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args1.getCookie() == 20);
    assert(args1.getCookieverf().equals(new NFS_cookieverf3(new byte[8])));
    assert(args1.getCount() == 10);
    assert(args1.equals(args1));

    // Test 2
    args2 = new READDIR3args(NFS_nfs_fh3.makeAllZeroHandle(),
                             15,
                             new NFS_cookieverf3(new byte[8]),
                             38);
    assert(args2.getDir().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args2.getCookie() == 15);
    assert(args2.getCookieverf().equals(new NFS_cookieverf3(new byte[8])));
    assert(args2.getCount() == 38);
    assert(args2.equals(args2));
    assert(!args2.equals(args1));
    assert(!args1.equals(args2));
    assert(args2.hashCode() != args1.hashCode());

    // Test 3
    args3 = new READDIR3args(NFS_nfs_fh3.makeAllZeroHandle(),
                             20,
                             new NFS_cookieverf3(new byte[8]),
                             10);
    assert(args3.getDir().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args3.getCookie() == 20);
    assert(args3.getCookieverf().equals(new NFS_cookieverf3(new byte[8])));
    assert(args3.getCount() == 10);
    assert(args3.equals(args3));
    assert(args3.equals(args1));
    assert(args1.equals(args3));
    assert(!args3.equals(args2));
    assert(!args2.equals(args3));
    assert(args3.hashCode() == args1.hashCode());
    assert(args3.hashCode() != args2.hashCode());
  }

 /** 
 *  Test the NFS_entry3 class 
 **/ 
  private static void
  testNFS_entry3(){
    NFS_entry3 entry1 = null;   
    NFS_entry3 entry2 = null;   
    NFS_entry3 entry3 = null;   

    // Test 1
    entry1 = new NFS_entry3(104, "a", 57, null);
    assert(entry1.getFileId() == 104);
    assert(entry1.getName().equals("a"));
    assert(entry1.getCookie() == 57);
    assert(entry1.getNextEntry() == null);
    assert(entry1.equals(entry1));

    // Test 2
    entry2 = new NFS_entry3(108, "b", 38, entry1);
    assert(entry2.getFileId() == 108);
    assert(entry2.getName().equals("b"));
    assert(entry2.getCookie() == 38);
    assert(entry2.getNextEntry().equals(entry1));
    assert(entry2.equals(entry2));
    assert(!entry2.equals(entry1));
    assert(!entry1.equals(entry2));
    assert(entry2.getNextEntry().equals(entry1));
    assert(entry1.equals(entry2.getNextEntry()));
    assert(entry2.getNextEntry().hashCode() == entry1.hashCode());
    assert(entry2.hashCode() != entry1.hashCode());

    // Test 3
    entry3 = new NFS_entry3(104, "a", 57, null);
    assert(entry3.getFileId() == 104);
    assert(entry3.getName().equals("a"));
    assert(entry3.getCookie() == 57);
    assert(entry3.getNextEntry() == null);
    assert(entry3.equals(entry3));
    assert(entry3.equals(entry1));
    assert(entry1.equals(entry3));
    assert(!entry3.equals(entry2));
    assert(!entry2.equals(entry3));
    assert(entry3.equals(entry2.getNextEntry()));
    assert(entry2.getNextEntry().equals(entry3));
    assert(entry3.hashCode() == entry1.hashCode());
    assert(entry3.hashCode() != entry2.hashCode());
    assert(entry3.hashCode() == entry2.getNextEntry().hashCode());
  }

 /** 
 *  Test the NFS_dirlist3 class 
 **/ 
  private static void
  testNFS_dirlist3(){
    NFS_dirlist3 list1 = null;
    NFS_dirlist3 list2 = null;
    NFS_dirlist3 list3 = null;

    // Test 1
    list1 = new NFS_dirlist3(new NFS_entry3(1, "apple", 10, null), false);
    assert(list1.getEntries().equals(new NFS_entry3(1, "apple", 10, null)));
    assert(!list1.getEOF());
    assert(list1.equals(list1));

    // Test 2
    list2 = new NFS_dirlist3(new NFS_entry3(2, "banana", 5, null), true);
    assert(list2.getEntries().equals(new NFS_entry3(2, "banana", 5, null)));
    assert(list2.getEOF());
    assert(list2.equals(list2));
    assert(!list2.equals(list1));
    assert(!list1.equals(list2));
    assert(list2.hashCode() != list1.hashCode());

    // Test 3
    list3 = new NFS_dirlist3(new NFS_entry3(1, "apple", 10, null), false);
    assert(list3.getEntries().equals(new NFS_entry3(1, "apple", 10, null)));
    assert(!list3.getEOF());
    assert(list3.equals(list3));
    assert(list3.equals(list1));
    assert(list1.equals(list3));
    assert(!list3.equals(list2));
    assert(!list2.equals(list3));
    assert(list3.hashCode() == list1.hashCode());
    assert(list3.hashCode() != list2.hashCode());
  }

 /** 
 *  Test the READDIR3resok class 
 **/ 
  private static void
  testREADDIR3resok(){
    READDIR3resok resok1 = null;
    READDIR3resok resok2 = null;
    READDIR3resok resok3 = null;

    // Test 1
    resok1 = new READDIR3resok(new NFS_post_op_attr(false, null),
                               new NFS_cookieverf3(new byte[8]),
                               new NFS_dirlist3(new NFS_entry3(104,
                                                              "a",
                                                              57,
                                                              null),
                                               false));
    assert(resok1.getDirAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok1.getCookieverf().equals(new NFS_cookieverf3(new byte[8])));
    assert(resok1.getReply().equals(new NFS_dirlist3(new NFS_entry3(104,
                                                                    "a",
                                                                    57,
                                                                    null),
                                                     false)));
    assert(resok1.equals(resok1));

    // Test 2
    resok2 = new READDIR3resok(new NFS_post_op_attr(false, null),
                               new NFS_cookieverf3(new byte[8]),
                               new NFS_dirlist3(new NFS_entry3(108,
                                                               "a",
                                                               47,
                                                               null),
                                                true));
    assert(resok2.getDirAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok2.getCookieverf().equals(new NFS_cookieverf3(new byte[8])));
    assert(resok2.getReply().equals(new NFS_dirlist3(new NFS_entry3(108,
                                                                    "a",
                                                                    47,
                                                                    null),
                                                     true)));
    assert(resok2.equals(resok2));
    assert(!resok2.equals(resok1));
    assert(!resok1.equals(resok2));
    assert(resok2.hashCode() != resok1.hashCode());

    // Test 3
    resok3 = new READDIR3resok(new NFS_post_op_attr(false, null),
                               new NFS_cookieverf3(new byte[8]),
                               new NFS_dirlist3(new NFS_entry3(104,
                                                               "a",
                                                               57,
                                                               null),
                                                false));
    assert(resok3.getDirAttributes().equals(new NFS_post_op_attr(false,
                                                                 null)));
    assert(resok3.getCookieverf().equals(new NFS_cookieverf3(new byte[8])));
    assert(resok3.getReply().equals(new NFS_dirlist3(new NFS_entry3(104,
                                                                    "a",
                                                                    57,
                                                                    null),
                                                     false)));
    assert(resok3.equals(resok3));
    assert(resok3.equals(resok1));
    assert(resok1.equals(resok3));
    assert(!resok3.equals(resok2));
    assert(!resok2.equals(resok3));
    assert(resok3.hashCode() == resok1.hashCode());
    assert(resok3.hashCode() != resok2.hashCode());
  }

 /** 
 *  Test the READDIR3resfail class 
 **/ 
  private static void
  testREADDIR3resfail(){
    READDIR3resfail resfail1 = null;
    READDIR3resfail resfail2 = null;
    READDIR3resfail resfail3 = null;
    NFS_post_op_attr postAttr1 = null;
    NFS_post_op_attr postAttr2 = null;

    // Test 1
    postAttr1 = new NFS_post_op_attr(true,
                                     new NFS_fattr3(NFS_ftype3.NF3REG,
                                                    36,
                                                    63,
                                                    71,
                                                    85,
                                                    (72L << 32) + 83L,
                                                    (81L << 32) + 66L,
                                                    new NFS_specdata3(55, 49),
                                                    (22L << 32) + 68L,
                                                    (99L << 32) + 163L,
                                                    new NFS_nfstime3(43, 7),
                                                    new NFS_nfstime3(31, 48),
                                                    new NFS_nfstime3(12, 8)));
    postAttr2 = new NFS_post_op_attr(true,
                                     new NFS_fattr3(NFS_ftype3.NF3REG,
                                                    63,
                                                    36,
                                                    17,
                                                    58,
                                                    (27L << 23) + 38L,
                                                    (18L << 23) + 66L,
                                                    new NFS_specdata3(55, 94),
                                                    (22L << 23) + 86L,
                                                    (99L << 23) + 361L,
                                                    new NFS_nfstime3(34, 70),
                                                    new NFS_nfstime3(14, 84),
                                                    new NFS_nfstime3(21, 80)));
    resfail1 = new READDIR3resfail(postAttr1);
    assert(resfail1.getDirAttributes().equals(postAttr1));
    assert(resfail1.equals(resfail1));

    // Test 2
    resfail2 = new READDIR3resfail(postAttr2);
    assert(resfail2.getDirAttributes().equals(postAttr2));
    assert(resfail2.equals(resfail2));
    assert(!resfail2.equals(resfail1));
    assert(!resfail1.equals(resfail2));
    assert(resfail2.hashCode() != resfail1.hashCode());

    // Test 3
    resfail3 = new READDIR3resfail(postAttr1);
    assert(resfail3.getDirAttributes().equals(postAttr1));
    assert(resfail3.equals(resfail3));
    assert(resfail3.equals(resfail1));
    assert(resfail1.equals(resfail3));
    assert(!resfail3.equals(resfail2));
    assert(!resfail2.equals(resfail3));
    assert(resfail3.hashCode() == resfail1.hashCode());
    assert(resfail3.hashCode() != resfail2.hashCode());
  }

 /** 
 *  Test the READDIR3res class 
 **/ 
  private static void
  testREADDIR3res(){
    READDIR3res res1 = null;
    READDIR3res res2 = null;
    READDIR3res res3 = null;
    READDIR3resok resok = null;
    READDIR3resfail resfail = null;

    // Test 1
    resok = new READDIR3resok(new NFS_post_op_attr(false, null),
                              new NFS_cookieverf3(new byte[8]),
                              new NFS_dirlist3(new NFS_entry3(104,
                                                              "a",
                                                              57,
                                                              null),
                                               false));
    res1 = new READDIR3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res1.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res1.getResOk().equals(resok));
    assert(res1.equals(res1));

    // Test 2
    resfail = new READDIR3resfail(new NFS_post_op_attr(false, null));
    res2 = new READDIR3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    assert(res2.getStatus() == NFS_nfsstat3.NFS3ERR_IO);
    assert(res2.getResFail().equals(resfail));
    assert(res2.equals(res2));
    assert(!res2.equals(res1));
    assert(!res1.equals(res2));
    assert(res2.hashCode() != res1.hashCode());

    // Test 3
    res3 = new READDIR3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res3.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res3.getResOk().equals(resok));
    assert(res3.equals(res3));
    assert(res3.equals(res1));
    assert(res1.equals(res3));
    assert(!res3.equals(res2));
    assert(!res2.equals(res3));
    assert(res3.hashCode() == res1.hashCode());
    assert(res3.hashCode() != res2.hashCode());
  }
}

 /** 
 *  $Log: READDIR3.java,v $ 
 *  Revision 1.1  2006/06/12 20:07:20  nayate 
 *  First version of the Java NFS library 
 *  
 **/ 
