 /** 
 *  Implement a way to store NFS symbolic link information. This currently 
 *  involves serializing and deserializing a Java String 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

public class SymlinkData{

 /** 
 *  Constants 
 **/ 
  public static byte MAGIC_NUM = 6;

 /** 
 *  Data members 
 **/ 
  private String dataStr;
  private byte[] serializedData;

 /** 
 *  Constructor 
 **/ 
  public
  SymlinkData(String newDataStr){
    try{
      this.dataStr = newDataStr;
      this.serializedData = this.getDataBytes();
    }catch(IOException e){
      assert false : ("" + e);
    }
  }

 /** 
 *  Constructor 
 *  
 *  Read this object from an InputStream 
 **/ 
  public
  SymlinkData(InputStream is) throws IOException{
    DataInputStream dis = null;
    ObjectInputStream ois = null;
    byte magicNum = (byte)0;
    ByteArrayInputStream bais = null;

    try{
      dis = new DataInputStream(is);
      magicNum = dis.readByte();
      assert(magicNum == SymlinkData.MAGIC_NUM);
      ois = new ObjectInputStream(dis);
      this.dataStr = (String)ois.readObject();
      this.serializedData = this.getDataBytes();
    }catch(ClassNotFoundException e){
      assert false : ("" + e);
    }
  }

 /** 
 *  Write this object to an OutputStream 
 *  
 *  Note: Writing a symlink data to the output stream isn't enough. We need 
 *  to change the size attribute for the file metadata as well. 
 *  
 *  Note: We write data to a byte array first and then send the byte array 
 *  to the output stream both (1) for efficiency and (2) to get the size of 
 *  data written. 
 **/ 
  public void
  toOutputStream(OutputStream os) throws IOException{
    os.write(this.serializedData);
  }

 /** 
 *  Retrieve a directory entry 
 **/ 
  public String
  getData(){
    return(this.dataStr);
  }

 /** 
 *  Return the amount of space needed to serialized this symlink 
 **/ 
  public long
  getSize(){
    return(this.serializedData.length);
  }

 /** 
 *  Return true if this object equals "obj" 
 **/ 
  public boolean
  equals(Object obj){
    SymlinkData symlinkData = null;
    boolean eq = false;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof SymlinkData){
      symlinkData = (SymlinkData)obj;
      eq = this.dataStr.equals(symlinkData.dataStr);
    }else{
      // Different object types
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    code = this.dataStr.hashCode();
    return(code);
  }

 /** 
 *  Fill a byte array with this object serialized 
 **/ 
  private byte[]
  getDataBytes() throws IOException{
    byte[] dataBytes = null;
    ByteArrayOutputStream baos = null;
    ObjectOutputStream oos = null;
    DataOutputStream dos = null;

    try{
      baos = new ByteArrayOutputStream();
      dos = new DataOutputStream(baos);
      dos.writeByte(SymlinkData.MAGIC_NUM);
      oos = new ObjectOutputStream(dos);
      oos.writeObject(this.dataStr);
      oos.flush();
      dos.flush();
      baos.flush();
      dataBytes = baos.toByteArray();
    }finally{
      if(oos != null){
        try{
          oos.close();
        }catch(IOException e){
          // Do nothing; not much that we can do
        }
      }
      if(dos != null){
        try{
          dos.close();
        }catch(IOException e){
          // Do nothing; not much that we can do
        }
      }
      if(baos != null){
        try{
          baos.close();
        }catch(IOException e){
          // Do nothing; not much that we can do
        }
      }
    }
    return(dataBytes);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing SymlinkData.java...");
    SymlinkData.testSimple();
    SymlinkData.testSerialize();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    SymlinkData sym1 = null;
    SymlinkData sym2 = null;
    SymlinkData sym3 = null;

    // Test 1
    sym1 = new SymlinkData("a");
    sym2 = new SymlinkData("a");
    assert(sym1.equals(sym2));
    assert(sym2.equals(sym1));

    // Test 2
    sym1 = new SymlinkData("a");
    assert(sym1.equals(sym1));
    assert(sym1.getData().equals("a"));

    // Test 3
    sym2 = new SymlinkData("b");
    assert(sym2.equals(sym2));
    assert(sym2.getData().equals("b"));
    assert(!sym1.equals(sym2));
    assert(!sym2.equals(sym1));
    assert(sym1.hashCode() != sym2.hashCode());

    // Test 4
    sym3 = new SymlinkData("a");
    assert(sym3.getData().equals("a"));
    assert(sym3.equals(sym3));
    assert(sym3.equals(sym1));
    assert(sym1.equals(sym3));
    assert(!sym3.equals(sym2));
    assert(!sym2.equals(sym3));
    assert(sym3.hashCode() != sym2.hashCode());
    assert(sym3.hashCode() == sym1.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSerialize(){
    ByteArrayInputStream bais = null;
    ByteArrayOutputStream baos = null;
    SymlinkData sym1 = null;
    SymlinkData sym2 = null;
    byte[] bArray = null;

    // Test 1
    try{
      baos = new ByteArrayOutputStream();
      sym1 = new SymlinkData("a");
      sym1.toOutputStream(baos);
      baos.flush();
      bArray = baos.toByteArray();
      bais = new ByteArrayInputStream(bArray);
      sym2 = new SymlinkData(bais);
      assert(sym1.equals(sym2));
      assert(sym2.equals(sym1));
      assert(sym1.getData().equals("a"));
      assert(sym2.getData().equals("a"));
    }catch(IOException e){
      assert false : ("" + e);
    }

    // Test 2
    try{
      baos = new ByteArrayOutputStream();
      sym1 = new SymlinkData("b");
      sym1.toOutputStream(baos);
      baos.flush();
      bArray = baos.toByteArray();
      bais = new ByteArrayInputStream(bArray);
      sym2 = new SymlinkData(bais);
      assert(sym1.equals(sym2));
      assert(sym2.equals(sym1));
      assert(sym1.getData().equals("b"));
      assert(sym2.getData().equals("b"));
    }catch(IOException e){
      assert false : ("" + e);
    }
  }
}