 /** 
 *  Represents the data structures used by the WRITE NFS call 
 **/ 
import java.util.Arrays;

 /** 
 *  Store constants for representing the desired cache consistency 
 **/ 
class NFS_stable_how{

 /** 
 *  Data members 
 **/ 
  public static final int UNSTABLE = 0;
  public static final int DATA_SYNC = 1;
  public static final int FILE_SYNC = 2;
}

 /** 
 *  Store the arguments to the WRITE NFS call 
 **/ 
class WRITE3args{

 /** 
 *  Data members 
 **/ 
  private final NFS_nfs_fh3 file;
  private final long offset;
  private final int count;
  private final int stable;
  private final byte[] data;

 /** 
 *  Constructor 
 **/ 
  public
  WRITE3args(NFS_nfs_fh3 newFile,
             long newOffset,
             int newCount,
             int newStable,
             byte[] newData){
    this.file = newFile;
    this.offset = newOffset;
    this.count = newCount;
    this.stable = newStable;
    this.data = newData;
  }

 /** 
 *  Return the file handle 
 **/ 
  public NFS_nfs_fh3
  getFile(){
    return(this.file);
  }

 /** 
 *  Return the offset 
 **/ 
  public long
  getOffset(){
    return(this.offset);
  }

 /** 
 *  Return the count 
 **/ 
  public int
  getCount(){
    return(this.count);
  }

 /** 
 *  Return the stable flag value 
 **/ 
  public int
  getStable(){
    return(this.stable);
  }

 /** 
 *  Return the data 
 **/ 
  public byte[]
  getData(){
    return(this.data);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public boolean
  equals(Object obj){
    boolean eq = false;
    WRITE3args args = null;

    // 2 Java-declared fields, 5 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 7);
    if(obj instanceof WRITE3args){
      args = (WRITE3args)obj;
      eq = (this.file.equals(args.file) &&
            (this.offset == args.offset) &&
            (this.count == args.count) &&
            (this.stable == args.stable) &&
            (this.data.length == args.data.length));
      if(eq){
        for(int i = 0; eq && (i < this.data.length); i++){
          eq = (this.data[i] == args.data[i]);
        }
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 5 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 7);
    code = (this.file.hashCode() ^
            (new Long(this.offset)).hashCode() ^
            (new Integer(this.count)).hashCode() ^
            (new Integer(this.stable)).hashCode() ^
            Arrays.hashCode(this.data));

    return(code);
  }
}

 /** 
 *  Store the result of the WRITE NFS call when successful 
 **/ 
class WRITE3resok{

 /** 
 *  Data members 
 **/ 
  private final NFS_wcc_data fileWCC;
  private final int count;
  private final int committed;
  private final NFS_writeverf3 verf;

 /** 
 *  Constructor 
 **/ 
  public
  WRITE3resok(NFS_wcc_data newFileWCC,
              int newCount,
              int newCommitted,
              NFS_writeverf3 newVerf){
    this.fileWCC = newFileWCC;
    this.count = newCount;
    this.committed = newCommitted;
    this.verf = newVerf;
  }

 /** 
 *  Return the weak-cache consistency data value 
 **/ 
  public NFS_wcc_data
  getFileWCC(){
    return(this.fileWCC);
  }

 /** 
 *  Return the count 
 **/ 
  public int
  getCount(){
    return(this.count);
  }

 /** 
 *  Return the value of the committed flag 
 **/ 
  public int
  getCommitted(){
    return(this.committed);
  }

 /** 
 *  Return the write verification data 
 **/ 
  public NFS_writeverf3
  getVerf(){
    return(this.verf);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public boolean
  equals(Object obj){
    boolean eq = false;
    WRITE3resok resok = null;

    // 2 Java-declared fields, 4 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 6);
    if(obj instanceof WRITE3resok){
      resok = (WRITE3resok)obj;
      eq = (this.fileWCC.equals(resok.fileWCC) &&
            (this.count == resok.count) &&
            (this.committed == resok.committed) &&
            this.verf.equals(resok.verf));
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 6 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 6);
    code = (this.fileWCC.hashCode() ^
            (new Integer(this.count)).hashCode() ^
            (new Integer(this.committed)).hashCode() ^
            this.verf.hashCode());

    return(code);
  }
}

 /** 
 *  Store the result of the WRITE NFS call when unsuccessful 
 **/ 
class WRITE3resfail{

 /** 
 *  Data members 
 **/ 
  private final NFS_wcc_data fileWCC;

 /** 
 *  Constructor 
 **/ 
  public
  WRITE3resfail(NFS_wcc_data newFileWCC){
    this.fileWCC = newFileWCC;
  }

 /** 
 *  Return the file weak-cache consistency data 
 **/ 
  public NFS_wcc_data
  getFileWCC(){
    return(this.fileWCC);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public boolean
  equals(Object obj){
    boolean eq = false;
    WRITE3resfail resfail = null;

    // 2 Java-declared fields, 1 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 3);
    if(obj instanceof WRITE3resfail){
      resfail = (WRITE3resfail)obj;
      eq = this.fileWCC.equals(resfail.fileWCC);
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 1 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 3);
    code = this.fileWCC.hashCode();

    return(code);
  }
}

 /** 
 *  Store the result of the WRITE NFS call 
 **/ 
class WRITE3res{

 /** 
 *  Data members 
 **/ 
  private final int status;
  private final WRITE3resok resok;
  private final WRITE3resfail resfail;

 /** 
 *  Constructor 
 **/ 
  public
  WRITE3res(int newStatus, WRITE3resok newResOk, WRITE3resfail newResFail){
    this.status = newStatus;
    this.resok = newResOk;
    this.resfail = newResFail;
  }

 /** 
 *  Return the status 
 **/ 
  public int
  getStatus(){
    return(this.status);
  }

 /** 
 *  Return the result of the operation if successful 
 **/ 
  public WRITE3resok
  getResOk(){
    assert(this.status == NFS_nfsstat3.NFS3_OK);
    return(this.resok);
  }

 /** 
 *  Return the result of the operation if unsuccessful 
 **/ 
  public WRITE3resfail
  getResFail(){
    assert(this.status != NFS_nfsstat3.NFS3_OK);
    return(this.resfail);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public boolean
  equals(Object obj){
    boolean eq = false;
    WRITE3res res = null;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(obj instanceof WRITE3res){
      res = (WRITE3res)obj;
      if(this.status == res.status){
        if(this.status == NFS_nfsstat3.NFS3_OK){
          eq = this.resok.equals(res.resok);
        }else{
          eq = this.resfail.equals(resfail);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-declared fields, 3 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 5);
    if(this.status == NFS_nfsstat3.NFS3_OK){
      code = (new Integer(this.status)).hashCode() ^ this.resok.hashCode();
    }else{
      code = (new Integer(this.status)).hashCode() ^ this.resfail.hashCode();
    }
    return(code);
  }
}

 /** 
 *  Used for testing 
 **/ 
public class WRITE3{

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("WRITE3 self test...");
    WRITE3.testWRITE3args();
    WRITE3.testWRITE3resok();
    WRITE3.testWRITE3resfail();
    WRITE3.testWRITE3res();
    System.err.println("...WRITE3 self test succeeds");
    System.exit(0);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testWRITE3args(){
    WRITE3args args1 = null;
    WRITE3args args2 = null;
    WRITE3args args3 = null;

    // Test 1
    args1 = new WRITE3args(NFS_nfs_fh3.makeAllZeroHandle(),
                           78,
                           80,
                           NFS_stable_how.DATA_SYNC,
                           new byte[80]);
    assert(args1.getFile().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args1.getOffset() == 78);
    assert(args1.getCount() == 80);
    assert(args1.getStable() == NFS_stable_how.DATA_SYNC);
    assert(Arrays.equals(args1.getData(), new byte[80]));
    assert(args1.equals(args1));

    // Test 2
    args2 = new WRITE3args(NFS_nfs_fh3.makeAllZeroHandle(),
                           77,
                           78,
                           NFS_stable_how.FILE_SYNC,
                           new byte[78]);
    assert(args2.getFile().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args2.getOffset() == 77);
    assert(args2.getCount() == 78);
    assert(args2.getStable() == NFS_stable_how.FILE_SYNC);
    assert(Arrays.equals(args2.getData(), new byte[78]));
    assert(args2.equals(args2));
    assert(!args2.equals(args1));
    assert(!args1.equals(args2));
    assert(args2.hashCode() != args1.hashCode());

    // Test 3
    args3 = new WRITE3args(NFS_nfs_fh3.makeAllZeroHandle(),
                           78,
                           80,
                           NFS_stable_how.DATA_SYNC,
                           new byte[80]);
    assert(args3.getFile().equals(NFS_nfs_fh3.makeAllZeroHandle()));
    assert(args3.getOffset() == 78);
    assert(args3.getCount() == 80);
    assert(args3.getStable() == NFS_stable_how.DATA_SYNC);
    assert(Arrays.equals(args3.getData(), new byte[80]));
    assert(args3.equals(args3));
    assert(args3.equals(args1));
    assert(args1.equals(args3));
    assert(!args3.equals(args2));
    assert(!args2.equals(args3));
    assert(args3.hashCode() == args1.hashCode());
    assert(args3.hashCode() != args2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testWRITE3resok(){
    WRITE3resok resok1 = null;
    WRITE3resok resok2 = null;
    WRITE3resok resok3 = null;
    NFS_wcc_data data = null;

    // Test 1
    data = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                            new NFS_post_op_attr(false, null));
    resok1 = new WRITE3resok(data,
                             55,
                             NFS_stable_how.UNSTABLE,
                             new NFS_writeverf3(new byte[8]));
    assert(resok1.getFileWCC().equals(data));
    assert(resok1.getCount() == 55);
    assert(resok1.getCommitted() == NFS_stable_how.UNSTABLE);
    assert(resok1.getVerf().equals(new NFS_writeverf3(new byte[8])));
    assert(resok1.equals(resok1));

    // Test 2
    resok2 = new WRITE3resok(data,
                             33,
                             NFS_stable_how.UNSTABLE,
                             new NFS_writeverf3(new byte[8]));
    assert(resok2.getFileWCC().equals(data));
    assert(resok2.getCount() == 33);
    assert(resok2.getCommitted() == NFS_stable_how.UNSTABLE);
    assert(resok2.getVerf().equals(new NFS_writeverf3(new byte[8])));
    assert(resok2.equals(resok2));
    assert(!resok2.equals(resok1));
    assert(!resok1.equals(resok2));
    assert(resok2.hashCode() != resok1.hashCode());

    // Test 3
    resok3 = new WRITE3resok(data,
                             55,
                             NFS_stable_how.UNSTABLE,
                             new NFS_writeverf3(new byte[8]));
    assert(resok3.getFileWCC().equals(data));
    assert(resok3.getCount() == 55);
    assert(resok3.getCommitted() == NFS_stable_how.UNSTABLE);
    assert(resok3.getVerf().equals(new NFS_writeverf3(new byte[8])));
    assert(resok3.equals(resok3));
    assert(resok3.equals(resok1));
    assert(resok1.equals(resok3));
    assert(!resok3.equals(resok2));
    assert(!resok2.equals(resok3));
    assert(resok3.hashCode() == resok1.hashCode());
    assert(resok3.hashCode() != resok2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testWRITE3resfail(){
    WRITE3resfail resfail1 = null;
    WRITE3resfail resfail2 = null;
    WRITE3resfail resfail3 = null;
    NFS_wcc_data data1 = null;
    NFS_wcc_data data2 = null;
    NFS_wcc_attr attr = null;

    // Test 1
    data1 = new NFS_wcc_data(new NFS_pre_op_attr(true, null),
                             new NFS_post_op_attr(false, null));
    resfail1 = new WRITE3resfail(data1);
    assert(resfail1.getFileWCC().equals(data1));

    // Test 2
    attr = new NFS_wcc_attr(7, new NFS_nfstime3(1, 2), new NFS_nfstime3(3, 4));
    data2 = new NFS_wcc_data(new NFS_pre_op_attr(true, attr),
                             new NFS_post_op_attr(false, null));
    resfail2 = new WRITE3resfail(data2);
    assert(resfail2.getFileWCC().equals(data2));
    assert(resfail2.equals(resfail2));
    assert(!resfail2.equals(resfail1));
    assert(!resfail1.equals(resfail2));
    assert(resfail2.hashCode() != resfail1.hashCode());

    // Test 3
    resfail3 = new WRITE3resfail(data1);
    assert(resfail3.getFileWCC().equals(data1));
    assert(resfail3.equals(resfail3));
    assert(resfail3.equals(resfail1));
    assert(resfail1.equals(resfail3));
    assert(!resfail3.equals(resfail2));
    assert(!resfail2.equals(resfail3));
    assert(resfail3.hashCode() == resfail1.hashCode());
    assert(resfail3.hashCode() != resfail2.hashCode());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testWRITE3res(){
    WRITE3res res1 = null;
    WRITE3res res2 = null;
    WRITE3res res3 = null;
    NFS_wcc_data data = null;
    WRITE3resok resok = null;
    WRITE3resfail resfail = null;

    // Test 1
    data = new NFS_wcc_data(new NFS_pre_op_attr(false, null),
                            new NFS_post_op_attr(false, null));
    resok = new WRITE3resok(data,
                            55,
                            NFS_stable_how.UNSTABLE,
                            new NFS_writeverf3(new byte[8]));
    res1 = new WRITE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res1.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res1.getResOk().equals(resok));
    assert(res1.equals(res1));

    // Test 2
    resfail = new WRITE3resfail(data);
    res2 = new WRITE3res(NFS_nfsstat3.NFS3ERR_IO, null, resfail);
    assert(res2.getStatus() == NFS_nfsstat3.NFS3ERR_IO);
    assert(res2.getResFail().equals(resfail));
    assert(res2.equals(res2));
    assert(!res2.equals(res1));
    assert(!res1.equals(res2));
    assert(res2.hashCode() != res1.hashCode());

    // Test 3
    res3 = new WRITE3res(NFS_nfsstat3.NFS3_OK, resok, null);
    assert(res3.getStatus() == NFS_nfsstat3.NFS3_OK);
    assert(res3.getResOk().equals(resok));
    assert(res3.equals(res3));
    assert(res3.equals(res1));
    assert(res1.equals(res3));
    assert(!res3.equals(res2));
    assert(!res2.equals(res3));
    assert(res3.hashCode() == res1.hashCode());
    assert(res3.hashCode() != res2.hashCode());
  }
}

 /** 
 *  $Log: WRITE3.java,v $ 
 *  Revision 1.2  2006/08/10 21:34:05  nayate 
 *  Minor change 
 *  
 *  Revision 1.1  2006/06/12 20:07:21  nayate 
 *  First version of the Java NFS library 
 *  
 **/ 
