package code;
 /** 
 *  A data structure that maps ObjectIDs and FileHandles in both directions 
 *  
 *  Note that we do not currently bound the size of this data structure 
 *  though a real implementation would normally do so. 
 **/ 
import java.util.HashMap;

public class FHCache{

 /** 
 *  Data members 
 **/ 
  private HashMap objIdHandleTable;
  private HashMap handleObjIdTable;
  private fhandle lastHandle;
  private LRUQueue lruQueue;
  private int maxEntries;

 /** 
 *  Constructor 
 **/ 
  public
  FHCache(String rootDirName, int newMaxEntries){
    this.objIdHandleTable = new HashMap();
    this.handleObjIdTable = new HashMap();
    this.lruQueue = new LRUQueue();
    this.maxEntries = newMaxEntries;
    // Make an all-zero handle
    this.lastHandle = fhandle.makeAllZeroHandle();
    // Add an entry for this handle (root directory)
    this.objIdHandleTable.put(new ObjId(rootDirName), this.lastHandle);
    this.handleObjIdTable.put(this.lastHandle, new ObjId(rootDirName));
    this.lruQueue.add(this.lastHandle);
  }

 /** 
 *  Return true if this cache contains the given ObjId 
 **/ 
  public synchronized boolean
  containsPrefix(ObjId objIdPrefix){
    boolean doesContain = false;

    doesContain = this.objIdHandleTable.containsKey(objIdPrefix);
    return(doesContain);
  }

 /** 
 *  Return a file handle from an objId 
 *  Note: We assume that objIdPrefix is valid 
 **/ 
  public synchronized fhandle
  getFileHandle(ObjId objIdPrefix){
    fhandle handle = null;
    fhandle removedHandle = null;
    ObjId removedPrefix = null;

    handle = (fhandle)this.objIdHandleTable.get(objIdPrefix);
    if(handle == null){
      handle = this.lastHandle.cloneIncOne();
      this.lastHandle = handle;
      this.objIdHandleTable.put(objIdPrefix, handle);
      this.handleObjIdTable.put(handle, objIdPrefix);
      this.lruQueue.add(handle);
      if(this.lruQueue.getSize() > this.maxEntries){
        removedHandle = (fhandle)this.lruQueue.evict();
        removedPrefix = (ObjId)this.handleObjIdTable.remove(removedHandle);
        assert(removedPrefix != null);
        removedHandle = (fhandle)this.objIdHandleTable.remove(removedPrefix);
        assert(removedHandle != null);
      }
    }else{
      assert(this.handleObjIdTable.get(handle) != null);
      assert(this.handleObjIdTable.get(handle).equals(objIdPrefix));
      this.lruQueue.touch(handle);
    }
    assert(this.lruQueue.getSize() == this.handleObjIdTable.size());
    assert(this.lruQueue.getSize() == this.objIdHandleTable.size());
    return(handle);
  }

 /** 
 *  Return an objId from a file handle 
 **/ 
  public synchronized ObjId
  getObjIdPrefix(fhandle handle) throws BadFileHandleException{
    ObjId objId = null;

    objId = (ObjId)this.handleObjIdTable.get(handle);
    if(objId == null){
      throw new BadFileHandleException("Bad handle: " + handle);
    }
    this.lruQueue.touch(handle);
    return(objId);
  }

 /** 
 *  Remove an entry from the table (if present) 
 **/ 
  public synchronized void
  removeObjIdPrefix(ObjId objIdPrefix){
    fhandle handle = null;
    fhandle removedHandle = null;
    ObjId removedObjIdPrefix = null;

    handle = (fhandle)this.objIdHandleTable.get(objIdPrefix);
    if(handle != null){
      removedHandle = (fhandle)this.objIdHandleTable.remove(objIdPrefix);
      removedObjIdPrefix = (ObjId)this.handleObjIdTable.remove(handle);
      assert(removedHandle.equals(handle));
      assert(removedObjIdPrefix.equals(objIdPrefix));
      this.lruQueue.remove(removedHandle);
    }
    assert(this.lruQueue.getSize() == this.handleObjIdTable.size());
    assert(this.lruQueue.getSize() == this.objIdHandleTable.size());
  }

 /** 
 *  Return true if "this" equals "obj" (note: currently unimplemented) 
 **/ 
  public boolean
  equals(Object obj){
    assert false:"Unimplemented";
    return(false);
  }
 /** 
 *  Return a hash code for "this" (note: currently unimplemented) 
 **/ 
  public int
  hashCode(Object obj){
    assert false:"Unimplemented";
    return(0);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("FHCache self test...");
    FHCache.testSimple();
    FHCache.testBoundedCache();
    System.err.println("...FHCache self test succeeds");
    System.exit(0);
  }

 /** 
 *  Test the simple methods of the class 
 **/ 
  private static void
  testSimple(){
    FHCache cache1 = null;
    FHCache cache2 = null;
    fhandle handle1 = null;
    fhandle handle2 = null;
    fhandle handle3 = null;
    fhandle handle4 = null;
    fhandle handle5 = null;
    fhandle handle6 = null;
    fhandle compHandle1 = null;
    fhandle compHandle2 = null;
    fhandle compHandle3 = null;
    byte[] b = null;

    cache1 = new FHCache("/root", 1000);
    handle1 = cache1.getFileHandle(new ObjId("a"));
    handle2 = cache1.getFileHandle(new ObjId("b"));
    assert(cache1.containsPrefix(new ObjId("a")));
    assert(cache1.containsPrefix(new ObjId("b")));
    assert(!cache1.containsPrefix(new ObjId("c")));

    try{
      assert(cache1.getObjIdPrefix(handle1).equals(new ObjId("a")));
    }catch(BadFileHandleException e){
      assert(false);
    }
    try{
      assert(cache1.getObjIdPrefix(handle2).equals(new ObjId("b")));
    }catch(BadFileHandleException e){
      assert(false);
    }

    // Make sure the handles we got back follow what we expect
    b = new byte[fhandle.FHSIZE];
    b[fhandle.FHSIZE - 1] = 1;
    compHandle1 = fhandle.makeHandle(b);
    assert(compHandle1.equals(handle1));
    b[fhandle.FHSIZE - 1] = 2;
    compHandle2 = fhandle.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Try to pass in a bad file name
    try{
      b[fhandle.FHSIZE - 1] = 3;
      cache1.getObjIdPrefix(fhandle.makeHandle(b));
      assert(false);
    }catch(BadFileHandleException e){
      // We should be here
    }

    // Try to overflow the file handle
    for(int i = 0; i < fhandle.FHSIZE; i++){
      b[i] = -1;
    }
    handle3 = fhandle.makeHandle(b);
    handle4 = handle3.cloneIncOne();
    for(int i = 0; i < fhandle.FHSIZE; i++){
      b[i] = 0;
    }
    compHandle3 = fhandle.makeHandle(b);
    assert(handle4.equals(compHandle3));

    // Try to remove entries
    cache2 = new FHCache("/root", 1000);
    handle5 = cache2.getFileHandle(new ObjId("a"));
    handle6 = cache2.getFileHandle(new ObjId("b"));
    assert(cache2.containsPrefix(new ObjId("a")));
    assert(cache2.containsPrefix(new ObjId("b")));
    assert(!cache2.containsPrefix(new ObjId("c")));

    try{
      assert(cache2.getObjIdPrefix(handle5).equals(new ObjId("a")));
      assert(cache2.getObjIdPrefix(handle6).equals(new ObjId("b")));
      cache2.removeObjIdPrefix(new ObjId("a"));
      assert(cache2.getObjIdPrefix(handle6).equals(new ObjId("b")));
      try{
        cache2.getObjIdPrefix(handle5);
        assert false;
      }catch(BadFileHandleException e){
        // This was expected
      }
    }catch(BadFileHandleException e2){
      assert false : ("" + e2);
    }
  }

 /** 
 *  Test to make sure the cache works even if bounded in size 
 **/ 
  private static void
  testBoundedCache(){
    FHCache cache = null;
    fhandle handle1 = null;
    fhandle handle2 = null;
    fhandle handle3 = null;
    fhandle handle4 = null;

    // Test 1
    // Insert k + 1 items in the cache
    // Make sure it has only k entries.
    // Ask for the evicted objIdPrefix
    // Make sure that a new handle is generated
    // Make sure the cache has k entries
    cache = new FHCache("/root", 2);
    handle1 = cache.getFileHandle(new ObjId("a"));
    handle2 = cache.getFileHandle(new ObjId("b"));
    handle3 = cache.getFileHandle(new ObjId("c"));
    assert(cache.objIdHandleTable.size() == 2);
    assert(handle2.equals(cache.getFileHandle(new ObjId("b"))));
    assert(handle3.equals(cache.getFileHandle(new ObjId("c"))));
    handle4 = cache.getFileHandle(new ObjId("a"));
    assert(!handle4.equals(handle1));
    assert(handle3.equals(cache.getFileHandle(new ObjId("c"))));
    assert(cache.objIdHandleTable.size() == 2);
  }
}
