package code;
 /** 
 *  Local interface created for the PRACTI FS 
 **/ 
import java.util.HashMap;
import java.util.Arrays;
import java.util.Map;
import java.util.Iterator;
import java.io.IOException;
import java.io.EOFException;

public class FakePRACTIFSLocalInterface extends PRACTIFSLocalInterface{

 /** 
 *  Data members 
 **/ 
  private HashMap buffer;

 /** 
 *  Constructor 
 **/ 
  public
  FakePRACTIFSLocalInterface(NodeId myNodeId){
    super(myNodeId);
    this.buffer = new HashMap();
  }

 /** 
 *  Acquire a read lock 
 **/ 
  public PRACTIFSReadLockToken
  acquireReadLock(ObjId objId){
    PRACTIFSReadLockToken token = null;

    token = new PRACTIFSReadLockToken(objId, true);
    return(token);
  }

 /** 
 *  Release a read lock 
 **/ 
  public void
  releaseReadLock(PRACTIFSReadLockToken token, ObjId objId){
    assert(token.getObjId().equals(objId));
    token.setIsAcquired(false);
  }

 /** 
 *  Acquire a write (exclusive) lock 
 **/ 
  public PRACTIFSWriteLockToken
  acquireWriteLock(ObjId objId){
    PRACTIFSWriteLockToken token = null;

    token = new PRACTIFSWriteLockToken(objId, true);
    return(token);
  }

 /** 
 *  Release a write lock 
 **/ 
  public void
  releaseWriteLock(PRACTIFSReadLockToken token, ObjId objId){
    assert(token.getObjId().equals(objId));
    token.setIsAcquired(false);
  }

 /** 
 *  Read from an object 
 **/ 
  public BodyMsg
  read(ObjId objId,
       long offset,
       long length) throws ObjNotFoundException, IOException, EOFException{
    byte[] objData = null;
    byte[] retData = null;
    BodyMsg bm = null;

    if(!this.buffer.containsKey(objId)){
      throw new ObjNotFoundException("" + objId + " was not found");
    }
    objData = (byte[])this.buffer.get(objId);
    if(offset >= objData.length){
      throw new EOFException("Trying to read beyond the end of " + objId);
    }
    if(objData.length <= (offset + length)){
      length = objData.length - offset;
    }
    retData = new byte[(int)length];
    for(int i = 0; i < length; i++){
      retData[i] = objData[(int)offset + i];
    }
    bm = new BodyMsg(objId,
                     offset,
                     length,
                     null,
                     new ImmutableBytes(retData),
                     false);
    return(bm);
  }

 /** 
 *  Write method called by PRACTIFS 
 **/ 
  public long[] write(FSWriteEntry[] fwe)
    throws IOException {
    double priority = 0.0;
    boolean embargoed = false;
    //Env.dprintln(dbgLock, "PRACTIFSLocalInterface::write" + fwe[0].getObjInvalTarget().toString());
    MultiWriteEntry[] mwe = createMultiWriteEntry(fwe, priority, false);
    long[] result = this.writeMulti(mwe, embargoed);
    //Env.dprintln(dbgLock, "return PRACTIFSLocalInterface::write"+fwe[0].getObjInvalTarget().toString());
    return result;    

  }

 /** 
 *  Write to multiple objects 
 **/ 
  protected long[]
  writeMulti(MultiWriteEntry[] mwe, boolean embargoed)
    throws IOException{
    long[] written = null;
    MultiWriteEntry entry = null;

    written = new long[mwe.length];
    for(int i = 0; i < mwe.length; i++){
      entry = mwe[i];
      if(!entry.getDelete()){
        written[i] = this.modifyBytes(entry.getObjInvalTarget().getObjId(),
                                      entry.getObjInvalTarget().getOffset(),
                                      entry.getObjInvalTarget().getLength(),
                                      entry.getImmutableBytes());
      }
    }
    return(written);
  }


 /** 
 *  createMultiWriteEntry - converts FSWriteEntry to MultiWriteEntry 
 *   incorporating the priority and bound parameters 
 **/ 
  protected MultiWriteEntry[] 
  createMultiWriteEntry(FSWriteEntry[] fwe, double priority, boolean useBound) {
    MultiWriteEntry[] mwe = new MultiWriteEntry[fwe.length];
    boolean useBValue;

    for(int i=0; i<mwe.length; i++) {
      // try to avoid bounded deletes!
      useBValue = useBound;
      if(fwe[i].getDelete()) {
        useBValue = false;
      } 
      mwe[i] = new MultiWriteEntry(fwe[i].getObjInvalTarget(),
                                   priority,
                                   fwe[i].getImmutableBytes(),
                                   useBValue,
                                   fwe[i].getDelete());
    }
    
    return mwe;
  }

 /** 
 *  Write to an object 
 **/ 
  public long
  write(ObjId objId, long offset, long length, byte buffer[]) 
    throws IOException{
    long numBytes = 0;

    numBytes = this.modifyBytes(objId,
                                offset,
                                length,
                                new ImmutableBytes(buffer));
    return(numBytes);
  }

 /** 
 *  Write to an object 
 **/ 
  private long
  modifyBytes(ObjId objId, long offset, long length, ImmutableBytes ib){
    byte[] data = null;
    byte[] objData = null;
    byte[] newObjData = null;

    data = ib.getCopyBytes();
    objData = (byte[])this.buffer.get(objId);
    if(objData == null){
      newObjData = new byte[(int)(offset + length)];
    }else{
      if(objData.length < (int)(offset + length)){
        // New bytes will cause object size to grow
        newObjData = new byte[(int)(offset + length)];
        for(int i = 0; i < objData.length; i++){
          // Copy original bytes
          newObjData[i] = objData[i];
        }
      }else{
        newObjData = objData;
      }
    }
    for(int i = 0; i < (int)length; i++){
      newObjData[i + (int)offset] = data[i];
    }
    this.buffer.put(objId, newObjData);
    return(length);
  }

 /** 
 *  Return true if this object equals another (note: current unimplemented) 
 **/ 
  public boolean
  equals(Object obj){
    System.err.println("Unimplemented");
    assert(false);
    return(false);
  }

 /** 
 *  Return a hash code for this object 
 **/ 
  public int
  hashCode(){
    System.err.println("Unimplemented");
    assert(false);
    return(0);
  }

 /** 
 *  Return a String representation of this class (used for testing) 
 **/ 
  public String
  toString(){
    String str = null;
    Map.Entry entry = null;

    str = "[";
    for(Iterator i = this.buffer.entrySet().iterator(); i.hasNext();){
      entry = (Map.Entry)i.next();
      str += ("(" + entry.getKey() + ", " +
              ((byte[])entry.getValue()).length + ") ");
    }
    str += "]";
    return(str);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing FakePRACTIFSLocalInterface.java...");
    FakePRACTIFSLocalInterface.testSimple();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    FakePRACTIFSLocalInterface fli = null;
    MultiWriteEntry[] mwe = null;
    long bytesWritten[] = null;
    BodyMsg bm1 = null;
    BodyMsg bm2 = null;
    byte[] data1 = null;
    byte[] data2 = null;
    byte[] data3 = null;

    // Test 1: Write two objects and read them
    fli = new FakePRACTIFSLocalInterface(new NodeId(10));
    data1 = new byte[100];
    data2 = new byte[100];
    data1[0] = 1;
    data2[0] = 2;
    mwe = new MultiWriteEntry[2];
    mwe[0] = new MultiWriteEntry(new ObjId("a"),
                                 0,
                                 100,
                                 0.1,
                                 new ImmutableBytes(data1),
                                 false,
                                 false);
    mwe[1] = new MultiWriteEntry(new ObjId("b"),
                                 0,
                                 100,
                                 0.1,
                                 new ImmutableBytes(data2),
                                 false,
                                 false);
    try{
      bytesWritten = fli.writeMulti(mwe, false);
      bm1 = fli.read(new ObjId("a"), 0, 100);
      assert(Arrays.equals(bm1.getBody().getCopyBytes(), data1));
      bm2 = fli.read(new ObjId("b"), 0, 100);
      assert(Arrays.equals(bm2.getBody().getCopyBytes(), data2));
    }catch(ObjNotFoundException e){
      assert(false);
    }catch(IOException e){
      assert(false);
    }

    // Test 2: Cause an object size to grow
    mwe = new MultiWriteEntry[1];
    mwe[0] = new MultiWriteEntry(new ObjId("a"),
                                 100,
                                 100,
                                 0.1,
                                 new ImmutableBytes(data1),
                                 false,
                                 false);
    try{
      data3 = new byte[200];
      for(int i = 0; i < data1.length; i++){
        data3[i] = data1[i];
        data3[i + 100] = data1[i];
      }
      bytesWritten = fli.writeMulti(mwe, false);
      bm1 = fli.read(new ObjId("a"), 0, 200);
      assert(Arrays.equals(bm1.getBody().getCopyBytes(), data3));
    }catch(ObjNotFoundException e){
      assert(false);
    }catch(IOException e){
      assert(false);
    }
  }
}
