package code;
 /** 
 *  Used to build a module that implements the LRU cache replacement scheme 
 **/ 
import java.util.HashMap;

public class LRUQueue{

 /** 
 *  Data members 
 **/ 
  private HashMap table;
  private LRUNode head;
  private LRUNode tail;

 /** 
 *  Internal class to build a doubly-linked list 
 **/ 
  private static class LRUNode{

 /** 
 *  Data members 
 **/ 
    public Object obj;
    public LRUNode next;
    public LRUNode prev;

 /** 
 *  Constructor 
 **/ 
    public
    LRUNode(Object newObj, LRUNode newNext, LRUNode newPrev){
      this.obj = newObj;
      this.next = newNext;
      this.prev = newPrev;
    }
  }

 /** 
 *  Contructor 
 **/ 
  public
  LRUQueue(){
    this.table = new HashMap();
    this.head = null;
    this.tail = null;
  }

 /** 
 *  Add a new entry 
 **/ 
  public void
  add(Object obj){
    LRUNode node = null;

    assert(!this.table.containsKey(obj));
    node = new LRUNode(obj, null, null);
    this.addLastToList(node);
    this.table.put(obj, node);
  }

 /** 
 *  Return the size 
 **/ 
  public int
  getSize(){
    return(this.table.size());
  }

 /** 
 *  Evict the least recently used object from this queue 
 **/ 
  public Object
  evict(){
    Object evictedObj = null;
    LRUNode node = null;

    assert(this.head != null);
    evictedObj = this.head.obj;
    node = (LRUNode)this.table.remove(evictedObj);
    assert(node == this.head);
    this.removeFromList(node);
    return(evictedObj);
  }

 /** 
 *  Indicate that this object was just used 
 **/ 
  public void
  touch(Object obj){
    LRUNode node = null;

    node = (LRUNode)this.table.get(obj);
    assert(node != null);
    this.removeFromList(node);
    this.addLastToList(node);
  }

 /** 
 *  Remove this object from the queue 
 **/ 
  public void
  remove(Object obj){
    LRUNode node = null;

    assert(this.table.containsKey(obj));
    node = (LRUNode)this.table.remove(obj);
    this.removeFromList(node);
  }

 /** 
 *  Add this node to the end of the internal linked list 
 *  Note: Modifies the parameter 
 **/ 
  private void
  addLastToList(LRUNode node){
    assert(node != null);
    node.prev = null;
    node.next = null;
    if(this.head == null){
      assert(this.tail == null);
      this.head = node;
      this.tail = node;
    }else{
      assert(this.head.prev == null);
      assert(this.tail != null);
      assert(this.tail.next == null);
      node.prev = this.tail;
      this.tail.next = node;
      this.tail = node;
    }
  }

 /** 
 *  Remove this node from the internal linked list 
 *  Note: Modifies the parameter 
 **/ 
  private void
  removeFromList(LRUNode node){
    // Check simple data structure integrity
    assert(node != null);
    assert(this.head != null);
    assert(this.tail != null);
    assert(this.head.prev == null);
    assert(this.tail.next == null);

    if(node == this.head){
      this.head = this.head.next;
      if(this.head == null){
        this.tail = null;
      }else{
        this.head.prev = null;
      }
    }else if(node == this.tail){
      // We know that (head != tail); otherwise we would have been in the
      // previous "if" block
      assert(this.head != this.tail);
      this.tail = this.tail.prev;
      this.tail.next = null;
    }else{
      // Node lies somewhere between head and tail
      assert(this.head != this.tail); // Node has to be somewhere in list
      node.prev.next = node.next;
      node.next.prev = node.prev;
    }
    node.prev = null;
    node.next = null;
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("LRUQueue self test...");
    LRUQueue.testSimple();
    System.err.println("...LRUQueue self test succeeds");
    System.exit(0);
  }

 /** 
 *  Run some simple tests 
 **/ 
  private static void
  testSimple(){
    LRUQueue l = null;
    Object o1 = null;

    // Test 1:
    // Make sure the size is 0
    // Insert into the queue
    // Make sure the size is correct
    // Evict from the queue
    // Make sure the size is 0
    // Add back to the queue
    // Make sure the size is 1
    // Remove from the queue
    // Make sure the size is 0
    l = new LRUQueue();

    assert(l.getSize() == 0);
    l.add(new Integer(100));
    assert(l.getSize() == 1);
    o1 = l.evict();
    assert(o1.equals(new Integer(100)));
    assert(l.getSize() == 0);
    l.add(new Integer(101));
    assert(l.getSize() == 1);
    l.remove(new Integer(101));
    assert(l.getSize() == 0);

    // Test 2
    // Add three entries
    // Make sure the size is 3
    // Touch the second entry
    // Touch the first entry
    // Evict an entry
    // Make sure the removed entry is the third one
    // Make sure the size is 2
    // Evict an entry
    // Make sure the removed entry is the second one
    // Make sure the size is 1
    // Evict the final entry
    // Make sure it is the first one
    // Make sure the size is 0
    l = new LRUQueue();

    assert(l.getSize() == 0);
    l.add(new String("a"));
    l.add(new String("b"));
    l.add(new String("c"));
    assert(l.getSize() == 3);
    l.touch(new String("b"));
    l.touch(new String("a"));
    o1 = l.evict();
    assert(o1.equals(new String("c")));
    assert(l.getSize() == 2);
    o1 = l.evict();
    assert(o1.equals(new String("b")));
    assert(l.getSize() == 1);
    o1 = l.evict();
    assert(o1.equals(new String("a")));
    assert(l.getSize() == 0);

    // Test 3
    // Add three entries
    // Make sure the size is 3
    // Touch the first entry
    // Remove the second entry
    // Make sure the size is 2
    // Evict an entry
    // Make sure it is the third one
    // Make sure the size is 1
    // Remove the first entry
    // Make sure the size is 0
    l = new LRUQueue();

    assert(l.getSize() == 0);
    l.add(new String("a"));
    l.add(new String("b"));
    l.add(new String("c"));
    assert(l.getSize() == 3);
    l.touch(new String("a"));
    l.remove(new String("b"));
    assert(l.getSize() == 2);
    o1 = l.evict();
    assert(o1.equals(new String("c")));
    assert(l.getSize() == 1);
    l.remove(new String("a"));
    assert(l.getSize() == 0);
  }
}
