package code;
 /** 
 *  A table of locks for various objects 
 *  
 *  Note that we do not currently bound the size of this data structure 
 *  though a real implementation would normally do so. 
 **/ 
import java.util.HashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;

// For testing
import java.util.LinkedList;

public class LockTable{

 /** 
 *  Constants 
 **/ 
  private static boolean PRINT_DEBUG = false;

 /** 
 *  Data members 
 **/ 
  private HashMap table;

 /** 
 *  Constructor 
 **/ 
  public
  LockTable(){
    this.table = new HashMap();
  }

 /** 
 *  Acquire a read lock on an object 
 **/ 
  public synchronized void
  acquireReadLock(ObjId objId) throws InterruptedException{
    ReentrantReadWriteLock lock = null;
    ReentrantReadWriteLock.ReadLock readLock = null;

    lock = this.getLock(objId);
    readLock = lock.readLock();
    while(!readLock.tryLock()){
      this.wait();
      lock = this.getLock(objId);
      readLock = lock.readLock();
    }
    assert(!lock.isWriteLocked());

    Env.dprinterrln(PRINT_DEBUG,
                    "LockTable: " + Thread.currentThread().getName() +
                    " acquired read lock on " + objId);
  }

 /** 
 *  Acquire a write lock on an object 
 **/ 
  public synchronized void
  acquireWriteLock(ObjId objId) throws InterruptedException{
    ReentrantReadWriteLock lock = null;
    ReentrantReadWriteLock.WriteLock writeLock = null;

    lock = this.getLock(objId);
    writeLock = lock.writeLock();
    while(!writeLock.tryLock()){
      this.wait();
      lock = this.getLock(objId);
      writeLock = lock.writeLock();
    }
    assert(lock.getReadLockCount() == 0);

    Env.dprinterrln(PRINT_DEBUG,
                    "LockTable: " + Thread.currentThread().getName() +
                    " acquired write lock on " + objId);
  }

 /** 
 *  Release a read lock on an object 
 **/ 
  public synchronized void
  releaseReadLock(ObjId objId){
    ReentrantReadWriteLock lock = null;
    ReentrantReadWriteLock.ReadLock readLock = null;

    lock = this.getLock(objId);
    assert(lock != null);
    assert(lock.getReadLockCount() > 0);
    assert(!lock.isWriteLocked());
    readLock = lock.readLock();
    readLock.unlock();
    this.notifyAll();
    if(lock.getReadLockCount() == 0){
      this.table.remove(objId);
    }
    Env.dprinterrln(PRINT_DEBUG,
                    "LockTable: " + Thread.currentThread().getName() +
                    " releasing read lock on " + objId);
  }

 /** 
 *  Release a write lock on an object 
 **/ 
  public synchronized void
  releaseWriteLock(ObjId objId){
    ReentrantReadWriteLock lock = null;
    ReentrantReadWriteLock.WriteLock writeLock = null;

    lock = this.getLock(objId);
    assert(lock != null);
    assert(lock.getReadLockCount() == 0);
    assert(lock.isWriteLocked());
    writeLock = lock.writeLock();
    writeLock.unlock();
    this.notifyAll();
    this.table.remove(objId);
    Env.dprinterrln(PRINT_DEBUG,
                    "LockTable: " + Thread.currentThread().getName() +
                    " releasing write lock on " + objId);
  }

 /** 
 *  Return a lock table entry 
 **/ 
  private ReentrantReadWriteLock
  getLock(ObjId objId){
    ReentrantReadWriteLock lock = null;

    lock = (ReentrantReadWriteLock)this.table.get(objId);
    if(lock == null){
      lock = new ReentrantReadWriteLock();
      this.table.put(objId, lock);
    }
    return(lock);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing LockTable.java...");
    LockTable.testSimple();
    LockTable.testDoubleThread();
    System.out.println("...LockTable.java");
    System.exit(0);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    LockTable table = null;
    ProdConsQueue queue1 = null;
    ProdConsQueue queue2 = null;
    ProdConsQueue queue3 = null;
    ProdConsQueue queue4 = null;
    LockTableTestThread lttt1 = null;
    LockTableTestThread lttt2 = null;
    LockTableTestThread lttt3 = null;
    LockTableTestThread lttt4 = null;

    try{
      // Test 1
      // Thread 1: Acquire a read lock
      // Thread 2: Acquire a write lock
      // Thread 2: Make sure thread is blocked
      // Thread 1: Release read lock
      // Thread 2: Make sure thread is unblocked
      table = new LockTable();
      queue1 = new ProdConsQueue();
      queue2 = new ProdConsQueue();
      lttt1 = new LockTableTestThread(table, queue1, 1);
      lttt2 = new LockTableTestThread(table, queue2, 2);

      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());

      // Thread 1: Request read lock
      queue1.enqueue(new LTTTReq(LTTTReq.ACQUIRE_READ_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());

      // Thread 2: Request write lock
      queue2.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(lttt2.getIsBlocked());

      // Thread 1: Release read lock
      queue1.enqueue(new LTTTReq(LTTTReq.RELEASE_READ_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());

      // Thread 2: Release write lock
      queue2.enqueue(new LTTTReq(LTTTReq.RELEASE_WRITE_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());

      // Test 2
      // Thread 1: Acquire a read lock
      // Thread 2: Acquire another read lock
      // Thread 3: Acquire a write lock
      // Thread 3: Make sure thread is blocked
      // Thread 1: Release read lock
      // Thread 2: Release read lock
      // Thread 3: Make sure thread is unblocked
      queue3 = new ProdConsQueue();
      lttt3 = new LockTableTestThread(table, queue3, 3);

      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());

      // Thread 1: Request read lock
      queue1.enqueue(new LTTTReq(LTTTReq.ACQUIRE_READ_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());

      // Thread 2: Request read lock
      queue2.enqueue(new LTTTReq(LTTTReq.ACQUIRE_READ_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());

      // Thread 3: Request write lock
      queue3.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(lttt3.getIsBlocked());

      // Thread 1: Release read lock
      queue1.enqueue(new LTTTReq(LTTTReq.RELEASE_READ_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(lttt3.getIsBlocked());

      // Thread 2: Release read lock
      queue2.enqueue(new LTTTReq(LTTTReq.RELEASE_READ_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());

      // Thread 3: Release write lock
      queue3.enqueue(new LTTTReq(LTTTReq.RELEASE_WRITE_LOCK, new ObjId("a")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());

      // Test 3: Acquire locks on unrelated files
      // Thread 1: Acquire a write lock for "o1"
      // Thread 2: Acquire a write lock for "o2"
      // Thread 3: Request a read lock for "o1"
      // Thread 4: Request a read lock for "o2"
      // Thread 3: Make sure thread is blocked
      // Thread 4: Make sure thread is blocked
      // Thread 1: Release the write lock
      // Thread 3: Make sure thread is unblocked
      // Thread 2: Release the write lock
      // Thread 4: Make sure thread is unblocked
      queue4 = new ProdConsQueue();
      lttt4 = new LockTableTestThread(table, queue4, 4);

      // Thread 1: Acquire a write lock for "o1"
      // Thread 2: Acquire a write lock for "o2"
      queue1.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("o1")));
      queue2.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("o2")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());
      assert(!lttt4.getIsBlocked());

      // Thread 3: Request a read lock for "o1"; make sure thread is blocked
      // Thread 4: Request a read lock for "o2"; make sure thread is blocked
      queue3.enqueue(new LTTTReq(LTTTReq.ACQUIRE_READ_LOCK, new ObjId("o1")));
      queue4.enqueue(new LTTTReq(LTTTReq.ACQUIRE_READ_LOCK, new ObjId("o2")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(lttt3.getIsBlocked());
      assert(lttt4.getIsBlocked());

      // Thread 1: Release the write lock for "o1"
      // Thread 3: Make sure thread is unlocked
      queue1.enqueue(new LTTTReq(LTTTReq.RELEASE_WRITE_LOCK, new ObjId("o1")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());
      assert(lttt4.getIsBlocked());

      // Thread 2: Release the write lock for "o2"
      // Thread 4: Make sure thread is unlocked
      queue2.enqueue(new LTTTReq(LTTTReq.RELEASE_WRITE_LOCK, new ObjId("o2")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
      assert(!lttt3.getIsBlocked());
      assert(!lttt4.getIsBlocked());
    }catch(InterruptedException e){
      System.err.println("" + e);
    }
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testDoubleThread(){
    LockTable table = null;
    ProdConsQueue queue1 = null;
    ProdConsQueue queue2 = null;
    ProdConsQueue queue3 = null;
    ProdConsQueue queue4 = null;
    LockTableTestThread lttt1 = null;
    LockTableTestThread lttt2 = null;
    LockTableTestThread lttt3 = null;
    LockTableTestThread lttt4 = null;

    try{
      // Test 1
      // Thread 1: Acquire a write lock for o1
      // Thread 1: Acquire a write lock for o2
      // Thread 2: Acquire a write lock for o1
      // Thread 2: Make sure thread is blocked
      // Thread 2: Acquire a write lock for o2
      // Thread 2: Make sure thread is blocked
      // Thread 1: Release write lock for o2
      // Thread 2: Make sure thread is blocked
      // Thread 1: Release write lock for o1
      // Thread 2: Make sure thread is unblocked
      table = new LockTable();
      queue1 = new ProdConsQueue();
      queue2 = new ProdConsQueue();
      lttt1 = new LockTableTestThread(table, queue1, 1);
      lttt2 = new LockTableTestThread(table, queue2, 2);

      // Thread 1: Acquire write locks for o1 and o2
      queue1.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("o1")));
      queue1.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("o2")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());

      // Thread 2: Acquire write lock for o1
      queue2.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("o1")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(lttt2.getIsBlocked());

      // Thread 2: Acquire write lock for o2
      queue2.enqueue(new LTTTReq(LTTTReq.ACQUIRE_WRITE_LOCK, new ObjId("o2")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(lttt2.getIsBlocked());

      // Thread 1: Release write lock for o2
      queue1.enqueue(new LTTTReq(LTTTReq.RELEASE_WRITE_LOCK, new ObjId("o2")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(lttt2.getIsBlocked());

      // Thread 1: Release write lock for o1
      queue1.enqueue(new LTTTReq(LTTTReq.RELEASE_WRITE_LOCK, new ObjId("o1")));
      Thread.sleep(100);
      assert(!lttt1.getIsBlocked());
      assert(!lttt2.getIsBlocked());
    }catch(InterruptedException e){
      System.err.println("" + e);
    }
  }
}

 /** 
 *  A table of locks for various objects 
 **/ 
class LockTableTestThread implements Runnable{

 /** 
 *  Data members 
 **/ 
  private LockTable lockTable;
  private ProdConsQueue list;
  private boolean isBlocked;
  private int threadNum;

 /** 
 *  Constructor 
 **/ 
  public
  LockTableTestThread(LockTable newLockTable,
                      ProdConsQueue newList,
                      int newThreadNum){
    this.lockTable = newLockTable;
    this.list = newList;
    this.isBlocked = false;
    this.threadNum = newThreadNum;
    (new Thread(this)).start();
  }

 /** 
 *  Main thread method 
 **/ 
  public void
  run(){
    LTTTReq req = null;
    ObjId objId = null;

    while(true){
      try{
        req = (LTTTReq)this.list.dequeue();
        this.setIsBlocked(true);
        objId = req.getObjId();
        if(req.getReqType() == LTTTReq.ACQUIRE_READ_LOCK){
          this.lockTable.acquireReadLock(objId);
        }else if(req.getReqType() == LTTTReq.ACQUIRE_WRITE_LOCK){
          this.lockTable.acquireWriteLock(objId);
        }else if(req.getReqType() == LTTTReq.RELEASE_READ_LOCK){
          this.lockTable.releaseReadLock(objId);
        }else{
          assert(req.getReqType() == LTTTReq.RELEASE_WRITE_LOCK);
          this.lockTable.releaseWriteLock(objId);
        }
      }catch(InterruptedException e){
        System.err.println("" + e);
      }finally{
        this.setIsBlocked(false);
      }
    }
  }

 /** 
 *  Return true if this thread is blocked 
 **/ 
  public synchronized boolean
  getIsBlocked(){
    return(this.isBlocked);
  }

 /** 
 *  Set the value of the blocked flag 
 **/ 
  private synchronized void
  setIsBlocked(boolean newIsBlocked){
    this.isBlocked = newIsBlocked;
  }

 /** 
 *  Return the thread number 
 **/ 
  public int
  getThreadNum(){
    return(this.threadNum);
  }
}


 /** 
 *  Store requests for the LockTableTestThread 
 **/ 
class LTTTReq{

 /** 
 *  Constants 
 **/ 
  public static final int ACQUIRE_READ_LOCK = 1;
  public static final int ACQUIRE_WRITE_LOCK = 2;
  public static final int RELEASE_READ_LOCK = 3;
  public static final int RELEASE_WRITE_LOCK = 4;

 /** 
 *  Data members 
 **/ 
  private int reqType;
  private ObjId objId;

 /** 
 *  Constructor 
 **/ 
  public
  LTTTReq(int newReqType, ObjId newObjId){
    this.reqType = newReqType;
    this.objId = newObjId;
  }

 /** 
 *  Get the object ID 
 **/ 
  public ObjId
  getObjId(){
    return(this.objId);
  }

 /** 
 *  Get the request type 
 **/ 
  public int
  getReqType(){
    return(this.reqType);
  }
}


 /** 
 *  A producer/consumer queue 
 **/ 
class ProdConsQueue{

 /** 
 *  Data members 
 **/ 
  private LinkedList list;

 /** 
 *  Constructor 
 **/ 
  public
  ProdConsQueue(){
    this.list = new LinkedList();
  }

 /** 
 *  Dequeue the next entry (wait until something available) 
 **/ 
  public synchronized Object
  dequeue() throws InterruptedException{
    Object obj = null;

    while(this.list.isEmpty()){
      this.wait();
    }
    obj = this.list.removeFirst();
    return(obj);
  }

 /** 
 *  Enqueue the next entry 
 **/ 
  public synchronized void
  enqueue(Object obj){
    this.list.addLast(obj);
    this.notifyAll();
  }
}
