package code;
 /** 
 *  Implement the Java version of the NFS loopback server 
 **/ 
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.ByteArrayOutputStream;
import java.util.Vector;
import java.util.Collections;

public class PRACTIFS implements NFS2FS{

 /** 
 *  Constants 
 **/ 
  
  //
  // NOTE:
  // don't change the META_SUFFIX, DATA_SUFFIX, ROOT_DIR_NAME
  // otherwise make sure that any overlog files have the same definition
  // currently CODE/case-studies/Pangaea/PangaeaTopology.olg is depending
  // on these definition
  //
  public static String META_SUFFIX = ".meta";
  public static String DATA_SUFFIX = ".data";
  public static String ROOT_DIR_NAME = "/publications/nfsroot/root";
  public static int DEFAULT_READ_BLOCK_SIZE = 2000;
  public static long FSID = 1;
  public static long FS_BLOCK_SIZE = 512;
  public static int MAX_FHCACHE_SIZE = 5000;

  private static final boolean PRINT_ALL_FILE_NAMES = false;
  private static String allFileNames = ROOT_DIR_NAME + META_SUFFIX+":" + ROOT_DIR_NAME + DATA_SUFFIX;
  private static final boolean DEBUG = false;
  protected static final boolean PRINT_METHODS = false;
  protected static final boolean PRINT_METHODS_VERBOSE = false;
  protected static final boolean PRINT_TIME = true;
  protected static final long PRINT_TIME_FILTER = 500;//ignore event takes less than PRINT_TIME_FILTER time
 
 /** 
 *  Data members 
 **/ 
  protected PRACTIFSLocalInterface localInterface;
  protected FHCache fhCache;
  
 /** 
 *  Constructor -- add <makeRoot> parameter to allow construction without 
 *                 creating the root directory 
 **/ 
  public
  PRACTIFS(PRACTIFSLocalInterface newLocalInterface,
           boolean makeRoot){
    Env.dprintln(PRINT_METHODS, "PRACTIFS creating");

    byte[] b = null;

    Env.dprintln(DEBUG, "PRACTIFS creating");
    this.localInterface = newLocalInterface;
    this.fhCache = new FHCache(ROOT_DIR_NAME, MAX_FHCACHE_SIZE);

    //this.fileid = 0;


   //Env.dprintln(DEBUG, "PRACTIFS make root");

    if(makeRoot){
      System.out.println("Making root");
      this.makeRootDirectory();
    }
    Env.dprintln(PRINT_METHODS, "PRACTIFS construction done");
  }

 /** 
 *  Implement the GETATTR NFS method 
 **/ 
  public attrstat
  getAttributes(fhandle fh){
    Env.dprintln(PRINT_METHODS, "-------PRACTIFS getAttributes " + fh);
    long start = System.currentTimeMillis();

    boolean dbgMe = true;
    ObjId prefix = null;
    ObjId metaDataObjId = null;
    PRACTIFSReadLockToken token = null;
    PRACTIFSFileAttributes attr = null;
    attrstat result = null;
    
    try{
      prefix = this.fhCache.getObjIdPrefix(fh);
      metaDataObjId = new ObjId(prefix.getPath() + META_SUFFIX);
      token = this.localInterface.acquireReadLock(metaDataObjId);
      attr = this.readAttributes(token, metaDataObjId);
      result = new attrstat(stat.NFS_OK, attr);
    }catch(BadFileHandleException e){
      // The handle was bad
      result = new attrstat(stat.NFSERR_STALE, null);
    }catch(FileNotFoundException e){
      // The file has already been deleted
      result = new attrstat(stat.NFSERR_STALE, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new attrstat(stat.NFSERR_IO, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((metaDataObjId != null) && (token != null)){
        this.localInterface.releaseReadLock(token, metaDataObjId);
      }
    }
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS getAttributes " + fh + " took "+totalTime + "ms");
    }
    Env.dprintln(PRINT_METHODS, "........PRACTIFS getAttributes " + fh + " done.");
    return(result);
  }

 /** 
 *  Implement the SETATTR NFS method 
 *  NOTE: This method does not currently allow increasing file sizes. 
 *  NOTE: We always set the time to the server time. Need to change this in 
 *  the future 
 *  Note: This method currently ignores the guard parameter. In the future 
 *  it should not do so. 
 *  Note: This method does not return attributes at all. Might want to 
 *  change this in the future. 
 **/ 
  public attrstat
  setAttributes(sattrargs args){
    Env.dprintln(PRINT_METHODS, "---------PRACTIFS setAttributes " + args.getFile());
    long start = System.currentTimeMillis();
    ObjId prefix = null;
    ObjId metaDataObjId = null;
    PRACTIFSWriteLockToken token = null;
    attrstat result = null;
    PRACTIFSFileAttributes attr = null;

    try{
      prefix = this.fhCache.getObjIdPrefix(args.getFile());
      Env.dprinterrln(PRINT_METHODS_VERBOSE,
                      "PRACTIFS: " + Thread.currentThread().getName() +
                      " Before setAttributes(" + prefix +
                      ", " + args.getAttributes());
      metaDataObjId = new ObjId(prefix.getPath() + META_SUFFIX);
      token = this.localInterface.acquireWriteLock(metaDataObjId);
      attr = this.writeAttributes(token, metaDataObjId, args.getAttributes());
      assert attr!=null;
      result = new attrstat(stat.NFS_OK, attr);
      
    }catch(BadFileHandleException e){
      Env.dprintln(DEBUG, "set 1");
      // The file handle was bad
      result = new attrstat(stat.NFSERR_STALE, null);
    }catch(IOException e){
      Env.dprintln(DEBUG, "set 2");
      if(DEBUG){
        e.printStackTrace();
      }
      // Don't know what caused this; generic IO exception
      result = new attrstat(stat.NFSERR_IO, null);
    }catch(InterruptedException e){
      Env.dprintln(DEBUG, "set 3");
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((metaDataObjId != null) && (token != null)){
        this.localInterface.releaseWriteLock(token, metaDataObjId);
      }
    }

    Env.dprinterrln(PRINT_METHODS_VERBOSE,
                    "PRACTIFS: " + Thread.currentThread().getName() +
                    " After setAttributes(" + prefix +
                    ", " + result);
    Env.dprintln(PRINT_METHODS, "..........PRACTIFS setAttributes " + args.getFile() + "done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS setAttributes " + args.getFile()
		   + " took "+totalTime + "ms");
    }
    return(result);
  }

 /** 
 *  Implement the NFS ROOT method 
 *  
 *  Note: This method is now obsolete 
 **/ 
  public void
  root(){
    assert false : "Called obsolete NFS method";
  }

 /** 
 *  Implement the LOOKUP NFS method 
 *  
 *  NOTE: This method does not currently check whether the file being 
 *    opened is really a directory (by checking its attributes). It only 
 *    does a simple test of trying to open it as a directory and 
 *    complaining if it fails. 
 *  
 *  update NOTE: 
 *    if need to make any change in this method,  
 *    make sure make the same change in ../Pangaea/PangaeaFS.lookup() 
 *     
 *    tbd: need a better way to cope with this dependency  
 **/ 
  public diropres
  lookup(diropargs args){
    Env.dprintln(PRINT_METHODS, "----------PRACTIFS lookup " + args.getDir() + " " + args.getName());
    long start = System.currentTimeMillis();
    ObjId dirPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSDirectory dir = null;
    ObjId filePrefix = null;
    fhandle objHandle = null;
    diropres result = null;
    PRACTIFSFileAttributes objAttr = null;
    PRACTIFSFileAttributes dirAttr = null;

    try{
      dirPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireReadLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireReadLock(dirDataObjId);
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("" + dirPrefix + " not a directory");
      }
      dir = this.readDir(dirDataToken, dirDataObjId);
      objIdPrefix = dir.getEntry(args.getName()).getObjIdPrefix();
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objMetaToken = this.localInterface.acquireReadLock(objMetaObjId);

      objHandle = this.fhCache.getFileHandle(objIdPrefix);
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      result = new diropres(stat.NFS_OK, objHandle, objAttr);
    }catch(BadFileHandleException e){
      // Bad directory handle
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(IsNotDirectoryException e){
      // This object is not a directory
      result = new diropres(stat.NFSERR_NOTDIR, null, null);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(ObjNotInDirectoryException e){
      // Directory has no problem; the file just doesn't exist
      result = new diropres(stat.NFSERR_NOENT, null, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new diropres(stat.NFSERR_IO, null, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseReadLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseReadLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseReadLock(dirMetaToken, dirMetaObjId);
      }
    }

    
    Env.dprintln(PRINT_METHODS, ".........PRACTIFS lookup " + args.getDir() + " " + args.getName()
      + " done.");
    
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS lookup " + args.getDir() + " " + args.getName()
		   + " took "+totalTime + "ms");
    }

    return(result);
  }

 /** 
 *  Implement the READLINK NFS method 
 *  
 *  NOTE: This method does not currently correctly check whether the object 
 *    is in fact a symbolic link. It should check with the file attributes. 
 **/ 
  public readlinkres
  readlink(fhandle fh){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "-----------------PRACTIFS readlink " + fh);
    ObjId symPrefix = null;
    ObjId symlinkDataObjId = null;
    String symlinkDataStr = null;
    PRACTIFSReadLockToken symlinkToken = null;
    readlinkres result = null;

    try{
      symPrefix = this.fhCache.getObjIdPrefix(fh);
      symlinkDataObjId = new ObjId(symPrefix.getPath() + DATA_SUFFIX);
      symlinkToken = this.localInterface.acquireReadLock(symlinkDataObjId);
      symlinkDataStr = this.readSymlinkData(symlinkToken, symlinkDataObjId);
      result = new readlinkres(stat.NFS_OK, symlinkDataStr);
    }catch(BadFileHandleException e){
      // Got a bad file handlee
      result = new readlinkres(stat.NFSERR_STALE, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new readlinkres(stat.NFSERR_IO, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if(symlinkToken != null){
        assert(symPrefix != null);
        this.localInterface.releaseReadLock(symlinkToken, symPrefix);
      }
    }
    Env.dprintln(PRINT_METHODS, "...........PRACTIFS readlink " + fh + "done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS readlink " + fh
		   + " took "+totalTime + "ms");
    }
    return(result);
  }

 /** 
 *  Implement the READ NFS method 
 *  
 *  NOTE: This method does not check whether the read is to a real file 
 *    or not. Need to change this in the future. 
 *  Note: This method makes a copy of the data bytes. This could be made  
 *    more efficient in the future (though it requires handling with care). 
 **/ 
  public readres
  read(readargs args){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "--------PRACTIFS read " + args.getFile());
    BodyMsg bodyMsg = null;
    ObjId objIdPrefix = null;
    ObjId objDataObjId = null;
    ObjId objMetaObjId = null;
    PRACTIFSReadLockToken objDataToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    long size = 0;
    long numBytesToRead = 0;
    long numBytesRead = 0;
    long length = 0;
    byte[] data = null;
    byte[] b = null;
    PRACTIFSFileAttributes objAttr = null;
    readres result = null;
    int count = 0;
    try{
      objIdPrefix = this.fhCache.getObjIdPrefix(args.getFile());
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      objMetaToken = this.localInterface.acquireReadLock(objMetaObjId);
      objDataToken = this.localInterface.acquireReadLock(objDataObjId);
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      size = objAttr.getSize();
      
      if(args.getOffset() < size){
        numBytesToRead = (long)args.getCount();
        if((args.getOffset() + numBytesToRead) > size){
          numBytesToRead = (long)(size - args.getOffset());
        }
        data = new byte[(int)numBytesToRead];
        numBytesRead = 0;
	
        while(numBytesRead < numBytesToRead){
	  count++;
	  try{
	    long readStart = System.currentTimeMillis();
	    bodyMsg = this.localInterface.read(objDataObjId,
					       args.getOffset() + numBytesRead,
					       numBytesToRead - numBytesRead);
	    long readEnd = System.currentTimeMillis();
	    if((readEnd-readStart) > PRINT_TIME_FILTER ){
	      Env.dprintln(PRINT_TIME, "li.read " + args.getFile() + " size=" + size
			   + " off= " + (args.getOffset()+numBytesRead) + " len=" + bodyMsg.getLength()
			   + " took "+(readEnd-readStart) + "ms");
	    }
	  }catch(ReadOfHoleException rhe){
	    long nextOffset=rhe.getNextValidBytePosition();
	    length = nextOffset -( args.getOffset()+ numBytesRead);//num of holes
	    if(length > (numBytesToRead - numBytesRead)){
	      length = numBytesToRead - numBytesRead;
	    }
	    for(int i = 0; i < length; i++){
	      data[(int)(numBytesRead+i)] = 0;//hole value
	    }
	    numBytesRead += length;
	    continue;
	  }
	  
	  
          assert(bodyMsg.getObjInvalTarget().getOffset() ==
                 (args.getOffset() + numBytesRead));
          b = bodyMsg.getBody().dangerousGetReferenceToInternalByteArray();
          length = b.length;
          if(length > (numBytesToRead - numBytesRead)){
            length = numBytesToRead - numBytesRead;
          }
          System.arraycopy(b,
                           0,
                           data,
                           (int)numBytesRead,
                           (int)length);
          b = null;
          numBytesRead += length;
        }
        result = new readres(stat.NFS_OK, objAttr, data);
      }else{
        // Tried to read outside byte range; allow this but specify 0 bytes
        // read (as per NFS specification)
        data = new byte[0];
      }
      result = new readres(stat.NFS_OK, objAttr, data);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = new readres(stat.NFSERR_STALE, null, null);
    }catch(ObjNotFoundException e){
      // File has already been deleted
      result = new readres(stat.NFSERR_STALE, null, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new readres(stat.NFSERR_IO, null, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      // Release the data lock
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseReadLock(objDataToken, objDataObjId);
      }
      // Release the metadata lock
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseReadLock(objMetaToken, objMetaObjId);
      }
    }
    Env.dprintln(PRINT_METHODS, ".........PRACTIFS read " + args.getFile() + " done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS read " + args.getFile() + " size=" + size
		   + " off= " + args.getOffset() + " len=" + args.getCount()
		   + " took "+totalTime + "ms" + "total li.read(): " +count);
    }
    return(result);
  }

 /** 
 *  Implement the WRITECACHE NFS method 
 *  
 *  Note: This method is now obsolete 
 **/ 
  public void
  writeCache(){
    assert false : "Called obsolete NFS method";
  }

 /** 
 *  Implement the WRITE NFS method 
 *  
 *  NOTE: This method does not change any attributes except the size; 
 *    need to change this 
 *  NOTE: This method does not check whether the write is to a real file 
 *    or not. Need to change this in the future. 
 *  NOTE: This method currently ignores the stability flag and makes all 
 *    writes DATA_SYNC stable 
 *  Note: This method makes a copy of the data bytes. This could be made  
 *    more efficient in the future (though it requires handling with care). 
 **/ 
  public attrstat
  write(fhandle file,
        long beginOffset,
        long offset,
        long totalCount,
        byte[] data){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "--------------------PRACTIFS write " + file);
    ObjId objIdPrefix = null;
    ObjId objDataObjId = null;
    ObjId objMetaObjId = null;
    PRACTIFSReadLockToken objDataToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] metaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes objAttr = null;
    attrstat result = null;

    try{
      objIdPrefix = this.fhCache.getObjIdPrefix(file);
      Env.dprinterrln(PRINT_METHODS_VERBOSE,
                      "PRACTIFS: " + Thread.currentThread().getName() +
                      " Before write(" + objIdPrefix +
                      ", " + offset +
                      ", " + data.length + ")");
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);
      fswArr = new FSWriteEntry[2];

      // Set new attributes
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      if((offset + (long)data.length) > objAttr.getSize()){
        objAttr.setSize(offset + (long)data.length);
        long now = System.currentTimeMillis();
        objAttr.setATime(new timeval(now));
        objAttr.setMTime(new timeval(now));
        
      }
      metaBytes = this.getAttributeBytes(objAttr);
      fswArr[0] = new FSWriteEntry(objMetaObjId,
                                   0,
                                   metaBytes.length,
                                   new ImmutableBytes(metaBytes),
                                   false);

      // Write the actual data
      fswArr[1] = new FSWriteEntry(objDataObjId,
                                   offset,
                                   (long)data.length,
                                   new ImmutableBytes(data),
                                   false);

      // Perform the write
      Env.dprinterrln(DEBUG, "PRACTIFS: Calling localInterface.write()");
      writeLengths = this.localInterface.write(fswArr);
      Env.dprinterrln(DEBUG, "PRACTIFS: Called localInterface.write()");
      result = new attrstat(stat.NFS_OK, objAttr);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = new attrstat(stat.NFSERR_STALE, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new attrstat(stat.NFSERR_IO, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
    }
    Env.dprinterrln(PRINT_METHODS_VERBOSE,
                    "PRACTIFS: " + Thread.currentThread().getName() +
                    " After write(" + objIdPrefix +
                    ", " + result);
    Env.dprintln(PRINT_METHODS, "..........PRACTIFS write " + file + " done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
        if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS write " + file
		   + " off= " + offset + " len=" + totalCount
		   + " took "+totalTime + "ms");
    }
    return result;
  }

 /** 
 *  Implement the CREATE NFS method 
 *  
 *  NOTE: This method does not check whether the write is to an existing 
 *    file or not. Need to change this in the future. 
 *  NOTE: This method does not work for the EXCLUSIVE createhow switch 
 *  NOTE: This method does not change the directory attributes timestamp 
 *  NOTE: The newly created file attributes give everyone permission to 
 *    do everything 
 **/ 
  public diropres
  create(createargs args){
    long start = System.currentTimeMillis();
    boolean dbgMe = false;
    Env.dprintln(PRINT_METHODS,
                 "-----------PRACTIFS create " 
                 + args.getWhere().getDir() + " " + args.getWhere().getName());
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes objAttr = null;
    PRACTIFSFileAttributes dirAttr = null;
    //long now = 0;
    ImmutableBytes dirIB = null;
    fhandle objFH = null;
    diropres result = null;
    long numBlocks = 0;

    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getWhere().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      objIdPrefix = this.makeNewObjIdPrefix(dirDataObjId,
                                            args.getWhere().getName());
      Env.dprinterrln(PRINT_METHODS_VERBOSE||dbgMe,
                      "PRACTIFS: " + Thread.currentThread().getName() +
                      " Before create(" + dirIdPrefix +
                      ", " + args.getWhere().getName() +
                      ", " + args.getAttributes() +
                      ")");
      Env.dprinterrln(PRINT_METHODS_VERBOSE||dbgMe,
                      "PRACTIFS: " + Thread.currentThread().getName() +
                      " creating " + objIdPrefix);

      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);
      
      Env.dprintln(DEBUG||dbgMe, "grabed all the lock");
      objFH = this.fhCache.getFileHandle(objIdPrefix);
      
      fswArr = new FSWriteEntry[4];

      // Read the directory, add the entry, and prepare to write it again
      Env.dprintln(DEBUG||dbgMe, "***readAttributes "+ dirMetaObjId);
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      Env.dprintln(DEBUG||dbgMe, "***readAttributes "+ dirMetaObjId + " done");
      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      
      dir = this.readDir(dirDataToken, dirDataObjId);
      
      PRACTIFSDirEntry dirEntry = createDirEntry(objIdPrefix);
      dir.addEntry(args.getWhere().getName(), 
                   dirEntry);
      
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      fswArr[0] = new FSWriteEntry(dirDataObjId,
                                     0,
                                     dir.getSize(),
                                     dirIB,
                                     false);

      // Prepare to write the new directory attributes
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                   0,
                                   dirMetaBytes.length,
                                   new ImmutableBytes(dirMetaBytes),
                                   false);
      
      // Write the new file attributes

      if((args.getAttributes().getSize() <= Integer.MAX_VALUE) &&
         (args.getAttributes().getSize() >= 0)){
        numBlocks = ((args.getAttributes().getSize() + FS_BLOCK_SIZE - 1) /
                     FS_BLOCK_SIZE);
      }
      objAttr = createFileAttributes(args.getAttributes(),
                                     ftype.NFREG,
                                     0, // size
                                     numBlocks, // blocks
                                     (long)objDataObjId.hashCode(), // fileid
                                     dirIdPrefix);//this file's parrent

      objMetaBytes = this.getAttributeBytes(objAttr);
      fswArr[2] = new FSWriteEntry(objMetaObjId,
                                   0,
                                   objMetaBytes.length,
                                   new ImmutableBytes(objMetaBytes),
                                   false);

      fswArr[3] = new FSWriteEntry(objDataObjId,
                                   0,
                                   1, // 1 byte
                                   new ImmutableBytes(new byte[1]),
                                   false);  // not delete

      Env.dprintln(dbgMe, "44444444");
      // Perform the writes
      writeLengths = this.localInterface.write(fswArr);
      Env.dprintln(dbgMe, "555555555");
      result = new diropres(stat.NFS_OK, objFH, objAttr);
    }catch(BadFileHandleException e){
      Env.dprintln(DEBUG, "BadFileHandleException");
      // Got a bad file handle
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(IsNotDirectoryException e){
      Env.dprintln(DEBUG, "IsNotDirectoryException");
      // Trying to add to a non-directory
      result = new diropres(stat.NFSERR_NOTDIR, null, null);
    }catch(ObjAlreadyPresentException e){
      // Adding an existing file to a directory
      Env.dprintln(DEBUG, "ObjAlreadyPresentException");
      result = new diropres(stat.NFSERR_EXIST, null, null);
    }catch(FileNotFoundException e){
      Env.dprintln(DEBUG, "FileNotFoundException");
      // Handle for directory exists, but directory doesn't
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(IOException e){
      Env.dprintln(DEBUG, "IOException ");
      e.printStackTrace();
      // Don't know what caused this; generic IO exception
      result = new diropres(stat.NFSERR_IO, null, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    Env.dprinterrln(PRINT_METHODS_VERBOSE||dbgMe,
                    "PRACTIFS: " + Thread.currentThread().getName() +
                    " After create(" + dirIdPrefix +
                    ", " + result +
                    ")");
    Env.dprintln(PRINT_METHODS,
                 "........PRACTIFS create " + args.getWhere().getDir() + " " 
                 + args.getWhere().getName() + " done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS create " + args.getWhere().getDir() + " " 
                 + args.getWhere().getName()
		   + " took "+totalTime + "ms");
    }
    return result;
  }

 /** 
 *  Implement the REMOVE NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public int
  remove(diropargs args){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "---------PRACTIFS remove " + args.getDir() + " " + args.getName());
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory obj = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes objAttr = null;
    PRACTIFSFileAttributes dirAttr = null;
    long now = 0;
    int result = 0;
    ImmutableBytes dirIB = null;

    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);

      fswArr = new FSWriteEntry[4];

      // Read the outer directory, remove the file, and prepare to write
      // it again
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      dir = this.readDir(dirDataToken, dirDataObjId);
      objIdPrefix = dir.getEntry(args.getName()).getObjIdPrefix();
      dir.removeEntry(args.getName());
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      fswArr[0] = new FSWriteEntry(dirDataObjId,
                                     0,
                                     dir.getSize(),
                                     dirIB,
                                     false);

      // Prepare to write the new directory attributes
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                     0,
                                     dirMetaBytes.length,
                                     new ImmutableBytes(dirMetaBytes),
                                     false);

      // Read the file metadata to know how many bytes to delete, and then
      // delete the file
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      fswArr[2] = new FSWriteEntry(objDataObjId,
                                     0,
                                     objAttr.getSize(),
                                     null,
                                     true);

      // Delete the file metadata
      objMetaBytes = this.getAttributeBytes(objAttr);
      fswArr[3] = new FSWriteEntry(objMetaObjId,
                                   0,
                                   objMetaBytes.length,
                                   null,
                                   true);

      // Perform the writes
      writeLengths = this.localInterface.write(fswArr);
      result = stat.NFS_OK;
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = stat.NFSERR_STALE;
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      result = stat.NFSERR_NOTDIR;
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      result = stat.NFSERR_NOENT;
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = stat.NFSERR_STALE;
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = stat.NFSERR_IO;
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    Env.dprintln(PRINT_METHODS, "........PRACTIFS remove " + args.getDir() 
                 + " " + args.getName() + " done. ");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS remove " + args.getDir() + " " 
                 + args.getName()
		   + " took "+totalTime + "ms");
    }
    return result;
  }


 /** 
 *  Implement the RENAME NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public int
  rename(renameargs args){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "---------PRACTIFS remove file " + args.getFrom().getName() 
                 + " from " + args.getFrom().getDir()
                 + " to dir " + args.getTo().getDir()+ " file: " +args.getTo().getName());
    int result = 0;

    if(args.getFrom().getDir().equals(args.getTo().getDir())){
      result = this.renameSameDir(args);
    }else{
      result = this.renameAcrossDirs(args);
    }
    Env.dprintln(PRINT_METHODS, "...........PRACTIFS rename" + args.getFrom().getName() 
                 + " from " + args.getFrom().getDir()
                 + " to " + args.getTo().getDir() + " done.");
    
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS rename " + args.getFrom().getName() 
                 + " from " + args.getFrom().getDir()
                 + " to " + args.getTo().getDir()
		   + " took "+totalTime + "ms");
    }
    return(result);
  }

 /** 
 *  Implement the RENAME NFS method when renaming across directories 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  protected int
  renameAcrossDirs(renameargs args){
    
    ObjId fromDirIdPrefix = null;
    ObjId fromDirDataObjId = null;
    ObjId fromDirMetaObjId = null;
    ObjId toDirIdPrefix = null;
    ObjId toDirDataObjId = null;
    ObjId toDirMetaObjId = null;
    PRACTIFSReadLockToken fromDirDataToken = null;
    PRACTIFSReadLockToken fromDirMetaToken = null;
    PRACTIFSReadLockToken toDirMetaToken = null;
    PRACTIFSReadLockToken toDirDataToken = null;
    PRACTIFSDirectory fromDir = null;
    PRACTIFSDirectory toDir = null;
    ImmutableBytes fromDirIB = null;
    ImmutableBytes toDirIB = null;
    PRACTIFSFileAttributes fromDirAttr = null;
    PRACTIFSFileAttributes toDirAttr = null;
    byte[] fromDirMetaBytes = null;
    byte[] toDirMetaBytes = null;
    ObjId objIdPrefix = null;
    FSWriteEntry[] fswArr = null;
    long[] writeLengths = null;
    int result = 0;

    try{
      fromDirIdPrefix = this.fhCache.getObjIdPrefix(args.getFrom().getDir());
      fromDirMetaObjId = new ObjId(fromDirIdPrefix.getPath() + META_SUFFIX);
      fromDirDataObjId = new ObjId(fromDirIdPrefix.getPath() + DATA_SUFFIX);
      toDirIdPrefix = this.fhCache.getObjIdPrefix(args.getTo().getDir());
      toDirMetaObjId = new ObjId(toDirIdPrefix.getPath() + META_SUFFIX);
      toDirDataObjId = new ObjId(toDirIdPrefix.getPath() + DATA_SUFFIX);

      // TODO: Check if the to directory is the same as the from directory

      fromDirMetaToken =
        this.localInterface.acquireWriteLock(fromDirMetaObjId);
      fromDirDataToken =
        this.localInterface.acquireWriteLock(fromDirDataObjId);
      toDirMetaToken = this.localInterface.acquireWriteLock(toDirMetaObjId);
      toDirDataToken = this.localInterface.acquireWriteLock(toDirDataObjId);

      fswArr = new FSWriteEntry[4];

      // Read the source directory, remove the entry, and prepare to write
      // it again
      fromDirAttr = this.readAttributes(fromDirMetaToken, fromDirMetaObjId);
      if(fromDirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      fromDir = this.readDir(fromDirDataToken, fromDirDataObjId);
      objIdPrefix = fromDir.getEntry(args.getFrom().getName()).getObjIdPrefix();
      fromDir.removeEntry(args.getFrom().getName());
      fromDirIB = new ImmutableBytes(fromDir.dangerousGetBytes());
      fswArr[0] = new FSWriteEntry(fromDirDataObjId,
                                   0,
                                   fromDir.getSize(),
                                   fromDirIB,
                                   false);

      // Prepare to write the source directory attributes
      fromDirAttr.setSize(fromDir.getSize());
      fromDirMetaBytes = this.getAttributeBytes(fromDirAttr);
      fswArr[1] = new FSWriteEntry(fromDirMetaObjId,
                                   0,
                                   fromDirMetaBytes.length,
                                   new ImmutableBytes(fromDirMetaBytes),
                                   false);

      // Read the target directory, add the entry, and prepare to write
      // it again
      toDir = this.readDir(toDirDataToken, toDirDataObjId);
      PRACTIFSDirEntry dirEntry = createDirEntry(objIdPrefix);
      
      try{
        toDir.addEntry(args.getTo().getName(), dirEntry);
        //in linux, it is allowed to overwritten an existing file by rename
      }catch(ObjAlreadyPresentException e){
        // Moving a file to a location where it already exists

          toDir.removeEntry(args.getTo().getName());
          //retry
          try{
            toDir.addEntry(args.getTo().getName(), dirEntry);
          }catch(ObjAlreadyPresentException ee){
          //impossible
            assert false;
          }

      }
      /*
       replacing this one with above by zjd
       in linux, it is ok to remove a file to an existing file
       e.g. andrew benchmark phase V

      toDir.addEntry(args.getTo().getName(), 
                     dirEntry);
      */
      toDirIB = new ImmutableBytes(toDir.dangerousGetBytes());
      fswArr[2] = new FSWriteEntry(toDirDataObjId,
                                   0,
                                   toDir.getSize(),
                                   toDirIB,
                                   false);

      // Prepare to write the target directory attributes
      toDirAttr = this.readAttributes(toDirMetaToken, toDirMetaObjId);
      if(toDirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      toDirAttr.setSize(toDir.getSize());
      toDirMetaBytes = this.getAttributeBytes(toDirAttr);
      fswArr[3] = new FSWriteEntry(toDirMetaObjId,
                                   0,
                                   toDirMetaBytes.length,
                                   new ImmutableBytes(toDirMetaBytes),
                                   false);

      // Perform the writes
      writeLengths = this.localInterface.write(fswArr);
      result = stat.NFS_OK;
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = stat.NFSERR_STALE;
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      result = stat.NFSERR_NOTDIR;
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      result = stat.NFSERR_NOENT;
    /*
     commented by zjd, see comments inside the method
    }catch(ObjAlreadyPresentException e){
      // Moving a file to a location where it already exists
      result = stat.NFSERR_EXIST;
    */
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = stat.NFSERR_STALE;
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = stat.NFSERR_IO;
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((toDirDataObjId != null) && (toDirDataToken != null)){
        this.localInterface.releaseWriteLock(toDirDataToken, toDirDataObjId);
      }
      if((toDirMetaObjId != null) && (toDirMetaToken != null)){
        this.localInterface.releaseWriteLock(toDirMetaToken, toDirMetaObjId);
      }
      if((fromDirDataObjId != null) && (fromDirDataToken != null)){
        this.localInterface.releaseWriteLock(fromDirDataToken,
                                             fromDirDataObjId);
      }
      if((fromDirMetaObjId != null) && (fromDirMetaToken != null)){
        this.localInterface.releaseWriteLock(fromDirMetaToken,
                                             fromDirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the RENAME NFS method when renaming within the same directory 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  protected int
  renameSameDir(renameargs args){
    ObjId dirIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSDirectory dir = null;
    ImmutableBytes dirIB = null;
    PRACTIFSFileAttributes dirAttr = null;
    byte[] dirMetaBytes = null;
    ObjId objIdPrefix = null;
    FSWriteEntry[] fswArr = null;
    long[] writeLengths = null;
    int result = 0;

    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getFrom().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);

      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);

      fswArr = new FSWriteEntry[2];

      // Read the source directory, remove the entry, and prepare to write
      // it again
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      dir = this.readDir(dirDataToken, dirDataObjId);
      objIdPrefix = dir.getEntry(args.getFrom().getName()).getObjIdPrefix();
      dir.removeEntry(args.getFrom().getName());
      PRACTIFSDirEntry dirEntry = createDirEntry(objIdPrefix);
      try{
        dir.addEntry(args.getTo().getName(), dirEntry);
        //in linux, it is allowed to overwritten an existing file by rename
      }catch(ObjAlreadyPresentException e){
        // Moving a file to a location where it already exists
        
          dir.removeEntry(args.getTo().getName());
	  //retry
	  try{
	    dir.addEntry(args.getTo().getName(), dirEntry);
          }catch(ObjAlreadyPresentException ee){
	  //impossible
	    assert false;
          }

      }

      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      fswArr[0] = new FSWriteEntry(dirDataObjId,
                                   0,
                                   dir.getSize(),
                                   dirIB,
                                   false);

      // Prepare to write the source directory attributes
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                   0,
                                   dirMetaBytes.length,
                                   new ImmutableBytes(dirMetaBytes),
                                   false);

      // Perform the writes
      writeLengths = this.localInterface.write(fswArr);
      result = stat.NFS_OK;
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = stat.NFSERR_STALE;
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      result = stat.NFSERR_NOTDIR;
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      result = stat.NFSERR_NOENT;
    /*
    //commented by zjd
    //in linux, it is allowed to overwritten an existing file by rename
    }catch(ObjAlreadyPresentException e){
      // Moving a file to a location where it already exists
      result = stat.NFSERR_EXIST;
    */
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = stat.NFSERR_STALE;
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = stat.NFSERR_IO;
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    return result;
  }

 /** 
 *  Implement the LINK NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public int
  link(linkargs args){
    return(stat.NFSERR_IO);
  }

 /** 
 *  Implement the SYMLINK NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 **/ 
  public int
  symlink(symlinkargs args){
    return(stat.NFSERR_IO);
  }

 /** 
 *  Implement the MKDIR NFS method 
 *  
 *  NOTE: This method does not check whether the write is to an existing 
 *    file or not. Need to change this in the future. 
 *  NOTE: This method does not change the directory attributes timestamp 
 *  NOTE: The newly created directory attributes give everyone permission to 
 *    do everything 
 **/ 
  public diropres
  mkdir(createargs args){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "------------PRACTIFS mkdir " + args.getWhere().getName() 
                 + " in " + args.getWhere().getDir());
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory obj = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes objAttr = null;
    PRACTIFSFileAttributes dirAttr = null;
    diropres result = null;
    //long now = 0;
    ImmutableBytes dirIB = null;
    ImmutableBytes objIB = null;
    fhandle objFH = null;
    long numBlocks = 0;

    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getWhere().getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      objIdPrefix = this.makeNewObjIdPrefix(dirDataObjId,
                                            args.getWhere().getName());
      Env.dprinterrln(PRINT_METHODS_VERBOSE,
                      "PRACTIFS: " + Thread.currentThread().getName() +
                      " Before mkdir(" + dirIdPrefix +
                      ", " + args.getWhere().getName() +
                      ", " + args.getAttributes() +
                      ")");

      /*
        objIdPrefix = new ObjId(dirDataObjId.getPath() +
        "/" +
        args.getWhere().getName());
      */
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);
      objFH = this.fhCache.getFileHandle(objIdPrefix);

      fswArr = new FSWriteEntry[4];

      // Read the outer directory, add the entry, and prepare to write it again
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      dir = this.readDir(dirDataToken, dirDataObjId);
      PRACTIFSDirEntry dirEntry = createDirEntry(objIdPrefix);
      dir.addEntry(args.getWhere().getName(), dirEntry);
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      fswArr[0] = new FSWriteEntry(dirDataObjId,
                                   0,
                                   dir.getSize(),
                                   dirIB,
                                   false);

      // Prepare to write the new outer directory attributes
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                   0,
                                   dirMetaBytes.length,
                                   new ImmutableBytes(dirMetaBytes),
                                   false);

      // Write the inner directory data
      obj = new PRACTIFSDirectory();
      objIB = new ImmutableBytes(obj.dangerousGetBytes());
      fswArr[2] = new FSWriteEntry(objDataObjId,
                                   0,
                                   obj.getSize(),
                                   objIB,
                                   false);

      // Write the new inner directory attributes
      if((args.getAttributes().getSize() <= Integer.MAX_VALUE) &&
         (args.getAttributes().getSize() >= 0)){
        numBlocks = ((args.getAttributes().getSize() + FS_BLOCK_SIZE - 1) /
                     FS_BLOCK_SIZE);
      }else{
        numBlocks = (obj.getSize() + FS_BLOCK_SIZE - 1) / FS_BLOCK_SIZE;
      }
      objAttr = createFileAttributes(args.getAttributes(),
                                     ftype.NFDIR,
                                     obj.getSize(),//size
                                     numBlocks, //blocks
                                     (long)objDataObjId.hashCode(),//fileid
                                     dirIdPrefix);//outer dir
      objMetaBytes = this.getAttributeBytes(objAttr);
      fswArr[3] = new FSWriteEntry(objMetaObjId,
                                   0,
                                   objMetaBytes.length,
                                   new ImmutableBytes(objMetaBytes),
                                   false);

      // Perform the writes
      writeLengths = this.localInterface.write(fswArr);
      result = new diropres(stat.NFS_OK, objFH, objAttr);
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      result = new diropres(stat.NFSERR_NOTDIR, null, null);
    }catch(ObjAlreadyPresentException e){
      // Adding an existing file to a directory
      result = new diropres(stat.NFSERR_EXIST, null, null);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new diropres(stat.NFSERR_IO, null, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    Env.dprinterrln(PRINT_METHODS_VERBOSE,
                    "PRACTIFS: " + Thread.currentThread().getName() +
                    " After mkdir(" + dirIdPrefix +
                    ", " + result +
                    ")");
    Env.dprintln(PRINT_METHODS, "..........PRACTIFS mkdir " + args.getWhere().getName() 
                 + " in " + args.getWhere().getDir() + " done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS mkdir " + args.getWhere().getName() 
                 + " in " + args.getWhere().getDir() 
		   + " took "+totalTime + "ms");
    }
    return result;
  }

 /** 
 *  Implement the RMDIR NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public int
  rmdir(diropargs args){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "--------------PRACTIFS rmdir " + args.getName() 
                 + " from " + args.getDir());
    ObjId dirIdPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory obj = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSReadLockToken objDataToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] dirMetaBytes = null;
    byte[] objMetaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes objAttr = null;
    PRACTIFSFileAttributes dirAttr = null;
    ImmutableBytes dirIB = null;
    int result = 0;

    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);

      fswArr = new FSWriteEntry[4];

      // Read the outer directory, remove the entry, and prepare to write
      // it again
      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);
      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("Not a directory");
      }
      dir = this.readDir(dirDataToken, dirDataObjId);
      objIdPrefix = dir.getEntry(args.getName()).getObjIdPrefix();
      dir.removeEntry(args.getName());
      dirIB = new ImmutableBytes(dir.dangerousGetBytes());
      fswArr[0] = new FSWriteEntry(dirDataObjId,
                                   0,
                                   dir.getSize(),
                                   dirIB,
                                   false);

      // Prepare to write the new outer directory attributes
      dirAttr.setSize(dir.getSize());
      dirMetaBytes = this.getAttributeBytes(dirAttr);
      fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                   0,
                                   dirMetaBytes.length,
                                   new ImmutableBytes(dirMetaBytes),
                                   false);

      // Read the inner directory metadata to know how many bytes to
      // delete, and then delete the directory
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
      objMetaToken = this.localInterface.acquireWriteLock(objMetaObjId);
      objDataToken = this.localInterface.acquireWriteLock(objDataObjId);
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      fswArr[2] = new FSWriteEntry(objDataObjId,
                                   0,
                                   objAttr.getSize(),
                                   null,
                                   true);
      // Make sure the inner directory is empty
      obj = this.readDir(objDataToken, objDataObjId);
      if(!obj.isEmpty()){
        throw new NotEmptyException("" + objIdPrefix + " is not empty");
      }

      // Delete the file metadata
      objMetaBytes = this.getAttributeBytes(objAttr);
      fswArr[3] = new FSWriteEntry(objMetaObjId,
                                   0,
                                   objMetaBytes.length,
                                   null,
                                   true);

      // Perform the writes
      writeLengths = this.localInterface.write(fswArr);
      result = stat.NFS_OK;
    }catch(BadFileHandleException e){
      // Got a bad file handle
      result = stat.NFSERR_STALE;
    }catch(IsNotDirectoryException e){
      // Trying to add to a non-directory
      result = stat.NFSERR_NOTDIR;
    }catch(ObjNotInDirectoryException e){
      // Removing a non-existant file from the directory
      result = stat.NFSERR_NOENT;
    }catch(NotEmptyException e){
      // Removing a non-empty directory
      result = stat.NFSERR_NOTEMPTY;
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = stat.NFSERR_STALE;
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = stat.NFSERR_IO;
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objDataObjId != null) && (objDataToken != null)){
        this.localInterface.releaseWriteLock(objDataToken, objDataObjId);
      }
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseWriteLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
    Env.dprintln(PRINT_METHODS, "............PRACTIFS rmdir " + args.getName() 
                 + " from " + args.getDir() + " done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS rmdir " + args.getName() 
                 + " from " + args.getDir()
		   + " took "+totalTime + "ms");
    }
    return result;
  }

 /** 
 *  Implement the READDIR NFS method 
 *  
 *  NOTE: This method isn't currently implemented 
 *  NOTE: We currently ignore the cookieverf token 
 *  NOTE: We currently ignore the usage of cookies (cookies must be 0) 
 **/ 
  public readdirres
  readdir(readdirargs args){
    long start = System.currentTimeMillis();
    Env.dprintln(PRINT_METHODS, "-------PRACTIFS readdir " + args.getDir());
    ObjId dirObjIdPrefix = null;
    ObjId dirMetaObjId = null;
    ObjId dirDataObjId = null;
    ObjId objIdPrefix = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory.DirIterator it = null;
    entry ent = null;
    String name = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    long cookie = 0;
    readdirres result = null;
    Vector entryVec = null;

    try{
      dirObjIdPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirObjIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirObjIdPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireReadLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireReadLock(dirDataObjId);
      dir = this.readDir(dirDataToken, dirDataObjId);
      it = dir.getIterator();
      cookie = args.getCookie();
      assert(cookie == 0);  // For now!
      entryVec = new Vector();
      while(it.hasNext()){
        name = it.getNext();
        entryVec.add(name);
      }
      Collections.sort(entryVec);
      cookie = entryVec.size() + 3; // Room for ".", "..", compensate for --
      ent = null;
      for(int i = entryVec.size() - 1; i >= 0; i--){
        cookie--;
        name = (String)entryVec.get(i);
        try{
          objIdPrefix = dir.getEntry(name).getObjIdPrefix();
        }catch(ObjNotInDirectoryException e){
          // This cannot happen
          assert(false);
        }
        ent = new entry((long)objIdPrefix.hashCode(), name, cookie, ent);
      }
      cookie--;
      ent = new entry((long)(new ObjId("..")).hashCode(), "..", cookie, ent);
      cookie--;
      ent = new entry((long)(new ObjId(".")).hashCode(), ".", cookie, ent);
      result = new readdirres(stat.NFS_OK, ent, true);
    }catch(BadFileHandleException e){
      // Got a bad file handle for the directory
      result = new readdirres(stat.NFSERR_STALE, null, true);
    }catch(IsNotDirectoryException e){
      // Got a file handle that wasn't for a directory
      result = new readdirres(stat.NFSERR_NOTDIR, null, true);
    }catch(IOException e){
      // Don't know what caused this; generic IOException
      result = new readdirres(stat.NFSERR_IO, null, true);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseReadLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseReadLock(dirMetaToken, dirMetaObjId);
      }
    }
    Env.dprintln(PRINT_METHODS, "......PRACTIFS readdir " + args.getDir() + " done.");
    long end = System.currentTimeMillis();
    long totalTime = end-start;
    if(totalTime > PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS readdir " + args.getDir()
		   + " took "+totalTime + "ms");
    }
    return(result);
  }

 /** 
 *  Implement the FSSTAT NFS method 
 *  
 *  NOTE: This method isn't currently implemented properly 
 **/ 
  public statfsres
  statfs(fhandle fh){
    Env.dprintln(PRINT_METHODS, "-------PRACTIFS statfs " + fh.toString());
    statfsres result = null;

    Env.dprinterrln(PRINT_METHODS_VERBOSE,
                    "PRACTIFS: " + Thread.currentThread().getName() +
                    " Before statfs(" + fh + ")");
    result = new statfsres(stat.NFS_OK,
                           8192, // optimum transfer size
                           512,   // block size
                           0,   // total number of blocks
                           0,   // free number of blocks
                           0);  // available number of blocks
    Env.dprinterrln(PRINT_METHODS_VERBOSE,
                    "PRACTIFS: " + Thread.currentThread().getName() +
                    " After statfs(" + result + ")");
    Env.dprintln(PRINT_METHODS, "......PRACTIFS statfs " + fh.toString() + " done.");
    return(result);
  }

 /** 
 *  Return the attributes for this object 
 **/ 
  protected PRACTIFSFileAttributes
  readAttributes(PRACTIFSReadLockToken token, ObjId metaDataObjId)
    throws IOException{
    PRACTIFSFileAttributes attr = null;
    PRACTIFileInputStream pfis = null;

    assert(token.isAcquired());
    try{
      pfis = new PRACTIFileInputStream(this.localInterface,
                                       metaDataObjId,
                                       PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      //Env.dprintln(DEBUG, "pfis created");
      attr = new PRACTIFSFileAttributes(pfis);
      //Env.dprintln(DEBUG, "attr read succeed");
      //Env.dprintln(DEBUG, "attr.getOtherAttr=" + attr.getOtherAttr()); 
    }finally{
      if(pfis != null){
        //Close the stream
        pfis.close();
      }
    }
    return(attr);
  }

 /** 
 *  Write the attributes for this object 
 **/ 
  protected PRACTIFSFileAttributes
  writeAttributes(PRACTIFSWriteLockToken token,
                  ObjId metaDataObjId,
                  sattr newAttributes) throws IOException{
    PRACTIFSFileAttributes attr = null;
    long modeTypeValue = 0;
    PRACTIFileOutputStream pfos = null;
    ByteArrayOutputStream baos = null;
    byte[] dataBytes = null;

    assert(token.isAcquired());
    try{
      attr = this.readAttributes(token, metaDataObjId);
      if((newAttributes.getMode() <= Integer.MAX_VALUE) &&
         (newAttributes.getMode() >= 0)){
        modeTypeValue = this.getModeTypeValue(attr.getType());
        attr.setMode(newAttributes.getMode() | modeTypeValue);
      }
      if((newAttributes.getUID() <= Integer.MAX_VALUE) &&
         (newAttributes.getUID() >= 0)){
        attr.setUID(newAttributes.getUID());
      }
      if((newAttributes.getGID() <= Integer.MAX_VALUE) &&
         (newAttributes.getGID() >= 0)){
        attr.setGID(newAttributes.getGID());
      }
      if((newAttributes.getSize() <= Integer.MAX_VALUE) &&
         (newAttributes.getSize() >= 0)){
        attr.setSize(newAttributes.getSize());
      }
      if((newAttributes.getATime().getSeconds() <= Integer.MAX_VALUE) &&
         (newAttributes.getATime().getSeconds() >= 0)){
        attr.setATime(newAttributes.getATime());
      }
      if((newAttributes.getMTime().getSeconds() <= Integer.MAX_VALUE) &&
         (newAttributes.getMTime().getSeconds() >= 0)){
        attr.setMTime(newAttributes.getMTime());
      }

      baos = new ByteArrayOutputStream();
      attr.toOutputStream(baos);
      baos.flush();
      dataBytes = baos.toByteArray();
      baos.close();
      pfos = new PRACTIFileOutputStream(this.localInterface, metaDataObjId);
      pfos.write(dataBytes);
    }finally{
      if(pfos != null){
        pfos.close();
      }
    }
    return(attr);
  }

 /** 
 *  Return the value of the attribute "mode" depending on the file type 
 **/ 
  protected long
  getModeTypeValue(long type){
    long modeTypeValue = 0;

    switch((int)type){
    case ftype.NFREG:
      modeTypeValue = 0100000;
      break;
    case ftype.NFDIR:
      modeTypeValue = 0040000;
      break;
    case ftype.NFBLK:
      modeTypeValue = 0060000;
      break;
    case ftype.NFCHR:
      modeTypeValue = 0020000;
      break;
    case ftype.NFLNK:
      modeTypeValue = 0120000;
      break;
    default:
      modeTypeValue = 0;
      break;
    }
    return(modeTypeValue);
  }

 /** 
 *  Make a PRACTIFSDirectory for the root directory 
 **/ 
  protected void
  makeRootDirectory(){
    ObjId dirIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    PRACTIFSDirectory rootDir = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] dirMetaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes dirAttr = null;
    ImmutableBytes dirIB = null;
    long numBlocks = 0;

    try{
      dirIdPrefix = this.fhCache.getObjIdPrefix(fhandle.makeAllZeroHandle());
      assert(dirIdPrefix.equals(new ObjId(PRACTIFS.ROOT_DIR_NAME)));
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      
      dirMetaToken = this.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireWriteLock(dirDataObjId);
      
      try{
        Env.dprintln(DEBUG, "readDir in makeRootDirectory, expect FileNotFoundException");
        rootDir = this.readDir(dirDataToken, dirDataObjId);
        Env.dprintln(DEBUG, "root already exist");
        
      }catch(FileNotFoundException e){
        Env.dprintln(DEBUG, "FileNotFoundException expected");
        // Create the directory
        rootDir = new PRACTIFSDirectory();
        fswArr = new FSWriteEntry[2];

        // Prepare to write the root directory data
        dirIB = new ImmutableBytes(rootDir.dangerousGetBytes());
        fswArr[0] = new FSWriteEntry(dirDataObjId,
                                       0,
                                       rootDir.getSize(),                                       
                                       dirIB,
                                       false);

        // Write the new inner directory attributes
        numBlocks = (dirIB.getLength() + FS_BLOCK_SIZE - 1) / FS_BLOCK_SIZE;
        dirAttr = createFileAttributes(null,//newAttr 
                                       ftype.NFDIR,
                                       dirIB.getLength(), // size
                                       numBlocks, // blocks
                                       (long)dirDataObjId.hashCode(), // fileid
                                       null);//I'm the root
        dirMetaBytes = this.getAttributeBytes(dirAttr);
        fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                       0,
                                       dirMetaBytes.length,
                                       new ImmutableBytes(dirMetaBytes),
                                       false);
        
        // Perform the writes
        writeLengths = this.localInterface.write(fswArr);
        /*
          System.out.println("PRACTIFS: Wrote: " +
          fswArr[0].getObjInvalTarget().getObjId() +
          ":" + writeLengths[0]);
          System.out.println("PRACTIFS: Wrote: " +
          fswArr[1].getObjInvalTarget().getObjId() +
          ":" + writeLengths[1]);
        */
      }
    }catch(BadFileHandleException e){
      // Did not find all-zero-file-handle in cache; this is an error!
      assert(false);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      assert(false);
    }catch(Exception e){
      // Program error
      e.printStackTrace();
      assert false : ("" + e);
    }finally{
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
  }

 /** 
 *  Return the attributes for this object 
 **/ 
  protected PRACTIFSDirectory
  readDir(PRACTIFSReadLockToken token, ObjId dirDataObjId)
    throws IOException, IsNotDirectoryException{
    PRACTIFSDirectory dir = null;
    PRACTIFileInputStream pfis = null;

    //Env.dprintln(DEBUG, "readDir " + dirDataObjId);
    //assert(token.isAcquired());
    try{
      pfis = new PRACTIFileInputStream(this.localInterface,
                                       dirDataObjId,
                                       PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      //Env.dprintln(DEBUG, "readDir " + dirDataObjId + " 1 ");
      dir = new PRACTIFSDirectory(pfis);
      //Env.dprintln(DEBUG, "readDir " + dirDataObjId + " 2");
      
    }finally{
      if(pfis != null){
        // Close the stream
        pfis.close();
      }
    }
    return(dir);
  }

 /** 
 *  Read symbolic link data 
 **/ 
  protected String
  readSymlinkData(PRACTIFSReadLockToken token, ObjId symlinkDataObjId)
    throws IOException{
    SymlinkData symlinkData = null;
    PRACTIFileInputStream pfis = null;
    String symlinkDataStr = null;

    assert(token.isAcquired());
    try{
      pfis = new PRACTIFileInputStream(this.localInterface,
                                       symlinkDataObjId,
                                       PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      symlinkData = new SymlinkData(pfis);
      symlinkDataStr = symlinkData.getData();
    }finally{
      if(pfis != null){
        // Close the stream
        pfis.close();
      }
    }
    return(symlinkDataStr);
  }

 /** 
 *  Serialize a given set of attributes into an array 
 **/ 
  protected byte[]
  getAttributeBytes(PRACTIFSFileAttributes attr){
    ByteArrayOutputStream baos = null;
    byte[] barr = null;

    try{
      baos = new ByteArrayOutputStream();
      attr.toOutputStream(baos);
      baos.flush();
      barr = baos.toByteArray();
    }catch(IOException e){
      assert false : ("" + e);
    }finally{
      if(baos != null){
        try{
          baos.close();
        }catch(IOException e){
          // Do nothing; not much we can do anyway
        }
      }
    }
    return(barr);
  }

/*

  //-------------------------------------------------------------------------
  // Make a new object ID
  //-------------------------------------------------------------------------
  protected ObjId
  makeNewObjIdPrefix(ObjId dirDataObjId, String name) throws IOException{
    ObjId objIdPrefix = null;
    ObjId objDataObjId = null;
    PRACTIFSReadLockToken objDataToken = null;
    int instanceNum = 0;
    boolean done = false;
    boolean checkObj = false;

    Env.dprintln(DEBUG, "makeNewObjPrefix");
    instanceNum = 1;
    while(!done){
      objIdPrefix = new ObjId(dirDataObjId.getPath() +
                              "/" + name +
                              "." + instanceNum);
      // Check the cache
      if(this.fhCache.containsPrefix(objIdPrefix)){
        // Increment the instance number
        instanceNum++;
      }else{
        // Try to read this object to see if it exists
        try{
          objDataObjId = new ObjId(objIdPrefix.getPath() + DATA_SUFFIX);
          objDataToken = this.localInterface.acquireReadLock(objDataObjId);
          checkObj = this.localInterface.isValid(objDataObjId,
                                                0,
                                                1);
          instanceNum++;
        }catch(ObjNotFoundException e){
          // Found a usable entry
          done = true;
        
        }catch(InterruptedException e){
          // This came from trying to acquire locks
          assert false : ("" + e);
        }finally{
          if((objDataObjId != null) && (objDataToken != null)){
            this.localInterface.releaseReadLock(objDataToken, objDataObjId);
          }
        }
      }
    }
    Env.dprintln(DEBUG, "makeNewObjPrefix done return " + objIdPrefix);
    return(objIdPrefix);
  }
*/

 /** 
 *  Make a new object ID 
 *  Use dirObjId, fileName, AcceptStamp 
 **/ 
  protected ObjId
  makeNewObjIdPrefix(ObjId dirDataObjId, String name) throws IOException{
    ObjId objIdPrefix = null;
    ObjId objDataObjId = null;
    PRACTIFSReadLockToken objDataToken = null;
    int instanceNum = 0;
    boolean done = false;
    BodyMsg bodyMsg = null;

    objIdPrefix = new ObjId(dirDataObjId.getPath() +
                            "/" + name 
                            //+ "." + this.localInterface.getMyNodeId() 
                            //+"." + System.currentTimeMillis()
			    );
    //for debug
    if(PRINT_ALL_FILE_NAMES){
      allFileNames = allFileNames + ":" + objIdPrefix + META_SUFFIX + ":" + objIdPrefix + DATA_SUFFIX;
    }
    Env.dprintln(PRINT_ALL_FILE_NAMES, "\n\n "+allFileNames);

    return(objIdPrefix);
  }

 /** 
 *  Create new dir entry for an object 
 *  by default there's no additional item in the entry except the  
 *  actual PRACTI objId. 
 **/ 
  protected PRACTIFSDirEntry createDirEntry(ObjId objPrefix){
    return new PRACTIFSDirEntry(objPrefix, null);
  }

 /** 
 *  Create a new file attribute for an object 
 *  by default there's no additional item in the entry except the  
 *  standard nfs attributes. 
 *  Note: dirPrefix is the parent of the file 
 *        it is reserved here to provide enough information for customizing 
 *        FileAttributes for different file systems. 
 *        if dirPrefix = null, then this fileAttributes belong to the root 
 **/ 
  protected PRACTIFSFileAttributes createFileAttributes(sattr newAttr, 
                                                        long fileType,
                                                        long size,
                                                        long numBlocks,
                                                        long myfileid,
                                                        ObjId dirPrefix){
    PRACTIFSFileAttributes ret = null;
    long now = System.currentTimeMillis();
    if (newAttr == null){
      ret = new PRACTIFSFileAttributes(fileType,
                                       0x041FF, // Giva all full permission
                                       1, // Only one link to this file so far
                                       0, // uid
                                       0, // gid
                                       size, // size
                                       FS_BLOCK_SIZE, // blocksize
                                       0, // rdev
                                       numBlocks, // blocks
                                       PRACTIFS.FSID, // fsid
                                       myfileid, // fileid
                                       new timeval(now), // atime
                                       new timeval(now), // mtime
                                       new timeval(now), // ctime
                                       null);//otherAttributes if any
    }else{
      ret = new PRACTIFSFileAttributes(newAttr,
                                       fileType,
                                       0x041FF, // Giva all full permission
                                       1, // Only one link to this file so far
                                       0, // uid
                                       0, // gid
                                       size, // size
                                       FS_BLOCK_SIZE, // blocksize
                                       0, // rdev
                                       numBlocks, // blocks
                                       PRACTIFS.FSID, // fsid
                                       myfileid, // fileid
                                       new timeval(now), // atime
                                       new timeval(now), // mtime
                                       new timeval(now), // ctime
                                       null);//otherAttributes if any
    }
      return ret;
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PRACTIFS.java...");
    PRACTIFS.testSimple();
    System.out.println("...Finished");
    System.exit(0);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    PRACTIFSLocalInterface li = null;
    PRACTIFS pfs = null;
    fhandle handle = null;
    fhandle handle2 = null;
    URANode uraNode = null;

    /*
      li = new FakePRACTIFSLocalInterface();
      pfs = new PRACTIFS(li, true);
    */

    // Note: Works with a FakePRACTIFSLocalInterface; trying using a real
    // one
    uraNode = new URANode("AmolTestConfig.config",
                          new NodeId(11),

                          Controller.NULL_CONTROLLER,
                          true);
    li = new PRACTIFSLocalInterface(uraNode.getLocalInterface(), new NodeId(11));
    pfs = new PRACTIFS(li, true);

    //System.out.println("" + li);
    PRACTIFS.testCreate(li, pfs, "a");
    handle = PRACTIFS.testLookup(li, pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle == null);
    handle = PRACTIFS.testLookup(li, pfs, fhandle.makeAllZeroHandle(), "a");
    assert(handle != null);
    handle2 = PRACTIFS.testLookup(li, pfs, handle, "b");
    assert(handle2 == null);
    
    PRACTIFS.testGetAttributes(li, pfs, handle, ftype.NFREG);
    PRACTIFS.testGetAttributes(li, pfs, fhandle.makeAllZeroHandle(), ftype.NFDIR);
    PRACTIFS.testWrite(li, pfs, handle);
    PRACTIFS.testGetAttributes(li, pfs, handle,  ftype.NFREG);
    PRACTIFS.testRead(li, pfs, handle);
    PRACTIFS.testReadDir(li, pfs, fhandle.makeAllZeroHandle());
    PRACTIFS.testRename(li,
                        pfs,
                        fhandle.makeAllZeroHandle(),
                        "a",
                        fhandle.makeAllZeroHandle(),
                        "b");
    handle = PRACTIFS.testLookup(li, pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle != null);
    PRACTIFS.testMkdir(li, pfs, fhandle.makeAllZeroHandle(), "subDir");
    handle = PRACTIFS.testLookup(li, pfs, fhandle.makeAllZeroHandle(), "subDir");
    assert(handle != null);
    PRACTIFS.testRename(li, pfs, fhandle.makeAllZeroHandle(), "b", handle, "newB");
    handle2 = PRACTIFS.testLookup(li, pfs, handle, "newB");
    assert(handle2 != null);
    handle2 = PRACTIFS.testLookup(li, pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle2 == null);
    PRACTIFS.testSetAttributes(li, pfs, handle);
  }

 /** 
 *  Test the create method 
 **/ 
  protected static void
  testCreate(PRACTIFSLocalInterface li, PRACTIFS pfs, String name){
    createargs args = null;
    diropres result = null;
    diropargs dirOpArgs = null;
    sattr attr = null;

    dirOpArgs = new diropargs(fhandle.makeAllZeroHandle(), name);
    attr = new sattr(0, 0, 0, 0, new timeval(1, 2), new timeval(3, 4));

    args = new createargs(dirOpArgs, attr);
    result = pfs.create(args);

    // Make sure creation succeeded
    assert(result.getStatus() == stat.NFS_OK);

    // Check to see what our fake local interface has
    //System.out.println("" + li);
  }

 /** 
 *  Test the lookup method 
 **/ 
  protected static fhandle
  testLookup(PRACTIFSLocalInterface li,
             PRACTIFS pfs,
             fhandle fh,
             String name){
    diropargs args = null;
    diropres result = null;

    args = new diropargs(fh, name);
    result = pfs.lookup(args);
    if(result.getStatus() == stat.NFS_OK){
      fh = result.getFile();
    }else{
      fh = null;
    }

    return(fh);
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  protected static void
  testGetAttributes(PRACTIFSLocalInterface li,
                    PRACTIFS pfs,
                    fhandle fh,
                    long expectedType){
    attrstat result = null;
    fattr attr = null;

    result = pfs.getAttributes(fh);
    assert(result.getStatus() == stat.NFS_OK);

    attr = result.getAttributes();
    
    assert attr.getType() == expectedType;
    /*
    assert attr.getMode() == expectedAttr.getMode();
    assert attr.getNLink() == expectedAttr.getNLink();
    assert attr.getUID() == expectedAttr.getUID();
    assert attr.getGID() == expectedAttr.getGID();
    assert attr.getSize() == expectedAttr.getSize();
    assert attr.getBlockSize() == expectedAttr.getBlockSize();
    assert attr.getBlocks() == expectedAttr.getBlocks();
    assert attr.getFSID() == expectedAttr.getFSID();
    assert attr.getFileId() == expectedAttr.getFileId();
    */
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  protected static void
  testSetAttributes(PRACTIFSLocalInterface li,
                    PRACTIFS pfs,
                    fhandle fh){
    attrstat result1 = null;
    attrstat result2 = null;
    fattr attr1 = null;
    fattr attr2 = null;
    sattrargs args = new sattrargs(fh,
                                  new sattr(1, 2, 3, 4, 
                                            new timeval(5, 6), 
                                            new timeval(7, 8)));
    result1 = pfs.setAttributes(args);
    assert(result1.getStatus() == stat.NFS_OK);
    
    attr1 = result1.getAttributes();
    
    result2 = pfs.getAttributes(fh);
    assert(result2.getStatus() == stat.NFS_OK);

    attr2 = result2.getAttributes();
    assert attr1.equals(attr2);
    //System.out.println("Size = " + attr.getSize());
  }

 /** 
 *  Test the write method 
 **/ 
  protected static void
  testWrite(PRACTIFSLocalInterface li,
            PRACTIFS pfs,
            fhandle fh){
    byte[] data = null;
    attrstat result = null;

    data = new byte[100];
    data[0] = 7;
    data[52] = 3;
    result = pfs.write(fh, 0, 0, 100, data);
    assert(result.getStatus() == stat.NFS_OK);

    // See what our fake local interface has
    //System.out.println("" + li);
  }

 /** 
 *  Test the read method 
 **/ 
  protected static void
  testRead(PRACTIFSLocalInterface li,
           PRACTIFS pfs,
           fhandle fh){
    byte[] data = null;
    readargs args = null;
    readres result = null;

    // Read the first 50 bytes
    args = new readargs(fh, 0, 50, 100);
    result = pfs.read(args);
    assert(result.getStatus() == stat.NFS_OK);

    data = result.getData();
    //assert(data.length == 50);
    //System.out.println("" + data.length);
    assert(data[0] == 7);

    // Read the next 50 bytes
    args = new readargs(fh, 50, 50, 100);
    result = pfs.read(args);
    assert(result.getStatus() == stat.NFS_OK);

    data = result.getData();
    //System.out.println("data.length = " + data.length);
    assert(data.length == 50);
    assert(data[2] == 3);
  }

 /** 
 *  Test the readdir method 
 **/ 
  protected static void
  testReadDir(PRACTIFSLocalInterface li,
              PRACTIFS pfs,
              fhandle fh){
    readdirargs args = null;
    readdirres result = null;
    entry entryList = null;

    args = new readdirargs(fh, 0, 100);
    result = pfs.readdir(args);
    assert(result.getStatus() == stat.NFS_OK);
    assert(result.getEOF());

    /*
    entryList = result.getEntries();
    while(entryList != null){
      System.out.println("entryList: (" + entryList.getFileId() +
                         ", " + entryList.getName() +
                         ", " + entryList.getCookie() + ")");
      entryList = entryList.getNextEntry();
    }
    */

    entryList = result.getEntries();
    assert(entryList != null);
    assert(entryList.getName().equals("."));
    entryList = entryList.getNextEntry();
    assert(entryList != null);
    assert(entryList.getName().equals(".."));
    entryList = entryList.getNextEntry();
    assert(entryList != null);
    assert(entryList.getName().equals("a"));
    entryList = entryList.getNextEntry();
    assert(entryList == null);
  }

 /** 
 *  Test the rename method 
 **/ 
  protected static void
  testRename(PRACTIFSLocalInterface li,
             PRACTIFS pfs,
             fhandle origFH,
             String origName,
             fhandle destFH,
             String destName){
    int result = 0;
    renameargs args = null;

    args = new renameargs(new diropargs(origFH, origName),
                          new diropargs(destFH, destName));
    result = pfs.rename(args);
    //System.out.println("" + li);
  }

 /** 
 *  Test the mkdir method 
 **/ 
  protected static void
  testMkdir(PRACTIFSLocalInterface li,
            PRACTIFS pfs,
            fhandle dirHandle,
            String newSubDirName){
    createargs args = null;
    diropres result = null;
    args = new createargs(new diropargs(dirHandle, newSubDirName),
                          new sattr(10, 20, 30, 40, new timeval(4, 8), new timeval(1, 2)));
    result = pfs.mkdir(args);
    assert(result.getStatus() == stat.NFS_OK);
  }

  
}


 /** 
/* $Log: PRACTIFS.java,v $
/* Revision 1.48  2007/04/02 21:11:39  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.47  2007/03/06 18:25:51  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.46  2007/03/02 01:35:29  zjiandan
/* Yeah! Working Simplified TierStore.
/*
/* Revision 1.45  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.44  2007/02/23 20:54:29  zjiandan
/* Fixed mem leak problems in NFS2Interface and some other bugs.
/* Andrew benchmark passed, performance still needs to tune.
/*
/* Revision 1.43  2007/02/13 03:00:02  zjiandan
/* minor changes.
/*
/* Revision 1.42  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.41  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.40  2007/01/22 16:45:42  nalini
/* changed testSimple from protected to private
/*
/* Revision 1.39  2007/01/21 19:23:19  nalini
/* ChainReplication works with NFS -- needed change NFSlib so that chainRep node makes root locally
/*
/* Revision 1.38  2007/01/10 21:07:39  nalini
/* cleaned up NFS interface so that all policy is in PRACTIFSLocalInterface
/*
/* Revision 1.37  2007/01/10 06:34:01  zjiandan
/* fixed "mount: localhost:practidir: can't read superblock" problem
/* and the Java P2 wrapper string problem.
/*
/* Revision 1.36  2006/11/18 15:54:53  nayate
/* Made writing attributes more efficient
/*
/* Revision 1.35  2006/11/18 08:11:12  nayate
/* Minor debugging print statement added
/*
/* Revision 1.34  2006/11/17 20:04:33  zjiandan
/* Tested new PRACTIFS.
/*
/* Revision 1.33  2006/11/16 21:22:57  zjiandan
/* *** empty log message ***
/* */
 **/ 
