package code;
 /** 
 *  Local interface created for the PRACTI FS 
 *   -- by default bound is set to true 
 *   -- but can be set to false by constructor 
 **/ 
import java.io.IOException;
import java.io.EOFException;
import java.util.HashSet;

public class PRACTIFSLocalInterface{

  private static final boolean dbgLock = false;

 /** 
 *  Data members 
 **/ 
  protected LocalInterface li;  // Redirect calls to li
  private LockTable lockTable; // Table of locked objects
  private NodeId myNodeId; // ID of this node
  private HashSet acquiredReadLocks; // Used for debugging
  private HashSet acquiredWriteLocks; // Used for debugging
  private boolean useBound;


 /** 
 *  Constructor -- default bound is true 
 **/ 
  public
  PRACTIFSLocalInterface(LocalInterface newLI, NodeId newMyNodeId){
    this(newLI, newMyNodeId, true);
  }

 /** 
 *  Constructor - to set bound policy 
 **/ 
  public
  PRACTIFSLocalInterface(LocalInterface newLI, NodeId newMyNodeId, 
			 boolean useBound){
    this.li = newLI;
    this.myNodeId = newMyNodeId;
    this.lockTable = new LockTable();
    this.acquiredReadLocks = new HashSet();
    this.acquiredWriteLocks = new HashSet();
    this.useBound = useBound;
  }

 /** 
 *  Constructor  -- to get older test cases to work 
 **/ 
  public
  PRACTIFSLocalInterface(PRACTIFSLocalInterface fsli, NodeId newMyNodeId){
    this.li = fsli.li;
    this.myNodeId = newMyNodeId;
    this.lockTable = new LockTable();
    this.acquiredReadLocks = new HashSet();
    this.acquiredWriteLocks = new HashSet();
  }

 /** 
 *  Constructor  -- to get older test cases to work 
 **/ 
  public
  PRACTIFSLocalInterface(NodeId newMyNodeId){
    this.li = new LocalInterface(null, null);
    this.myNodeId = newMyNodeId;
    this.lockTable = new LockTable();
    this.acquiredReadLocks = new HashSet();
    this.acquiredWriteLocks = new HashSet();
  }

 /** 
 *  Acquire a read lock 
 **/ 
  public PRACTIFSReadLockToken
  acquireReadLock(ObjId objId) throws InterruptedException{
    PRACTIFSReadLockToken token = null;
    token = new PRACTIFSReadLockToken(objId, true);
    Env.dprintln(dbgLock, "acquireReadLock for "+objId.toString());
    this.lockTable.acquireReadLock(objId); // Wait until lock acquired
    assert(this.notifyReadLockAcquired(token));
    Env.dprintln(dbgLock, "return acquireReadLock for "+objId.toString());
    return(token);
  }

 /** 
 *  Release a read lock 
 **/ 
  public void
  releaseReadLock(PRACTIFSReadLockToken token, ObjId objId){
    assert(token.getObjId().equals(objId));
    Env.dprintln(dbgLock, "release ReadLock for "+objId.toString());
    token.setIsAcquired(false);
    this.lockTable.releaseReadLock(objId);
    Env.dprintln(dbgLock, "return release ReadLock for "+objId.toString());
    assert(this.notifyReadLockReleased(token));
  }

 /** 
 *  Acquire a write (exclusive) lock 
 **/ 
  public PRACTIFSWriteLockToken
  acquireWriteLock(ObjId objId) throws InterruptedException{
    PRACTIFSWriteLockToken token = null;
    Env.dprintln(dbgLock, "acquireWriteLock for "+objId.toString());
    token = new PRACTIFSWriteLockToken(objId, true);
    this.lockTable.acquireWriteLock(objId); // Wait until lock acquired
    assert(this.notifyWriteLockAcquired(token));
    Env.dprintln(dbgLock, "return acquireWriteLock for "+objId.toString());
    return(token);
  }

 /** 
 *  Release a read lock 
 **/ 
  public void
  releaseWriteLock(PRACTIFSReadLockToken token, ObjId objId){
    assert(token.getObjId().equals(objId));
    Env.dprintln(dbgLock, "releaseWriteLock for "+objId.toString());
    token.setIsAcquired(false);
    this.lockTable.releaseWriteLock(objId);
    assert this.notifyWriteLockReleased(token) : "Failed on objId = " + objId;
      Env.dprintln(dbgLock, "return releaseWriteLock for "+objId.toString());
  }

 /** 
 *  Read from an object 
 **/ 
  public synchronized BodyMsg
  read(ObjId objId,
       long offset,
       long length)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    BodyMsg result = null;
    Env.dprintln(dbgLock, "PRACTIFSLocalInterface::read "+objId.toString());
    try{
      result = this.li.read(objId,
			    offset,
			    length,
			    true,//blockInvalid
			    true);//blockPrecise
    }catch(ReadOfInvalidRangeException roire){
      assert(false); // this should not happen
    }

    Env.dprintln(dbgLock, "return PRACTIFSLocalInterface::read"+objId.toString());
    return(result);
  }

 /** 
 *  Write method called by PRACTIFS 
 **/ 
  public long[] write(FSWriteEntry[] fwe)
    throws IOException {
    double priority = 0.0;
    boolean embargoed = false;
    Env.dprintln(dbgLock, "PRACTIFSLocalInterface::write" + fwe[0].getObjInvalTarget().toString());
    MultiWriteEntry[] mwe = createMultiWriteEntry(fwe, priority, useBound);
    long[] result = this.writeMulti(mwe, embargoed);
    Env.dprintln(dbgLock, "return PRACTIFSLocalInterface::write"+fwe[0].getObjInvalTarget().toString());
    return result;    

  }

 /** 
 *  Version of the write() method that allows writing to multiple 
 *  object atomically 
 **/ 
  protected long[]
  writeMulti(MultiWriteEntry[] mwe, boolean embargoed)
    throws IOException{
    long[] result = null;
    Env.dprintln(dbgLock, "PRACTIFSLocalInterface::writeMulti");
    // Pass the call onto the real LocalInterface
    result = this.li.writeMulti(mwe, embargoed);
    Env.dprintln(dbgLock, "return PRACTIFSLocalInterface::writeMulti");
    return(result);
  }

 /** 
 *  createMultiWriteEntry - converts FSWriteEntry to MultiWriteEntry 
 *   incorporating the priority and bound parameters 
 **/ 
  protected MultiWriteEntry[] 
  createMultiWriteEntry(FSWriteEntry[] fwe, double priority, boolean useBound) {
    MultiWriteEntry[] mwe = new MultiWriteEntry[fwe.length];
    boolean useBValue;

    for(int i=0; i<mwe.length; i++) {
      // try to avoid bounded deletes!
      useBValue = useBound;
      if(fwe[i].getDelete()) {
        useBValue = false;
      } 
      mwe[i] = new MultiWriteEntry(fwe[i].getObjInvalTarget(),
                                   priority,
                                   fwe[i].getImmutableBytes(),
                                   useBValue,
                                   fwe[i].getDelete());
    }

    return mwe;
  }

                                   

 /** 
 *  Write method 
 **/ 
  public long
  write(ObjId objId, long offset, long length, byte buffer[]) 
    throws IOException {
    long numBytes = 0;
    double priority = 0.0;
    Env.dprintln(dbgLock, "PRACTIFSLocalInterface::write"+objId.toString());
    // Pass the call onto the real LocalInterface object
    numBytes = this.li.write(objId,
                             offset,
                             length,
                             priority,
                             buffer,
                             useBound);
    Env.dprintln(dbgLock, "return PRACTIFSLocalInterface::write"+objId.toString());
    return(numBytes);
  }

 
 /** 
 *  Return the local node ID 
 **/ 
  public NodeId
  getMyNodeId(){
    return(this.myNodeId);
  }


 
 /** 
 *  checks if the object is valid 
 *  
 *  NOTE: Currently unimplemented; not needed 
 **/ 
  
  public boolean
  isValid(ObjId objId, long offset, long length)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException {
    
    return this.li.isValid(objId, offset, length);
  }

 /** 
 *  Checks if the object is precise 
 *  
 *  NOTE: Currently unimplemented; not needed 
 **/ 
  public boolean
  isPrecise(ObjId objId){
    return this.li.isPrecise(objId);
  }

 /** 
 *  Account for the creation of a read lock (for debugging) 
 **/ 
  private boolean
  notifyReadLockAcquired(PRACTIFSReadLockToken token){
    this.acquiredReadLocks.add(token);
    return(true);
  }

 /** 
 *  Account for the creation of a write lock (for debugging) 
 **/ 
  private boolean
  notifyWriteLockAcquired(PRACTIFSReadLockToken token){
    this.acquiredWriteLocks.add(token);
    return(true);
  }

 /** 
 *  Account for the release of a read lock (for debugging) 
 **/ 
  private boolean
  notifyReadLockReleased(PRACTIFSReadLockToken token){
    return(this.acquiredReadLocks.remove(token));
  }

 /** 
 *  Account for the release of a write lock (for debugging) 
 **/ 
  private boolean
  notifyWriteLockReleased(PRACTIFSReadLockToken token){
    return(this.acquiredWriteLocks.remove(token));
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PRACTIFSLocalInterface.java...");
    PRACTIFSLocalInterface.testSimple();
    System.out.println("...PRACTIFSLocalInterface.java");
    System.exit(0);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    PRACTIFSLocalInterface li = null;
    FakePRACTIFSLocalInterface fakeLI = null;
    PRACTIFSReadLockToken readToken = null;
    PRACTIFSWriteLockToken writeToken = null;

    fakeLI = new FakePRACTIFSLocalInterface(new NodeId(30));
    li = new PRACTIFSLocalInterface(fakeLI, new NodeId(30));
    try{
      // General test of LocalInterface
      assert(li.getMyNodeId().equals(new NodeId(30)));

      // Try one objId
      readToken = li.acquireReadLock(new ObjId("a"));
      assert(readToken.getObjId().equals(new ObjId("a")));
      assert(readToken.isAcquired());
      li.releaseReadLock(readToken, new ObjId("a"));
      assert(!readToken.isAcquired());
      writeToken = li.acquireWriteLock(new ObjId("a"));
      assert(writeToken.getObjId().equals(new ObjId("a")));
      assert(writeToken.isAcquired());
      li.releaseWriteLock(writeToken, new ObjId("a"));
      assert(!writeToken.isAcquired());

      // Try two objIds
      readToken = li.acquireReadLock(new ObjId("a"));
      assert(readToken.getObjId().equals(new ObjId("a")));
      assert(readToken.isAcquired());
      writeToken = li.acquireWriteLock(new ObjId("b"));
      assert(writeToken.getObjId().equals(new ObjId("b")));
      assert(writeToken.isAcquired());
      li.releaseWriteLock(writeToken, new ObjId("b"));
      li.releaseReadLock(readToken, new ObjId("a"));
    }catch(InterruptedException e){
      System.err.println("" + e);
    }
  }
}
