package code;
 /** 
 *  Implement an InputStream interface to a PRACTI object 
 *  
 *  Note: This class is not thread safe! 
 **/ 
import java.io.StreamCorruptedException;
import java.io.IOException;
import java.io.EOFException;
import java.io.InputStream;
import java.io.FileNotFoundException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

public class PRACTIFileInputStream extends InputStream{
 /** 
 *  Data members 
 **/ 
  protected final ObjId objId; 
  protected final int blockSize;
  protected ImmutableBytes ib;
  protected long objOffset;
  protected int ibOffset;
  protected PRACTIFSLocalInterface localInterface;

  static boolean DEBUG = false;
 /** 
 *  Constructor 
 **/ 
  public
  PRACTIFileInputStream(PRACTIFSLocalInterface newLocalInterface,
                        ObjId newObjId,
                        int newBlockSize)
    throws IOException{
    
    this.objId = newObjId;
    this.objOffset = 0;
    this.ibOffset = 0;
    this.blockSize = newBlockSize;
    assert(this.blockSize > 0);
    this.localInterface = newLocalInterface;

    iniRead();
  }

  void iniRead() throws IOException  {
    try{
      BodyMsg bodyMsg = this.localInterface.read(this.objId,
                                         0,
                                         this.blockSize);
      
      this.ib = bodyMsg.getBody();
      this.objOffset += this.ib.getLength();
    }catch(EOFException eofe){
      // Note that we want to allow an application to create a
      // PRACTIFileInputStream object even for an object that is empty. So,
      // the constructor should not throw an EOFException. The read methods,
      // on the other hand, will throw EOFException if an applcation tries to
      // read a non-zero number of bytes from an empty object.
      this.ib = new ImmutableBytes(new byte[0]);
    }catch(ReadOfHoleException rhe){
      //same as EOFException
      this.ib = new ImmutableBytes(new byte[0]);


    }catch(ObjNotFoundException onfe){
      
      // Convert this to an exception that is thrown by InputStream
      throw new FileNotFoundException("" + onfe);
    }
    Env.dprintln(DEBUG, "ib=" + ib);
  }

                        

 /** 
 *  Return the number of available bytes 
 **/ 
  public int
  available() throws StreamCorruptedException{
    if(this.ib == null){
      throw new StreamCorruptedException("Stream for " + this.objId +
                                         " closed");
    }
    return(this.ib.getLength() - this.ibOffset);
  }

 /** 
 *  Close the stream 
 **/ 
  public void
  close() throws StreamCorruptedException{
    if(this.ib == null){
      throw new StreamCorruptedException("Stream  for " + this.objId +
                                         " closed");
    }
    // Allow Java to garbage-collect the buffer
    this.ib = null;
  }

 /** 
 *  Mark this position in the stream (note: unsupported operation) 
 **/ 
  public void
  mark(){
    // Do nothing
  }

 /** 
 *  Return true if this stream supports mark/reset operations (which it 
 *  doesn't) 
 **/ 
  public boolean
  markSupported(){
    return(false);
  }

 /** 
 *  Read a byte of data 
 **/ 
  public int
  read() throws
    StreamCorruptedException,
    EOFException,
    IOException,
    FileNotFoundException{
    BodyMsg bodyMsg = null;
    byte[] data = null;
    int val = 0;

    if(this.ib == null){
      throw new StreamCorruptedException("Stream for " + this.objId +
                                         " closed");
    }
    while(this.ibOffset >= this.ib.getLength()){
      try{
        // The while loops keeps reading until we receive a non-zero number
        // of bytes
	try{
	  bodyMsg = this.localInterface.read(this.objId,
                                           this.objOffset,
                                           this.blockSize);
	}catch(ReadOfHoleException e){
	  //assume that this File InputStream is only used for read meta data
	  //which is well defined and never has a hole
	  assert false;
	}
        this.ib = bodyMsg.getBody();
        this.objOffset += this.ib.getLength();
        this.ibOffset = 0;

      }catch(ObjNotFoundException onfe){
        // Convert this to an exception that is thrown by InputStream
        throw new FileNotFoundException("" + onfe);
      }
    }
    data = this.ib.dangerousGetReferenceToInternalByteArray();
    // Cut off the series of 1's that appear because of sign-extensions during
    // typecasting a byte to an int
    val = ((int)data[this.ibOffset]) & 0xFF;
    this.ibOffset++;
    return(val);
  }

 /** 
 *  Populate the array of bytes with data and return the number of bytes 
 *  read 
 **/ 
  public int
  read(byte[] b) throws
    StreamCorruptedException,
    EOFException,
    IOException,
    FileNotFoundException{
    int numBytesRead = 0;

    numBytesRead = this.read(b, 0, b.length);
    return(numBytesRead);
  }

 /** 
 *  Populate the array of bytes with data and return the number of bytes 
 *  read 
 **/ 
  public int
  read(byte[] b, int off, int len) throws
    StreamCorruptedException,
    EOFException,
    IOException,
    FileNotFoundException{
    BodyMsg bodyMsg = null;
    byte[] data = null;
    int numBytesToCopy = 0;

    assert(b.length >= len);
    if(this.ib == null){
      throw new StreamCorruptedException("Stream for " + this.objId +
                                         " closed");
    }
    while(this.ibOffset >= this.ib.getLength()){
      try{
        // The while loop keeps reading until we receive a non-zero number
        // of bytes
        bodyMsg = this.localInterface.read(this.objId,
                                           this.objOffset,
                                           this.blockSize);
        this.ib = bodyMsg.getBody();
        this.objOffset += this.ib.getLength();
        this.ibOffset = 0;
      }catch(ObjNotFoundException onfe){
        // Convert this to an exception that is thrown by InputStream
        throw new FileNotFoundException("" + onfe);
      }catch(ReadOfHoleException rhe){
	assert false;
      }
    }
    numBytesToCopy = (this.ib.getLength() - this.ibOffset);
    if(numBytesToCopy > len){
      numBytesToCopy = len;
    }
    data = this.ib.dangerousGetReferenceToInternalByteArray();
    System.arraycopy(data, this.ibOffset, b, off, numBytesToCopy);
    this.ibOffset += numBytesToCopy;
    return(numBytesToCopy);
  }

 /** 
 *  Reset the stream (note: unsupported operation) 
 **/ 
  public void
  reset() throws IOException{
    // Indicate that this operation is not supported
    throw new IOException("Operation not supported");
  }

 /** 
 *  Skip n bytes in the stream 
 *  Note: We currently borrow the default inefficient implementation for 
 *  simplicity. In the future we may change this. 
 **/ 
  public long
  skip(long n) throws IOException{
    long numSkipped = 0;

    numSkipped = super.skip(n);
    return(numSkipped);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv)
  throws Exception{
    Env.verifyAssertEnabled();
    System.out.println("Testing PRACTIFileInputStream.java...");
    PRACTIFileInputStream.testSimple();
    PRACTIFileInputStream.testWithObjectInputStream();
    System.out.println("...Finished");
  }

 /** 
 *  Run tests with ObjectInputStream and ObjectOutputStream 
 **/ 
  protected static void
  testWithObjectInputStream() throws Exception{
    FakePRACTIFSLocalInterface fpli = new FakePRACTIFSLocalInterface(new NodeId(1));
    PRACTIFileOutputStream pos = new PRACTIFileOutputStream(fpli, new ObjId("a"));
    ObjectOutputStream oos = new ObjectOutputStream(pos);
    String str = "children parentId childId ak;ldfkja;lkgjadflfgh";
    oos.writeObject(str);
    oos.flush();
    
    PRACTIFileInputStream pis = new PRACTIFileInputStream(fpli, new ObjId("a"), 10);
    ObjectInputStream ois = new ObjectInputStream(pis);
    String ret = (String)(ois.readObject());
    
    System.out.println(ret);
   
    assert (ret.equals(str)) : 
      "ret=\"" + ret + "\" expected:\""+str + "\"";
      try{
	ret = (String)(ois.readObject());
	assert false;
      }catch(EOFException e){
	//expected
	str = "new str appended";
	oos.writeObject(str);
	oos.flush();
      }
      try{
	ret = (String)(ois.readObject());
	assert ret.equals(str) :"ret=\"" + ret + "\" expected:\""+str + "\"";
	  System.out.println(ret);
      }catch(EOFException e){
	assert false;
      }
      
  }

 /** 
 *  Run simple tests 
 **/ 
  protected static void
  testSimple(){
    PRACTIFileInputStream fis = null;
    byte[] data = null;
    int bInt = 0;
    byte[] b = null;
    int numBytesRead = 0;
    long numBytesSkipped = 0;
    PFISFakeLocalInterface pfisFLI = null;

    data = PRACTIFileInputStream.makeRandomData(1000);
    try{
      pfisFLI = new PFISFakeLocalInterface(data);
      fis = new PRACTIFileInputStream(pfisFLI, new ObjId("a"), 10);

      // Test 1: Read the first two bytes
      assert(fis.available() == 10);
      bInt = fis.read();
      assert(bInt == (((int)data[0]) & 0xFF));
      bInt = fis.read();
      assert(bInt == (((int)data[1]) & 0xFF));

      // Test 2: Read a few bytes at a time
      // We should only read 8 bytes because (1) the internal buffer
      // should be 10 bytes, and (2) we have already read 2 bytes
      assert(fis.available() == 8);
      b = new byte[15];
      numBytesRead = fis.read(b);
      assert(numBytesRead == 8);
      for(int i = 0; i < numBytesRead; i++){
        assert(b[i] == data[i + 2]);
      }

      // Test 3: Ask for a small buffer to be populated
      // We have read 10 bytes by now; read the next two
      assert(fis.available() == 0);
      b = new byte[2];
      numBytesRead = fis.read(b);
      assert(numBytesRead == 2);
      assert(b[0] == data[10]);
      assert(b[1] == data[11]);

      // Test 4: Ask for a subset of a buffer to be populated
      // Populate 5 bytes starting from offset 2
      // Make sure offset 0 and 1 are unchanged
      // We should have read 12 bytes by now
      assert(fis.available() == 8);
      b = new byte[7];
      b[0] = (byte)45;
      b[1] = (byte)72;
      numBytesRead = fis.read(b, 2, 5);
      assert(numBytesRead == 5);
      assert(b[0] == (byte)45);
      assert(b[1] == (byte)72);
      for(int i = 0; i < numBytesRead; i++){
        assert(b[i + 2] == data[i + 12]);
      }

      // Test 5: Return fewer than a full buffer of bytes
      // We should have read 17 bytes by now
      // We should only be able to read 3 bytes right now
      assert(fis.available() == 3);
      b = new byte[4];
      numBytesRead = fis.read(b);
      assert(numBytesRead == 3);
      assert(b[0] == data[17]);
      assert(b[1] == data[18]);
      assert(b[2] == data[19]);

      // Test 6: Skip to nearly the end
      // We have read 20 bytes by now
      // We should be able to skip 975 bytes exception-free
      assert(fis.available() == 0);
      numBytesSkipped = fis.skip(975);

      // Test 7: Return an incomplete buffer due to EOF
      // We have read 995 bytes by now
      // We should only be able to read 5 more bytes before EOF
      assert(fis.available() == 5);
      b = new byte[1000];
      numBytesRead = fis.read(b);
      assert(numBytesRead == 5);
      for(int i = 0; i < numBytesRead; i++){
        assert(b[i] == data[i + 995]);
      }
      assert(fis.available() == 0);

      // Test 8: Force an EOF exception
      try{
        bInt = fis.read();
        assert(false);
      }catch(EOFException e){
        // Do nothing; this was expected
      }

      // Test 9: Create a 0 byte stream
      try{
        pfisFLI = new PFISFakeLocalInterface(new byte[0]);
        fis = new PRACTIFileInputStream(pfisFLI, new ObjId("a"), 10);
      }catch(IOException e){
        assert(false);
      }

      // Test 10: Force and EOF exception by reading from it
      try{
        bInt = fis.read();
        assert(false);
      }catch(EOFException e){
        // Do nothing; expected behavior
      }
    }catch(IOException e){
      assert false : "" + e;
    }
  }

 /** 
 *  Make some bytes of data 
 *  Note: Make this more random later 
 **/ 
  protected static byte[]
  makeRandomData(int numBytes){
    byte[] data = null;
    double rand = 0;

    data = new byte[numBytes];
    for(int i = 0; i < numBytes; i++){
      data[i] = (byte)(i & 0xFF);
    }
    return(data);
  }
}

 /** 
 *  Make a fake local interface 
 **/ 
class PFISFakeLocalInterface extends PRACTIFSLocalInterface{

 /** 
 *  Data members 
 **/ 
  protected byte[] data;
  protected int maxNumBytesToReturn;

 /** 
 *  Constructor 
 **/ 
  public
  PFISFakeLocalInterface(byte[] newData){
    super(new NodeId(1));
    this.data = newData;
    this.maxNumBytesToReturn = Integer.MAX_VALUE;
  }

 /** 
 *  Set a bound on the maximum number of bytes we can return from a read 
 **/ 
  public void
  setMaxNumBytesToReturn(int newMaxNumBytesToReturn){
    this.maxNumBytesToReturn = newMaxNumBytesToReturn;
  }

 /** 
 *  Get the bound on the maximum number of bytes we can return from a read 
 **/ 
  public int
  getMaxNumBytesToReturn(){
    return(this.maxNumBytesToReturn);
  }

 /** 
 *  Read a certain number of bytes 
 **/ 
  public BodyMsg
  read(ObjId objId, long offset, long length)
    throws EOFException{
    BodyMsg bodyMsg = null;
    ImmutableBytes dataIB = null;
    byte[] retData = null;
    int numBytesToReturn = 0;

    if(offset >= this.data.length){
      throw new EOFException("Stream offset " + offset +
                             " is beyond the stream end");
    }
    numBytesToReturn = (int)length;
    if(this.maxNumBytesToReturn < length){
      numBytesToReturn = this.maxNumBytesToReturn;
    }
    if(((int)(this.data.length - offset)) < numBytesToReturn){
      numBytesToReturn = (int)(this.data.length - offset);
    }
    retData = new byte[(int)numBytesToReturn];
    System.arraycopy(this.data, (int)offset, retData, 0, numBytesToReturn);
    dataIB = new ImmutableBytes(retData);
    bodyMsg = new BodyMsg(objId,
                          offset,
                          (long)numBytesToReturn,
                          new AcceptStamp(100, new NodeId(1)),
                          dataIB,
                          false);
    return(bodyMsg);
  }
}
