package code;
 /** 
 *  Implement an OutputStream interface to a PRACTI object 
 *  
 *  Atomicity: This stream does not provide atomicity. If a write results 
 *  in an exception being thrown before completion, it is possible that some 
 *  of the data is written. 
 *  
 *  Note: This class is not thread-safe! 
 **/ 
import java.io.IOException;
import java.io.OutputStream;

public class PRACTIFileOutputStream extends OutputStream{

 /** 
 *  Constants 
 **/ 
  private final static boolean PRINT_DEBUG = false;

 /** 
 *  Data members 
 **/ 
  private ObjId objId;
  private boolean isClosed;
  private long objOffset;
  private PRACTIFSLocalInterface localInterface;

 /** 
 *  Constructor 
 **/ 
  public
  PRACTIFileOutputStream(PRACTIFSLocalInterface newLocalInterface, ObjId newObjId){
    this.localInterface = newLocalInterface;
    this.isClosed = false;
    this.objId = newObjId;
    this.objOffset = 0;
  }

 /** 
 *  Close this stream 
 **/ 
  public void
  close() throws IOException{
    if(this.isClosed){
      System.out.println("Stream closed twice!!!!!!!!!!");
      throw new IOException("Stream for " + this.objId + " already closed");
    }
    this.isClosed = true;
  }

 /** 
 *  Flush this stream to disk 
 *  Note: Currently doesn't do anything because we flush every write anyway 
 **/ 
  public void
  flush(){
    // Do nothing
  }

 /** 
 *  Write len bytes to this stream from b starting from offset off 
 **/ 
  public void
  write(byte[] b, int off, int len) throws IOException{
    byte newBytes[] = null;
    long numBytes = 0;
    int totalNumBytes = 0;
    int numBytesToWrite = 0;

    if(this.isClosed){
      throw new IOException("Stream for " + this.objId + " already closed");
    }
    numBytesToWrite = len - totalNumBytes;
    while(numBytesToWrite > 0){
      newBytes = new byte[numBytesToWrite];
      System.arraycopy(b, off + totalNumBytes, newBytes, 0, numBytesToWrite);
      Env.dprinterrln(PRINT_DEBUG,
                      "PRACTIFileOutputStream: writing " +
                      this.objId + ", " +
                      this.objOffset + ", " +
                      numBytesToWrite);
      numBytes = this.localInterface.write(this.objId,
                                           this.objOffset,
                                           numBytesToWrite,
                                           newBytes);
      Env.dprinterrln(PRINT_DEBUG,
                      "PRACTIFileOutputStream: wrote " + numBytes);
      this.objOffset += numBytes;
      totalNumBytes += numBytes;
      numBytesToWrite = len - totalNumBytes;
      assert(numBytesToWrite >= 0);
    }
  }

 /** 
 *  Write b.length bytes to this stream from b 
 **/ 
  public void
  write(byte[] b) throws IOException{
    this.write(b, 0, b.length);
  }

 /** 
 *  Write a single byte to the stream 
 **/ 
  public void
  write(int b) throws IOException{
    byte[] bytes = null;
    long numBytes = 0;
    boolean done = false;

    if(this.isClosed){
      throw new IOException("Stream for " + this.objId + " already closed");
    }
    done = false;
    while(!done){
      bytes = new byte[1];
      bytes[0] = (byte)b;
      numBytes = this.localInterface.write(this.objId,
                                           this.objOffset,
                                           1,
                                           bytes);
      assert((numBytes == 0) || (numBytes == 1));
      if(numBytes > 0){
        this.objOffset++;
        done = true;
      }
    }    
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PRACTIFileOutputStream.java...");
    PRACTIFileOutputStream.testSimple();
    System.out.println("...Finished");
  }

 /** 
 *  Run simple tests 
 **/ 
  private static void
  testSimple(){
    PRACTIFileOutputStream fos = null;
    PFOSFakeLocalInterface pfosFLI = null;
    byte[] data = null;
    byte[] localData = null;

    try{
      data = new byte[1000];
      pfosFLI = new PFOSFakeLocalInterface(data);
      fos = new PRACTIFileOutputStream(pfosFLI, new ObjId("a"));

      // Test 1: Write two bytes
      fos.write((byte)5);
      fos.write((byte)7);
      assert(data[0] == (byte)5);
      assert(data[1] == (byte)7);

      // Test 2: Write a series of bytes; let them all be written at once
      // We have written 2 bytes so far
      localData = new byte[10];
      for(int i = 0; i < localData.length; i++){
        localData[i] = (byte)i;
      }
      fos.write(localData);
      for(int i = 0; i < localData.length; i++){
        assert(localData[i] == data[i + 2]);
      }

      // Test 3: Force the stream to break up a write into multiple writes
      // We have written 12 bytes so far
      pfosFLI.setMaxNumBytesToWrite(5);
      localData = new byte[10];
      for(int i = 0; i < localData.length; i++){
        localData[i] = (byte)(i + 20);
      }
      fos.write(localData);
      for(int i = 0; i < localData.length; i++){
        assert(localData[i] == data[i + 12]);
      }

      // Test 4: Write from a subset of a byte array
      // Write 8 bytes of data and make sure that only those 8 bytes
      // are copied.
      // The maximum number of bytes written at a time is still 5
      // We have written 22 bytes so far
      localData = new byte[100];
      data[30] = 56;
      data[31] = 77;
      for(int i = 2; i < 10; i++){
        localData[i] = (byte)(i + 38);
      }
      fos.write(localData, 2, 8);
      for(int i = 0; i < 8; i++){
        assert(localData[i + 2] == data[i + 22]);
      }
      assert(data[30] == 56);
      assert(data[31] == 77);
    }catch(IOException e){
      assert(false);
    }
  }
}

 /** 
 *  Make a fake local interface 
 **/ 
class PFOSFakeLocalInterface extends PRACTIFSLocalInterface{

 /** 
 *  Data members 
 **/ 
  private byte[] data;
  private int maxNumBytesToWrite;

 /** 
 *  Constructor 
 **/ 
  public
  PFOSFakeLocalInterface(byte[] newData){
    super(null);
    this.data = newData;
    this.maxNumBytesToWrite = Integer.MAX_VALUE;
  }

 /** 
 *  Set a bound on the maximum number of bytes we can write at a time 
 **/ 
  public void
  setMaxNumBytesToWrite(int newMaxNumBytesToWrite){
    this.maxNumBytesToWrite = newMaxNumBytesToWrite;
  }

 /** 
 *  Get the bound on the maximum number of bytes we can write at a time 
 **/ 
  public int
  getMaxNumBytesToWrite(){
    return(this.maxNumBytesToWrite);
  }

 /** 
 *  Write a certain number of bytes 
 **/ 
  public long
  write(ObjId objId, long offset, long length, byte buffer[])
    throws IOException{
    int numBytesToWrite = 0;

    numBytesToWrite = (int)length;
    if(numBytesToWrite > this.maxNumBytesToWrite){
      numBytesToWrite = this.maxNumBytesToWrite;
    }
    System.arraycopy(buffer, 0, this.data, (int)offset, numBytesToWrite);
    if(false){
      throw new IOException();
    }
    return(numBytesToWrite);
  }
}
