package code;
 /** 
 *  Represent an NFS file handle 
 **/ 
public class fhandle implements Immutable{

 /** 
 *  Constants 
 **/ 
  public static final int FHSIZE = 32;

 /** 
 *  Local data members 
 **/ 
  private final byte[] data;

 /** 
 *  Constructor 
 **/ 
  private
  fhandle(byte[] newData){
    this.data = newData;
  }

 /** 
 *  Create a new filehandle with all zeros 
 **/ 
  public static fhandle
  makeAllZeroHandle(){
    byte[] data = null;
    fhandle handle = null;

    data = new byte[fhandle.FHSIZE];
    handle = new fhandle(data);
    return(handle);
  }

 /** 
 *  Create a new fhandle object without copying the array 
 **/ 
  public static fhandle
  makeHandleDangerous(byte[] data){
    fhandle handle = null;

    assert(data.length == fhandle.FHSIZE);
    handle = new fhandle(data);
    return(handle);
  }

 /** 
 *  Create a new fhandle object without copying the array 
 **/ 
  public static fhandle
  makeHandle(byte[] data){
    byte[] dataCopy = null;
    fhandle handle = null;

    assert(data.length == fhandle.FHSIZE);
    dataCopy = new byte[fhandle.FHSIZE];
    for(int i = 0; i < fhandle.FHSIZE; i++){
      dataCopy[i] = data[i];
    }
    handle = new fhandle(dataCopy);
    return(handle);
  }

 /** 
 *  Make a handle whose binary value is one larger than this one 
 **/ 
  public fhandle
  cloneIncOne(){
    fhandle newHandle = null;
    byte[] dataCopy = null;
    byte carry = 0;
    int val = 0;

    // Note: We can play around with dataCopy because getData() returns a copy
    dataCopy = this.getData();
    carry = 1;
    for(int i = (fhandle.FHSIZE - 1); (carry > 0) && (i >= 0); i--){
      val = ((int)dataCopy[i]) & 0xFF;
      val += carry;
      carry = (byte)((val & 0xFFF0) >> 8);
      dataCopy[i] = (byte)(val & 0xFF);
    }
    newHandle = fhandle.makeHandleDangerous(dataCopy);
    return(newHandle);
  }

 /** 
 *  Return the data 
 **/ 
  public byte[]
  getDataDangerous(){
    return(this.data);
  }

 /** 
 *  Return a copy of the data 
 **/ 
  public byte[]
  getData(){
    byte[] dataCopy = null;
    fhandle handle = null;

    dataCopy = new byte[fhandle.FHSIZE];
    for(int i = 0; i < fhandle.FHSIZE; i++){
      dataCopy[i] = this.data[i];
    }
    return(dataCopy);
  }

 /** 
 *  Return true if "obj" equals "this" 
 **/ 
  public final boolean
  equals(Object obj){
    fhandle fh = null;
    boolean eq = false;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    if(obj instanceof fhandle){
      fh = (fhandle)obj;
      if(fh.data.length == fhandle.FHSIZE){
        eq = true;
        for(int i = 0; (i < fhandle.FHSIZE) && eq; i++){
          eq = (this.data[i] == fh.data[i]);
        }
      }else{
        eq = false;
      }
    }else{
      eq = false;
    }
    return(eq);
  }

 /** 
 *  Compute and return a hashcode for this object 
 **/ 
  public int
  hashCode(){
    int code = 0;
    int newCode = 0;

    // 2 Java-declared fields, 2 self-declared fields
    assert(this.getClass().getDeclaredFields().length == 4);
    assert((fhandle.FHSIZE % 4) == 0);

    for(int i = 0; i < (fhandle.FHSIZE / 4); i++){
      newCode = ((((int)this.data[i * 4]) << 24) |
                 (((int)this.data[i * 4 + 1]) << 16) |
                 (((int)this.data[i * 4 + 2]) << 8) |
                 ((int)this.data[i * 4 + 3]));
      code = code ^ newCode;
    }
    return(code);
  }

 /** 
 *  Create and return a String representation for this object 
 **/ 
  public String
  toString(){
    String str = null;

    str = "[";
    for(int i = 0; i < fhandle.FHSIZE; i++){
      str += this.data[i];
    }
    str += "]";
    return(str);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.err.println("fhandle self test...");
    fhandle.testSimple();
    fhandle.testCloneIncOne();
    System.err.println("...fhandle self test succeeds");
    System.exit(0);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    byte[] data1 = null;
    byte[] data2 = null;
    byte[] data3 = null;
    fhandle fh1 = null;    
    fhandle fh2 = null;    
    fhandle fh3 = null;
    fhandle fh4 = null;
    fhandle fh5 = null;

    // Test 1
    data1 = new byte[fhandle.FHSIZE];
    for(int i = 0; i < fhandle.FHSIZE; i++){
      data1[i] = (byte)i;
    }
    fh1 = fhandle.makeHandle(data1);
    assert(fh1.equals(fh1));
    fh4 = fhandle.makeHandleDangerous(data1);
    assert(fh4.equals(fh4));
    assert(fh4.equals(fh1));
    assert(fh1.equals(fh4));

    // Test 2
    data2 = new byte[fhandle.FHSIZE];
    for(int i = 0; i < fhandle.FHSIZE; i++){
      data2[i] = (byte)i;
    }
    fh2 = fhandle.makeHandle(data2);
    assert(fh2.equals(fh2));
    assert(fh1.equals(fh2));
    assert(fh2.equals(fh1));
    assert(fh1.hashCode() == fh2.hashCode());

    // Test 3
    data3 = new byte[fhandle.FHSIZE];
    for(int i = 0; i < fhandle.FHSIZE; i++){
      data3[i] = (byte)(fhandle.FHSIZE - i);
    }
    data3[5] = 10; // Random change
    fh3 = fhandle.makeHandle(data3);
    assert(fh3.equals(fh3));
    assert(!fh1.equals(fh3));
    assert(!fh3.equals(fh1));
    assert(!fh2.equals(fh3));
    assert(!fh3.equals(fh2));
    assert(fh1.hashCode() != fh3.hashCode());
    assert(fh2.hashCode() != fh3.hashCode());
    fh5 = fhandle.makeHandleDangerous(data3);
    assert(fh5.equals(fh5));
    assert(fh5.equals(fh3));
    assert(fh3.equals(fh5));
    assert(!fh5.equals(fh2));
    assert(!fh2.equals(fh5));
    assert(fh5.hashCode() != fh1.hashCode());
    assert(fh5.hashCode() != fh2.hashCode());
    assert(fh5.hashCode() == fh3.hashCode());
  }

 /** 
 *  Test the cloneIncOne method 
 **/ 
  private static void
  testCloneIncOne(){
    fhandle handle1 = null;
    fhandle handle2 = null;
    fhandle compHandle2 = null;
    byte[] b = null;

    // Test 1
    b = new byte[fhandle.FHSIZE];
    handle1 = fhandle.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    b[fhandle.FHSIZE - 1] = 1;
    compHandle2 = fhandle.makeHandle(b);
    assert(handle2.equals(compHandle2));

    // Test 2
    b[fhandle.FHSIZE - 1] = (byte)-1;
    handle1 = fhandle.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    b[fhandle.FHSIZE - 1] = 0;
    b[fhandle.FHSIZE - 2] = 1;
    compHandle2 = fhandle.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Test 3
    b[fhandle.FHSIZE - 1] = 0;
    b[fhandle.FHSIZE - 2] = 10;
    handle1 = fhandle.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    b[fhandle.FHSIZE - 1] = 1;
    compHandle2 = fhandle.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Test 4
    for(int i = 0; i < 10; i++){
      b[fhandle.FHSIZE - i - 1] = -1;
    }
    handle1 = fhandle.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    for(int i = 0; i < 10; i++){
      b[fhandle.FHSIZE - i - 1] = 0;
    }
    b[fhandle.FHSIZE - 11] = 1;
    compHandle2 = fhandle.makeHandle(b);
    assert(compHandle2.equals(handle2));

    // Test 5
    for(int i = 0; i < fhandle.FHSIZE; i++){
      b[i] = (byte)-1;
    }
    handle1 = fhandle.makeHandle(b);
    handle2 = handle1.cloneIncOne();
    assert(handle2.equals(fhandle.makeAllZeroHandle()));
  }
}

 /** 
 *  $Log: fhandle.java,v $ 
 *  Revision 1.1  2006/07/15 02:54:21  nayate 
 *  Added version 1 of the Java library corresponding to NFS 2 
 *  
 **/ 
