package code;
 /** 
 *  Implementation of a worker thread that handles receiving a stream 
 *  of update messages 
 **/ 
import java.net.Socket;
import java.io.IOException;
import java.io.ObjectInputStream;

public class NiceRecvWorker extends Thread implements SocketMagicConstant{
  private TaggedInputStream s;
  private Core core;
  private Controller controller;
  private Socket underlyingSocket;
  
  private boolean dbg = false;

 /** 
 *  Constructor 
 **/ 
  public
  NiceRecvWorker(TaggedInputStream s,
                 Core core,
                 Controller controller,
                 Socket underlyingSocket){
    Env.performanceWarning("NiceRecvWorker TBD: add a 'negotiation' " +
                           "exchange between SubscribeBodyWorker and " +
                           "NiceRecvWorker to avoid resending data that " +
                           "the receiver already has");
    this.s = s;
    this.core = core;
    this.controller = controller;
    this.underlyingSocket = underlyingSocket;
  }

 /** 
 *  Main thread method 
 **/ 
  public void
  run(){
    NodeId senderNodeId = null;

    //
    // Read sender nodeId from stream.
    //
    try{
      Long magic = (Long)s.readTaggedObject();
      if(magic.longValue() != PREFETCH_SOCKET_MAGIC){
        Env.remoteAssert(false, "descriptive error msg");
      }

      senderNodeId = (NodeId)(s.readTaggedObject());

    }catch(Exception e){
      // 
      // Attempt to connect failed and we don't even know
      // who it was that was trying. Nothing to do but
      // have this thread die. If this was important,
      // then a controller should timeout and retry. 
      //
      System.err.println(" ATTEMPT TO CONNECT FAILED------------");
      this.closeStreamAndSocket(null, 
                                "NiceRecvWorker cannot handshake " +
                                "with sender " + e);
      return; // Thread exit.
    }
    if( dbg ){
      Env.dprintln(dbg, "NiceRecvWorker::Initiated stream to" + senderNodeId);
    }
    controller.informBodyStreamInitiated(senderNodeId);

    while(true){
      try{
        
        Object next = s.readTaggedObject();       
        if( dbg ){
	  Env.dprintln(dbg, "NiceRecvWorker::Received msg: " + next);
        }

        if(next instanceof StartBodyConnMsg) {
          //ignore
        }
        if(next instanceof BodySSStartMsg){
          BodySSStartMsg ssStart = (BodySSStartMsg) next;
          controller.informSubscribeBodySucceeded(senderNodeId, 
                                                   ssStart.getSS());
        }
        if(next instanceof BodySSEndMsg) {
          BodySSEndMsg ssEnd = (BodySSEndMsg) next;
          controller.informSubscribeBodyRemoved(senderNodeId,
                                                 ssEnd.getSS());
        }
        if(next instanceof TerminateBodyConnMsg) {
          TerminateBodyConnMsg terminateConn = (TerminateBodyConnMsg) next;
          controller.informSubscribeBodyRemoved(senderNodeId,
                                                terminateConn.getSS());
          this.closeStreamAndSocket(senderNodeId, "NiceRecvWorker received " +
                                    "terminate msg.");
          return;
        }
        if(next instanceof BodyMsg) {
          
          BodyMsg msg = (BodyMsg)next;
          long bytes = 0;
          try{
            Env.tbd("NiceRecvWorker ends up applying bodies in FIFO order."
                    + " The fix is: all NiceRecvWorker threads should"
                    + " insert the BodyMsg's they pull off the network into "
                    + " a *shared* set of priority queues, one per NodeId, "
                    + " this set of priority queues should be notified whenever "
                    + " an element of currentVV is advanced "
                    + " e.g., pending.notify(nodeId, localClock), " 
                    + " and a worker thread(s) should pull causally legal events "
                    + " out of these shared queues and call core.applyBody with "
                    + " them. May want multiple worker threads since "
                    + " DataStore:applyBody can block.");
            bytes = core.applyBody(msg);	  
         
          }catch(InterruptedException e){
            System.err.println("" + e);
            assert(false);
          }
          if(bytes < msg.getLength()){
            Env.performanceWarning("NiceRecvWorker received " +
                                   msg.getLength() + " bytes, but " +
                                   "only " + bytes + " were useful. " +
                                   "TBD: add a negotiation between " +
                                   "sender and receiver to avoid " +
                                   "sending redundant stuff.");
          }
          if(msg.isDemandReply()){
            controller.informReceiveDemandReply(senderNodeId, msg);
          }else{
            controller.informReceivePushBody(msg);
          }
        } // finished handling bodyMsg
      }catch(IOException e){
        // Includes ClosedByInterruptException
        //on exception:  call closeStreamAndSocket
        //and return (thread exit)
        this.closeStreamAndSocket(senderNodeId, 
                                  "NiceRecvWorker interrupted while " +
                                  "receiving data " + e);
        return;
      }catch(ClassNotFoundException e){
        this.closeStreamAndSocket(senderNodeId, 
                                  "NiceRecvWorker interrupted while " +
                                  "receiving data " + e);
        return;
      }catch(ClassCastException f){
        this.closeStreamAndSocket(senderNodeId, 
                                  "NiceRecvWorker interrupted while " +
                                  "receiving data " + f);
      }catch(Exception g){
        this.closeStreamAndSocket(senderNodeId, 
                                  "NiceRecvWorker interrupted while " +
                                  "receiving data " + g);
      }
    }
  }

 /** 
 *  Cleanly close the stream and socket 
 **/ 
  private void
  closeStreamAndSocket(NodeId node, String msg){
    if( dbg ){
      Env.dprintln(dbg, "NiceRecvWorker:: closing socket to "+ node);
    }
    try{
      s.close();
      underlyingSocket.close();
    }catch(Exception e){
      // Ignore errors when trying to close
    }
    if(node != null){
      controller.informBodyStreamTerminated(node);
    }
    Env.event("NiceRecvWorker closing stream: " + msg);
  }
}

//---------------------------------------------------------------------------
/* $Log: NiceRecvWorker.java,v $
/* Revision 1.9  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.8  2006/10/31 21:43:32  nalini
/* added control msgs to body streams
/*
/* Revision 1.7  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.6  2006/06/13 03:49:19  nalini
/* RMI for P2 Runtime Implemented
/*
/* Revision 1.5  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.4.2.1  2006/06/02 22:18:13  nalini
/* Supports addition and removeal of SS from Outgoing Body Streams
/*
/* Revision 1.4  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.3  2005/10/13 00:24:24  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.2  2005/03/13 09:48:12  nayate
/* Added code to work with TCP-LP
/*
/* Revision 1.1  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/* */
//---------------------------------------------------------------------------
