package code;
 /** 
 *  Implementation of a TCP Nice socket 
 **/ 
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.FileDescriptor;
import java.net.Socket;
import java.net.SocketImpl;
import java.net.InetAddress;
import java.net.SocketOptions;
import java.net.InetSocketAddress;

// For Java 1.4 compatibility
import java.net.SocketAddress;

public class NiceSocketImpl extends SocketImpl{

 /** 
 *  JNI stuff 
 **/ 
  static{
    try{
      System.loadLibrary("nicesock");
    }catch(Exception e){
      e.printStackTrace();
      System.exit(-1);
    }
    cSupportsNice = c_checkSupportNice() == 1 ? true : false;
  }

  public native static int c_checkSupportNice();
  public native int c_socket();
  public native int c_connect(int sock, String ipAddress, int port);
  public native int c_setNoDelay(int sock, int val);
  public native int c_getNoDelay(int sock);
  public native int c_close(int sock);

 /** 
 *  Constructor 
 **/ 
  public
  NiceSocketImpl(){
    this.sockFD = -1;
    this.is = null;
    this.os = null;
    this.stubSocket = null;
  }

 /** 
 *  Constructor 
 **/ 
  public
  NiceSocketImpl(int sockFD, InetAddress remoteAddress){
    if(cSupportsNice){
      this.sockFD = sockFD;
      this.remoteAddress = remoteAddress;
      this.is = new NiceSocketInputStream(this.sockFD);
      this.os = new NiceSocketOutputStream(this.sockFD);
    }
    else{
      assert(false); 
    }
  }

 /** 
 *  Accept a connection 
 **/ 
  protected
  void accept(SocketImpl s){
    assert(false);
  }

 /** 
 *  Returns the number of bytes that can be read from this socket 
 *  without blocking. 
 **/ 
  protected int
  available() throws IOException{
    assert(false);
    return(0);
  }

 /** 
 *  Binds this socket to the specified port number on the specified host. 
 **/ 
  protected void
  bind(InetAddress host, int port){
    assert(false);
  }

 /** 
 *  Closes this socket. 
 **/ 
  public void
  close() throws IOException{
    int retVal = 0;

    closeStreams();

    if(cSupportsNice){
      retVal = c_close(this.sockFD);
      if(retVal < 0){
        throw(new IOException("Error closing socket"));
      }
    }
    else{
      if(stubSocket != null){
        try{
          stubSocket.close();
        }
        finally{
          stubSocket = null;
        }
      }
    }
  }


 /** 
 *  Close my input and output streams (if any).  
 **/ 
  private void
  closeStreams() throws IOException{
    try{
      if(is != null){
        is.close();
      }
    }
    finally{
      is = null;
      try{
        if(os != null){
          os.close();
        }
      }
      finally{
        os = null;
      }
    }
  }

 /** 
 *  Connects this socket to the specified port number on the 
 *  specified host. For the cSupportsNice version, we  
 *  assume that sockFD is already set. For the !cSupportsNice 
 *  version, we assume that subSocket is already set. 
 **/ 
  public void
  connect(InetAddress address, int port) throws IOException{

    this.remoteAddress = address;

    if(cSupportsNice){
      String ipAddress = null;
      int retVal = 0;
      assert(this.sockFD >= 0);
      ipAddress = address.getHostAddress();
      retVal = c_connect(this.sockFD, ipAddress, port);
      if(retVal < 0){
        this.remoteAddress = null;
        throw(new IOException("Could not connect socket"));
      }
      this.is = new NiceSocketInputStream(this.sockFD);
      this.os = new NiceSocketOutputStream(this.sockFD);
    }
    else{
      InetSocketAddress isa = new InetSocketAddress(address, port);
      assert(this.stubSocket != null);
      this.stubSocket.connect(isa);
      this.is = this.stubSocket.getInputStream();
      this.os = this.stubSocket.getOutputStream();
    }
  }

 /** 
 *  Connects this socket to the specified port on the named host. 
 **/ 
  public void
  connect(String host, int port) throws IOException{
    InetAddress inetAddress = null;
    String ipAddress = null;
    int retVal = 0;

    inetAddress = InetAddress.getByName(host);
    connect(inetAddress, port);
  }

 /** 
 *  Creates either a stream or a datagram socket. 
 **/ 
  public void
  create(boolean stream) throws IOException{
    assert(stream);
    if(cSupportsNice){
      this.sockFD = c_socket();
      if(this.sockFD < 0){
        throw(new IOException("Could not create socket"));
      }
    }
    else{
      this.stubSocket = new Socket();
    }
  }


 /** 
 *  Returns the value of this socket's fd field. 
 **/ 
  protected FileDescriptor
  getFileDescriptor(){
    assert(false);
    return(null);
  }

 /** 
 *  Returns the value of this socket's address field. 
 **/ 
  protected InetAddress
  getInetAddress(){
    return(this.remoteAddress);
  }

 /** 
 *  Returns an input stream for this socket. 
 **/ 
  protected InputStream
  getInputStream(){
    return(this.is);
  }

 /** 
 *  Returns the value of this socket's localport field. 
 **/ 
  protected int
  getLocalPort(){
    assert(false);
    return(0);
  }

 /** 
 *  Returns an output stream for this socket. 
 **/ 
  protected OutputStream
  getOutputStream(){
    return(this.os);
  }

 /** 
 *  Returns the value of this socket's port field. 
 **/ 
  protected int
  getPort(){
    assert(false);
    return(0);
  }

 /** 
 *  Sets the maximum queue length for incoming connection indications 
 *  (a request to connect) to the count argument. 
 **/ 
  protected void
  listen(int backlog){
    assert(false);
  }

 /** 
 *  Places the input stream for this socket at "end of stream". 
 **/ 
  protected void
  shutdownInput(){
    assert(false);
  }

 /** 
 *  Disables the output stream for this socket. 
 **/ 
  protected void
  shutdownOutput(){
    assert(false);
  }

 /** 
 *  Returns the address and port of this socket as a String. 
 **/ 
  public String
  toString(){
    assert(false);
    return(null);
  }

 /** 
 *  FROM SocketOptions: 
 **/ 

 /** 
 *  Fetch the value of an option. 
 **/ 
  public Object
  getOption(int optID){
    assert(false);
    return(null);
  }

 /** 
 *  Enable/disable the option specified by optID. 
 **/ 
  public void
  setOption(int optID, Object value){
    int intVal = 0;
    Boolean boolVal = null;

    if(optID == SocketOptions.TCP_NODELAY){
      assert(value instanceof Boolean);
      boolVal = (Boolean)value;
      if(cSupportsNice){
        intVal = (boolVal.booleanValue()) ? 1 : 0;
        c_setNoDelay(this.sockFD, intVal);
        assert(c_getNoDelay(this.sockFD) == intVal);
      }
      else{
        assert(this.stubSocket != null);
        try{
          this.stubSocket.setTcpNoDelay(boolVal.booleanValue());
        }
        catch(java.net.SocketException se){
          Env.performanceWarning("NiceSocketImpl unable to setTcpNoDelay");
        }
      }
    }else{
      assert(false);
    }
  }

 /** 
 *  FROM Java 1.4: (Comment when compiling with Java 1.3) 
 **/ 

 /** 
 *  Connects this socket to the specified port on the named host. 
 **/ 
  protected void
  connect(SocketAddress address, int timeout) throws IOException{
    assert(false);
  }

 /** 
 *  Send one byte of urgent data on the socket 
 **/ 
  protected void
  sendUrgentData(int data){
    assert(false);
  }

 /** 
 *  Protected data 
 **/ 
  protected int sockFD = -1;
  protected InputStream is = null;
  protected OutputStream os = null;
  protected InetAddress remoteAddress;

  //
  // If native implementation doesn't support low priority
  // channels, then just use normal priority sockets.
  //
  public static boolean cSupportsNice;
  private Socket stubSocket = null;

}
