package code;
 /** 
 *  Represents an invalidation for a byte range of an object 
 **/ 
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;

// Used for testing
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.security.PublicKey;
import java.util.Enumeration;

import code.security.SangminConfig;
import code.security.SecurePreciseInv;

public final class ObjInvalTarget implements InvalTarget, Immutable{

 /** 
 *  Data members 
 **/ 
  private final ObjId objId;
  private final long offset;
  private final long length;
  
  private final static byte OBJ_INVAL_TARGET = 0x56; 
  
  public ObjInvalTarget(){
    objId = null;
    offset = -1;
    length = -1;
  }

 /** 
 *  Constructor: Save away passed-in values 
 **/ 
  public
  ObjInvalTarget(ObjId objId_, long offset_, long length_){
    this.objId = objId_;
    this.offset = offset_;
    this.length = length_;
  }

 /** 
 *  Constructor: Read an ObjInvalTarget object from the stream 
 **/ 
  public
  ObjInvalTarget(ObjectInputStream ois) throws IOException{
    ObjId newObjId = null;
    long newOffset = 0;
    long newLength = 0;

    try{
      newObjId = (ObjId)ois.readObject();
      newOffset = ois.readLong();
      newLength = ois.readLong();
    }catch(ClassNotFoundException e){
      System.err.println("" + e);
      assert(false);
    }
    this.objId = newObjId;
    this.offset = newOffset;
    this.length = newLength;
  }

 /** 
 *  Write this object onto the output stream 
 **/ 
  public void
  copySelfOntoOS(ObjectOutputStream oos) throws IOException{
    oos.writeObject(this.objId);
    oos.writeLong(this.offset);
    oos.writeLong(this.length);
  }

 /** 
 *  Return the object ID 
 **/ 
  public final ObjId
  getObjId(){
    return(this.objId);
  }

 /** 
 *  Return the offset 
 **/ 
  public final long
  getOffset(){
    return(this.offset);
  }

 /** 
 *  Return the length 
 **/ 
  public final long
  getLength(){
    return(this.length);
  }

 /** 
 *  Compute and return a hash code 
 **/ 
  public int
  hashCode(){
    int code = 0;

    // 2 Java-provided fields, 3 local fields
    assert(this.getClass().getDeclaredFields().length == 5);
    code = (this.objId.hashCode() ^
            (new Long(this.offset)).hashCode() ^
            (new Long(this.length)).hashCode());
    return(code);
  }

 /** 
 *  Returns true if the passed-in ObjInvalTarget is equal to this one 
 **/ 
  public boolean
  equals(Object o){
    boolean result = false;
    ObjInvalTarget oit2 = null;

    if(o instanceof ObjInvalTarget){
      oit2 = (ObjInvalTarget)o;
      result = ((this.objId.equals(oit2.objId)) &&
                (this.offset == oit2.offset) &&
                (this.length == oit2.length));
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Method inherited from InvalTarget (See InvalTarget.java) 
 **/ 
  public InvalTarget
  getIntersection(InvalTarget it){
    ObjInvalTarget oit = null;
    InvalTarget result = null;
    HierInvalTarget hit = null;

    assert((it instanceof HierInvalTarget) ||
           (it instanceof ObjInvalTarget) ||
           (it instanceof MultiObjInvalTarget));
    if(this.equals(it)){
      result = this;
    }else if(it instanceof HierInvalTarget){
      // Call this same method in HierInvalTarget.java
      hit = (HierInvalTarget)it;
      result = hit.getIntersection(this);
    }else if(it instanceof ObjInvalTarget){
      oit = (ObjInvalTarget)it;
      result = this.getObjInvalTargetIntersection(oit);
    }else{
      result = null;
      System.err.println("ObjInvalTarget.getIntersection" +
                         "(MultiObjInvalTarget) should never be called!");
      assert(false);
    }
    assert(result != null);
    return(result);
  }

 /** 
 *  Method inherited from InvalTarget (See InvalTarget.java) 
 **/ 
  public InvalTarget
  getUnion(InvalTarget it, SubscriptionSet ss){
    HierInvalTarget hit = null;
    MultiObjInvalTarget moit = null;
    InvalTarget result = null;

    // Simpler to keep union code in just one location. So,
    // we let the getUnion call of HierInvalTarget take care
    // of everything, since in our system we set up the union of
    // ObjInvalTarget and anything to be a HierInvalTarget. We note
    // that we ignore the case where we are trying to take the union
    // of two ObjInvalTargets that point to the same file. In this
    // case, although it may be possible to write getUnion to return an
    // ObjInvalTarget, we choose to return a HierInvalTarget for
    // simplicity. We do, however, make one exception; when we take
    // the union of two copies of the same ObjInvalTarget object, we
    // return one of the copies.

    assert((it instanceof ObjInvalTarget) ||
           (it instanceof HierInvalTarget) ||
           (it instanceof MultiObjInvalTarget));
    if(this.equals(it)){
      result = this;
    }else{
      // "it" is either a HierInvalTarget or a MultiObjInvalTarget
      hit = HierInvalTarget.makeHierInvalTarget(this.objId.getPath());
      result = hit.getUnion(it, ss);
    }
    return(result);
  }

 /** 
 *  Return true if ss intersects with this inval target 
 **/ 
  public final boolean intersects(SubscriptionSet ss){
    HierInvalTarget hit = null;

    // For simplicity, pass on the method to HierInvalTarget
    hit = HierInvalTarget.makeHierInvalTarget(this.objId.getPath());
    return(hit.intersects(ss));
  }

 /** 
 *  Return true if this region is empty (which it can't be!) 
 **/ 
  public final boolean
  isEmpty(){
    // NOTE: An ObjInvalTarget cannot be empty
    return(false);
  }

 /** 
 *  Return the intersected region with ObjInvalTargets 
 **/ 
  private InvalTarget
  getObjInvalTargetIntersection(ObjInvalTarget oit){
    long len = 0;
    long oitEnd = 0;
    long thisEnd = 0;
    InvalTarget isectInvTar = null;

    if(oit.objId.equals(this.objId)){
      oitEnd = oit.offset + oit.length; // NOTE: One beyond the end
      thisEnd = this.offset + this.length; // One beyond the end
      if((this.offset >= oit.offset) && (this.offset < oitEnd)){
        // Case 1: this.offset lies in the range covered by oit
        if(oitEnd < thisEnd){
          len = oitEnd - this.offset;
        }else{
          len = this.length;
        }
        isectInvTar = new ObjInvalTarget(this.objId, this.offset, len);
      }else if((oit.offset >= this.offset) && (oit.offset < thisEnd)){
        // Case 2: oit.offset lies in the range covered by this
        if(oitEnd < thisEnd){
          len = oit.length;
        }else{
          len = thisEnd - oit.offset;
        }
        isectInvTar = new ObjInvalTarget(this.objId, oit.offset, len);
      }else{
        // Either->
        // Case 3: oit's range completely preceeds this
        // Or->
        // Case 4: oit's range completely follow this
        //
        // There is no intersection. Hence, we return an empty
        // InvalTarget
        isectInvTar = new HierInvalTarget();
      }
    }else{
      // There is no intersection because the two invalidates are to
      // different objects. Hence, we return an empty InvalTarget
      isectInvTar = new HierInvalTarget();
    }
    assert(isectInvTar != null);
    return(isectInvTar);
  }

 /** 
 *  Convert to a string representation 
 **/ 
  public final String
  toString(){
    return("(" + this.objId +
           ", " + this.offset +
           ", " + this.length + ")");
  }

  public int onDiskSize(){
    return objId.toString().length()+7+8+8;
  }
  
  public static boolean matchClassCode(byte code){
    return code == OBJ_INVAL_TARGET;
    
  }
  
  public static byte getClassCode(){
    return OBJ_INVAL_TARGET;
  }
  
  public void writeExternal(ObjectOutput out) throws IOException{
    out.writeObject(this.getObjId().getPath());
    out.writeLong(this.getOffset());
    out.writeLong(this.getLength());
  }

  public void readExternal(ObjectInput in)
  throws IOException, ClassNotFoundException{
    String objIdString = (String)(in.readObject());
    long  offset = in.readLong();
    long  length = in.readLong();
    
    Field[] f = new Field[3];
    
    try{

      f[0] = ObjInvalTarget.class.getDeclaredField("objId");
      f[1] = ObjInvalTarget.class.getDeclaredField("length");
      f[2] = ObjInvalTarget.class.getDeclaredField("offset");
    }catch(NoSuchFieldException ne){
      System.err.println(ne.toString());
      ne.printStackTrace();
      System.exit(-1);

    }
    try{
      AccessibleObject.setAccessible(f, true);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    try{
      f[0].set(this, new ObjId(objIdString));
      f[1].set(this, length);
      f[2].set(this, offset);
    }catch(IllegalArgumentException ie){
      System.err.println(ie.toString());
      ie.printStackTrace();
      System.exit(-1);
    }catch(IllegalAccessException iae){
      System.err.println(iae.toString());
      iae.printStackTrace();
      System.exit(-1);
    }

    try{
      AccessibleObject.setAccessible(f, false);
    } catch (SecurityException se){
      System.err.println(se.toString());
      se.printStackTrace();
      System.exit(-1);
    }
    
  }
 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    ObjInvalTarget oit1 = null;
    ObjInvalTarget oit2 = null;
    InvalTarget result = null;
    SubscriptionSet ss = null;
    ObjectInputStream ois = null;
    ObjectOutputStream oos = null;
    ByteArrayInputStream bais = null;
    ByteArrayOutputStream baos = null;

    Env.verifyAssertEnabled();
    System.out.println("Testing ObjInvalTarget...");
    oit1 = new ObjInvalTarget(new ObjId("/rand1"), 10, 10);

    // Check simple methods
    assert(oit1.getObjId().equals(new ObjId("/rand1")));
    assert(oit1.getOffset() == 10);
    assert(oit1.getLength() == 10);

    // Check equals
    assert(oit1.equals(oit1));
    assert(oit1.equals(new ObjInvalTarget(new ObjId("/rand1"), 10, 10)));
    assert(!oit1.equals(new ObjInvalTarget(new ObjId("/rand2"), 10, 10)));

    // Check intersection routines
    oit2 = new ObjInvalTarget(new ObjId("rand2"), 10, 10);
    /*
      System.out.println(oit + ".intersects(" + oit2 +
      ") = " + oit.intersects(oit2));
      System.out.println(oit2 + ".intersects(" + oit +
      ") = " + oit.intersects(oit2));
      oit2 = new ObjInvalTarget(new ObjId("rand1"), 10, 10);
      System.out.println(oit + ".intersects(" + oit2 +
      ") = " + oit.intersects(oit2));
      System.out.println(oit + ".getIntersection(" + oit2 +
      ") = " + oit.getIntersection(oit2));
    */

    // Case 1a and 2a:
    oit1 = new ObjInvalTarget(new ObjId("/rand1"), 5, 10);
    oit2 = new ObjInvalTarget(new ObjId("/rand1"), 10, 10);
    result = oit1.getIntersection(oit2);
    assert(result.equals(new ObjInvalTarget(new ObjId("/rand1"), 10, 5)));
    assert(result.equals(oit2.getIntersection(oit1)));

    // Case 1b and 2b:
    oit1 = new ObjInvalTarget(new ObjId("/rand1"), 5, 20);
    oit2 = new ObjInvalTarget(new ObjId("/rand1"), 10, 10);
    result = oit1.getIntersection(oit2);
    assert(result.equals(new ObjInvalTarget(new ObjId("/rand1"), 10, 10)));
    assert(result.equals(oit2.getIntersection(oit1)));

    // Case 3 and 4: (Empty invalidate)
    oit1 = new ObjInvalTarget(new ObjId("/rand1"), 5, 10);
    oit2 = new ObjInvalTarget(new ObjId("/rand1"), 16, 10);
    result = oit1.getIntersection(oit2);
    assert(result.equals(new HierInvalTarget()));
    assert(result.equals(oit2.getIntersection(oit1)));

    // Check the union operation
    oit1 = new ObjInvalTarget(new ObjId("/rand1"), 5, 10);
    oit2 = new ObjInvalTarget(new ObjId("/rand1"), 5, 10);
    ss = SubscriptionSet.makeSubscriptionSet("/:/*");
    result = oit1.getUnion(oit2, ss);
    assert(result.equals(new ObjInvalTarget(new ObjId("/rand1"), 5, 10)));
    assert(result.equals(oit2.getUnion(oit1, ss)));

    // Force the union operation to return an imprecise invalidate
    oit1 = new ObjInvalTarget(new ObjId("/rand1"), 5, 10);
    oit2 = new ObjInvalTarget(new ObjId("/rand2"), 5, 10);
    ss = SubscriptionSet.makeSubscriptionSet("/asdf");
    result = oit1.getUnion(oit2, ss);
    assert(result.equals(HierInvalTarget.makeHierInvalTarget("/rand1:" +
                                                             "/rand2")));
    assert(result.equals(oit2.getUnion(oit1, ss)));

    // Test intersects
    oit1 = new ObjInvalTarget(new ObjId("/a/b"), 10, 20);
    ss = SubscriptionSet.makeSubscriptionSet("/a/b");
    assert(oit1.intersects(ss));

    oit1 = new ObjInvalTarget(new ObjId("/a/b/c"), 10, 20);
    ss = SubscriptionSet.makeSubscriptionSet("/a/*:/b/*");
    assert(oit1.intersects(ss));

    oit1 = new ObjInvalTarget(new ObjId("/a/b/c"), 10, 20);
    ss = SubscriptionSet.makeSubscriptionSet("/a/b:/a/b/c/d");
    assert(!oit1.intersects(ss));

    oit1 = new ObjInvalTarget(new ObjId("/a/b/c"), 10, 20);
    ss = SubscriptionSet.makeSubscriptionSet("/a/b/*:/a/c/d");
    assert(oit1.intersects(ss));

    try{
      // Test copySelfOntoOS and the constructor that reads from an InputStream
      oit1 = new ObjInvalTarget(new ObjId("/a/b"), 30, 37);
      baos = new ByteArrayOutputStream();
      oos = new ObjectOutputStream(baos);
      oit1.copySelfOntoOS(oos);
      oos.flush();
      bais = new ByteArrayInputStream(baos.toByteArray());
      ois = new ObjectInputStream(bais);
      oit2 = new ObjInvalTarget(ois);
      assert(oit1 != oit2);
      assert(oit1.equals(oit2));
      assert(oit2.equals(oit1));
    }catch(IOException e){
      System.err.println("" + e);
      assert(false);
    }

    // Test hashCode()
    oit1 = new ObjInvalTarget(new ObjId("/a/b"), 10, 20);
    oit2 = new ObjInvalTarget(new ObjId("/a/c"), 10, 20);
    assert(oit1.hashCode() != oit2.hashCode());

    oit1 = new ObjInvalTarget(new ObjId("/c/d"), 30, 20);
    oit2 = new ObjInvalTarget(new ObjId("/c/d"), 30, 20);
    assert(oit1.hashCode() == oit2.hashCode());

    System.out.println("...Finished");
  }
}

//---------------------------------------------------------------------------
/* $Log: ObjInvalTarget.java,v $
/* Revision 1.20  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.19  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.18  2005/03/02 00:25:33  nayate
/* Tested/fixed minor bugs
/*
/* Revision 1.17  2005/03/01 01:03:24  nayate
/* Modified for new code
/*
/* Revision 1.16  2005/02/28 22:59:12  nayate
/* Modified for new code
/*
/* Revision 1.15  2004/07/28 14:27:35  dahlin
/* Added sanity checks for immutable objects
/*
/* Revision 1.14  2004/05/20 04:56:23  nayate
/* Fixed a bug with the intersection routine for independent objects
/*
/* Revision 1.13  2004/05/11 03:49:45  zjiandan
/* Made some changes to get the first version to work.
/*
/* Revision 1.12  2004/05/04 06:26:42  nayate
/* Finished (but didn't test) ObjInvalTarget
/*
/* Revision 1.11  2004/05/03 19:35:54  nayate
/* Started making changes to allow passing an interest set to all union
/* operations.
/*
/* Revision 1.10  2004/04/30 19:23:22  nayate
/* Minor change
/*
/* Revision 1.9  2004/04/26 21:05:13  nayate
/* Changed "intersects" to allow intersection with DirectoryInterestSet
/*
/* Revision 1.8  2004/04/26 20:59:46  nayate
/* Changed code to allow intersection with DirectoryInterestSet
/*
/* Revision 1.7  2004/04/21 17:03:58  nayate
/* Added a new "isEmpty" method to InvalTarget and ObjInvalTarget
/*
/* Revision 1.6  2004/04/20 04:25:36  nayate
/* Tested ObjInvalTarget.java
/*
/* Revision 1.5  2004/04/18 07:15:58  nayate
/* Added a method to GeneralInv, finished (but didn't test) PreciseInv
/*
/* Revision 1.4  2004/04/18 04:13:03  nayate
/* Finished (but didn't test) ObjInvalTarget and associated classes
/*
/* Revision 1.3  2004/04/16 16:05:17  nayate
/* Subset of PreciseInv implemented + removed some compiler errors.
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
