package code;
 /**  OutgoingBodyConnection
 *
 *   A dynamic outgoing connection for subscribing bodies at the sender side.
 *   
 *   rewrite of SubsscribeBodyWorker
 * 
 *   we assume that there is only one body connection to a destination
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import java.net.Socket;
import java.util.Vector;
import java.util.Enumeration;
import java.io.IOException;
import java.io.EOFException;
import java.io.OutputStream;
import java.io.ObjectOutputStream;
import java.io.NotSerializableException;
import java.io.FileOutputStream;

public class OutgoingBodyConnection extends Thread{

  /*
   * This thing's job is to push updates to
   * another node.
   *
   * For now, a simple algorithm. We maintain
   * a local priority queue. When the subscription
   * starts, the local queue is *empty*. 
   * We ask the core to tell us about each
   * read that happens and we put the read into
   * a priority queue. A woker
   * thread drains the queue and sends the reads 
   * to the remote node.
   */

  private Core core;
  private Controller controller;
  private UpdatePriorityQueue upq;  // Body Updates that we have not sent
  private OutgoingBodyQueue sendQueue; // stuff to send on this connection
  private StreamPool streams;
  private NodeId destId;
  private String destDNS;
  private int portBody;
  private SubscriptionSet currentSS;

  private boolean timeToDie;

  private static final boolean dbg = false;


 /** 
 *  Constructor 
 **/ 
  public OutgoingBodyConnection(StreamPool updateStreams,
				Core core, Controller controller,
				NodeId nodeId,
				String nodeDNS,
				int portBody) {

    assert(core != null);
    this.core = core;
    this.destId = nodeId;
    this.controller = controller;
    this.upq = new UpdatePriorityQueue(destId);
    this.sendQueue = new OutgoingBodyQueue(upq);
    this.streams = updateStreams;
    this.destDNS = nodeDNS;
    this.portBody = portBody;
    this.currentSS = SubscriptionSet.makeEmptySet();

    Env.performanceWarning("TBD: persistent update log channel needs to " +
			   "add a 'negotiation' -- Before sending " +
			   "object X with timestamp Y, ask if doing so " +
			   "provides new information (same thing for " +
			   "bound invalidates) ... This will require a " +
			   "bit of rewriting -- either this thread can " +
			   "have a 2-way socket rather than a stream or " +
			   "we need two threads that rendevous via some " +
			   "data structure...");

    this.timeToDie = false;

  }



 /** 
 *  add new subscription set to the connection 
 **/ 
  
  public void addSubscriptionSet(SubscriptionSet newSubset, VV newStartVV){    
    assert !timeToDie;

    
    core.subscribeUpdates(upq, newSubset, newStartVV);
   
    //
    //  Derivation of the currentSS is also done in UPQNotifier.
    //  maybe we should combine it so that it happens in only one place
    //
    currentSS = currentSS.getCompleteUnion(newSubset);
    sendQueue.add(new BodySSStartMsg(newSubset));
    controller.informOutgoingSubscribeBodyInitiated(destId, newSubset);
    if(dbg){
      Env.dprintln(dbg,"OutgoingBodyConnection: added subscription set :( " + newSubset + ", " 
		   + newStartVV + ")"); 

    }
  }
  

 /** 
 *  removeSubset 
 **/ 
  public void removeSubscriptionSet(SubscriptionSet ss){
    
    assert !timeToDie;

    core.removeSubscribeUpdates(upq, ss);
    //
    //  Derivation of the currentSS is also done in UPQNotifier.
    //  maybe we should combine it so that it happens in only one place
    //
    try{
      currentSS = currentSS.remove(ss, true);
      sendQueue.add(new BodySSEndMsg(ss));
    }catch(IllegalRemoveSubscriptionSetException e){
      assert false : ("" + e);
    }

    if(dbg){
      Env.dprintln(dbg, "OutgoingBodyConnection: removed Subscription Set:( " 
		   + ss + ")"); 
    }

    controller.informOutgoingSubscribeBodyTerminated(destId, ss);
   
    // 
    // it should NOT automatically close the connection if the subsciption set
    // is empty, instead it should wait for directive from controller
    //

  }
  
 /** 
 * set timeToDie so that the thread would stop at some point 
 **/ 
  public void close(){
    assert !timeToDie;
    sendQueue.add(new TerminateBodyConnMsg(currentSS));
  }
  
 /** 
 *  start the worker thread 
 *  caller should call this one to start the worker immediately after the  
 *  object is created. 
 **/ 
  public void start(){
    (new Thread(this)).start();  
  }
				

 /** 
 *   main thread method  
 *   copied from subscribe body worker 
 **/ 
 

  public void run(){
    Object obj = null;
    Object msg = null;

    // send StartBodyConnMsg so as to  establish the connection 
    sendMsg(new StartBodyConnMsg());
    
    controller.informOutgoingBodyStreamInitiated(destId, true);
    if(dbg){
      Env.dprintln(dbg, "OutgoingBodyConnection: Established to "+ destId);
    }

    while(!timeToDie){
      Env.dprintln(dbg, "OutgoingBodyConnection: trying to get next from queue");
      obj = sendQueue.getNext();
      Env.dprintln(dbg, "OutgoingBodyConnection: got from queue " + obj);
        
      msg = null;
      if(obj instanceof MetaBody) {
	MetaBody mb = (MetaBody) obj;

	// note: MetaBody gives you a consolidated range of updates
	// we may need to do multiple reads to get all the 
	// bodies for the whole range
	ObjId reqObjId = mb.getObjId();
	long reqOffset = mb.getOffset();
	long reqLength = mb.getLength();
	  
	try{
	  while(reqLength > 0) {
	    BodyMsg bMsg = this.core.read(reqObjId,
					  reqOffset,
					  reqLength,
					  false, false, true, -1);
	    long bLen = bMsg.getLength();
	    reqOffset += bLen;
	    reqLength -= bLen;
	      
	    sendMsg(bMsg);
	  }
	}
	catch(EOFException e){
	  continue; // Data we prefetched is gone... Skip it and go on.
	}
	catch(ObjNotFoundException e){
	  continue; // Data we prefetched is gone... Skip it and go on.
	}
	catch(ReadOfInvalidRangeException e){
	  continue; // Data we prefetched is gone... Skip it and go on.
	}
	catch(ReadOfHoleException e){
	  continue; // Data we prefetched is gone... Skip it and go on.
	}
	catch(IOException e){
	  if(dbg){
	    Env.dprintln(dbg, "OutgoingBodyConnection: IO Exception");
	  }
	  continue; // Something went wrong.. skip and go on
	}
      } else {
	msg = obj;
	if(msg instanceof TerminateBodyConnMsg){
	  timeToDie = true;
	}
	sendMsg(msg);
      }
      
    }
    
    if(dbg){  
      Env.dprintln(dbg, "OutgoingBodyConnection: Cleaning up");
    }
    cleanup();
  }     
   

  private void sendMsg(Object msg){
    try{
      assert(msg != null);
      if(dbg){
	Env.dprintln(dbg, "OutgoingBodyConnection: Sending msg: " + msg + " to " + destId);
      }
      streams.send(destDNS, portBody, msg);
    }catch(UnableToConnectException e){
      if(dbg){
	Env.dprintln(dbg, "OutgoingBodyConnection: Unable to Connect Exception");
      }
    }
  }
 /** 
 *  Informs the core NOT to send updates to upq before closing 
 **/ 
  private void cleanup(){
    core.removeSubscribeUpdates(upq, currentSS);
    if(dbg){
      Env.dprintln(dbg, "OutgoingBodyConnection: To " + destId + " closed");
    }
    controller.informOutgoingBodyStreamTerminated(destId,currentSS, true);
  }

 /** 
 *  Methods used by OutgoingBodyConnectionUnit for testing 
 **/ 
 
 /** 
 *  testSetBytes 
 **/ 
  public void
  testSetBytes(byte b[], int len, byte value){
    int ii;
    for(ii = 0; ii < len; ii++){
      b[ii] = value;
    }
  }

 /** 
 *  testWriteBound 
 **/ 
 public void 
  testWriteBound(ObjId id, long offset, int len, byte val){
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    
    byte b[] = new byte[len];
    testSetBytes(b, len, val);
    ImmutableBytes ib = new ImmutableBytes(b);
    try{
      this.core.write(id, offset, len, ib, true, Long.MAX_VALUE);
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


 /** 
 *  testWriteUnBound 
 **/ 
 public void 
  testWriteUnBound(ObjId id, long offset, int len, byte val){
    ObjInvalTarget oit = new ObjInvalTarget(id, offset, len);
    
    byte b[] = new byte[len];
    testSetBytes(b, len, val);
    ImmutableBytes ib = new ImmutableBytes(b);
    try{
      this.core.write(id, offset, len, ib, false, 0);
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


 /** 
 *   Makes the config file for the experiment 
 **/ 
  private static void makePractiConfig(){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(0),
                            "localhost",
                            9588,
                            9589,
                            9591,
                            9592,
                            9590,
                            "test-node0.db",
                            "/*",
                            -1L,
                            "localhost",
                            9593,
                            9594,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(1),
			   "localhost",
			   9488,
			   9489,
			   9491,
			   9492,
			   9490,
                           "test-node1.db",
			   "/*",
			   -1L,
			   "localhost",
			   9493,
			   9494,
			   -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);

  }
					

}
//---------------------------------------------------------------------------    
/* $Log: OutgoingBodyConnection.java,v $
/* Revision 1.20  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
/* Revision 1.19  2007/05/11 21:07:09  nalini
/* fixed outgoing body connection, so that it tries multiple reads to send the whole range of the metabody received from upq
/*
/* Revision 1.18  2007/05/11 02:14:29  nalini
/* updated outgoing body subscription so that it sends bodies from startVV
/*
/* Revision 1.17  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.16  2007/03/11 04:16:54  zjiandan
/* Add timeout into read interface and expose acceptstamp in LocalInterface.write.
/*
/* Revision 1.15  2007/03/01 06:39:44  nalini
/* added support for maxBoundHop at Local Interface
/*
/* Revision 1.14  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.13  2006/12/08 21:36:26  nalini
/* reverting to the old read interface
/*
/* Revision 1.11  2006/11/14 18:22:59  nalini
/* added OutgoingBodyConnectionUnit
/*
/* Revision 1.10  2006/11/02 23:10:11  nalini
/* cleaned up testing code
/*
/* Revision 1.9  2006/11/01 17:04:32  nayate
/* Added a parameter to control log sync-ing
/*
/* Revision 1.8  2006/10/31 21:43:32  nalini
/* added control msgs to body streams
/*
/* Revision 1.7  2006/09/05 04:07:21  nayate
/* Fixed to handle new SubscriptionSet.remove() interface
/*
/* Revision 1.6  2006/09/01 21:32:24  dahlin
/* PicSharReader test now works for case when writer dies and restarts
/*
/* Revision 1.5  2006/08/15 21:46:24  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.4  2006/06/14 22:50:09  nalini
/* Changed nice sockets to normal sockets for outgoing body connections
/*
/* Revision 1.3  2006/06/13 03:49:19  nalini
/* RMI for P2 Runtime Implemented
/*
/* Revision 1.2  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.1.2.1  2006/06/02 22:18:13  nalini
/* Supports addition and removeal of SS from Outgoing Body Streams
*/
//---------------------------------------------------------------------------
