package code;
/* OutgoingBodyConnectionPool
 *
 *   Container for OutgoingBodyConnections of body subscriptions.
 *   
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import java.util.Hashtable;
import java.util.Enumeration;

public class OutgoingBodyConnectionPool
{
  private Hashtable pool; // netAddr --> OutgoingConnection
  private Core core;
  private Controller controller;
  private StreamPool updateStreams;
  private boolean shutdown = false;
  
  private boolean dbg = false;
  
 /** 
 *  Constructor 
 **/ 
  public OutgoingBodyConnectionPool(Core core, Controller controller,
				    StreamPool updateStreams){
    pool = new Hashtable();
    this.core = core;
    this.controller = controller;
    this.updateStreams = updateStreams;
  }
  
 /** 
 *   add a subscriptionset to a connection 
 *        
 *      For simplicity: here we just use one connection per (sender, receiver); 
 *      tbd: for multiple connections can take the simple policy such as: 
 *          always the most advanced stream i.e. the one with largest prevVV. 
 *          if the all prevVV is much less than cvv, then create a new stream? 
 *    Note: multi-stream might not worthwhile, because of overhead and the bottleneck 
 *    might be accessing the log, and there's redundant OUT imprecise invalidations. 
 *  
 *    currently, startVV is ignored 
 **/ 
  synchronized public void addToConnection(NodeId receiverId,
                                           String receiverDNS,
                                           int portBody,
                                           SubscriptionSet ss,
                                           VV startVV){
    if( dbg ){
      Env.dprintln(dbg,"OutgoingBodyConnectionPool: addToConnectionCAlled");
    }
    if(shutdown){
      return;
    }
    
    NetAddr netAddr = new NetAddr(receiverDNS, portBody);
    OutgoingBodyConnection oc = (OutgoingBodyConnection)pool.get(netAddr);
    if(oc == null){
      oc = new OutgoingBodyConnection(updateStreams, core, controller, 
                                  receiverId, receiverDNS, portBody);
      pool.put(netAddr, oc);      
      oc.start();
      if( dbg ){
	Env.dprintln(dbg,"OutgoingBodyConnectionPool: started Outgoing connection");
      }
    } 
    oc.addSubscriptionSet(ss, startVV);
    
  }

 /** 
 *  remove a subscriptionset from a connection 
 **/ 
  synchronized public void removeFromConnection(NodeId receiverId,
                                                String receiverDNS,
                                                int portBody,
                                                SubscriptionSet ss){
    if( dbg ){
      Env.dprintln(dbg,"OutgoingBodyConnectionPool: removeFromConnectionCalled");
    }

    if(shutdown){
      return;
    }
    NetAddr netAddr = new NetAddr(receiverDNS, portBody);
    OutgoingBodyConnection oc = (OutgoingBodyConnection)(pool.get(netAddr));
   

    if (oc != null){
      oc.removeSubscriptionSet(ss);
    }
    
  }

 /** 
 *  remove a connection 
 **/ 
  synchronized public void removeConnection(NodeId receiverId,
                                            String receiverDNS,
                                            int portBody){
    if(shutdown){
      return;
    }
    NetAddr netAddr = new NetAddr(receiverDNS, portBody);
    OutgoingBodyConnection oc = (OutgoingBodyConnection)(pool.get(netAddr));
    
    
    if (oc != null){
      oc.close();
      pool.remove(netAddr);
    }
  }

 /** 
 *  shutdown all connections  
 **/ 
  synchronized void shutdownAll(){
    for (Enumeration e = pool.keys(); e.hasMoreElements();){
      NetAddr key = (NetAddr)e.nextElement();
      OutgoingBodyConnection oc = (OutgoingBodyConnection)(pool.get(key));
      assert oc != null;
      oc.close();
      oc = null;
      pool.remove(key);
      
    }
  
    shutdown = true;
  }
}

//---------------------------------------------------------------------------
/* $Log: OutgoingBodyConnectionPool.java,v $
/* Revision 1.5  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.4  2006/06/14 22:50:09  nalini
/* Changed nice sockets to normal sockets for outgoing body connections
/*
/* Revision 1.3  2006/06/13 03:49:19  nalini
/* RMI for P2 Runtime Implemented
/*
/* Revision 1.2  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.1.2.1  2006/06/02 22:28:25  nalini
/* added outgoingBodyingConnectionPool
/*
/* 
/* */
//---------------------------------------------------------------------------
