package code;
/* OutgoingBodyConnectionUnit
 *
 *  Tests the OutgoingBodyConnectionClass
 *   
 *   we assume that there is only one body connection to a destination
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class OutgoingBodyConnectionUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals

  private NodeId senderId = new NodeId(1);
  private NodeId receiverId = new NodeId(0); 


  private OutgoingBodyConnection oc;
  private RMIClient rmiClient1;
  private RMIClient rmiClient2;
  private SocketServer socketServer1;
  private SocketServer socketServer2;
  private Core fakeCore1;
  private Core fakeCore2;
  private Controller controller1;
  private Controller controller2;
  private LocalInterface li2;
  private LinkedList receiverEvents;
  private LinkedList senderEvents;

 /** 
 *   Basic Constructor - called by test runners 
 **/ 

  public OutgoingBodyConnectionUnit(final String s){
    super(s);
  }
  
 /** 
 *   Set up environment in which tests must run 
 **/ 
  protected void setUp() throws Exception {
    super.setUp();

   //Env.verifyAssertEnabled();
    System.out.println("Testing OutgoingBodyConnection.java...");
    
    makePractiConfig();

    //Initialize sender
    rmiClient1 = new RMIClient();
    fakeCore1 = new Core(rmiClient1, false, true, senderId, false);
    fakeCore1.testSkipRecoverLocalState();
    controller1 = new OutgoingBodyConnectionUnitController(fakeCore1);
    StreamPool prefetchStreams1 = new StreamPool(fakeCore1.getMyNodeId());
    socketServer1 = new SocketServer(fakeCore1, controller1);


    //Initialize receiver
    rmiClient2 = new RMIClient();
    fakeCore2 = new Core(rmiClient2, false, true, receiverId, false);
    fakeCore2.testSkipRecoverLocalState();
    controller2 = new OutgoingBodyConnectionUnitController(fakeCore2);
    socketServer2 = new SocketServer(fakeCore2, controller2);
    li2 = new LocalInterface(controller2, fakeCore2);
    

    // set up inval subscription
    // OutgoingConnectionPool invalPool 

    OutgoingConnectionPool invalConnectionPool = null;
    OutgoingInvalConnectionPool newInvalConPool = null;
    SubscriptionSet allSet = SubscriptionSet.makeSubscriptionSet("/*");
    if(!Core.useNewIterator){
      invalConnectionPool = new OutgoingConnectionPool(fakeCore1, controller1);
      invalConnectionPool.addToConnection(receiverId, 
          Config.getDNS(receiverId), 
          Config.getPortInval(receiverId), 
          allSet,
          AcceptVV.makeVVAllNegatives(),
          true,
          true);
    }else{
      newInvalConPool = new OutgoingInvalConnectionPool(fakeCore1, controller1);
      newInvalConPool.addToConnection(receiverId, 
          Config.getDNS(receiverId), 
          Config.getPortInval(receiverId), 
          allSet,
          AcceptVV.makeVVAllNegatives(),
          true,
          true);
    }
    
    

    oc = new OutgoingBodyConnection(prefetchStreams1,						  
                                    fakeCore1, 
                                    controller1,
                                    receiverId,
                                    Config.getDNS(receiverId),
                                    Config.getPortBody(receiverId));
   
  }

//---------------------------------------------------------------------------
// Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {
    try{
      socketServer1.shutdown();
      socketServer1 = null;
      socketServer2.shutdown();
      socketServer2 = null;
      fakeCore1.syncStateToDisk();
      fakeCore1.close();
      fakeCore1 = null;
      fakeCore2.syncStateToDisk();
      fakeCore2.close();
      fakeCore2 = null;
      super.tearDown();
      Thread.currentThread().sleep(500);
    }catch(Exception e) {
      //ignore
    }

  }

  //----------------------------------------------------------------------
  // testOutgoingBodyConnection - main test code
  //----------------------------------------------------------------------
  public void 
  testOutgoingBodyConnection() {
  
  //populate expected events
    receiverEvents = new LinkedList();
    senderEvents = new LinkedList();
    populateExpectedEvents(receiverEvents, senderEvents);
    
    //setup controllers
    ((OutgoingBodyConnectionUnitController)controller1).setEvents(receiverEvents, senderEvents);
    ((OutgoingBodyConnectionUnitController)controller2).setEvents(receiverEvents, senderEvents);

    //establish connection!  
    oc.start();
    try{
      Thread.sleep(1000);
    } catch(Exception e){
    }

  
    byte aByte= 65;
    byte bByte = 66;
    byte cByte = 67;

    SubscriptionSet appleSet = SubscriptionSet.makeSubscriptionSet("/apple");
    SubscriptionSet bananaSet = SubscriptionSet.makeSubscriptionSet("/banana");
    SubscriptionSet cherrySet = SubscriptionSet.makeSubscriptionSet("/cherry");

    ObjId appleObj = new ObjId("/apple");
    ObjId bananaObj = new ObjId("/banana");
    ObjId cherryObj = new ObjId("/cherry");

    oc.addSubscriptionSet(appleSet, AcceptVV.makeVVAllNegatives());
    oc.testWriteUnBound(appleObj, 0, 5, aByte);
    oc.testWriteUnBound(bananaObj, 0, 5, aByte);
    oc.testWriteUnBound(cherryObj, 0, 5, aByte);
 
    System.out.println("waiting to send writes...1");
    try{
      Thread.sleep(1000);
    } catch(Exception e){
    }
    
    oc.addSubscriptionSet(bananaSet, AcceptVV.makeVVAllNegatives());    
    oc.testWriteUnBound(appleObj, 0, 5, bByte);
    oc.testWriteUnBound(bananaObj, 0, 5, bByte);
    oc.testWriteUnBound(cherryObj, 0, 5, bByte);
    System.out.println("waiting to send writes...2");

    try{
      Thread.sleep(1000);
    } catch(Exception e){
    }
   
    oc.removeSubscriptionSet(appleSet);
    oc.testWriteUnBound(appleObj, 0, 5, cByte);
    oc.testWriteUnBound(bananaObj, 0, 5, cByte);
    oc.testWriteUnBound(cherryObj, 0, 5, cByte);
    
    System.out.println("waiting to send writes...3");
    oc.close();
    
    if (senderEvents.size() > 0 || receiverEvents.size() > 0){
      //give it some time to handle all the events
      try{
        Thread.sleep(5000);
      }catch(Exception e){
      }
    }
    if(senderEvents.size() > 0 || receiverEvents.size() > 0) {
      System.out.println("SenderEvents left:" + senderEvents);
      System.out.println("ReceiverEvents left:" + receiverEvents);
      assert(false) : "Not all events have gone through";
    }

    System.out.println("...Finished");
 
 
  }

       
//---------------------------------------------------------------------------
//  Makes the config file for the experiment
//---------------------------------------------------------------------------
  private static void makePractiConfig(){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(0),
                            "localhost",
                            9501,
                            9502,
                            9503,
                            9504,
                            9505,
                            "test-node0.db",
                            "/*",
                            -1L,
                            "localhost",
                            9506,
                            9507,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(1),
			   "localhost",
			   9401,
			   9402,
			   9403,
			   9404,
			   9405,
                           "test-node1.db",
			   "/*",
			   -1L,
			   "localhost",
			   9406,
			   9407,
			   -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);

  }
	

  public static Test suite(){
    TestSuite suite = new TestSuite(OutgoingBodyConnectionUnit.class);
    return suite;
  }

  public static void main(String s[]) {
    String name = "OutgoingBodyConnectionUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
         vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new OutgoingBodyConnectionUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }

  public void populateExpectedEvents(LinkedList receiverEvents, LinkedList senderEvents) {
    //establishing connection
    senderEvents.add("informOutgoingBodyStreamInitiated:"+receiverId); 
    receiverEvents.add("informBodyStreamInitiated:"+senderId);
 
    
    //adding apple SS
    senderEvents.add("informOutgoingSubscribeBodyInitiated:"+receiverId+":/apple");
    receiverEvents.add("informSubscribeBodySucceeded:"+senderId+":/apple");
    
       

   //adding banana SS
    senderEvents.add("informOutgoingSubscribeBodyInitiated:"+receiverId+":/banana");
    receiverEvents.add("informSubscribeBodySucceeded:"+senderId+":/banana");
    
 
    //removing apple SS
    senderEvents.add("informOutgoingSubscribeBodyTerminated:"+receiverId+":/apple");
    receiverEvents.add("informSubscribeBodyRemoved:"+senderId+":/apple");

    //Terminating the connection
    senderEvents.add("informOutgoingBodyStreamTerminated:"+receiverId+":/banana");
    receiverEvents.add("informSubscribeBodyRemoved:"+senderId+":/banana");
    receiverEvents.add("informBodyStreamTerminated:"+senderId);

    
  }

//---------------------------------------------------------------------------
// Private class which is the controller for this test
//---------------------------------------------------------------------------

  class OutgoingBodyConnectionUnitController extends LocalController{
    LinkedList receiverEvents = null;
    LinkedList senderEvents = null;

    private boolean dbg = false;
    
    public OutgoingBodyConnectionUnitController(Core core){
      super(core);
    }

    public void setEvents(LinkedList receiverEvents, LinkedList senderEvents){
      this.receiverEvents = receiverEvents;
      this.senderEvents = senderEvents;
    }

    public void informOutgoingBodyStreamInitiated(NodeId targetNodeId, 
						  boolean placeholderWriterSet){
      if(dbg){
	Env.dprintln(dbg, "ReceivedEvent--informOutgoingBodyStreamInitiated:" + targetNodeId);
      }
      if(senderEvents!= null && senderEvents.size() > 0) {
        String str = (String) senderEvents.getFirst();
        if(dbg){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informOutgoingBodyStreamInitiated:"+targetNodeId)):
          "informOutgoingBodyStreamInitiated - not as expected";
        senderEvents.removeFirst();
      }
    }
    
    public void informBodyStreamInitiated(NodeId senderNodeId){
      if(dbg){
	Env.dprintln(dbg, "ReceivedEvent--informBodyStreamInitiated:" + senderNodeId);
      }
      if(receiverEvents!=null && receiverEvents.size() > 0) {
        String str = (String) receiverEvents.getFirst();
        if( dbg ){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informBodyStreamInitiated:"+senderNodeId)):
          "informBodyStreamInitiated - not as expected";
        receiverEvents.removeFirst();
      }
    }
    

    public void informOutgoingSubscribeBodyInitiated(NodeId targetNodeId, 
                                                     SubscriptionSet ss){
      if( dbg ){
	Env.dprintln(dbg, "ReceivedEvent--informOutgoingSubscribeBodyInitiated: " + targetNodeId + ":" + ss);
      }
      if(senderEvents != null && senderEvents.size() > 0){
        String str = (String) senderEvents.getFirst();
	if(dbg){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informOutgoingSubscribeBodyInitiated:"+targetNodeId+":"+ss)):
          "informOutgoingSubscribeBody - not as expected";
        senderEvents.removeFirst();
      }
    }
    
    
    public void informSubscribeBodySucceeded(NodeId senderNodeId, SubscriptionSet ss){
     if(dbg){
       Env.dprintln(dbg, "ReceivedEvent--informSubscribeBodySucceeded:" + senderNodeId + ":" + ss);
     }
     if(receiverEvents!=null && receiverEvents.size() > 0) {
       String str = (String) receiverEvents.getFirst();
       if(dbg){
	 Env.dprintln(dbg, "ExpectedEvent--" + str);
       }
       assert(str.equals("informSubscribeBodySucceeded:"+senderNodeId+":"+ss)):
         "informSubscribeBodySucceeded - not as expected";
       receiverEvents.removeFirst();
     }
    }
    
    public void informOutgoingSubscribeBodyTerminated(NodeId targetNodeId, 
                                                     SubscriptionSet ss){     
      if(dbg){
	Env.dprintln(dbg, "ReceivedEvent--informOutgoingSubscribeBodyTerminated: " + targetNodeId + ":" + ss);
      }
      if(senderEvents!=null && senderEvents.size() > 0 ) {
        String str = (String) senderEvents.getFirst();
	if(dbg){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informOutgoingSubscribeBodyTerminated:"+targetNodeId+":"+ss)):
          "informOutgoingSubscribeBodyTerminated - not as expected";
        senderEvents.removeFirst();
      }
    }
    
    public void informSubscribeBodyRemoved(NodeId senderNodeId, SubscriptionSet ss){
      if( dbg ){
	Env.dprintln(dbg, "ReceivedEvent--informSubscribeBodyRemoved:" + senderNodeId + ":" + ss); 
      }
      if(receiverEvents!=null && receiverEvents.size() > 0) {
        String str = (String) receiverEvents.getFirst();
        if( dbg ){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informSubscribeBodyRemoved:"+senderNodeId+":"+ss)):
          "informSubscribeBodyRemoved - not as expected";
        receiverEvents.removeFirst();
      }
    }
    
    public void informReceiveInval(GeneralInv inv , NodeId senderId) {
      ObjId objId = ((ObjInvalTarget)inv.getInvalTarget()).getObjId();
      if(dbg){
	Env.dprintln(dbg, "Controller: informReceiveInval:"+objId);
      }
    }

    public void informBodyStreamTerminated(NodeId senderId) {
      if(dbg){
	Env.dprintln(dbg, "ReceivedEvent--informBodyStreamTerminated:"+senderId);
      }
      if(receiverEvents !=null && receiverEvents.size() > 0) {
        String str = (String) receiverEvents.getFirst();
        if(dbg){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informBodyStreamTerminated:"+senderId)):
          "informBodyStreamTerminated - not as expected";
        receiverEvents.removeFirst();
      }
    }

    public void informOutgoingBodyStreamTerminated(NodeId targetNodeId, 
                                                   SubscriptionSet ss,                                                
                                                   boolean placeholderWriterSet){
      if(dbg){
	Env.dprintln(dbg, "ReceivedEvent--informOutgoingBodyStreamTerminated:" + targetNodeId + ":" + ss);
      }
      if(senderEvents !=null && senderEvents.size() > 0) {
        String str = (String) senderEvents.getFirst();
	if(dbg){
	  Env.dprintln(dbg, "ExpectedEvent--" + str);
	}
        assert(str.equals("informOutgoingBodyStreamTerminated:"+targetNodeId + ":" + ss)):
          "informOutgoingBodyStreamTerminated - not as expected";
        senderEvents.removeFirst();
      }
    }
  }  
}
//---------------------------------------------------------------------------    
/* $Log: OutgoingBodyConnectionUnit.java,v $
/* Revision 1.10  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.9  2007/06/04 21:40:59  zjiandan
/* expose stream catchup type CP|LOG option to rmiClient.subscribeInval().
/*
/* Revision 1.8  2007/05/30 20:30:19  dahlin
/* Added checkpoint to SubscribeBWUnit. Changed outgoingconnection checkpoint send to not send body by default. Told barrier to close sockets when done with them.
/*
/* Revision 1.7  2007/05/11 02:14:29  nalini
/* updated outgoing body subscription so that it sends bodies from startVV
/*
/* Revision 1.6  2007/04/05 21:49:25  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.5  2007/04/02 22:09:56  nalini
/* junit fixes
/*
/* Revision 1.4  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.3  2006/11/16 21:08:09  nalini
/* Makefile
/*
/* Revision 1.2  2006/11/15 17:48:46  nalini
/* fixing unit tests
/*
/* Revision 1.1  2006/11/14 18:22:59  nalini
/* added OutgoingBodyConnectionUnit
/*
*/
//---------------------------------------------------------------------------
