package code;
/* OutgoingConnection
 *
 *   A dynamic outgoing connection for subscribing invalidation at the sender side.
 *   
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import java.net.Socket;
import java.util.LinkedList;
import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.FileOutputStream;

public class OutgoingConnection{
  protected final static boolean measureCPCatchupTime = false;
  protected final static boolean measureLogCatchupTime = false;
  protected final static boolean trackPerRequestTime = false;
  public final static boolean dbgHang = false;
  public final static boolean dbgProgress = false;
  public final static boolean dbgPerformance = false;
  public final static boolean printTime = false;
  public final static boolean dbgRepeatInvalStream = false;
  public final static boolean dbgii = false;
  protected static boolean dbgWithFileOutputStream = false;

  // See inner class OutgoingConnectionWorker for more debug params
  private LinkedList<SubscriptionRequest> pendingReq;
  private boolean timeToDie;//true if close() called
  private Thread worker;

  private Core core;
  private Controller controller;
  private NodeId receiverId;
  private String receiverDNS;
  private int portInval;

  protected final boolean autoRemoveIfEmptySubset = true;
  // by default automatically remove empty connection

  public final static boolean USE_HEARTBEATS = false;
  public final static boolean USE_BW_CONSTRAINTS = true;

 /** 
 *  Constructor 
 **/ 
  public OutgoingConnection(Core core, 
      Controller controller,
      NodeId receiverId,
      String receiverDNS,
      int portInval){
    if(dbgProgress){
      System.out.println("OutgoingConnection(rid= " + receiverId 
          + ") is created"); 
    }


    this.pendingReq = new LinkedList<SubscriptionRequest>();
    this.timeToDie = false;
    this.core = core;
    this.controller = controller;
    this.receiverId = receiverId;
    this.receiverDNS = receiverDNS;
    this.portInval = portInval;
    this.worker = null;

  }


 /** 
 *  Start new worker thread 
 **/ 
  private synchronized void startNewWorker(){
    assert(worker == null);
    assert !timeToDie;
    worker = new OutgoingConnectionWorker(core, controller, 
        receiverId, receiverDNS,
        portInval, this);
    worker.start();

  }


 /** 
 *  add add-subscription-set request to the pending request queue 
 *  
 *  catchup with CP if  
 *   (1) catchupWithCP = true or  
 *   (2) startVV has any component <= omitVV 
 *  
 *  note: if omitVV >= startVV, CP is the only way 
 **/ 
  public synchronized void addSubscriptionSet(SubscriptionSet newSubset, 
      AcceptVV newStartVV,
      boolean includeBodiesIfCPSent,
      boolean catchupWithCP){
    long start, end;
    if(trackPerRequestTime || measureCPCatchupTime || measureLogCatchupTime){
      start = System.currentTimeMillis();
    }
    if(trackPerRequestTime){
      LatencyWatcher.put(newSubset.toString(), start);
    }

    if(timeToDie){//this stream is going to stop
      //
      // Although we never managed to get this part of the
      // inval stream started, the RMI that requested it
      // returned thinking that we were working on it.
      // So, generate a controller event in case some
      // action is desired.
      //
      controller.informOutgoingInvalStreamTerminated(receiverId,
          newSubset,
          newStartVV,
          true);
      return;
    }


    if(worker == null){
      startNewWorker();
    }

    if(catchupWithCP || (useCP(newSubset, newStartVV))){
      pendingReq.add(new SubscriptionRequest(SubscriptionRequest.CP, newSubset, 
          newStartVV, includeBodiesIfCPSent));
      if(dbgProgress||dbgPerformance){
        Env.dprintln(dbgProgress||dbgPerformance,
            "OutgoingConnection.addSubscriptionSet:( " + newSubset + ", " 
            + newStartVV + ") is called using ---- CP ------@" 
            + System.currentTimeMillis()); 
      }
    }else{      
      pendingReq.add(new SubscriptionRequest(SubscriptionRequest.LOG, newSubset, 
          newStartVV, includeBodiesIfCPSent));  
      if(dbgProgress||dbgPerformance){
        Env.dprintln(dbgProgress||dbgPerformance, 
            "OutgoingConnection.addSubscriptionSet:( " + newSubset + ", " 
            + newStartVV + ") is called using ----LOG ------@"
            +System.currentTimeMillis()); 
      }
    }   


    if(measureCPCatchupTime || measureLogCatchupTime){
      end = System.currentTimeMillis();

      LatencyWatcher.put("OC::addrequestToQueue", (end-start));
    }
  }

 /** 
 *  Add a subscription request to pending work. Used by worker thread. 
 **/ 
  protected synchronized void addSubscriptionRequest(SubscriptionRequest sr){
    if(timeToDie){//this stream is going to stop
      //
      // Although we never managed to get this part of the
      // inval stream started, the RMI that requested it
      // returned thinking that we were working on it.
      // So, generate a controller event in case some
      // action is desired.
      //
      controller.informOutgoingInvalStreamTerminated(receiverId,
          sr.getSubscriptionSet(),
          sr.getStartVV(),
          true);
      return;
    }
    pendingReq.add(sr);
  }


 /** 
 *  Decide whether use Checkpoint to catch up the stream 
 *        
 *  simple policy: use Checkpoint only if there's something newer than startVV is omitted 
 *                 return true, if the omivv is larger than the startvv 
 *                 return false, otherwise 
 **/ 
  private boolean useCP(SubscriptionSet ss, VV startVV){
    //
    // we expose the CPPolicy decision to each addInval Request
    // and leave it in P2 layer
    //
    //if(!warnedCPPolicy){
    //  Env.tbd("Need more sophisticated policy to choose CP or LOG to have a new"
    //          + "subscriptionset to catch up with an existing stream");
    //  warnedCPPolicy = true;
    //}
    //boolean ret = false;
    //
    //  replaced by exposing CP|LOG option to each subscribeInval request
    //  instead of set by Core for all.
    //
    //if(setCP){
    //  ret = core.getCP();
    //}
    //if (ret){
    //  return ret;
    //}

    AcceptVV currentOmitVV = core.getInMemOmitVV();
    if (currentOmitVV.includesAnyPartOf(startVV)){//log has missing information
      return true;
    } else {
      return false;
    }
  }

 /** 
 *  how many requests pending? 
 **/ 
  protected synchronized long countPending(){
    return pendingReq.size();
  }


 /** 
 *  add remove-subscription-set request to the pending request queue 
 **/ 
  public synchronized void removeSubscriptionSet(SubscriptionSet newSubset){

    if(timeToDie){//this stream is going to stop
      //
      // Although we never managed to get this part of the
      // inval stream started, the RMI that requested it
      // returned thinking that we were working on it.
      // So, generate a controller event in case some
      // action is desired.
      //
      controller.informOutgoingInvalStreamTerminated(receiverId,
          newSubset,
          null,
          true);
      return;
    }

    if(dbgProgress){
      System.out.println("OutgoingConnection.removeSubscriptionSet:( " + newSubset
          + ") is called"); 
    }

    assert !timeToDie;
    if(worker != null){
      pendingReq.add(new SubscriptionRequest(SubscriptionRequest.REMOVE, newSubset, 
          null, false));
    }else{
      //no need to remove success anyways
      controller.informOutgoingSubscribeInvalTerminated(receiverId,
          newSubset);
    }

  }


 /** 
 *  Get next request (does not block -- return null if empty) 
 **/ 
  protected synchronized SubscriptionRequest getNextNoBlock(){
    SubscriptionRequest nextRequest;
    try{
      nextRequest = (SubscriptionRequest)(pendingReq.removeFirst());
      if( dbgProgress||dbgPerformance){
        Env.dprintln(dbgProgress||dbgPerformance, "OutgoingConnection.remove requset" 
            + nextRequest +" ------@" + System.currentTimeMillis()); 
      }
      return nextRequest;
    }
    catch(NoSuchElementException e){
      return null;
    }
  }

 /** 
 * set timeToDie so that the thread would stop at some point 
 **/ 
  public synchronized void close(){
    if(dbgProgress){
      System.out.println("OutgoingConnection.close() is called"); 
    }
    timeToDie = true;
  }


 /** 
 * get timeToDie 
 **/ 
  protected synchronized boolean getTimeToDie(){
    return timeToDie;
  }

 /** 
 *  Connection died -- kill worker and empty out the pending work queue. 
 *  Next new request to arrive will allocate a new worker (and socket). 
 *  
 *  Note -- we empty the pending work queue because the alternative would 
 *  be to create a new worker right away. But since we just closed 
 *  (due to an exception), that could leave us spinning in repeated 
 *  calls to this function. This also helps avoid an infinite memory 
 *  leak in the case of never being able to make a connection to a node. 
 **/ 
  protected synchronized void connectionDied(){
    try{
      while(true){
        SubscriptionRequest nextRequest;
        nextRequest = (SubscriptionRequest)(pendingReq.removeFirst());
        //
        // Although we never managed to get this part of the
        // inval stream started, the RMI that requested it
        // returned thinking that we were working on it.
        // So, generate a controller event in case some
        // action is desired.
        //
        controller.informOutgoingInvalStreamTerminated(receiverId,
            nextRequest.getSubscriptionSet(),
            nextRequest.getStartVV(),
            true);
      }
    }
    catch(NoSuchElementException e){
      assert(pendingReq.size() == 0);
    }
    //
    // Tell the enclosing pool to remove this connection 
    // and create a new one the next time we get a request.
    // We do this rather than just leave this connection
    // in "disconnected" state to avoid memory leak
    // of large number of inactive OutgoingConnection objects.
    //
    //deadlock with OutgoingConnection
    //FIXME
    //   leave the "disconnected OutgoingConnection" in OutgoingConnectionPool.
    //   The OutgoingConnectionPool can periodically gabbage-collect
    //   the dead OutgoingConnection by checking(timeToDie) if needed.
    //   
    //   therefore we commented the following code which causes the deadlock
    //   situation
    //
    //if(pool != null){
    //  pool.removeConnection(receiverId, receiverDNS, portInval);
    //}
    worker = null;
  }


 /** 
 *  check if the OutgoingConnection has active worker 
 *  called by OutgoingConnectionPool gc thread 
 **/ 
  public synchronized boolean hasNoWorker(){
    return worker == null;
  }

  public static void main(String[] argv){
    OutgoingConnectionWorker.main(argv);
  }

  public static void setDbgWithFileOutputStream(boolean b){
    dbgWithFileOutputStream = b;
  }
}















 /** 
 * The real meat of OutgoingConnection is here 
 **/ 

class OutgoingConnectionWorker extends Thread implements SocketMagicConstant{

  protected final static boolean trackPerRequestTime = OutgoingConnection.trackPerRequestTime;
  protected final static boolean measureLogCatchupTime = OutgoingConnection.measureLogCatchupTime;
  protected final static boolean measureCPCatchupTime = OutgoingConnection.measureCPCatchupTime;
  protected static long totalTime = 0;
  protected static int count = 0;


  private final static boolean dbg = false; // Open, close, setup
  private final static String dbgTag = "DBG OutgoingConnectionWorker:: ";
  private final static boolean dbgProgress = OutgoingConnection.dbgProgress;

  private final static boolean printStream = false; //print objects written to network

  //
  // MDD: the following is a design flaw that should 
  // be fixed. Can we refactor the state so that we 
  // wait() for the next event rather than periodically 
  // poll? See Programming With Threads standards
  // document -- periodic polling is forbidden!
  // (Why -- one example -- is it reallyOK to add 500ms
  // of latency to a subscribe() request?)
  //
  private static final long TIMEOUTFORNEWINVAL = 100; //needed to break the waiting for new
  //invalidates inserted in the log, InvalIterator.getNext() can't wait for ever for the
  //new invalidates to come because there might be some new requests to add some
  //new subscription sets into the connection during this time. 

  private static boolean tbds = false;


  private FileOutputStream fos = null;
  private Socket sock = null;


  private Core core;
  private Controller controller;
  private NodeId receiverId;
  private String receiverDNS;
  private int portInval;
  private InvalIterator mainInvalIterator;
  private OutgoingConnection workQueue;

  //file name to store output objects -- for unit tests and dbgs only
  //name format: OutgoingConnection_SenderId_receiverId.out"
  public final static String outFilePrefix = "OutgoingConnection_";

 /** 
 *  Constructor for worker thread 
 **/ 
  OutgoingConnectionWorker(Core core, 
      Controller controller,
      NodeId receiverId,
      String receiverDNS,
      int portInval, 
      OutgoingConnection workQueue){
    this.core = core;
    this.controller = controller;
    this.receiverId = receiverId;
    this.receiverDNS = receiverDNS;
    this.portInval = portInval;
    this.workQueue = workQueue;
    try{
      // can't make the initial ss empty(ii.getNext() will complain), 
      // use a dumb subscriptoinset "/never", which will never overlap any interest sets.


      this.mainInvalIterator = core.makeInvalIterator(SubscriptionSet.makeSubscriptionSet("/never"),
          core.getCurrentVV(),
          receiverId);
    }catch(OmittedVVException e){
      assert false;//impossible
    }

    Env.performanceWarning("OutgoingConnectionWorker:: will delay for " + TIMEOUTFORNEWINVAL +
        " ms when it calls mainInvalIterator.getNext() " + 
        "and there's no new invalidate to send." +
    " See MDD in OutgoingConnection.java");
  }



 /** 
 *  send the checkpoint + CP_END 
 **/ 
  private boolean sendCP(TaggedOutputStream tos, 
      SubscriptionSet subset, 
      AcceptVV startvv, 
      boolean withBody,
      AcceptVV prevVV)
  throws IOException{
    if(dbgProgress){
      System.out.println("OutgoingConnection.sendCP:( " + subset + ", " + startvv 
          + ",\n @prevVV = " + prevVV
          + " and cvv = " + core.getCurrentVV() + ") is called"); 
    }
    long start, end;
    if(measureCPCatchupTime){
      start = System.currentTimeMillis();
    }
    boolean ret = core.sendCheckpoint(tos, subset, startvv, withBody, prevVV);
    if(dbgProgress){
      System.out.println("OutgoingConnection.sendCP:( " + subset + ", " + startvv 
          + ") return " + ret); 
    }
    if(measureCPCatchupTime){
      end = System.currentTimeMillis();
      LatencyWatcher.put("Core.sendCP", (end-start));
    }
    return ret;
    //send a hint to the receiver that it's time to 
    //check if the CPsubset is illegible to attach to the connection
  }



 /** 
 *  send an invalidate stream generated by an CatchupInvalIterator 
 *  
 *  return false,  if UpdateLog::gcWorker gc any invalidate not sent yet 
 *         true, otherwise 
 **/ 
  private boolean sendInvals(TaggedOutputStream tos, CatchupInvalIterator ii)
  throws IOException{

    long start, end;

    if(dbgProgress){
      System.out.println("OutgoingConnection.sendInvals() -- for catch up stream is called"); 
    }
    if(measureLogCatchupTime){
      start = System.currentTimeMillis();
    }
    AcceptVV prevVV = mainInvalIterator.getCurrentVV();
    assert (ii != null) && (prevVV!=null) && (ii.getEndVV().equals(prevVV));
    assert core.getCurrentVV().includes(prevVV);
    assert prevVV.includes(ii.getCurrentVV());
    Object nextObj = null;

    while(!ii.getCurrentVV().includes(prevVV)){
      if(workQueue.getTimeToDie()){
        throw new IOException(" OutgoingConnection shutdown with possible incomplete catchupStream.");
      }
      try{
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendInvals-preGetNext", (end-start));
          start = System.currentTimeMillis();
        }

        nextObj = ii.getNext();
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendInvals-GetNext", (end-start));
          start = System.currentTimeMillis();
        }


      }catch(OmittedVVException oe){
        Env.sprinterrln("OutgoingConnection catch up stream is behind log garbage "
            + "collector, has to stop:" + oe);

        controller.informGapExistForSubscribeInv(receiverId, 
            ii.getSubscriptionSet(), 
            ii.getCurrentVV(), 
            core.getInMemOmitVV());
        // it's ok only that the subscriptoinset is failed but the main invalIterator
        // can still going on. And this catchup request can be satisfied by using CP
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendInvals-postGetNext1", (end-start));
          start = System.currentTimeMillis();
        }
        return false; 
      }


      if(nextObj instanceof GeneralInv){
        GeneralInv gi = (GeneralInv)nextObj;

        //Chop the invalidate with the catchup's endVV(i.e. this.prevVV) if necessary
        // otherwise, the invalidate might kickoff the existing connection.ss
        if (gi.getEndVV().isAnyPartGreaterThan(prevVV)){//a component has catch up  
          assert gi instanceof ImpreciseInv;
          nextObj = ImpreciseInv.chopImpreciseInv((ImpreciseInv)gi, prevVV);
        }
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("postGetNext2-1", (end-start));
          start = System.currentTimeMillis();
        }
        assert nextObj != null;
        tos.writeObject(nextObj);
        if(dbgProgress || printStream){
          long time = System.currentTimeMillis();
          System.out.println("OutgoingConnection-"+ receiverId + ".writeObject(gi) for catchup (" 
              + ii.getSubscriptionSet() + "): " 
              + nextObj + " at " + time); 
        }
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("postGetNext2-2", (end-start));
          start = System.currentTimeMillis();
        }
      }else{
        //the catchupInvalIterator will send a "null" object to end the stream
        //when its cvv catchs up with prevVV

        assert ((nextObj == null)&&(ii.getCurrentVV().includes(prevVV)));
        break;
      }

    }
    return true;
  } 

  private static boolean warnedHB = false;
  private static boolean printedTBD = false;
 /** 
 *  Make connection with remote node. 
 *  Side effect -- sets fos and sock, which are global 
 *  so that we can clean them up later. 
 **/ 
  private TaggedOutputStream makeConnection() throws IOException{
    TaggedOutputStream tos = null;

    if(dbgProgress){
      Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-entered");
    }
    if(OutgoingConnection.dbgWithFileOutputStream) {
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-else to " + 
            outFilePrefix + core.getMyNodeId() 
            + "_" + receiverId + ".out");
      }
      try{
        fos = new FileOutputStream(outFilePrefix + core.getMyNodeId() 
            + "_" + receiverId + ".out");
        tos = new TaggedOutputStream(fos);
      }catch(Exception e){
        e.printStackTrace();
        assert false;
        return null;
      }

    }else{
      OutputStream os = null;
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-else to " + 
            receiverDNS + ":" + portInval);
      }
      if(OutgoingConnection.USE_HEARTBEATS){
        sock = new HeartbeatSocket(receiverDNS, portInval);
        if(!printedTBD){
          Env.tbd(" cleanly shutdown HeartbeatOutputStream");
          printedTBD = true;
        }
      }else{
        if(!warnedHB){
          Env.warn("OutgoingConnection -- heartbeats turned off");
          warnedHB = true;
        }
        sock = new Socket(receiverDNS, portInval);
      }
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-sock created");
      }
      sock.setTcpNoDelay(true);
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-sock nodelay");
      }
      if(OutgoingConnection.USE_BW_CONSTRAINTS){
        os = new ConstrainedOutputStream(sock.getOutputStream(),
            Config.getBandwidth(core.getMyNodeId()));
      }else{
        Env.warn("OutgoingConnection -- BW constraints turned off");
        os = sock.getOutputStream();
      }
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-got os");
      }
      tos = new TaggedOutputStream(os);
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingConnection::makeConnection-got tos");
      }
    }
    return tos;
  }


 /** 
 *  Send the handshake to start the connection. 
 **/ 
  private void handshake(TaggedOutputStream tos) throws IOException{
    tos.writeObject(new Long(INVAL_SOCKET_MAGIC));
    if(dbgProgress || printStream){
      System.out.println("OutgoingConnection-" + receiverId + ".writeObject(INVAL_SOCKET_MAGIC): " 
          + new Long(INVAL_SOCKET_MAGIC)); 
    }
    tos.writeObject(core.getMyNodeId());
    if(dbgProgress || printStream){
      System.out.println("OutgoingConnection-" + receiverId + ".writeObject(myNodeId): " 
          + core.getMyNodeId()); 
    }


    tos.writeObject(mainInvalIterator.getCurrentVV());//synchronize prevVV with receiver
    if(dbgProgress || printStream){
      System.out.println("OutgoingConnection-" + receiverId + ".writeObject(initial prevVV): " 
          + mainInvalIterator.getCurrentVV()); 
    }
  }


 /** 
 *  For a LOG request, send the events from startVV needed to catch up 
 *  Return true if "gap" has been filled by sending a catchup stream 
 *  (or true if no gap existed so no catchup stream was needed.) 
 *  
 *  Return false (and add a CP request in our place) if we were unable to fill  
 *  the gap. 
 **/ 
  private boolean sendCatchupStream(SubscriptionRequest nextRequest, 
      TaggedOutputStream tos,
      AcceptVV prevVV) throws IOException{
    boolean sendCPInstead = false;
    boolean ret = true;
    long start, end;
    if(measureLogCatchupTime){
      start = System.currentTimeMillis();
    }
    //
    // notify receiver that catchup stream is starting
    //
    CatchupStreamStartMsg cssm = new CatchupStreamStartMsg(nextRequest.getSubscriptionSet(), 
        nextRequest.getStartVV());
    tos.writeObject(cssm);

    if(dbgProgress || printStream){
      long time = System.currentTimeMillis();
      System.out.println("OutgoingConnection-" + receiverId + ".writeObject(CatchupStreamStartMsg): " 
          + cssm + " done sending at " + time); 
    }


    if(measureLogCatchupTime){
      end = System.currentTimeMillis();
      LatencyWatcher.put("sendCatchupStream-sendstartMsg", (end-start));
      start = System.currentTimeMillis();
    }

    //
    // Prepare and send the catchup invalIterator (if needed)
    //
    if(!nextRequest.getStartVV().includes(prevVV)){
      if(OutgoingConnection.dbgProgress){
        Env.dprintln(OutgoingConnection.dbgProgress, 
            "going to catch up the stream for "+ nextRequest.getSubscriptionSet() 
            + "from "+ nextRequest.getStartVV() + "to" + mainInvalIterator.getCurrentVV());
      }
      CatchupInvalIterator nextRequestII = null;
      try{
        // 
        // make sure that the catch up stream never sends any invalidate that
        // is not coverred by prevVV, otherwise it might kick off all the existing
        // subscriptionSet of the incomming connection.
        // Be careful about the mismatch component in startVV and prevVV
        //
        CounterVV catchupStart = new CounterVV(nextRequest.getStartVV());
        catchupStart.setToMinVV(prevVV);
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream-preInval1", (end-start));
          start = System.currentTimeMillis();
        }

        nextRequestII = core.makeCatchupInvalIterator(nextRequest.getSubscriptionSet(), 
            catchupStart.cloneAcceptVV(),
            receiverId,
            prevVV);//used for catchup
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream-preInval2", (end-start));
          start = System.currentTimeMillis();
        }

        sendCPInstead = !sendInvals(tos, nextRequestII);
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream-sendInv", (end-start));
          start = System.currentTimeMillis();
        }
      } catch(OmittedVVException oe){//garbage collection thread won
        sendCPInstead = true;
      }finally{
        if(nextRequestII != null){
          this.core.removeInvalIterator(receiverId, nextRequestII);
          nextRequestII = null;
        }
      }

      if(sendCPInstead){//If we failed, then add a checkpoint send in our place
        SubscriptionRequest sr = new SubscriptionRequest(SubscriptionRequest.CP, 
            nextRequest.getSubscriptionSet(), 
            nextRequest.getStartVV(),
            nextRequest.getCPBodies());
        workQueue.addSubscriptionRequest(sr);
        ret = false;        
      }
      if(measureLogCatchupTime){
        end = System.currentTimeMillis();
        LatencyWatcher.put("sendCatchupStream-postSendInv", (end-start));
        start = System.currentTimeMillis();
      }
    }


    //
    // Whether we succeeded or failed, send CatchupStreamEndMsg
    // Notify the receiver that the catch up stream has been stopped
    //
    CatchupStreamEndMsg csem = new CatchupStreamEndMsg();
    tos.writeObject(csem);
    if(dbgProgress || printStream){
      long time = System.currentTimeMillis();
      System.out.println("OutgoingConnection-" + receiverId + ".writeObject(CatchupStreamEndMsg): " 
          + csem + " done sending at " + time); 
    }
    assert(!sendCPInstead) || (!ret);
    if(measureLogCatchupTime){
      end = System.currentTimeMillis();
      LatencyWatcher.put("sendCatchupStream-sendEndMsg", (end-start));
      start = System.currentTimeMillis();
    }
    return ret;
  }              


 /** 
 *  remove a request from pendingQue 
 *  and process the request by sending catchupstream && add the subscriptionset 
 *                          or removing the subscriptionset 
 *  
 *  Side effect - update pendingCP with deferred checkpoints catchup requests. 
 **/ 
  private void serveNewRequests(TaggedOutputStream tos,
      AcceptVV prevVV,
      Vector<SubscriptionRequest> pendingCP) 
  throws IOException{
    SubscriptionRequest nextRequest; 

    //
    // Check if there're new subscription set requests
    //

    for(nextRequest = workQueue.getNextNoBlock();
    nextRequest != null;
    nextRequest = workQueue.getNextNoBlock()){

      //
      // CASE 1: New request is for CP
      //
      if(nextRequest.getType() == SubscriptionRequest.CP){
        if(dbgProgress){
          Env.dprintln(dbgProgress, "OutgoingConnection processing CP request ");
        }
        if(core.getCurrentVV().equalsIgnoreNegatives( mainInvalIterator.getCurrentVV())){ 
          if(sendCP(tos, nextRequest.getSubscriptionSet(), 
              nextRequest.getStartVV(), nextRequest.getCPBodies(), prevVV)){
            if(trackPerRequestTime){ 
              LatencyWatcher.getDiffAndPut(nextRequest.getSubscriptionSet().toString(),
                  System.currentTimeMillis());
            }

            mainInvalIterator.addSubscriptionSet(nextRequest.getSubscriptionSet());
            if(OutgoingConnection.dbgRepeatInvalStream){
              Env.dprintln(OutgoingConnection.dbgRepeatInvalStream, 
                  "mainInvalIterator.addSub("+nextRequest.getSubscriptionSet().toString());
            }
            controller.informOutgoingSubscribeInvalInitiated(receiverId,
                nextRequest.getSubscriptionSet());
          }else{
            pendingCP.add(nextRequest);
          }
        }
        else {//not ready to send CP
          pendingCP.add(nextRequest);
        }
      }


      //
      // CASE 2: New request is for log
      //
      else if (nextRequest.getType() == SubscriptionRequest.LOG){
        if(dbgProgress){
          Env.dprintln(dbgProgress, "OutgoingConnection processing LOG request");
        }
        long start, end;
        if(measureLogCatchupTime){
          start = System.currentTimeMillis();

        }
        boolean success = sendCatchupStream(nextRequest, tos, prevVV);
        if(measureLogCatchupTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream", (end-start));
        }
        if(OutgoingConnection.dbgRepeatInvalStream){
          Env.dprintln(OutgoingConnection.dbgRepeatInvalStream, 
              "sendCatchupStream for " + nextRequest.getSubscriptionSet().toString()
              + prevVV.toString());
        }
        if(success){
          mainInvalIterator.addSubscriptionSet(nextRequest.getSubscriptionSet());
          if(OutgoingConnection.dbgRepeatInvalStream){
            Env.dprintln(OutgoingConnection.dbgRepeatInvalStream, 
                "2mainInvalIterator.addSub("+nextRequest.getSubscriptionSet().toString());
          }
          controller.informOutgoingSubscribeInvalInitiated(receiverId,
              nextRequest.getSubscriptionSet());
        }
      }

      //
      // CASE 3: New request to REMOVE
      //
      else{
        if(dbgProgress){
          Env.dprintln(dbgProgress, "OutgoingConnection processing remove request");
        }
        assert nextRequest.getType() == SubscriptionRequest.REMOVE;
        mainInvalIterator.removeSubscriptionSet(nextRequest.getSubscriptionSet());
        controller.informOutgoingSubscribeInvalTerminated(receiverId,
            nextRequest.getSubscriptionSet());
        assert workQueue != null;
        if(dbgProgress){
          Env.dprintln(dbgProgress, "OutgoingConnection inform should have been called");
        }
        if(workQueue.autoRemoveIfEmptySubset  && mainInvalIterator.getSubscriptionSet().isEmpty()){
          if(dbgProgress){  
            Env.dprintln(dbgProgress, dbgTag 
                + " run-close and clean up because workQueue has empty subscriptionset");
          }


          this.core.removeInvalIterator(receiverId, mainInvalIterator);
          controller.informOutgoingInvalStreamTerminated(receiverId,
              mainInvalIterator.getSubscriptionSet(), 
              mainInvalIterator.getCurrentVV(), 
              true);
          mainInvalIterator = null;
          workQueue.close();
          if(dbgProgress){
            Env.dprintln(dbgProgress, dbgTag + "workQueue close called because of empty subscriptionset");
          }
        }
      }
    }
  }


 /** 
 *   main fuction 
 *  send a stream constructing of: 
 *  [INVAL_SOCKET_MAGIC, senderId, intial prevVV,  
 *   {(CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)| 
 *   (GeneralInv|UnbindMsg|DebargoMsg)*| 
 *   (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)}* 
 *   TerminateMsg] 
 **/ 
  private static boolean printTBDs = false;
  public void run(){

    TaggedOutputStream tos = null;
    Vector<SubscriptionRequest> pendingCP = new Vector<SubscriptionRequest>(); //for those CP requests that are waiting for
    // con.prevVV == core.getCVV();

    SubscriptionRequest nextRequest = null;
    Object nextObj = null;
    if(!printTBDs){
      Env.tbd("OutgoingConnection can be optimized by adding a set of buffers for \n"+
          "multiple catchup stream threads and a main network sender thread.\n" +
          "Each catchup stream threads can parallelly generate the objects for\n" +
          "a catchupstream in its own buffer and the main network sender thread\n"  +
          "picks one stream buffer at a time and sends it all then pick another one...\n"+
          "And Give the preferrence to the ready and short ones." +
          "This strategies will quickly drain the pending add/remove requests, at least"+
      "the remove request can be done quickly and don't have to wait for any thread");
      printTBDs=true;
    }

    if(measureLogCatchupTime){
    }
    try{
      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-starts");
      }
      tos = makeConnection();
      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-connected");
      }
      handshake(tos);
      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-handshake");
      }
      controller.informOutgoingInvalStreamInitiated(receiverId,
          mainInvalIterator.getCurrentVV(), 
          true);


      if(!tbds){
        Env.tbd(" Defer the support of clustering across intersected invalidates, i.e. the Delay"
            + "parameter. The basic algorithm, is send the interested invalidates first"
            +" then send the accumulated( might across multiple interseted invalidates)"
            + "imprecise invalidate"
            + " the lpvv should be fine, need to make sure that the log would be fine");
        Env.tbd("Design issue in OutgoingConnectionWorker -- we need to unify the"
            + " work queues so that new events are handled immediately"
            + " rather than relying on invalIterator TIMEOUTORNEWINVAL");
        tbds = true;
      }      

      //
      // main flow: send any pending CPs which is waiting for prevVV = cvv
      //            check if there's new requests
      //               if yes, send CP or catch up invalidation stream or remove subset
      //            
      //            send new invalidate(the invalidateIterator's interestset might have changed)
      while(!workQueue.getTimeToDie()){
        //
        //send the CP which is waiting for con.prevVV to match with node.cvv
        //note: ii.cvv is con.prevVV
        //     tricky: if we don't freeze datastore at the moment when 
        //             con.prevVV == datastore.cvv, then it is possible that
        //             at the time checkpoint is actually sent, cvv is advanced while prevv
        //             don't. Therefore
        //             the checkpoint would kickoff any existing subset for the connection.
        //            fix: inside Datastore.sendCheckpoint, check it again, if it doesn't hold
        //                 anymore, don't send checkpint, instead return false.
        //
        AcceptVV prevVV = mainInvalIterator.getCurrentVV();
        if(OutgoingConnection.dbgHang){
          Env.dprintln(OutgoingConnection.dbgHang, "OutgoingConnection has " 
              + pendingCP.size() + " pending CP requests"
              + " @ mainInvalIterator.cvv=" + mainInvalIterator.getCurrentVV() 
              + " ===== SS added :" + mainInvalIterator.getSubscriptionSet());
        }
        if (core.getCurrentVV().equalsIgnoreNegatives(prevVV)){ 
          //
          // Send all pending CPs
          //
          for ( Enumeration<SubscriptionRequest> e = pendingCP.elements(); e.hasMoreElements();){
            nextRequest = (SubscriptionRequest)(e.nextElement());
            assert nextRequest.getType() == SubscriptionRequest.CP;
            if(sendCP(tos, nextRequest.getSubscriptionSet(), 
                nextRequest.getStartVV(), nextRequest.getCPBodies(), prevVV)){
              if(trackPerRequestTime){ 
                LatencyWatcher.getDiffAndPut(nextRequest.getSubscriptionSet().toString(),
                    System.currentTimeMillis());
              }
              pendingCP.remove(nextRequest);
              mainInvalIterator.addSubscriptionSet(nextRequest.getSubscriptionSet());
              controller.informOutgoingSubscribeInvalInitiated(receiverId,
                  nextRequest.getSubscriptionSet());
            }
          } 
        }    


        //
        // process all requests in the pendingQue
        // might move some requests from pendingQue to pendingCP queue
        //

        serveNewRequests(tos, prevVV, pendingCP);

        //
        //resume the main stream
        //
        try{
          if(OutgoingConnection.dbgii){
            Env.dprintln(OutgoingConnection.dbgii, "before mainInvalIterator.getNext: " 
                + "ii.ss=" + mainInvalIterator.getSubscriptionSet()
                + "ii.cvv=" + mainInvalIterator.getCurrentVV());
          }
          if(mainInvalIterator == null){
            assert workQueue.getTimeToDie();
            return;
          }
          nextObj = mainInvalIterator.getNext(TIMEOUTFORNEWINVAL);

          if(nextObj instanceof ImpreciseInv){
            nextObj = ((code.security.SecureCore)core).securityFilter.createImpreciseInv((ImpreciseInv)nextObj);
          }else if(nextObj instanceof SingleWriterImpreciseInv){
            nextObj = ((code.security.SecureCore)core).securityFilter.createImpreciseInv(
                ((SingleWriterImpreciseInv)nextObj).cloneImpreciseInv());
          }

          if(OutgoingConnection.dbgii){
            Env.dprintln(OutgoingConnection.dbgii, "mainInvalIterator.getNext: " 
                + "ii.ss=" + mainInvalIterator.getSubscriptionSet()
                + "ii.cvv=" + mainInvalIterator.getCurrentVV()
                + "nextObj= " + nextObj);
          }

        }catch(OmittedVVException oe){
          Env.sprinterrln("OutgoingConnection is behind log garbage "
              + "collector, has to stop:" + oe);

          controller.informGapExistForSubscribeInv(receiverId, 
              mainInvalIterator.getSubscriptionSet(), 
              mainInvalIterator.getCurrentVV(), 
              core.getInMemOmitVV());
          this.core.removeInvalIterator(receiverId, mainInvalIterator); 
          controller.informOutgoingInvalStreamTerminated(receiverId,
              mainInvalIterator.getSubscriptionSet(), 
              mainInvalIterator.getCurrentVV(), 
              true);
          mainInvalIterator = null;
          return; //terminate the SubscribeInvalWorker thread.
        }  

        if (nextObj != null){
          if(OutgoingConnection.printTime && (nextObj instanceof PreciseInv)){
            System.out.println("OutgoingConnection-" + receiverId + ".writeObject(gi) for normal: "  
                + ((PreciseInv)nextObj).getAcceptStamp()
                + " send @ " + System.currentTimeMillis());
          }
          tos.writeObject(nextObj);
          if(dbgProgress || printStream){
            System.out.println("OutgoingConnection-" + receiverId + ".writeObject(gi) for normal: "  + nextObj); 
          }
        }
      } // time to die = true
      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-terminate no work todo");
      }
    }
    catch(IOException e){
      if(!workQueue.getTimeToDie()){
        e.printStackTrace();
        Env.inform("OutgoingConnection unable to send to node " +
            receiverDNS + ":" + portInval + ". Exception: "+ e.toString());

        Env.dprintln(true, "OutgoingConnection unable to send to node " +
            receiverDNS + ":" + portInval + ". Exception: "+ e.toString());

      }else{
        e.printStackTrace();
        System.err.println(" Ignore any exceptions during shutdown: " + e);
      }
      return;
    }
    catch(Exception e2){ // Unexpected exception
      if(!workQueue.getTimeToDie()){
        System.out.println("Unexpected exception in OutgoignConnection " + e2.toString());
        e2.printStackTrace();
        Env.remoteAssert(false,
        "Something broken in OutgoingConnection");
      }else{
        e2.printStackTrace();
        System.err.println(" Ignore any exceptions during shutdown: " + e2.toString());
      }

      return;

    }
    finally{

      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-close and clean");
      }
      if(mainInvalIterator != null){
        this.core.removeInvalIterator(receiverId, mainInvalIterator);
        controller.informOutgoingInvalStreamTerminated(receiverId,
            mainInvalIterator.getSubscriptionSet(), 
            mainInvalIterator.getCurrentVV(), 
            true);
        mainInvalIterator = null;
      }

      closeAndCleanupStreamAndSocket(tos, sock, fos);

    }
    assert workQueue.getTimeToDie();
    return; 
  }

 /** 
 *  Cleanly close the socket and stream 
 **/ 
  private void closeAndCleanupStreamAndSocket(TaggedOutputStream tos,
      Socket sock,
      FileOutputStream fos){
    if(dbg){
      Env.inform(dbgTag 
          + "<<<close and cleanup>>>"                
          + " target: " + receiverId.toString()
          + " dns: " + receiverDNS
          + " port: " + portInval
          + ")");
    }




    if(dbg){
      Env.sprinterrln("Closing socket: OutgoingConnection");
    }
    try{
      if(tos != null){
        tos.writeObject(new TerminateMsg());
        if(dbgProgress || printStream){
          System.out.println("OutgoingConnection-" + receiverId + ".writeObject(terminagemsg)"); 
        }
        tos.close();
      }
    }catch(IOException e){
      // Do nothing when closing
    }
    try{
      if(sock != null){
        sock.close();
      }
      if(fos != null){
        fos.close();
      }
    }catch(IOException e){
      // Do nothing when closing
    }
    workQueue.connectionDied();
    return;
  }

 /** 
 *  main() -- self test code 
 **/ 
  public static void 
  main(String[] argv) {
    //moved to OutgoingConnectionUnit.java
  }

 /** 
 *  Unit Test helper functions 
 *   
 *  set the outputstream to a tmp file 
 **/ 

  public static void 
  setDbgWithFileOutputStream(boolean withFile){
    OutgoingConnection.dbgWithFileOutputStream = withFile;
  }
}

//---------------------------------------------------------------------------
/* $Log: OutgoingConnection.java,v $
/* Revision 1.50  2007/11/05 23:32:42  zjiandan
/* fix SubscribeBWunit.
/*
/* Revision 1.49  2007/10/24 05:59:59  zjiandan
/* Fix junittest.
/*
/* */
//---------------------------------------------------------------------------
